use std::collections::BTreeMap;

type MLmap = BTreeMap<String, LinkCache>;

#[cfg(feature = "async")]
pub mod asynchronous;
#[cfg(not(feature = "async"))]
pub mod synchronous;
pub mod win;

use crate::cross_platform::cmltypes::CMumbleLink;
use crate::mlp::{MumbleLink, WindowDimensions};
use std::time::Instant;

#[derive(Debug, Clone)]
pub struct LinkCache {
    pub link: MumbleLink,
    pub win_dim: WindowDimensions,
    pub last_updated: Instant,
    pub link_ptr: *const CMumbleLink,
}

impl LinkCache {
    pub fn new(key: &str) -> anyhow::Result<Self> {
        let link_ptr = CMumbleLink::new_ptr(key)?;

        let mut link = MumbleLink::default();
        let mut win_dim = WindowDimensions::default();
        if CMumbleLink::is_valid(link_ptr) {
        link.update(link_ptr)?;
        win_dim.update(link_ptr)?;
        }

        let last_updated = Instant::now();

        Ok(LinkCache {
            link_ptr,
            link,
            last_updated,
            win_dim,
        })
    }
    pub fn update_link(&mut self) -> anyhow::Result<()> {
        if self.last_updated.elapsed() > std::time::Duration::from_millis(10) {
            return Ok(())
        } else {
            self.last_updated = Instant::now();
        }
        if !CMumbleLink::is_valid(self.link_ptr) {
            panic!("updating from invalid pointer is not okay :(");
        }
         self.link.update(self.link_ptr)?;
        Ok(())
    }
}

impl Default for LinkCache {
    fn default() -> Self {
        LinkCache {
            last_updated: Instant::now(),
            link_ptr: std::ptr::null::<CMumbleLink>(),
            ..Default::default()
        }
    }
}

unsafe impl Send for LinkCache {}
