//putting all the winapi specific stuff here. so that i can lock it all behind a cfg attr at the mod declaration

use crate::cross_platform::cmltypes::{CMumbleContext, CMumbleLink, C_MUMBLE_LINK_SIZE};
use crate::mlp::WindowDimensions;

use winapi::{
    shared::{
        minwindef::{BOOL, LPARAM, LPDWORD},
        ntdef::NULL,
        windef::{HWND, LPRECT, RECT},
    },
    um::{
        errhandlingapi::GetLastError,
        handleapi::INVALID_HANDLE_VALUE,
        memoryapi::{MapViewOfFile, FILE_MAP_ALL_ACCESS},
        minwinbase::SECURITY_ATTRIBUTES,
        winbase::CreateFileMappingA,
        winnt::PAGE_READWRITE,
        winuser::{EnumWindows, GetWindowRect, GetWindowThreadProcessId},
    },
};

pub fn create_link_shared_mem(key: &str) -> anyhow::Result<*const CMumbleLink> {
    let handle;
    let key_cstr = std::ffi::CString::new(key)?;
    let key_cstr_ptr = key_cstr.as_ptr();
    unsafe {
        let file_handle = CreateFileMappingA(
            INVALID_HANDLE_VALUE,
            NULL as *mut SECURITY_ATTRIBUTES,
            PAGE_READWRITE,
            0,
            C_MUMBLE_LINK_SIZE as u32,
            key_cstr_ptr,
        );
        if file_handle == NULL {
            anyhow::bail!(
                "could not create file map handle, error code: {}",
                GetLastError()
            );
        }
        handle = MapViewOfFile(file_handle, FILE_MAP_ALL_ACCESS, 0, 0, C_MUMBLE_LINK_SIZE);
        if handle == NULL {
            anyhow::bail!("could not map view of file, error code: {}", GetLastError());
        }
    }
    Ok(handle as *const CMumbleLink)
}
#[cfg(target_os = "windows")]
pub fn get_win_pos_dim(link_ptr: *const CMumbleLink) -> anyhow::Result<WindowDimensions> {
    unsafe {
        if !CMumbleLink::is_valid(link_ptr) {
            anyhow::bail!("the MumbleLink is not init yet. so, getting window position is not valid operation");
        }
        let context = (*link_ptr).context.as_ptr() as *const CMumbleContext;
        let mut pid: isize = (*context).process_id as isize;

        let result: BOOL = EnumWindows(Some(get_handle_by_pid), &mut pid as *mut isize as LPARAM);
        if result != 0 {
            anyhow::bail!("couldn't find gw2 window. error code: {}", GetLastError());
        }

        let mut rect: RECT = RECT {
            left: 0,
            top: 0,
            right: 0,
            bottom: 0,
        };
        let status = GetWindowRect(pid as isize as HWND, &mut rect as LPRECT);
        if status == 0 {
            anyhow::bail!("could not get gw2 window size");
        }
        Ok(WindowDimensions {
            x: rect.left as u32,
            y: rect.top as u32,
            width: (rect.right - rect.left) as u32,
            height: (rect.bottom - rect.top) as u32,
        })
    }
}
#[cfg(target_os = "windows")]
unsafe extern "system" fn get_handle_by_pid(handle: HWND, lp: LPARAM) -> BOOL {
    let mut handle_pid: u32 = 0;
    GetWindowThreadProcessId(handle, &mut handle_pid as LPDWORD);
    if handle_pid == 0 {
        return 1;
    }

    if *(lp as *const u32) == handle_pid {
        *(lp as *mut isize) = handle as isize;
        return 0;
    }
    return 1;
}
