use super::{LinkCache, MLmap};
use crate::{
    cross_platform::cmltypes::USEFUL_C_MUMBLE_LINK_SIZE, mlp::mumble_link_response::MumbleStatus,
};
use log::trace;
use num_traits::FromPrimitive;

use crate::cross_platform::{cmltypes::CMumbleLink, MLRequestCode};
use std::net::SocketAddr;

/// binds a udp server at the given socket_addr.
/// waits for a request of 64 bytes atmost with the format
/// first byte is the request type, second byte holds the length of the key string, and third/fourth bytes are left alone for padding.
/// the rest of the [4..64] bytes are for the key string in utf-8.

/// based on the request, tries to prepare a response based on request. response for cml is 1197 and response for win_dim is 20.
/// the first byte is response status. a padding of 3 bytes if success. and the rest of the data will follow.

pub fn start_udp_server_multi_sync(socket_addr: SocketAddr) -> anyhow::Result<()> {
    let mut ml_map = MLmap::new();
    let socket = std::net::UdpSocket::bind(socket_addr)?;
    trace!("server bound to socket: {:#?}", socket_addr);
    loop {
        let mut recv_buffer = [0_u8; 64];
        let (_, sender) = socket.recv_from(&mut recv_buffer)?;
        trace!("request from {:#?}. request buffer is {:#?}", sender, &recv_buffer[..]);
        match MLRequestCode::from_u8(recv_buffer[0]) {
            
            Some(MLRequestCode::CML) => {
                let key = std::str::from_utf8(&recv_buffer[4..(recv_buffer[1] as usize + 4)]);
                dbg!(recv_buffer[1]);
                match key {
                    Ok(k) => {
                        trace!("key for cml request is: {}", k);
                        //if valid key, either insert or get an existing linkptr from the map
                        let link = ml_map
                            .entry(k.to_string())
                            .or_insert(LinkCache::new(k).unwrap());
                        // if mumble not init
                        if !CMumbleLink::is_valid(link.link_ptr) {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                            socket.send_to(&send_buffer, sender)?;
                            continue;
                        }
                        link.update_link()?;

                        //if mumble is init, send the data with the first byte for response status, 3 bytes for padding and 1193 bytes for the useful mumble link data
                        let mut send_buffer = [0_u8; USEFUL_C_MUMBLE_LINK_SIZE + 4];
                        CMumbleLink::copy_raw_bytes_into(link.link_ptr, &mut send_buffer[4..]);
                        send_buffer[0] = MumbleStatus::Success as u8;
                        socket.send_to(&send_buffer, sender).unwrap();
                    }
                    Err(e) => {
                        log::error!("failed to parse key: {}", e);
                        let mut send_buffer = [0_u8; 1];
                        send_buffer[0] = MumbleStatus::Failed as u8;
                        socket.send_to(&send_buffer, sender)?;
                    }
                }
            }
            Some(MLRequestCode::WD) => {
                let key = std::str::from_utf8(&recv_buffer[4..recv_buffer[1] as usize + 4]);
                match key {
                    Ok(k) => {
                        trace!("key for win_dim request is: {}", k);

                        let link = ml_map
                            .entry(k.to_string())
                            .or_insert(LinkCache::new(k).unwrap());
                        if !CMumbleLink::is_valid(link.link_ptr) {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                            socket.send_to(&send_buffer, sender)?;
                            continue;
                        }

                        let mut send_buffer = [0_u8; 20];
                        link.win_dim.update(link.link_ptr)?;
                        link.win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
                        send_buffer[0] = MumbleStatus::Success as u8;
                        socket.send_to(&send_buffer, sender).unwrap();
                    }
                    Err(e) => {
                        log::error!("failed to parse key: {}", e);

                        let mut send_buffer = [0_u8; 1];
                        send_buffer[0] = MumbleStatus::Failed as u8;
                        socket.send_to(&send_buffer, sender)?;
                    }
                }
            }
            _ => {
                log::error!("invalid mumble request enum. ");
                let mut send_buffer = [0_u8; 1];
                send_buffer[0] = MumbleStatus::Failed as u8;
                socket.send_to(&send_buffer, sender)?;
            }
        }
    }
}

// pub fn start_udp_server_single_sync(&self) -> anyhow::Result<()> {
//     let mut single: *const CMumbleLink = CMumbleLink::new_ptr("MumbleLink")?;
//     let socket = std::net::UdpSocket::bind(self.socket_addr)?;

//     loop {
//         let mut recv_buffer = [0_u8; 64];
//         let (_, sender) = socket.recv_from(&mut recv_buffer)?;
//         match MLRequestCode::from_u8(recv_buffer[0]) {
//             Some(MLRequestCode::CML) => {
//                 let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
//                 match key {
//                     Ok(k) => {
//                         if single.is_null() {
//                             single = CMumbleLink::new_ptr(k)?;
//                         }
//                         unsafe {
//                             if (*single).ui_tick == 0 {
//                                 let mut send_buffer = [0_u8; 1];
//                                 send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
//                                 socket.send_to(&send_buffer, sender)?;
//                                 continue;
//                             }
//                         }
//                         let mut send_buffer = [0_u8; 1197];
//                         CMumbleLink::copy_raw_bytes_into(single, &mut send_buffer[4..]);
//                         send_buffer[0] = MumbleStatus::Success as u8;
//                         socket.send_to(&send_buffer, sender).unwrap();
//                     }
//                     Err(_) => {
//                         let mut send_buffer = [0_u8; 1];
//                         send_buffer[0] = MumbleStatus::Failed as u8;
//                         socket.send_to(&send_buffer, sender)?;
//                     }
//                 }
//             }
//             Some(MLRequestCode::WD) => {
//                 let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
//                 match key {
//                     Ok(k) => {
//                         if single.is_null() {
//                             single = CMumbleLink::new_ptr(k)?;
//                         }
//                         unsafe {
//                             if (*single).ui_tick == 0 {
//                                 let mut send_buffer = [0_u8; 1];
//                                 send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
//                                 socket.send_to(&send_buffer, sender)?;
//                                 continue;
//                             }
//                         }
//                         let mut send_buffer = [0_u8; 20];
//                         let mut win_dim = WindowDimensions::default();
//                         win_dim.update(single)?;
//                         win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
//                         send_buffer[0] = MumbleStatus::Success as u8;
//                         socket.send_to(&send_buffer, sender).unwrap();
//                     }
//                     Err(_) => {
//                         let mut send_buffer = [0_u8; 1];
//                         send_buffer[0] = MumbleStatus::Failed as u8;
//                         socket.send_to(&send_buffer, sender)?;
//                     }
//                 }
//             }
//             _ => {
//                 let mut send_buffer = [0_u8; 1];
//                 send_buffer[0] = MumbleStatus::Failed as u8;
//                 socket.send_to(&send_buffer, sender)?;
//             }
//         }
//     }
// }
