use std::sync::Arc;

use super::MLmap;
use crate::cross_platform::cmltypes::CMumbleLink;
use crate::mlp::{
    mumble_link_response::MumbleStatus, mumble_service_server::MumbleService, Empty,
    MumbleLinkRequest, MumbleLinkResponse, RawLink, WindowDimensionsResponse,
};
use crate::server::LinkCache;
use parking_lot::Mutex;

#[derive(Debug)]
pub struct Mumble {
    pub mlinks: Arc<Mutex<MLmap>>,
    // pub mlink: Arc<Mutex<LinkCache>>,
}
#[tonic::async_trait]
impl MumbleService for Mumble {
    async fn get_mumble_link(
        &self,
        request: tonic::Request<MumbleLinkRequest>,
    ) -> Result<tonic::Response<MumbleLinkResponse>, tonic::Status> {
        let key = request.into_inner().mumble_link_name;
        let mut ml_map_guard = self.mlinks.lock();
        let link = ml_map_guard
            .entry(key.clone())
            .or_insert(LinkCache::new(&key).unwrap());
        if link.last_updated.elapsed() > std::time::Duration::from_millis(20) {
            link.link.update(link.link_ptr).unwrap();
        }
        let res = MumbleLinkResponse {
            status: MumbleStatus::Success as i32,
            mumble_link: Some(link.link.clone()),
        };
        Ok(tonic::Response::new(res))
    }

    async fn get_raw_link(
        &self,
        request: tonic::Request<MumbleLinkRequest>,
    ) -> Result<tonic::Response<RawLink>, tonic::Status> {
        let key = request.into_inner().mumble_link_name;
        let mut ml_map_guard = self.mlinks.lock();
        let link = ml_map_guard
            .entry(key.clone())
            .or_insert(LinkCache::new(&key).unwrap());
        let mut buffer = vec![0_u32; 299];
        CMumbleLink::copy_raw_bytes_into(link.link_ptr, bytemuck::cast_slice_mut(&mut buffer));
        let res = RawLink { raw_bytes: buffer };
        Ok(tonic::Response::new(res))
    }

    async fn get_window_dimensions(
        &self,
        request: tonic::Request<MumbleLinkRequest>,
    ) -> Result<tonic::Response<WindowDimensionsResponse>, tonic::Status> {
        let key = request.into_inner().mumble_link_name;
        let mut ml_map_guard = self.mlinks.lock();
        let link = ml_map_guard
            .entry(key.clone())
            .or_insert(LinkCache::new(&key).unwrap());
        if link.last_updated.elapsed() > std::time::Duration::from_millis(20) {
            link.win_dim.update(link.link_ptr).unwrap();
        }
        let res = WindowDimensionsResponse {
            status: MumbleStatus::Success as i32,
            window_dimensions: Some(link.win_dim),
        };
        Ok(tonic::Response::new(res))
    }

    async fn test_call(
        &self,
        request: tonic::Request<Empty>,
    ) -> Result<tonic::Response<Empty>, tonic::Status> {
        let _ = request;
        Ok(tonic::Response::new(Empty {}))
    }

    async fn test_echo(
        &self,
        request: tonic::Request<MumbleLinkRequest>,
    ) -> Result<tonic::Response<MumbleLinkRequest>, tonic::Status> {
        let key = request.into_inner().mumble_link_name;

        let res = MumbleLinkRequest {
            mumble_link_name: key,
        };
        Ok(tonic::Response::new(res))
    }
}
