use crate::cross_platform::{cmltypes::CMumbleLink, MLRequestCode};
use crate::mlp::{mumble_link_response::MumbleStatus, mumble_service_server::MumbleServiceServer};
use num_traits::FromPrimitive;
use parking_lot::Mutex;
use std::{
    net::{IpAddr, SocketAddr},
    sync::Arc,
};
use tonic::transport::Server;

use super::{LinkCache, MLmap};
use rpc::Mumble;

pub mod rpc;

pub async fn start_grpc_server_multi_async(ip: IpAddr, port: u16) -> anyhow::Result<()> {
    let mumble_service = MumbleServiceServer::new(Mumble {
        mlinks: Arc::new(Mutex::new(MLmap::new())),
    });

    let router = Server::builder().add_service(mumble_service);

    router.serve(SocketAddr::new(ip, port)).await?;

    Ok(())
}

pub async fn start_udp_server_multi_async(ip: IpAddr, port: u16) -> anyhow::Result<()> {
    let mut ml_map = MLmap::new();

    let socket = tokio::net::UdpSocket::bind(SocketAddr::new(ip, port)).await?;
    loop {
        let mut recv_buffer = [0_u8; 64];
        let (_, sender) = socket.recv_from(&mut recv_buffer).await?;
        match MLRequestCode::from_u8(recv_buffer[0]) {
            Some(MLRequestCode::CML) => {
                let key = std::str::from_utf8(&recv_buffer[4..recv_buffer[1] as usize]);
                match key {
                    Ok(k) => {
                        let link = ml_map
                            .entry(k.to_string())
                            .or_insert(LinkCache::new(k).unwrap());
                        if CMumbleLink::is_valid(link.link_ptr) {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                            socket.send_to(&send_buffer, sender).await?;
                            continue;
                        }

                        let mut send_buffer = [0_u8; 1197];
                        CMumbleLink::copy_raw_bytes_into(link.link_ptr, &mut send_buffer[4..]);
                        send_buffer[0] = MumbleStatus::Success as u8;
                        socket.send_to(&send_buffer, sender).await.unwrap();
                    }
                    Err(_) => {
                        let mut send_buffer = [0_u8; 1];
                        send_buffer[0] = MumbleStatus::Failed as u8;
                        socket.send_to(&send_buffer, sender).await?;
                    }
                }
            }
            Some(MLRequestCode::WD) => {
                let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                match key {
                    Ok(k) => {
                        let link = ml_map
                            .entry(k.to_string())
                            .or_insert(LinkCache::new(k).unwrap());
                        if CMumbleLink::is_valid(link.link_ptr) {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                            socket.send_to(&send_buffer, sender).await?;
                            continue;
                        }

                        let mut send_buffer = [0_u8; 20];
                        link.win_dim.update(link.link_ptr)?;
                        link.win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
                        send_buffer[0] = MumbleStatus::Success as u8;
                        socket.send_to(&send_buffer, sender).await.unwrap();
                    }
                    Err(_) => {
                        let mut send_buffer = [0_u8; 1];
                        send_buffer[0] = MumbleStatus::Failed as u8;
                        socket.send_to(&send_buffer, sender).await?;
                    }
                }
            }
            _ => {
                let mut send_buffer = [0_u8; 1];
                send_buffer[0] = MumbleStatus::Failed as u8;
                socket.send_to(&send_buffer, sender).await?;
            }
        }
    }
}

// pub async fn start_udp_server_single_async(&self) -> anyhow::Result<()> {
//     let mut single: *const CMumbleLink = std::ptr::null();
//     let socket = tokio::net::UdpSocket::bind(self.socket_addr).await?;
//     loop {
//         let mut recv_buffer = [0_u8; 64];
//         let (_, sender) = socket.recv_from(&mut recv_buffer).await?;
//         match MLRequestCode::from_u8(recv_buffer[0]) {
//             Some(MLRequestCode::CML) => {
//                 let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
//                 match key {
//                     Ok(k) => {
//                         if single.is_null() {
//                             single = CMumbleLink::new_ptr(k)?;
//                         }

//                         unsafe {
//                             if (*single).ui_tick == 0 {
//                                 let mut send_buffer = [0_u8; 1];
//                                 send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
//                                 socket.send_to(&send_buffer, sender).await?;
//                                 continue;
//                             }
//                         }
//                         let mut send_buffer = [0_u8; 1197];
//                         CMumbleLink::copy_raw_bytes_into(single, &mut send_buffer[4..]);
//                         send_buffer[0] = MumbleStatus::Success as u8;
//                         socket.send_to(&send_buffer, sender).await.unwrap();
//                     }
//                     Err(_) => {
//                         let mut send_buffer = [0_u8; 1];
//                         send_buffer[0] = MumbleStatus::Failed as u8;
//                         socket.send_to(&send_buffer, sender).await?;
//                     }
//                 }
//             }
//             Some(MLRequestCode::WD) => {
//                 let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
//                 match key {
//                     Ok(k) => {
//                         if single.is_null() {
//                             single = CMumbleLink::new_ptr(k)?;
//                         }
//                         unsafe {
//                             if (*single).ui_tick == 0 {
//                                 let mut send_buffer = [0_u8; 1];
//                                 send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
//                                 socket.send_to(&send_buffer, sender).await?;
//                                 continue;
//                             }
//                         }
//                         let mut send_buffer = [0_u8; 20];
//                         let mut win_dim = WindowDimensions::default();
//                         win_dim.update(single)?;
//                         win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
//                         send_buffer[0] = MumbleStatus::Success as u8;
//                         socket.send_to(&send_buffer, sender).await.unwrap();
//                     }
//                     Err(_) => {
//                         let mut send_buffer = [0_u8; 1];
//                         send_buffer[0] = MumbleStatus::Failed as u8;
//                         socket.send_to(&send_buffer, sender).await?;
//                     }
//                 }
//             }
//             _ => {
//                 let mut send_buffer = [0_u8; 1];
//                 send_buffer[0] = MumbleStatus::Failed as u8;
//                 socket.send_to(&send_buffer, sender).await?;
//             }
//         }
//     }
// }

pub enum ServerMode {
    Udp,
    Grpc,
}
