//! module with types to be repr(C) or used to interact with C.

use serde::{Deserialize, Serialize};

// pub const MUMBLE_LINK_SIZE: usize = 1193;
pub const C_MUMBLE_LINK_SIZE: usize = std::mem::size_of::<CMumbleLink>();
pub const USEFUL_C_MUMBLE_LINK_SIZE: usize = 1193;

#[derive(Debug, Clone, Copy)]
#[repr(C)]
pub struct CMumbleLink {
    pub ui_version: u32,
    pub ui_tick: u32,
    pub f_avatar_position: [f32; 3],
    pub f_avatar_front: [f32; 3],
    pub f_avatar_top: [f32; 3],
    pub name: [u8; 512],
    pub f_camera_position: [f32; 3],
    pub f_camera_front: [f32; 3],
    pub f_camera_top: [f32; 3],
    pub identity: [u16; 256],
    pub context_len: u32,
    pub context: [u8; 256],
    pub description: [u8; 4096],
}
#[derive(Debug, Default, Clone, Copy)]
#[repr(C)]
/// The mumble context as stored inside the context field of CMumbleLink.
/// the first 48 bytes Mumble uses for identification is upto build_id field
/// the rest of the fields after build_id are provided by gw2 for addon devs.
pub struct CMumbleContext {
    /// first byte is `2` if ipv4. and `[4..7]` bytes contain the ipv4 octets.
    pub server_address: [u8; 28], // contains sockaddr_in or sockaddr_in6
    pub map_id: u32,
    pub map_type: u32,
    pub shard_id: u32,
    pub instance: u32,
    pub build_id: u32,
    // Additional data that gw2 provides us
    pub ui_state: u32, // Bitmask: Bit 1 = IsMapOpen, Bit 2 = IsCompassTopRight, Bit 3 = DoesCompassHaveRotationEnabled, Bit 4 = Game has focus, Bit 5 = Is in Competitive game mode, Bit 6 = Textbox has focus, Bit 7 = Is in Combat
    pub compass_width: u16, // pixels
    pub compass_height: u16, // pixels
    pub compass_rotation: f32, // radians
    pub player_x: f32, // continentCoords
    pub player_y: f32, // continentCoords
    pub map_center_x: f32, // continentCoords
    pub map_center_y: f32, // continentCoords
    pub map_scale: f32,
    pub process_id: u32,
    pub mount_index: u8,
}

#[derive(Debug, Default, Clone, Serialize, Deserialize)]
/// The json structure of the Identity field inside Cmumblelink.
/// the json string is null terminated and utf-16 encoded. so, need to use
/// Widestring crate's U16Cstring to first parse the bytes and then, convert to
/// String before deserializing to CIdentity
pub struct CIdentity {
    pub name: String,
    pub profession: u32,
    pub spec: u32,
    pub race: u32,
    pub map_id: u32,
    pub world_id: u32,
    pub team_color_id: u32,
    pub commander: bool,
    pub fov: f32,
    pub uisz: u32,
}

impl CMumbleLink {
    pub fn get_cmumble_link(link_ptr: *const CMumbleLink) -> CMumbleLink {
        unsafe { std::ptr::read_volatile(link_ptr) }
    }

    pub fn is_valid(link_ptr: *const CMumbleLink) -> bool {
        unsafe { (*link_ptr).ui_tick > 0 }
    }

    pub fn get_ui_tick(link_ptr: *const CMumbleLink) -> u32 {
        unsafe { (*link_ptr).ui_tick }
    }
    #[cfg(target_os = "windows")]
    pub fn new_ptr(key: &str) -> anyhow::Result<*const CMumbleLink> {
        crate::server::win::create_link_shared_mem(key)
    }
    pub fn copy_raw_bytes_into(link_ptr: *const CMumbleLink, buffer: &mut [u8]) {
        unsafe {
            std::ptr::copy_nonoverlapping(link_ptr as *const u8, buffer.as_mut_ptr(), buffer.len());
        }
    }
}

// mod private {

// #[derive(Debug, Default, Clone, Serialize, Deserialize)]
// #[repr(C)]
// pub struct MumbleLink {
//     //pub ui_version: u32,
//     pub ui_tick: u32,
//     pub f_avatar_position: [f32; 3],
//     pub f_avatar_front: [f32; 3],
//     //pub f_avatar_top: [f32; 3],
//     //pub name: [u16; 256],
//     pub f_camera_position: [f32; 3],
//     pub f_camera_front: [f32; 3],
//     //pub f_camera_top: [f32; 3],
//     pub identity: Identity,
//     //pub context_len: u32,
//     pub context: MumbleContext,
// }
// }

// #[cfg(test)]
// mod tests {
//     #[test]
//     fn validate_identity_json_str() {
//         let id = String::from(
//             r#"{
//             "name": "Irwene",
//             "profession": 4,
//             "spec": 55,
//             "race": 4,
//             "map_id": 50,
//             "world_id": 268435505,
//             "team_color_id": 0,
//             "commander": false,
//             "fov": 0.873,
//             "uisz": 1
//           }"#,
//         );
//         use super::Identity;
//         let identity: Identity = serde_json::from_str(&id).unwrap();
//         assert_eq!(identity.name, "Irwene");

//     }

//     #[test]
//     fn validate_identity_json_binary_utf16() {
//         let pure_utf16: Vec<u16> = vec![0x007b , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x006e , 0x0061 , 0x006d , 0x0065 , 0x0022 , 0x003a , 0x0020 , 0x0022 , 0x0049 , 0x0072 , 0x0077 , 0x0065 , 0x006e , 0x0065 , 0x0022 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0070 , 0x0072 , 0x006f , 0x0066 , 0x0065 , 0x0073 , 0x0073 , 0x0069 , 0x006f , 0x006e , 0x0022 , 0x003a , 0x0020 , 0x0034 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0073 , 0x0070 , 0x0065 , 0x0063 , 0x0022 , 0x003a , 0x0020 , 0x0035 , 0x0035 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0072 , 0x0061 , 0x0063 , 0x0065 , 0x0022 , 0x003a , 0x0020 , 0x0034 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x006d , 0x0061 , 0x0070 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0035 , 0x0030 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0077 , 0x006f , 0x0072 , 0x006c , 0x0064 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0032 , 0x0036 , 0x0038 , 0x0034 , 0x0033 , 0x0035 , 0x0035 , 0x0030 , 0x0035 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0074 , 0x0065 , 0x0061 , 0x006d , 0x005f , 0x0063 , 0x006f , 0x006c , 0x006f , 0x0072 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0030 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0063 , 0x006f , 0x006d , 0x006d , 0x0061 , 0x006e , 0x0064 , 0x0065 , 0x0072 , 0x0022 , 0x003a , 0x0020 , 0x0066 , 0x0061 , 0x006c , 0x0073 , 0x0065 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0066 , 0x006f , 0x0076 , 0x0022 , 0x003a , 0x0020 , 0x0030 , 0x002e , 0x0038 , 0x0037 , 0x0033 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0075 , 0x0069 , 0x0073 , 0x007a , 0x0022 , 0x003a , 0x0020 , 0x0031 , 0x000a , 0x007d];
//         let id = String::from_utf16_lossy(&pure_utf16);
//         println!("the string content in id is {}", id);
//         use super::Identity;
//         let identity: Identity = serde_json::from_str(&id).unwrap();
//         assert_eq!(identity.name, "meow");

//     }
// }
