use anyhow::Context;

use super::cmltypes::{CMumbleContext, CMumbleLink};
use crate::mlp::{MumbleContext, MumbleIdentity, MumbleLink, WindowDimensions};

impl MumbleLink {
    /// The most used function probably. will check if the `link_ptr->ui_tick > &self.ui_tick`
    /// and update self's fields based on that.
    pub fn update(&mut self, link_ptr: *const CMumbleLink) -> anyhow::Result<()> {
        let cmlink = match unsafe { link_ptr.as_ref() } {
            Some(thing) => thing,
            None => panic!("link_ptr.as_ref returned None when trying to update MumbleLink. ptr is null. something is very wrong"),
        };

        if self.ui_tick == cmlink.ui_tick {
            return Ok(());
        }

        self.ui_tick = cmlink.ui_tick;

        self.f_avatar_position_x = cmlink.f_avatar_position[0];
        self.f_avatar_position_y = cmlink.f_avatar_position[1];
        self.f_avatar_position_z = cmlink.f_avatar_position[2];

        self.f_avatar_front_x = cmlink.f_avatar_front[0];
        self.f_avatar_front_y = cmlink.f_avatar_front[1];
        self.f_avatar_front_z = cmlink.f_avatar_front[2];

        self.f_camera_position_x = cmlink.f_camera_position[0];
        self.f_camera_position_y = cmlink.f_camera_position[1];
        self.f_camera_position_z = cmlink.f_camera_position[2];

        self.f_camera_front_x = cmlink.f_camera_front[0];
        self.f_camera_front_y = cmlink.f_camera_front[1];
        self.f_camera_front_z = cmlink.f_camera_front[2];
        if self.identity.is_none() {
            self.identity = Some(MumbleIdentity::default());
        }
        if self.context.is_none() {
            self.context = Some(MumbleContext::default());
        }
        self.identity.as_mut().unwrap().update(link_ptr)?;
        self.context.as_mut().unwrap().update(link_ptr);
        Ok(())
    }
}

impl MumbleIdentity {
    pub fn update(&mut self, link_ptr: *const CMumbleLink) -> anyhow::Result<()> {
        use widestring::U16CStr;
        let id = U16CStr::from_slice_with_nul(unsafe { &(*link_ptr).identity })
            .context("missing null terminator in Identity &[u8] to construct u16cstr from")?;
        let id = id.to_string().context("utf16 to utf-8 encoding failed")?;
        *self = serde_json::from_str::<MumbleIdentity>(&id)
            .context("failed to read identity from CMumbleLink.Identity")?;
        Ok(())
    }
}

impl MumbleContext {
    pub fn update(&mut self, link_ptr: *const CMumbleLink) {
        let mc = unsafe {
            std::ptr::read_volatile(&(*link_ptr).context as *const u8 as *const CMumbleContext)
        };
        self.server_address = u32::from_ne_bytes([
            mc.server_address[4],
            mc.server_address[5],
            mc.server_address[6],
            mc.server_address[7],
        ]);
        self.map_id = mc.map_id;
        self.map_type = mc.map_type;
        self.shard_id = mc.shard_id;
        self.instance = mc.instance;
        self.build_id = mc.build_id;
        self.ui_state = mc.ui_state;
        self.compass_width = mc.compass_width as u32;
        self.compass_height = mc.compass_height as u32;
        self.compass_rotation = mc.compass_rotation;
        self.player_x = mc.player_x;
        self.player_y = mc.player_y;
        self.map_center_x = mc.map_center_x;
        self.map_center_y = mc.map_center_y;
        self.map_scale = mc.map_scale;
        self.process_id = mc.process_id;
        self.mount_index = mc.mount_index as u32;
    }
}

impl WindowDimensions {
    #[cfg(target_os="windows")]
    pub fn update(&mut self, link_ptr: *const CMumbleLink) -> anyhow::Result<()> {

        *self = crate::engine::win::get_win_pos_dim(link_ptr)?;
        Ok(())
    }

    /// buffer must be atleast 16 bytes long
    pub fn copy_raw_bytes_into(&self, buffer: &mut [u8]) {
        if buffer.len() < 16 {
            panic!("buffer length less than 16 bytes.");
        }
        buffer.copy_from_slice(bytemuck::bytes_of(self));
    }
}
unsafe impl bytemuck::Zeroable for WindowDimensions {
    fn zeroed() -> Self {
        unsafe { core::mem::zeroed() }
    }
}
unsafe impl bytemuck::Pod for WindowDimensions {}

