use crate::mlp::{
    mumble_link_response::MumbleStatus, mumble_service_server::MumbleServiceServer,
    WindowDimensions,
};

use link_cache::LinkCache;
use num_traits::FromPrimitive;
use parking_lot::Mutex;

use rpc::Mumble;
use std::{
    collections::BTreeMap,
    net::{IpAddr, SocketAddr},
    sync::Arc,
};
use tonic::transport::Server;

use super::cross_platform::{cmltypes::CMumbleLink, MLRequestCode};

type MLmap = BTreeMap<String, LinkCache>;

pub mod link_cache;

pub mod rpc;

pub mod win;

#[derive(Debug)]
pub struct Engine {
    pub mlinks: Arc<Mutex<MLmap>>,
    pub socket_addr: SocketAddr,
}

impl Engine {
    pub fn create(ip: IpAddr, port: u16) -> anyhow::Result<Self> {
        let socket_addr = SocketAddr::new(ip, port);
        let mlinks = Arc::new(Mutex::new(MLmap::new()));

        Ok(Engine {
            mlinks,
            socket_addr,
        })
    }
    pub fn start_server(&self, mode: ServerMode) -> anyhow::Result<()> {
        match mode {
            ServerMode::UdpSync => self.start_udp_server_single_sync(),
            ServerMode::UdpAsync => {
                let rt = tokio::runtime::Runtime::new()?;
                rt.block_on(async {
                    self.start_udp_server_single_async().await?;
                    Ok(())
                })
            },
            ServerMode::GrpcAsync => {let rt = tokio::runtime::Runtime::new()?;
            rt.block_on(async {
                self.start_grpc_server_multi_async().await?;
                Ok(())
            })},
        }
    }
    pub async fn start_grpc_server_multi_async(&self) -> anyhow::Result<()> {
        let mservice = MumbleServiceServer::new(Mumble {
            mlinks: self.mlinks.clone(),
            mlink: Arc::new(Mutex::new(LinkCache::default())),
        });
        let router = Server::builder().add_service(mservice);

        router.serve(self.socket_addr).await?;

        Ok(())
    }

    pub async fn start_udp_server_single_async(&self) -> anyhow::Result<()> {
        let mut single: *const CMumbleLink = std::ptr::null();
        let socket = tokio::net::UdpSocket::bind(self.socket_addr).await?;
        loop {
            let mut recv_buffer = [0_u8; 64];
            let (_, sender) = socket.recv_from(&mut recv_buffer).await?;
            match MLRequestCode::from_u8(recv_buffer[0]) {
                Some(MLRequestCode::CML) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            if single.is_null() {
                                single = CMumbleLink::new_ptr(k)?;
                            }

                            unsafe {
                                if (*single).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender).await?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 1197];
                            CMumbleLink::copy_raw_bytes_into(single, &mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).await.unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender).await?;
                        }
                    }
                }
                Some(MLRequestCode::WD) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            if single.is_null() {
                                single = CMumbleLink::new_ptr(k)?;
                            }
                            unsafe {
                                if (*single).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender).await?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 20];
                            let mut win_dim = WindowDimensions::default();
                            win_dim.update(single)?;
                            win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).await.unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender).await?;
                        }
                    }
                }
                _ => {
                    let mut send_buffer = [0_u8; 1];
                    send_buffer[0] = MumbleStatus::Failed as u8;
                    socket.send_to(&send_buffer, sender).await?;
                }
            }
        }
    }

    pub fn start_udp_server_single_sync(&self) -> anyhow::Result<()> {
        let mut single: *const CMumbleLink = CMumbleLink::new_ptr("MumbleLink")?;
        let socket = std::net::UdpSocket::bind(self.socket_addr)?;

        
        loop {
            let mut recv_buffer = [0_u8; 64];
            let (_, sender) = socket.recv_from(&mut recv_buffer)?;
            match MLRequestCode::from_u8(recv_buffer[0]) {
                Some(MLRequestCode::CML) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            if single.is_null() {
                                single = CMumbleLink::new_ptr(k)?;
                            }
                            unsafe {
                                if (*single).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender)?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 1197];
                            CMumbleLink::copy_raw_bytes_into(single, &mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender)?;
                        }
                    }
                }
                Some(MLRequestCode::WD) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            if single.is_null() {
                                single = CMumbleLink::new_ptr(k)?;
                            }
                            unsafe {
                                if (*single).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender)?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 20];
                            let mut win_dim = WindowDimensions::default();
                            win_dim.update(single)?;
                            win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender)?;
                        }
                    }
                }
                _ => {
                    let mut send_buffer = [0_u8; 1];
                    send_buffer[0] = MumbleStatus::Failed as u8;
                    socket.send_to(&send_buffer, sender)?;
                }
            }
        }
    }

    pub fn start_udp_server_multi_sync(&self) -> anyhow::Result<()> {
        let mut ml_map = MLmap::new();

        let socket = std::net::UdpSocket::bind(self.socket_addr)?;
        loop {
            let mut recv_buffer = [0_u8; 64];
            let (_, sender) = socket.recv_from(&mut recv_buffer)?;
            match MLRequestCode::from_u8(recv_buffer[0]) {
                Some(MLRequestCode::CML) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            let link = ml_map
                                .entry(k.to_string())
                                .or_insert(LinkCache::new(k).unwrap());
                            unsafe {
                                if (*link.link_ptr).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender)?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 1197];
                            CMumbleLink::copy_raw_bytes_into(link.link_ptr, &mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender)?;
                        }
                    }
                }
                Some(MLRequestCode::WD) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            let link = ml_map
                                .entry(k.to_string())
                                .or_insert(LinkCache::new(k).unwrap());
                            unsafe {
                                if (*link.link_ptr).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender)?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 20];
                            link.win_dim.update(link.link_ptr)?;
                            link.win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender)?;
                        }
                    }
                }
                _ => {
                    let mut send_buffer = [0_u8; 1];
                    send_buffer[0] = MumbleStatus::Failed as u8;
                    socket.send_to(&send_buffer, sender)?;
                }
            }
        }
    }

    pub async fn start_udp_server_multi_async(&self) -> anyhow::Result<()> {
        let mut ml_map = MLmap::new();

        let socket = tokio::net::UdpSocket::bind(self.socket_addr).await?;
        loop {
            let mut recv_buffer = [0_u8; 64];
            let (_, sender) = socket.recv_from(&mut recv_buffer).await?;
            match MLRequestCode::from_u8(recv_buffer[0]) {
                Some(MLRequestCode::CML) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            let link = ml_map
                                .entry(k.to_string())
                                .or_insert(LinkCache::new(k).unwrap());
                            unsafe {
                                if (*link.link_ptr).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender).await?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 1197];
                            CMumbleLink::copy_raw_bytes_into(link.link_ptr, &mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).await.unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender).await?;
                        }
                    }
                }
                Some(MLRequestCode::WD) => {
                    let key = std::str::from_utf8(&recv_buffer[2..recv_buffer[1] as usize]);
                    match key {
                        Ok(k) => {
                            let link = ml_map
                                .entry(k.to_string())
                                .or_insert(LinkCache::new(k).unwrap());
                            unsafe {
                                if (*link.link_ptr).ui_tick == 0 {
                                    let mut send_buffer = [0_u8; 1];
                                    send_buffer[0] = MumbleStatus::MumbleNotInitialized as u8;
                                    socket.send_to(&send_buffer, sender).await?;
                                    continue;
                                }
                            }
                            let mut send_buffer = [0_u8; 20];
                            link.win_dim.update(link.link_ptr)?;
                            link.win_dim.copy_raw_bytes_into(&mut send_buffer[4..]);
                            send_buffer[0] = MumbleStatus::Success as u8;
                            socket.send_to(&send_buffer, sender).await.unwrap();
                        }
                        Err(_) => {
                            let mut send_buffer = [0_u8; 1];
                            send_buffer[0] = MumbleStatus::Failed as u8;
                            socket.send_to(&send_buffer, sender).await?;
                        }
                    }
                }
                _ => {
                    let mut send_buffer = [0_u8; 1];
                    send_buffer[0] = MumbleStatus::Failed as u8;
                    socket.send_to(&send_buffer, sender).await?;
                }
            }
        }
    }
}
pub enum ServerMode {
    UdpSync,
    UdpAsync,
    GrpcAsync,
    
}