use std::{
    collections::BTreeMap,
    net::{IpAddr, SocketAddr},
    sync::Arc,
};

use parking_lot::Mutex;
use tokio::time::Instant;
use tonic::transport::Server;

use crate::mlp::{
    mumble_link_response::MumbleStatus,
    mumble_service_server::{MumbleService, MumbleServiceServer},
    MumbleLink, MumbleLinkRequest, MumbleLinkResponse, RawLink, WindowDimensions,
};
use mlink::CMumbleLink;

type MLmap = BTreeMap<String, LinkCache>;

pub mod mlink;
#[cfg(target_os = "windows")]
pub mod win;

#[derive(Debug, Default)]
pub struct Mumble {
    pub mlinks: Arc<Mutex<MLmap>>,
}

#[tonic::async_trait]
impl MumbleService for Mumble {
    async fn get_mumble_link(
        &self,
        request: tonic::Request<MumbleLinkRequest>,
    ) -> Result<tonic::Response<MumbleLinkResponse>, tonic::Status> {
        let key = request.into_inner().mumble_link_name;
        let mut ml_map_guard = self.mlinks.lock();
        let link = ml_map_guard
            .entry(key.clone())
            .or_insert(LinkCache::new(&key).unwrap());
        if link.last_updated.elapsed() > std::time::Duration::from_millis(20) {
            link.link.update(link.link_ptr).unwrap();
        }
        let res = MumbleLinkResponse {
            status: MumbleStatus::Success as i32,
            mumble_link: Some(link.link.clone()),
        };
        Ok(tonic::Response::new(res))
    }

    async fn get_raw_link(
        &self,
        request: tonic::Request<crate::mlp::Empty>,
    ) -> Result<tonic::Response<crate::mlp::RawLink>, tonic::Status> {
        let mut ml_map_guard = self.mlinks.lock();
        let link = ml_map_guard
            .entry("MumbleLink".to_string())
            .or_insert(LinkCache::new("MumbleLink").unwrap());
        let b = unsafe {
            std::ptr::slice_from_raw_parts(link.link_ptr as *const u32, 512)
                .as_ref()
                .unwrap()
                .to_vec()
        };
        let res = RawLink { raw_bytes: b };
        Ok(tonic::Response::new(res))
    }
}

#[derive(Debug)]
pub struct Engine {
    pub mservice: MumbleServiceServer<Mumble>,
    pub mlinks: Arc<Mutex<MLmap>>,
    pub socket_addr: SocketAddr,
}

#[derive(Debug, Clone)]
pub struct LinkCache {
    pub link: MumbleLink,
    pub win_dim: WindowDimensions,
    pub last_updated: Instant,
    pub link_ptr: *const CMumbleLink,
}
impl LinkCache {
    #[allow(unused_assignments)]
    pub fn new(key: &str) -> anyhow::Result<Self> {
        let mut link_ptr = std::ptr::null();
        let mut link = MumbleLink::default();
        let last_updated = Instant::now();
        let mut win_dim = WindowDimensions::default();
        // filling vector to be size of 3 elements, so that it can be updated from MumbleLink::upadte();
        let temp = [0.0_f32; 3];
        link.f_avatar_position.extend_from_slice(&temp);
        link.f_avatar_front.extend_from_slice(&temp);
        link.f_camera_front.extend_from_slice(&temp);
        link.f_camera_position.extend_from_slice(&temp);

        #[cfg(target_os = "windows")]
        {
            link_ptr = win::create_link_shared_mem(key)?;
            link.update(link_ptr)?;
            win_dim = win::get_win_pos_dim(link_ptr)?;
        }

        Ok(LinkCache {
            link_ptr,
            link,
            last_updated,
            win_dim,
        })
    }
}
unsafe impl Send for LinkCache {}

impl Engine {
    pub fn create(ip: IpAddr, port: u16) -> anyhow::Result<Self> {
        let socket_addr = SocketAddr::new(ip, port);
        let mlinks = Arc::new(Mutex::new(MLmap::new()));
        let mservice = MumbleServiceServer::new(Mumble {
            mlinks: mlinks.clone(),
        });
        Ok(Engine {
            mservice,
            mlinks,
            socket_addr,
        })
    }

    pub async fn start_server(&self) -> anyhow::Result<()> {
        Server::builder()
            .add_service(self.mservice.clone())
            .serve(self.socket_addr)
            .await?;
        Ok(())
    }
    pub async fn start_udp_server(&self) -> anyhow::Result<()> {
        let socket = tokio::net::UdpSocket::bind(self.socket_addr).await.unwrap();
        let p = win::create_link_shared_mem("MumbleLink").unwrap();

        loop {
            let mut recv_buffer = [0_u8; 64];
            let (_, sender) = socket.recv_from(&mut recv_buffer).await.unwrap();
            let mut send_buffer = [0_u8; 1197];
            send_buffer[0] = 5;
            unsafe {
                std::ptr::copy_nonoverlapping(
                    p as *const u8,
                    send_buffer[4..].as_mut_ptr(),
                    send_buffer.len() - 4,
                )
            }
            socket.send_to(&send_buffer, sender).await.unwrap();
        }
    }
}
