use crate::mlp::{MumbleContext, MumbleIdentity, MumbleLink};
use anyhow::Context;
use serde::{Deserialize, Serialize};

// pub const MUMBLE_LINK_SIZE: usize = 1193;
pub const C_MUMBLE_LINK_SIZE: usize = std::mem::size_of::<CMumbleLink>();

#[derive(Debug, Clone, Copy)]
#[repr(C)]
pub struct CMumbleLink {
    pub ui_version: u32,
    pub ui_tick: u32,
    pub f_avatar_position: [f32; 3],
    pub f_avatar_front: [f32; 3],
    pub f_avatar_top: [f32; 3],
    pub name: [u8; 512],
    pub f_camera_position: [f32; 3],
    pub f_camera_front: [f32; 3],
    pub f_camera_top: [f32; 3],
    pub identity: [u16; 256],
    pub context_len: u32,
    pub context: [u8; 256],
    pub description: [u8; 4096],
}
#[derive(Debug, Default, Clone, Copy, serde::Serialize, serde::Deserialize)]
#[repr(C)]
pub struct CMumbleContext {
    //the 48 bytes Mumble uses for identification
    pub server_address: [u8; 28], // contains sockaddr_in or sockaddr_in6
    pub map_id: u32,
    pub map_type: u32,
    pub shard_id: u32,
    pub instance: u32,
    pub build_id: u32,
    // Additional data that gw2 provides us
    pub ui_state: u32, // Bitmask: Bit 1 = IsMapOpen, Bit 2 = IsCompassTopRight, Bit 3 = DoesCompassHaveRotationEnabled, Bit 4 = Game has focus, Bit 5 = Is in Competitive game mode, Bit 6 = Textbox has focus, Bit 7 = Is in Combat
    pub compass_width: u16, // pixels
    pub compass_height: u16, // pixels
    pub compass_rotation: f32, // radians
    pub player_x: f32, // continentCoords
    pub player_y: f32, // continentCoords
    pub map_center_x: f32, // continentCoords
    pub map_center_y: f32, // continentCoords
    pub map_scale: f32,
    pub process_id: u32,
    pub mount_index: u8,
}

#[derive(Debug, Default, Clone, Serialize, Deserialize)]
pub struct CIdentity {
    pub name: String,
    pub profession: u32,
    pub spec: u32,
    pub race: u32,
    pub map_id: u32,
    pub world_id: u32,
    pub team_color_id: u32,
    pub commander: bool,
    pub fov: f32,
    pub uisz: u32,
}

impl MumbleLink {
    pub fn update(&mut self, link_ptr: *const CMumbleLink) -> anyhow::Result<()> {
        let cmlink = match unsafe { link_ptr.as_ref() } {
            Some(thing) => thing,
            None => anyhow::bail!("link_ptr.as_ref returned None when trying to update MumbleLink"),
        };

        if !CMumbleLink::is_valid(link_ptr) {
            anyhow::bail!("MumbleLink is not valid. could not create Mumble Link");
        }
        self.ui_tick = cmlink.ui_tick;

        self.f_avatar_position[..3].copy_from_slice(&cmlink.f_avatar_position);
        self.f_avatar_front[..3].copy_from_slice(&cmlink.f_avatar_front);
        self.f_camera_position[..3].copy_from_slice(&cmlink.f_camera_position);
        self.f_camera_front[..3].copy_from_slice(&cmlink.f_camera_front);

        self.identity = Some(CMumbleLink::get_identity(link_ptr)?);
        self.context = Some(CMumbleLink::get_context(link_ptr)?);
        Ok(())
    }
}
impl CMumbleLink {
    pub fn get_cmumble_link(link_ptr: *const CMumbleLink) -> CMumbleLink {
        unsafe { std::ptr::read_volatile(link_ptr) }
    }

    pub fn is_valid(link_ptr: *const CMumbleLink) -> bool {
        unsafe { (*link_ptr).ui_tick > 0 }
    }

    pub fn get_ui_tick(link_ptr: *const CMumbleLink) -> u32 {
        unsafe { (*link_ptr).ui_tick }
    }

    pub fn get_identity(link_ptr: *const CMumbleLink) -> anyhow::Result<MumbleIdentity> {
        use widestring::U16CStr;
        let id = U16CStr::from_slice_with_nul(unsafe { &(*link_ptr).identity })
            .context("missing null to construct u16cstr")?;
        let id = id.to_string().context("utf16 to utf-8 encoding failed")?;
        let identity = serde_json::from_str::<MumbleIdentity>(&id)
            .context("failed to read identity from CMumbleLink.Identity")?;
        Ok(identity)
    }

    pub fn get_context(link_ptr: *const CMumbleLink) -> anyhow::Result<MumbleContext> {
        let mc = unsafe {
            std::ptr::read_volatile(&(*link_ptr).context as *const u8 as *const CMumbleContext)
        };
        let mut c = MumbleContext::default();
        c.server_address = u32::from_ne_bytes([
            mc.server_address[4],
            mc.server_address[5],
            mc.server_address[6],
            mc.server_address[7],
        ]);
        c.map_id = mc.map_id;
        c.map_type = mc.map_type;
        c.shard_id = mc.shard_id;
        c.instance = mc.instance;
        c.build_id = mc.build_id;
        c.ui_state = mc.ui_state;
        c.compass_width = mc.compass_width as u32;
        c.compass_height = mc.compass_height as u32;
        c.compass_rotation = mc.compass_rotation;
        c.player_x = mc.player_x;
        c.player_y = mc.player_y;
        c.map_center_x = mc.map_center_x;
        c.map_center_y = mc.map_center_y;
        c.map_scale = mc.map_scale;
        c.process_id = mc.process_id;
        c.mount_index = mc.mount_index as u32;

        Ok(c)
    }
    pub fn copy_raw_bytes_into(
        link_ptr: *const CMumbleLink,
        buffer: &mut [u8],
    ) -> anyhow::Result<()> {
        unsafe {
            std::ptr::copy_nonoverlapping(link_ptr as *const u8, buffer.as_mut_ptr(), buffer.len());
        }
        Ok(())
    }
}

// unsafe impl bytemuck::Zeroable for WindowDimensions {
//     fn zeroed() -> Self {
//         unsafe { core::mem::zeroed() }
//     }
// }
// unsafe impl bytemuck::Pod for WindowDimensions {}
// impl Copy for WindowDimensions {

// }
// impl WindowDimensions {
//     /// buffer must be atleast 16 bytes long
//     pub fn copy_raw_bytes_into(&self, buffer: &mut [u8]) {
//         if buffer.len() < 16 {
//             panic!("buffer length less than 16 bytes.");
//         }
//         buffer.copy_from_slice(bytemuck::bytes_of(self));
//     }
// }
// mod private {

// #[derive(Debug, Default, Clone, Serialize, Deserialize)]
// #[repr(C)]
// pub struct MumbleLink {
//     //pub ui_version: u32,
//     pub ui_tick: u32,
//     pub f_avatar_position: [f32; 3],
//     pub f_avatar_front: [f32; 3],
//     //pub f_avatar_top: [f32; 3],
//     //pub name: [u16; 256],
//     pub f_camera_position: [f32; 3],
//     pub f_camera_front: [f32; 3],
//     //pub f_camera_top: [f32; 3],
//     pub identity: Identity,
//     //pub context_len: u32,
//     pub context: MumbleContext,
// }
// }

// #[cfg(test)]
// mod tests {
//     #[test]
//     fn validate_identity_json_str() {
//         let id = String::from(
//             r#"{
//             "name": "Irwene",
//             "profession": 4,
//             "spec": 55,
//             "race": 4,
//             "map_id": 50,
//             "world_id": 268435505,
//             "team_color_id": 0,
//             "commander": false,
//             "fov": 0.873,
//             "uisz": 1
//           }"#,
//         );
//         use super::Identity;
//         let identity: Identity = serde_json::from_str(&id).unwrap();
//         assert_eq!(identity.name, "Irwene");

//     }

//     #[test]
//     fn validate_identity_json_binary_utf16() {
//         let pure_utf16: Vec<u16> = vec![0x007b , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x006e , 0x0061 , 0x006d , 0x0065 , 0x0022 , 0x003a , 0x0020 , 0x0022 , 0x0049 , 0x0072 , 0x0077 , 0x0065 , 0x006e , 0x0065 , 0x0022 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0070 , 0x0072 , 0x006f , 0x0066 , 0x0065 , 0x0073 , 0x0073 , 0x0069 , 0x006f , 0x006e , 0x0022 , 0x003a , 0x0020 , 0x0034 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0073 , 0x0070 , 0x0065 , 0x0063 , 0x0022 , 0x003a , 0x0020 , 0x0035 , 0x0035 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0072 , 0x0061 , 0x0063 , 0x0065 , 0x0022 , 0x003a , 0x0020 , 0x0034 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x006d , 0x0061 , 0x0070 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0035 , 0x0030 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0077 , 0x006f , 0x0072 , 0x006c , 0x0064 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0032 , 0x0036 , 0x0038 , 0x0034 , 0x0033 , 0x0035 , 0x0035 , 0x0030 , 0x0035 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0074 , 0x0065 , 0x0061 , 0x006d , 0x005f , 0x0063 , 0x006f , 0x006c , 0x006f , 0x0072 , 0x005f , 0x0069 , 0x0064 , 0x0022 , 0x003a , 0x0020 , 0x0030 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0063 , 0x006f , 0x006d , 0x006d , 0x0061 , 0x006e , 0x0064 , 0x0065 , 0x0072 , 0x0022 , 0x003a , 0x0020 , 0x0066 , 0x0061 , 0x006c , 0x0073 , 0x0065 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0066 , 0x006f , 0x0076 , 0x0022 , 0x003a , 0x0020 , 0x0030 , 0x002e , 0x0038 , 0x0037 , 0x0033 , 0x002c , 0x000a , 0x0020 , 0x0020 , 0x0022 , 0x0075 , 0x0069 , 0x0073 , 0x007a , 0x0022 , 0x003a , 0x0020 , 0x0031 , 0x000a , 0x007d];
//         let id = String::from_utf16_lossy(&pure_utf16);
//         println!("the string content in id is {}", id);
//         use super::Identity;
//         let identity: Identity = serde_json::from_str(&id).unwrap();
//         assert_eq!(identity.name, "meow");

//     }
// }
