use serde::{Deserialize, Serialize};
use winapi::shared::windef::LPRECT;
use winapi::um::minwinbase::SECURITY_ATTRIBUTES;
use winapi::um::winuser::GetWindowRect;
use winapi::um::{
    errhandlingapi::GetLastError,
    handleapi::INVALID_HANDLE_VALUE,
    memoryapi::{MapViewOfFile, FILE_MAP_ALL_ACCESS},
    winbase::CreateFileMappingA,
    winnt::HANDLE,
    winnt::PAGE_READWRITE,
    winuser::{GetWindowThreadProcessId, WNDENUMPROC},
};
use winapi::{
    shared::{
        minwindef::{BOOL, LPARAM, LPDWORD},
        ntdef::NULL,
        windef::HWND,
        windef::RECT,
    },
    um::winuser::EnumWindows,
};

pub const LINKSIZE: usize = std::mem::size_of::<MumbleLink>();
#[derive(Debug, Clone)]
pub struct GW2Handle {
    link_name: String,
    handle: HANDLE,
}

impl GW2Handle {
    pub fn new(key: &str) -> Self {
        let handle;
        let key_cstr = std::ffi::CString::new(key).unwrap();
        let key_cstr_ptr = key_cstr.as_ptr();
        unsafe {
            let file_handle = CreateFileMappingA(
                INVALID_HANDLE_VALUE,
                NULL as *mut SECURITY_ATTRIBUTES,
                PAGE_READWRITE,
                0,
                LINKSIZE as u32,
                key_cstr_ptr,
            );
            if file_handle == NULL {
                panic!(
                    "could not create file map handle, error code: {}",
                    GetLastError()
                );
            }
            handle = MapViewOfFile(file_handle, FILE_MAP_ALL_ACCESS, 0, 0, LINKSIZE);
            if handle == NULL {
                panic!("could not map view of file, error code: {}", GetLastError());
            }
        }
        GW2Handle {
            link_name: key.to_owned(),
            handle,
        }
    }

    pub fn read_raw(&self, buffer: &mut [u8; LINKSIZE]) {
        unsafe {
            buffer.copy_from_slice(&std::ptr::read_volatile::<[u8; LINKSIZE]>(
                self.handle as *const [u8; LINKSIZE],
            ));
        }
    }

    pub fn get_mumble_link(&self) -> MumbleLink {
        unsafe { std::ptr::read_volatile(self.handle as *const MumbleLink) }
    }
}

impl Default for GW2Handle {
    fn default() -> Self {
        GW2Handle::new("MumbleLink")
    }
}
impl MumbleLink {
    
    pub fn get_win_pos_dim(&self) -> Option<WinPosDim> {
        if !(self.ui_tick > 0) {
            return None;
        }
        let context = self.context.as_ptr() as *const MumbleContext;
        let mut pid = unsafe { (*context).process_id };

        let window_handle: LPARAM = &mut pid as *mut u32 as LPARAM;
        unsafe {
            let result: BOOL = EnumWindows(Some(Self::get_handle_by_pid), window_handle);
            if result != 0 {
                print!("couldn't find gw2 window. error code: {}", GetLastError());
                return None;
            }
            return None;

            // let rect: LPRECT = NULL as LPRECT;
            // let status = GetWindowRect(window_handle as HWND, rect);
            // if status == 0 {
            //     return None;
            // }
            // Some(WinPosDim {
            //     x: (*rect).left as u32,
            //     y: (*rect).top as u32,
            //     width: ((*rect).right -(*rect).left) as u32,
            //     height: ((*rect).bottom - (*rect).top) as u32,
            // })

        }



    }
    unsafe extern "system" fn get_handle_by_pid(handle: HWND, mut lp: LPARAM) -> BOOL {
        let handle_pid = NULL as *mut u32;
        GetWindowThreadProcessId(handle, handle_pid);
        if *(lp as *const u32) == *(handle_pid ) {
            lp = handle as isize;
            return 0;
        }
        return 1;
    }
}
#[derive(Debug, Clone, Copy, Default, Serialize, Deserialize)]
pub struct WinPosDim {
    x: u32,
    y: u32,
    width: u32,
    height: u32,
}
#[derive(Debug)]
#[repr(C)]
pub struct MumbleLink {
    pub ui_version: u32,
    pub ui_tick: u32,
    pub f_avatar_position: [f32; 3],
    pub f_avatar_front: [f32; 3],
    pub f_avatar_top: [f32; 3],
    pub name: [u8; 512],
    pub f_camera_position: [f32; 3],
    pub f_camera_front: [f32; 3],
    pub f_camera_top: [f32; 3],
    pub identity: [u16; 256],
    pub context_len: u32,
    pub context: [u8; 256],
    pub description: [u8; 4096],
}
#[allow(dead_code)]
#[repr(C)]
struct MumbleContext {
    //the 48 bytes Mumble uses for identification
    server_address: [u8; 28], // contains sockaddr_in or sockaddr_in6
    map_id: u32,
    map_type: u32,
    shard_id: u32,
    instance: u32,
    build_id: u32,
    // Additional data that gw2 provides us
    ui_state: u32, // Bitmask: Bit 1 = IsMapOpen, Bit 2 = IsCompassTopRight, Bit 3 = DoesCompassHaveRotationEnabled, Bit 4 = Game has focus, Bit 5 = Is in Competitive game mode, Bit 6 = Textbox has focus, Bit 7 = Is in Combat
    compass_width: u16, // pixels
    compass_height: u16, // pixels
    compass_rotation: f32, // radians
    player_x: f32, // continentCoords
    player_y: f32, // continentCoords
    map_center_x: f32, // continentCoords
    map_center_y: f32, // continentCoords
    map_scale: f32,
    process_id: u32,
    mount_index: u8,
}
#[allow(dead_code)]
#[derive(Serialize, Deserialize)]
pub struct Identity {
    name: String,
    profession: u8,
    spec: u8,
    race: u8,
    map_id: u16,
    world_id: u16,
    team_color_id: u16,
    commander: bool,
    fov: f32,
    uisz: u8,
}

