//! jlrs is a crate that provides access to most of the Julia C API, it can be used to embed Julia
//! in Rust applications and to use functionality from the Julia C API when writing `ccall`able
//! functions in Rust. Currently this crate is only tested on Linux and Windows in combination
//! with Julia 1.6 and 1.7 and is not compatible with other versions of Julia.
//!
//! The documentation assumes you're already familiar with the Julia programming language.
//!
//! An incomplete list of features that are currently supported by jlrs:
//!
//!  - Access arbitrary Julia modules and their contents.
//!  - Call Julia functions, including functions that take keyword arguments.
//!  - Exceptions can be handled or converted to their error message, optionally with color.
//!  - Include and call your own Julia code.
//!  - Use a custom system image.
//!  - Create values that Julia can use, and convert them back to Rust, from Rust.
//!  - Access the type information and fields of values. The contents of inline and bits-union
//!    fields can be accessed directly.
//!  - Create and use n-dimensional arrays. The `jlrs-ndarray` feature can be enabled for
//!    integration with ndarray.
//!  - Support for mapping Julia structs to Rust structs that can be generated by JlrsReflect.jl.
//!  - Structs that can be mapped to Rust include those with type parameters and bits unions.
//!  - An async runtime is available when the `async-std-rt` or `tokio-rt` feature is enabled,
//!    which can be used from multiple threads and supports scheduling Julia `Task`s and
//!    `await`ing the result without blocking the runtime.
//!
//!
//! # Prerequisites
//!
//! Julia must be installed before jlrs can be used. Only version 1.6 and 1.7 are supported,
//!
//! #### Linux
//!
//! The recommended way to install Julia is to download the binaries from the official website,
//! which is distributed in an archive containing a directory called `julia-x.y.z`. This directory
//! contains several other directories, including a `bin` directory containing the `julia`
//! executable.
//!
//! During compilation, the paths to the header and library are normally detected automatically by
//! executing the command `which julia`. The path to `julia.h` must be
//! `$(which julia)/../include/julia/julia.h` and the path to the library
//! `$(which julia)/../lib/libjulia.so`. If you want to override this default behaviour the
//! `JULIA_DIR` environment variable must be set to the path to the appropriate `julia.x-y-z`
//! directory; in this case `$JULIA_DIR/include/julia/julia.h` and
//! `$JULIA_DIR/lib/libjulia.so` are used instead.
//!
//! In order to be able to load `libjulia.so`, this file must be on the library search path. If
//! this is not the case you must add `/path/to/julia-x.y.z/lib` to the `LD_LIBRARY_PATH`
//! environment variable. When the `uv` feature is enabled, `/path/to/julia-x.y.z/lib/julia` must
//! also be added to `LD_LIBRARY_PATH`. The latter path should not be added to the default path
//! because this can break tools currently installed on your system.
//!
//! #### Windows
//!
//! Julia can be installed using the installer or portable installation downloaded from the
//! official website. After installation or extraction a folder called `Julia-x.y.z` exists, which
//! contains several folders including a `bin` folder containing `julia.exe`. The path to the
//! `bin` folder must be added to the `Path` environment variable.
//!
//! Julia is automatically detected by executing the command `where julia`. If this returns
//! multiple locations the first one is used. The default can be overridden by setting the
//! `JULIA_DIR` environment variable.
//!
//! Note that while both Julia 1.6 and 1.7 are supported on Windows, several methods are currently
//! unavailable when the LTS version is used.
//!
//! If you use the MSVC target, you must create two or three lib files using `lib.exe`. The def
//! files required for this can be found in the `defs` folder in the jl-sys crate. To create the
//! lib files, copy the three files from either the `lts` or `stable` folder to the `bin` folder
//! where Julia is installed. Afterwards, open a Developer Command Prompt for VS19 and execute the
//! following commands:
//!
//! ```cmd
//! cd C:\Path\To\Julia-x.y.z\bin
//! lib /def:libjulia.def /out:libjulia.lib /machine:x64
//! lib /def:libopenlibm.def /out:libopenlibm.lib /machine:x64
//! lib /def:libuv-2.def /out:libuv-2.lib /machine:x64
//! ```
//!
//! If you use the GNU target these lib files must not exist.
//!
//! # Features
//!
//! Much functionality of jlrs is unavailable unless the proper features are enabled. These
//! features generally belong to one of two categories: runtimes and utilities.
//!
//! A runtime lets you call Julia from Rust, you must enable one of them if you want to embed
//! Julia in a Rust application. The following features enable a runtime:
//!
//! - `sync-rt`
//!   This is the sync runtime. The struct [`Julia`] is available and can be used to
//!   initialize Julia on some thread, this instance can then be used from that thread.
//!
//! - `tokio-rt` and `async-std-rt`
//!   These are both async runtimes, the first is powered by tokio and the other by async-std. The
//!   struct [`AsyncJulia`] is available and can be used to initialize Julia on another thread.
//!   Unlike `Julia`, `AsyncJulia` can be shared across threads. The main additional feature
//!   that's available when an async runtime is used is the ability to schedule a new Julia
//!   `Task`, which is returned as a `Future` that resolves when this task has completed.
//!
//! If you're writing a library, either one that will be called from Julia or one that will be
//! used by an application that embeds Julia, no runtime is required.
//!
//! In addition to these runtimes, the following utility features are available:
//!
//! - `lts`
//!   Use the current LTS version of Julia (1.6) instead of the current stable version (1.7).
//!
//! - `async`
//!   Enable the features of the async runtime which don't depend on the backing runtime. This
//!   can be used in libraries which provide implementations of tasks that an async runtime can
//!   execute.
//!
//! - `jlrs-derive`
//!   This features should be used in combination with the JlrsReflect.jl package. This package
//!   generates Rust bindings for Julia structs, these bindings use the custom derive macros to
//!   enable the safe conversion of data from Julia to Rust, and from Rust to Julia in some cases.
//!
//! - `jlrs-ndarray`
//!   Access the contents of a Julia array as an `ArrayView` or `ArrayViewMut` from ndarray.
//!
//! - `f16`
//!   Adds support for working with Julia's `Float16` type from Rust using half's `f16` type.
//!
//! - `ccall`
//!   Julia's `ccall` interface can be used to call functions written in Rust from Julia. No
//!   runtime can be used in this case because Julia has already been initialized, when this
//!   feature is enabled the `CCall` struct is available which offers the same functionality as
//!   the sync runtime without initializing Julia.
//!
//! - `uv`
//!   When `ccall` has been enabled, this feature enables the method `CCall::uv_async_send`, which
//!   can be used to wake a Julia `AsyncCondition` from Rust.
//!
//! - `pyplot`
//!   This feature lets you plot data using the Pyplot package and Gtk 3 from Rust.
//!
//!
//! # Using this crate
//!
//! If you want to embed Julia and call it from Rust, you must enable a runtime feature:
//!
//! `jlrs = {version = "0.13", features = ["sync-rt"]}`
//!
//! `jlrs = {version = "0.13", features = ["tokio-rt"]}`
//!
//! `jlrs = {version = "0.13", features = ["async-std-rt"]}`
//!
//! A `prelude` is available which provides access to most of the structs and traits you're likely
//! to need. When embedding Julia, it must be initialized before it can be used.
//!
//! If you use the sync runtime, you can do this by calling [`Julia::init`] which returns an
//! instance of [`Julia`]. Note that this method can only be called once while the application is
//! running, if you drop it you won't be able to create a new instance but have to restart the
//! application. If you want to use a custom system image you must call [`Julia::init_with_image`]
//! instead of `Julia::init`.
//!
//! The async runtimes provide the same methods, [`AsyncJulia::init`] and
//! [`AsyncJulia::init_with_image`], and also add async variants, [`AsyncJulia::init_async`] and
//! [`AsyncJulia::init_with_image_async`]. All of these methods return an instance of
//! [`AsyncJulia`].
//!
//! When you're calling Rust from Julia everything has already been initialized. If the `ccall`
//! feature is enabled [`CCall`] is available which provides the same functionality as the sync
//! runtime.
//!
//! ## Calling Julia from Rust
//!
//! This section will focus on some topics that are common between the sync and async runtimes.
//!
//! After initialization you have an instance of [`Julia`] or [`AsyncJulia`], both provide a
//! method called `include` that lets you include files with custom Julia code. In order to
//! create Julia data and call Julia functions, a scope must be created.
//!
//! When the sync runtime is used this can be done by calling the methods [`Julia::scope`] and
//! [`Julia::scope_with_slots`]. These methods take a closure with two arguments, a [`Global`] and
//! a mutable reference to a [`GcFrame`] (frame). The first is an access token for global Julia
//! data, the second is used to root non-global data. While non-global data is rooted, it won't be
//! freed by Julia's garbage collector (GC). The frame is created when `Julia::scope(_with_slots)`
//! is called and dropped when it returns. This means that any data rooted in the frame associated
//! with a scope won't be freed by the GC until leaving that scope.
//!
//! Because `AsyncJulia` is a handle to the async runtime which runs on another thread it's not
//! possible to directly create a scope. Rather, the async runtime deals with tasks. The simplest
//! of these is a blocking task, which can be executed by calling
//! `AsyncJulia::(try_)blocking_task(_with_slots)`. This method accepts any closure
//! `Julia::scope(_with_slots)` can handle with the additional requirement that they're `Send` and
//! `Sync`. It's called a blocking task because the runtime is blocked while executing this task.
//! The other kinds of tasks that the async runtime can handle will be introduced later.
//!
//! Inside the closure provided to `Julia::scope(_with_slots)` or
//! `AsyncJulia::(try_)blocking_task(_with_slots)` it's possible to interact with Julia. Global
//! Julia data can be accessed through its module system, the methods [`Module::main`],
//! [`Module::base`], and [`Module::core`] can be used to access the `Main`, `Base`, and `Core`
//! modules respectively. The contents of these modules can then be accessed by calling
//! [`Module::function`] which returns a [`Function`], [`Module::global`] which returns a
//! [`Value`], and [`Module::submodule`] which returns another `Module`.
//!
//! `Value`, `Module`, and `Function` are all examples of pointer wrapper types. Pointer wrapper
//! types wrap a pointer to some data "owned" by the GC. Other important examples of pointer
//! wrapper types are [`Array`], [`JuliaString`] and [`DataType`]. `Value` wraps arbitrary Julia
//! data, all other pointer wrapper types can always be converted to a `Value`. All pointer
//! wrapper types wrap a type defined by the Julia C API.
//!
//! In addition to pointer wrapper types there are inline wrapper types. The most important
//! difference between the two kinds of wrapper types is that a pointer wrapper type wraps a
//! mutable type, while inline wrappers wrap immutable types. Examples are primitive types like
//! `Float32` and `UInt`, the inline wrapper types of these types are their counterparts, `f32`
//! and `usize`.
//!
//! `Value` provides several methods to allocate new Julia data. The simplest one is
//! [`Value::eval_string`], which evaluates the contents of the string passed to it and returns
//! the result as a `Value`. For example, if you evaluate `sqrt(2)` the returned [`Value`]
//! contains its result.
//!
//! In practice, [`Value::eval_string`] is relatively limited. It can be used to evaluate simple
//! function calls like `sqrt(2)`, but it must be parsed, compiled, and can't take any arguments.
//! Its most important use case is importing installed and standard library packages by evaluating
//! an `import` or `using` statement. A more interesting method, [`Value::new`], can be used with
//! data of any type that implements [`IntoJulia`]. This trait is implemented by primitive types
//! like `i8` and `char`. Any type that implements [`IntoJulia`] also implements [`Unbox`] which
//!  is used to extract the contents of a `Value`. Because `sqrt(2)` returns a `Float64`, it can
//! be unboxed as an `f64`. Pointer wrapper types don't implement [`IntoJulia`] or [`Unbox`], if
//! they can be created from Rust the provide methods to do so.
//!
//! It's possible to call anything that implements [`Call`] as a Julia function. In addition to
//! `Function`, this trait is implemented by `Value` because any Julia value is potentially
//! callable as a function. Functions can be called with any number of positional arguments and
//! can be provided with keyword arguments. Keywords must be provided as a `NamedTuple`, which can
//! be created with the [`named_tuple`] macro.
//!
//! Evaluating raw code and calling Julia functions is always unsafe. Nothing prevents you from
//! calling a function like `nasaldemons() = unsafe_load(Ptr{Float64}(C_NULL))`.
//!
//! As a simple example, let's convert two numbers to Julia values and add them:
//!
//! ```no_run
//! use jlrs::prelude::*;
//!
//! # fn main() {
//! // Initializing Julia is unsafe because it can race with another crate that does
//! // the same.
//! let mut julia = unsafe { Julia::init().unwrap() };
//! let res = julia.scope(|global, frame| {
//!     // Create the two arguments. Note that the first argument, something that
//!     // implements Scope, is taken by value and mutable references don't implement
//!     // Copy, so it's necessary to mutably reborrow the frame.
//!     let i = Value::new(&mut *frame, 2u64)?;
//!     let j = Value::new(&mut *frame, 1u32)?;
//!
//!     // The `+` function can be found in the base module.
//!     let func = Module::base(global).function(&mut *frame, "+")?;
//!
//!     // Call the function and unbox the result as a `u64`. The result of the function
//!     // call is a nested `Result`; the outer error doesn't contain to any Julia
//!     // data, while the inner error contains the exception if one is thrown. Here the
//!     // exception is converted to the outer error type by calling `into_jlrs_result`, this new
//!     // error contains the error message Julia would have shown. Colors can be enabled by
//!     // calling `Julia::error_color`.
//!     unsafe {
//!         func.call2(&mut *frame, i, j)?
//!             .into_jlrs_result()?
//!             .unbox::<u64>()
//!     }
//! }).unwrap();
//!
//! assert_eq!(res, 3);
//! # }
//! ```
//!
//! ### Async and persistent tasks
//!
//! In addition to blocking tasks, the async runtimes let you execute async tasks which implement
//! the [`AsyncTask`] trait, and persistent tasks which implement [`PersistentTask`]. Both of
//! these traits are async traits.
//!
//! An async task is similar to a blocking task, except that you must implement the async `run`
//! method instead of providing a closure. This method takes a `Global` and a mutable reference to
//! an [`AsyncGcFrame`]. This new frame type not only provides access to the same features as
//! [`GcFrame`] does, it can also be used to call async methods provided by the [`CallAsync`]
//! trait. These methods schedule a function call as a new Julia `Task` and can be `await`ed until
//! this task has completed. The async runtimes can switch to another task while the result is
//! pending, allowing multiple tasks to progress.
//!
//! The previous example can be rewritten as an async task:
//!
//! ```
//! use jlrs::prelude::*;
//!
//! struct AdditionTask {
//!     a: u64,
//!     b: u32,
//! }
//!
//! // Only the runtime thread can call the Julia C API, so the async
//! // trait methods of `AsyncTask` must not return a future that
//! // implements `Send` or `Sync`.
//! #[async_trait(?Send)]
//! impl AsyncTask for AdditionTask {
//!     // The type of the result of this task if it succeeds.
//!     type Output = u64;
//!
//!     // The contents of the `run` method can mostly be copied
//!     // from the previous example.
//!     async fn run<'base>(
//!         &mut self,
//!         global: Global<'base>,
//!         frame: &mut AsyncGcFrame<'base>,
//!     ) -> JlrsResult<Self::Output> {
//!         let a = Value::new(&mut *frame, self.a)?;
//!         let b = Value::new(&mut *frame, self.b)?;
//!             
//!         let func = Module::base(global).function(&mut *frame, "+")?;
//!
//!         // Here, the `CallAsync::call_async` trait method is
//!         // used to schedule the function call on another thread.
//!         // This method can only be used with an `AsyncGcFrame`.
//!         unsafe { func.call_async(&mut *frame, &mut [a, b]) }
//!             .await?
//!             .into_jlrs_result()?
//!             .unbox::<u64>()
//!     }
//! }
//! ```
//!
//! While blocking and async tasks run once and return their result, a persistent task returns a
//! handle after setting up its initial state. This handle can be shared across threads and used
//! to call its `run` method. In addition to a global and an async frame, this method can use the
//! state and input data provided by the caller.
//!
//! As an example, let's accumulate some number of values in a Julia array and return the sum of
//! its contents:
//!
//! ```
//! use jlrs::prelude::*;
//!
//! struct AccumulatorTask {
//!     n_values: usize
//! }
//!
//! struct AccumulatorTaskState {
//!     array: TypedArray<'static, 'static, usize>,
//!     offset: usize
//! }
//!
//! // Only the runtime thread can call the Julia C API, so the async trait
//! // methods of `PersistentTask` must not return a future that implements
//! // `Send` or `Sync`.
//! #[async_trait(?Send)]
//! impl PersistentTask for AccumulatorTask {
//!     // The type of the result of the task if it succeeds.
//!     type Output = usize;
//!     // The type of the task's internal state.
//!     type State = AccumulatorTaskState;
//!     // The type of the additional data that the task must be called with.
//!     type Input = usize;
//!
//!     // This method is called before the handle is returned. Note that the
//!     // lifetime of the frame is `'static`: the frame is not dropped until
//!     // the task has completed, so the task's internal state can contain
//!     // Julia data rooted in this frame.
//!     async fn init<'inner>(
//!         &'inner mut self,
//!         _global: Global<'static>,
//!         frame: &'inner mut AsyncGcFrame<'static>,
//!     ) -> JlrsResult<Self::State> {
//!         // A `Vec` can be moved from Rust to Julia if the element type
//!         // implements `IntoJulia`. When a new array is allocated it's
//!         // returned as a `Value` and can be cast to `Array` or
//!         // `TypedArray`.
//!         let data = vec![0usize; self.n_values];
//!         let array = Array::from_vec(&mut *frame, data, self.n_values)?
//!             .cast::<TypedArray<usize>>()?;
//!     
//!         Ok(AccumulatorTaskState {
//!             array,
//!             offset: 0
//!         })
//!     }
//!     
//!     // Whenever the task is called through its handle this method
//!     // is called. Unlike `init`, the frame that this method can use
//!     // is dropped after `run` returns.
//!     async fn run<'inner, 'frame>(
//!         &'inner mut self,
//!         global: Global<'frame>,
//!         frame: &'inner mut AsyncGcFrame<'frame>,
//!         state: &'inner mut Self::State,
//!         input: Self::Input,
//!     ) -> JlrsResult<Self::Output> {
//!         {
//!             // Array data can be directly accessed from Rust.
//!             // TypedArray::inline_data_mut can be used if the type
//!             // of the elements is concrete and immutable.
//!             let mut data = state.array.inline_data_mut(frame)?;
//!             data[state.offset] = input;
//!
//!             state.offset += 1;
//!             if (state.offset == self.n_values) {
//!                 state.offset = 0;
//!             }
//!         }
//!
//!         // Return the sum of the contents of `state.array`. To
//!         // use it as a function argument it must be converted
//!         // to a `Value` first, which is possible because
//!         // `TypedArray` is a pointer wrapper type.
//!         unsafe {
//!             Module::base(global)
//!                 .function(&mut *frame, "sum")?
//!                 .call1(&mut *frame, state.array.as_value())?
//!                 .into_jlrs_result()?
//!                 .unbox::<usize>()
//!         }
//!     }
//! }
//! ```
//!
//! ## Calling Rust from Julia
//!
//! Julia's `ccall` interface can be used to call `extern "C"` functions defined in Rust, for most
//! use-cases you shouldn't need jlrs. There are two major ways to use `ccall`, with a pointer to
//! the function or a `(:function, "library")` pair.
//!
//! A function can be cast to a void pointer and converted to a [`Value`]:
//!
//! ```no_run
//! # use jlrs::prelude::*;
//! // This function will be provided to Julia as a pointer, so its name can be mangled.
//! unsafe extern "C" fn call_me(arg: bool) -> isize {
//!     if arg {
//!         1
//!     } else {
//!         -1
//!     }
//! }
//!
//! # fn main() {
//! let mut julia = unsafe { Julia::init().unwrap() };
//! julia.scope(|global, frame| unsafe {
//!     // Cast the function to a void pointer
//!     let call_me_val = Value::new(&mut *frame, call_me as *mut std::ffi::c_void)?;
//!
//!     // Value::eval_string can be used to create new functions.
//!     let func = Value::eval_string(
//!         &mut *frame,
//!         "myfunc(callme::Ptr{Cvoid})::Int = ccall(callme, Int, (Bool,), true)"
//!     )?.into_jlrs_result()?;
//!
//!     // Call the function and unbox the result.
//!     let output = func.call1(&mut *frame, call_me_val)?
//!         .into_jlrs_result()?
//!         .unbox::<isize>()?;
//!
//!     assert_eq!(output, 1);
//!
//!     Ok(())
//! }).unwrap();
//! # }
//! ```
//!
//! You can also use functions defined in `dylib` and `cdylib` libraries. In order to create such
//! a library you need to add
//!
//! ```toml
//! [lib]
//! crate-type = ["dylib"]
//! ```
//!
//! or
//!
//! ```toml
//! [lib]
//! crate-type = ["cdylib"]
//! ```
//!
//! respectively to your crate's `Cargo.toml`. Use a `dylib` if you want to use the crate in other
//! Rust crates, but if it's only intended to be called through `ccall` a `cdylib` is the better
//! choice. On Linux, compiling such a crate will be compiled to `lib<crate_name>.so`.
//!
//! The functions you want to use with `ccall` must be both `extern "C"` functions to ensure the C
//! ABI is used, and annotated with `#[no_mangle]` to prevent name mangling. Julia can find
//! libraries in directories that are either on the default library search path or included by
//! setting the `LD_LIBRARY_PATH` environment variable on Linux. If the compiled library is not
//! directly visible to Julia, you can open it with `Libdl.dlopen` and acquire function pointers
//! with `Libdl.dlsym`. These pointers can be called the same way as the pointer in the previous
//! example.
//!
//! If the library is visible to Julia you can access it using the library name. If `call_me` is
//! defined in a crate called `foo`, the following should work:
//!
//! ```julia
//! ccall((:call_me, "libfoo"), Int, (Bool,), false)
//! ```
//!
//! One important aspect of calling Rust from other languages in general is that panicking across
//! an FFI boundary is undefined behaviour. If you're not sure your code will never panic, wrap it
//! with `std::panic::catch_unwind`.
//!
//! Most features provided by jlrs including accessing modules, calling functions, and borrowing
//! array data require a [`Global`] or a frame. You can access these by creating an instance of
//! [`CCall`] first. Another method provided by [`CCall`] is [`CCall::uv_async_send`], this method
//! can be used to wake an `Base.AsyncCondition`. In particular, it can be used to write a
//! `ccall`able function that does its actual work on another thread, returns early and then
//! `wait`ing on the async condition from Julia. The advantage of this is that the long-running
//! function will not block the Julia runtime.
//!
//!
//! # Testing
//!
//! The restriction that Julia can be initialized once must be taken into account when running
//! tests that use `jlrs`. The recommended approach is to create a thread-local static `RefCell`:
//!
//! ```no_run
//! use jlrs::prelude::*;
//! use std::cell::RefCell;
//! thread_local! {
//!     pub static JULIA: RefCell<Julia> = {
//!         let julia = RefCell::new(unsafe { Julia::init().unwrap() });
//!         julia.borrow_mut().scope(|_global, _frame| {
//!             /* include everything you need to use */
//!             Ok(())
//!         }).unwrap();
//!         julia
//!     };
//! }
//! ```
//!
//! Tests that use this construct can only use one thread for testing, so you must use
//! `cargo test -- --test-threads=1`, otherwise the code above will panic when a test
//! tries to call `Julia::init` a second time from another thread.
//!
//! If these tests also involve the async runtime, the `JULIA_NUM_THREADS` environment
//! variable must be set to a value larger than 2.
//!
//! If you want to run jlrs's tests, both these requirements must be taken into account, for
//! example:
//! `JULIA_NUM_THREADS=3 cargo test --features jlrs-ndarray,f16,ccall,uv,jlrs-derive,tokio-rt -- --test-threads=1`
//!
//!
//! # Custom types
//!
//! In order to map a struct in Rust to one in Julia you can derive [`ValidLayout`], [`Unbox`],
//! and [`Typecheck`]. If the struct in Julia has no type parameters and is a bits type you can
//! also derive [`IntoJulia`], which lets you use the type in combination with [`Value::new`].
//!
//! You should normally not need to implement these structs or traits manually. The JlrsReflect.jl
//! package can generate the correct Rust struct and automatically derive the supported traits for
//! types that have no tuple or union fields with type parameters. The reason for this restriction
//! is that the layout of tuple and union fields can be very different depending on these
//! parameters in a way that can't be represented by a single Rust struct.
//!
//! These custom types can also be used when you call Rust from Julia with `ccall`.
//!
//!
//! [their User Guide]: https://rust-lang.github.io/rust-bindgen/requirements.html
//! [on Microsoft's website]: https://docs.microsoft.com/en-us/windows/wsl/install-win10
//! [the examples directory of the repo]: https://github.com/Taaitaaiger/jlrs/tree/master/examples
//! [`Julia`]: crate::julia::Julia
//! [`Julia::scope`]: crate::julia::Julia::scope
//! [`Julia::scope_with_slots`]: crate::julia::Julia::scope_with_slots
//! [`Julia::init`]: crate::julia::Julia::init
//! [`AsyncJulia::init`]: crate::extensions::multitask::runtime::AsyncJulia::init
//! [`AsyncJulia::init_async`]: crate::extensions::multitask::runtime::AsyncJulia::init_async
//! [`Julia::init_with_image`]: crate::julia::Julia::init_with_image
//! [`CCall`]: crate::ccall::CCall
//! [`CCall::uv_async_send`]: crate::ccall::CCall::uv_async_send
//! [`Global`]: crate::memory::global::Global
//! [`GcFrame`]: crate::memory::frame::GcFrame
//! [`Module`]: crate::wrappers::ptr::module::Module
//! [`Function`]: crate::wrappers::ptr::function::Function
//! [`Value`]: crate::wrappers::ptr::value::Value
//! [`Call`]: crate::wrappers::ptr::call::Call
//! [`Value::eval_string`]: crate::wrappers::ptr::value::Value::eval_string
//! [`Value::new`]: crate::wrappers::ptr::value::Value::new
//! [`Array`]: crate::wrappers::ptr::array::Array
//! [`JuliaString`]: crate::wrappers::ptr::string::JuliaString
//! [`Module::main`]: crate::wrappers::ptr::module::Module::main
//! [`Module::base`]: crate::wrappers::ptr::module::Module::base
//! [`Module::core`]: crate::wrappers::ptr::module::Module::core
//! [`Module::function`]: crate::wrappers::ptr::module::Module::function
//! [`Module::global`]: crate::wrappers::ptr::module::Module::global
//! [`Module::submodule`]: crate::wrappers::ptr::module::Module::submodule
//! [`AsyncJulia::init_with_image`]: crate::extensions::multitask::runtime::AsyncJulia::init_with_image
//! [`AsyncJulia::init_with_image_async`]: crate::extensions::multitask::runtime::AsyncJulia::init_with_image_async
//! [`IntoJulia`]: crate::convert::into_julia::IntoJulia
//! [`Typecheck`]: crate::layout::typecheck::Typecheck
//! [`ValidLayout`]: crate::layout::valid_layout::ValidLayout
//! [`Unbox`]: crate::convert::unbox::Unbox
//! [`CallAsync::call_async`]: crate::extensions::multitask::call_async::CallAsync
//! [`AsyncGcFrame`]: crate::extensions::multitask::async_frame::AsyncGcFrame
//! [`Frame`]: crate::memory::frame::Frame
//! [`AsyncTask`]: crate::extensions::multitask::async_task::AsyncTask
//! [`PersistentTask`]: crate::extensions::multitask::async_task::PersistentTask
//! [`PersistentHandle`]: crate::extensions::multitask::async_task::PersistentHandle
//! [`AsyncJulia`]: crate::extensions::multitask::runtime::AsyncJulia
//! [`CallAsync`]: crate::extensions::multitask::call_async::CallAsync
//! [`DataType`]: crate::wrappers::ptr::datatype::DataType
//! [`TypedArray`]: crate::wrappers::ptr::array::TypedArray
//! [`Output`]: crate::memory::output::Output
//! [`OutputScope`]: crate::memory::output::OutputScope
//! [`ScopeExt`]: crate::memory::scope::ScopeExt
//! [`ScopeExt::scope`]: crate::memory::scope::ScopeExt::scope
//! [`Scope`]: crate::memory::scope::Scope
//! [`Scope::value_scope`]: crate::memory::scope::Scope::value_scope
//! [`Scope::result_scope`]: crate::memory::scope::Scope::result_scope

#![forbid(rustdoc::broken_intra_doc_links)]

#[cfg(any(feature = "sync-rt", feature = "async"))]
use std::sync::atomic::AtomicBool;

#[cfg(any(feature = "sync-rt", feature = "async"))]
macro_rules! init_fn {
    ($name:ident, $include:ident, $file:expr) => {
        pub(crate) static $include: &'static str = include_str!($file);
        pub(crate) fn $name<'frame, F: $crate::memory::frame::Frame<'frame>>(frame: &mut F) -> () {
            unsafe {
                match $crate::wrappers::ptr::value::Value::eval_string(frame, $include) {
                    Ok(Ok(_)) => (),
                    Ok(Err(e)) => {
                        panic!(
                            "{}",
                            $crate::wrappers::ptr::Wrapper::error_string_or(
                                e,
                                $crate::error::CANNOT_DISPLAY_VALUE
                            )
                        )
                    }
                    Err(_) => panic!("AllocError during initialization of {}", $file),
                }
            }
        }
    };
}

#[cfg(feature = "ccall")]
pub mod ccall;
pub mod convert;
pub mod error;
pub mod extensions;
pub mod info;
#[cfg(feature = "sync-rt")]
pub mod julia;
pub mod layout;
pub mod memory;
pub mod prelude;
pub(crate) mod private;
#[doc(hidden)]
#[cfg(feature = "sync-rt")]
pub mod util;
pub mod wrappers;

#[cfg(any(feature = "sync-rt", feature = "tokio-rt", feature = "async-std-rt"))]
pub(crate) static INIT: AtomicBool = AtomicBool::new(false);

#[cfg(any(feature = "sync-rt", feature = "tokio-rt", feature = "async-std-rt"))]
init_fn!(init_jlrs, JLRS_JL, "Jlrs.jl");
