//! Trait for checking Julia type properties.
//!
//! Several properties of Julia data can be checked by using [`Value::is`] and [`DataType::is`],
//! these methods must be used in combination with a type that implements the [`Typecheck`] trait.
//! Most types that implement this trait also implement [`Wrapper`] or [`Unbox`], for these types
//! the typecheck indicates whether or not it's valid to cast the value to or unbox it as that
//! type.
//!
//! Other important typechecks are [`Mutable`], which checks if the value is mutable when used
//! with [`Value::is`] and if instances of the type are mutable when used with [`DataType::is`];
//! [`Concrete`], which only makes sense with [`DataType::is`] and indicates whether instances of
//! that type can exist.; [`Nothing`], which checks if a value is `nothing` or a type is
//! `Nothing`.
//!
//! `Typecheck` is automatically implemented when [`ValidLayout`] is derived. All wrappers
//! generated by JlrsReflect.jl derive `ValidLayout`, you shouldn't implement it manually.
//!
//! [`Value::is`]: crate::wrappers::ptr::value::Value::is
//! [`Wrapper`]: crate::wrappers::ptr::Wrapper
//! [`Unbox`]: crate::convert::unbox::Unbox
//! [`ValidLayout`]: crate::layout::valid_layout::ValidLayout
use jl_sys::{
    jl_code_info_type, jl_datatype_type, jl_globalref_type, jl_gotonode_type, jl_intrinsic_type,
    jl_linenumbernode_type, jl_namedtuple_typename, jl_newvarnode_type, jl_nothing_type,
    jl_phicnode_type, jl_phinode_type, jl_pinode_type, jl_quotenode_type, jl_slotnumber_type,
    jl_string_type, jl_typedslot_type, jl_upsilonnode_type,
};

use crate::{
    convert::into_julia::IntoJulia,
    memory::global::Global,
    private::Private,
    wrappers::ptr::Wrapper,
    wrappers::ptr::{
        datatype::DataType, private::Wrapper as _, type_name::TypeName, union_all::UnionAll,
    },
};
use std::ffi::c_void;

/// This trait is used in combination with [`Value::is`] and [`DataType::is`] to check if that
/// property holds true.
///
/// Most types that implement this trait also implement [`Wrapper`] or [`Unbox`], for these types
/// the typecheck indicates whether or not it's valid to cast the value to or unbox it as that
/// type. This trait is automatically derived for wrappers generated with JlrsReflect.jl, in this
/// case the typecheck performs a [`ValidLayout`]-check.
///
/// [`Value::is`]: crate::wrappers::ptr::value::Value::is
/// [`DataType::cast`]: crate::wrappers::ptr::datatype::DataType::cast
/// [`Unbox`]: crate::convert::unbox::Unbox
/// [`Wrapper`]: crate::wrappers::ptr::Wrapper
/// [`ValidLayout`]: crate::layout::valid_layout::ValidLayout
pub unsafe trait Typecheck {
    #[doc(hidden)]
    fn typecheck(t: DataType) -> bool;
}

#[doc(hidden)]
#[macro_export]
macro_rules! impl_julia_typecheck {
    ($type:ty, $jl_type:expr, $($lt:lifetime),+) => {
        unsafe impl<$($lt),+> crate::layout::typecheck::Typecheck for $type {
            fn typecheck(t: $crate::wrappers::ptr::datatype::DataType) -> bool {
                unsafe {
                    <$crate::wrappers::ptr::datatype::DataType as $crate::wrappers::ptr::private::Wrapper>::unwrap(t, crate::private::Private) == $jl_type
                }
            }
        }
    };
    ($type:ty, $jl_type:expr) => {
        unsafe impl crate::layout::typecheck::Typecheck for $type {
            fn typecheck(t: $crate::wrappers::ptr::datatype::DataType) -> bool {
                unsafe {
                    <$crate::wrappers::ptr::datatype::DataType as $crate::wrappers::ptr::private::Wrapper>::unwrap(t, crate::private::Private) == $jl_type
                }
            }
        }
    };
    ($type:ty) => {
        unsafe impl crate::layout::typecheck::Typecheck for $type {
            fn typecheck(t: crate::wrappers::ptr::datatype::DataType) -> bool {
                unsafe {
                    let global = $crate::memory::global::Global::new();
                    <$crate::wrappers::ptr::datatype::DataType as $crate::wrappers::ptr::private::Wrapper>::unwrap(t, crate::private::Private) == <$type as $crate::convert::into_julia::IntoJulia>::julia_type(global).ptr()
                }
            }
        }
    };
}

impl_julia_typecheck!(i8);
impl_julia_typecheck!(i16);
impl_julia_typecheck!(i32);
impl_julia_typecheck!(i64);
impl_julia_typecheck!(isize);
impl_julia_typecheck!(u8);
impl_julia_typecheck!(u16);
impl_julia_typecheck!(u32);
impl_julia_typecheck!(u64);
impl_julia_typecheck!(usize);
impl_julia_typecheck!(f32);
impl_julia_typecheck!(f64);
impl_julia_typecheck!(bool);
impl_julia_typecheck!(char);
impl_julia_typecheck!(*mut c_void);

unsafe impl<T: IntoJulia> Typecheck for *mut T {
    fn typecheck(t: DataType) -> bool {
        unsafe {
            let global = Global::new();
            let ptr_tname = TypeName::of_pointer(global);

            if t.type_name().wrapper_unchecked() != ptr_tname {
                return false;
            }

            let params = t.parameters().wrapper_unchecked().data();
            if params.len() != 1 {
                return false;
            }

            let inner_ty = T::julia_type(global);
            if params[0].value_unchecked() != inner_ty.value_unchecked() {
                return false;
            }

            true
        }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the `DataType` (or the `DataType` of the `Value`) is a kind, i.e. its the type of a
/// `DataType`, a `UnionAll`, a `Union` or a `Union{}`.
pub struct Type;
unsafe impl Typecheck for Type {
    fn typecheck(t: DataType) -> bool {
        t.as_value().is_kind()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the `DataType` (or the `DataType` of the `Value`) is a bits type.
pub struct Bits;
unsafe impl Typecheck for Bits {
    fn typecheck(t: DataType) -> bool {
        t.is_bits()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the `DataType` is abstract. If it's invoked through `Value::is` it will always return false.
pub struct Abstract;
unsafe impl Typecheck for Abstract {
    fn typecheck(t: DataType) -> bool {
        t.is_abstract()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the value is a `Ref`.
pub struct AbstractRef;
unsafe impl Typecheck for AbstractRef {
    fn typecheck(t: DataType) -> bool {
        unsafe {
            t.type_name().wrapper_unchecked()
                == UnionAll::ref_type(Global::new())
                    .body()
                    .value_unchecked()
                    .cast_unchecked::<DataType>()
                    .type_name()
                    .wrapper_unchecked()
        }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the value is a `VecElement`.
pub struct VecElement;
unsafe impl Typecheck for VecElement {
    fn typecheck(t: DataType) -> bool {
        unsafe { t.type_name().wrapper_unchecked() == TypeName::of_vecelement(Global::new()) }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the value is a `Type{T}`.
pub struct TypeType;
unsafe impl Typecheck for TypeType {
    fn typecheck(t: DataType) -> bool {
        unsafe {
            t.type_name().wrapper_unchecked()
                == UnionAll::type_type(Global::new())
                    .body()
                    .value_unchecked()
                    .cast_unchecked::<DataType>()
                    .type_name()
                    .wrapper_unchecked()
        }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the value is a dispatch tuple.
pub struct DispatchTuple;
unsafe impl Typecheck for DispatchTuple {
    fn typecheck(t: DataType) -> bool {
        t.is_dispatch_tuple()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a named tuple.
pub struct NamedTuple;
unsafe impl Typecheck for NamedTuple {
    fn typecheck(t: DataType) -> bool {
        unsafe { t.unwrap_non_null(Private).as_ref().name == jl_namedtuple_typename }
    }
}

impl_julia_typecheck!(DataType<'frame>, jl_datatype_type, 'frame);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the fields of a value of this type can be modified.
pub struct Mutable;
unsafe impl Typecheck for Mutable {
    fn typecheck(t: DataType) -> bool {
        t.mutable()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the datatype is a mutable datatype.
pub struct MutableDatatype;
unsafe impl Typecheck for MutableDatatype {
    fn typecheck(t: DataType) -> bool {
        DataType::typecheck(t) && t.mutable()
    }
}
/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the datatype is `Nothing`.
pub struct Nothing;
impl_julia_typecheck!(Nothing, jl_nothing_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the fields of a value of this type cannot be modified.
pub struct Immutable;
unsafe impl Typecheck for Immutable {
    fn typecheck(t: DataType) -> bool {
        !t.mutable()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// the datatype is an immutable datatype.
pub struct ImmutableDatatype;
unsafe impl Typecheck for ImmutableDatatype {
    fn typecheck(t: DataType) -> bool {
        DataType::typecheck(t) && !t.mutable()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a primitive type.
pub struct PrimitiveType;
unsafe impl Typecheck for PrimitiveType {
    fn typecheck(t: DataType) -> bool {
        unsafe {
            t.is::<Immutable>()
                && !t.unwrap_non_null(Private).as_ref().layout.is_null()
                && t.n_fields() == 0
                && t.size() > 0
        }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a struct type.
pub struct StructType;
unsafe impl Typecheck for StructType {
    fn typecheck(t: DataType) -> bool {
        !t.is_abstract() && !t.is::<PrimitiveType>()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a struct type.
pub struct Singleton;
unsafe impl Typecheck for Singleton {
    fn typecheck(t: DataType) -> bool {
        !t.instance().is_undefined()
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a slot.
pub struct Slot;
unsafe impl Typecheck for Slot {
    fn typecheck(t: DataType) -> bool {
        unsafe { t.unwrap(Private) == jl_slotnumber_type || t.unwrap(Private) == jl_typedslot_type }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a global reference.
pub struct GlobalRef;
impl_julia_typecheck!(GlobalRef, jl_globalref_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a Goto node.
pub struct GotoNode;
impl_julia_typecheck!(GotoNode, jl_gotonode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a Pi node.
pub struct PiNode;
impl_julia_typecheck!(PiNode, jl_pinode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a Phi node.
pub struct PhiNode;
impl_julia_typecheck!(PhiNode, jl_phinode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a PhiC node.
pub struct PhiCNode;
impl_julia_typecheck!(PhiCNode, jl_phicnode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is an Upsilon node.
pub struct UpsilonNode;
impl_julia_typecheck!(UpsilonNode, jl_upsilonnode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a Quote node.
pub struct QuoteNode;
impl_julia_typecheck!(QuoteNode, jl_quotenode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is an NewVar node.
pub struct NewVarNode;
impl_julia_typecheck!(NewVarNode, jl_newvarnode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a Line node.
pub struct LineNode;
impl_julia_typecheck!(LineNode, jl_linenumbernode_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is code info.
pub struct CodeInfo;
impl_julia_typecheck!(CodeInfo, jl_code_info_type);

impl_julia_typecheck!(String, jl_string_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is a pointer to data not owned by Julia.
pub struct Pointer;
unsafe impl Typecheck for Pointer {
    fn typecheck(t: DataType) -> bool {
        unsafe { t.type_name().wrapper_unchecked() == TypeName::of_pointer(Global::new()) }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is an LLVM pointer.
pub struct LLVMPointer;
unsafe impl Typecheck for LLVMPointer {
    fn typecheck(t: DataType) -> bool {
        unsafe { t.type_name().wrapper_unchecked() == TypeName::of_llvmpointer(Global::new()) }
    }
}

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// a value of this type is an intrinsic.
pub struct Intrinsic;
impl_julia_typecheck!(Intrinsic, jl_intrinsic_type);

/// A typecheck that can be used in combination with `DataType::is`. This method returns true if
/// instances of the type can be created.
pub struct Concrete;
unsafe impl Typecheck for Concrete {
    fn typecheck(t: DataType) -> bool {
        t.is_concrete_type()
    }
}
