//! Trait to check if a Rust type and a Julia type have matching layouts.
//!
//! When working with Julia values, it's always possible to access their [`DataType`]. This
//! `DataType` contains all information about the value's fields and their layout. The
//! [`ValidLayout`] trait defined in this module is used to check if a type has the same layout
//! as a given Julia type. It is implemented automatically by JlrsReflect.jl, you should not
//! implement it manually.
//!
//! [`DataType`]: crate::wrappers::ptr::datatype::DataType

use crate::wrappers::ptr::value::Value;

/// Trait used to check if a Rust type and Julia type have matching layouts. All wrappers
/// generated by JlrsReflect.jl derive this trait. In this case all fields are checked recursively
/// to determine if the value can be unboxed as that type.
pub unsafe trait ValidLayout {
    /// Check if the layout of the implementor is compatible with the layout of `ty`. This
    /// argument is a `Value` to account for the fact that a field type can be a `Union`,
    /// `UnionAll` or `Union{}`.
    fn valid_layout(ty: Value) -> bool;
}

#[doc(hidden)]
#[macro_export]
macro_rules! impl_valid_layout {
    ($type:ty, $($lt:lifetime),+) => {
        unsafe impl<$($lt),+> $crate::layout::valid_layout::ValidLayout for $type {
            fn valid_layout(v: $crate::wrappers::ptr::value::Value) -> bool {
                if let Ok(dt) = v.cast::<$crate::wrappers::ptr::datatype::DataType>() {
                    dt.is::<$type>()
                } else {
                    false
                }
            }
        }
    };
    ($t:ty) => {
        unsafe impl $crate::layout::valid_layout::ValidLayout for $t {
            fn valid_layout(v: $crate::wrappers::ptr::value::Value) -> bool {
                if let Ok(dt) =  v.cast::<$crate::wrappers::ptr::datatype::DataType>() {
                    dt.is::<$t>()
                } else {
                    false
                }
            }
        }
    }
}

impl_valid_layout!(bool);
impl_valid_layout!(char);
impl_valid_layout!(i8);
impl_valid_layout!(i16);
impl_valid_layout!(i32);
impl_valid_layout!(i64);
impl_valid_layout!(isize);
impl_valid_layout!(u8);
impl_valid_layout!(u16);
impl_valid_layout!(u32);
impl_valid_layout!(u64);
impl_valid_layout!(usize);
impl_valid_layout!(f32);
impl_valid_layout!(f64);
