#![allow(non_snake_case)]

use ::std::future::Future;
use ::std::time::Duration;

use ::authentic::{AuthenticationProtocol, AuthenticationProtocolConfigure, AuthenticError, WithAuthentication};

use crate::v1_1_4::config::{Authentication, Configuration};

pub enum AuthScheme {
    None(::authentic::reqwest::NoAuthentication),
    AccessToken(::authentic::reqwest::BearerAuthentication<::authentic::credential::TokenCredential>),
    Basic(::authentic::reqwest::BasicAuthentication<::authentic::credential::UsernamePasswordCredential>),
    JWT(::authentic::reqwest::BearerAuthentication<::authentic::credential::JsonWebTokenCredential>),
}

impl From<&Authentication> for AuthScheme {
    fn from(authentication: &Authentication) -> Self {
        match authentication {
            Authentication::None => {
                AuthScheme::None(::authentic::reqwest::NoAuthentication::new())
            }
            Authentication::AccessToken(credential) => {
                AuthScheme::AccessToken(::authentic::reqwest::BearerAuthentication::new(credential.clone()).with_auth_scheme("token"))
            }
            Authentication::Basic(credential) => {
                AuthScheme::Basic(::authentic::reqwest::BasicAuthentication::new(credential.clone()))
            }
            Authentication::JWT(credential) => {
                AuthScheme::JWT(::authentic::reqwest::BearerAuthentication::new(credential.clone()))
            }
        }
    }
}

impl AuthenticationProtocol for AuthScheme {
    type Request = ::reqwest::Request;
    type Response = ::reqwest::Response;
    type Error = ::reqwest::Error;

    fn step(&self) -> Result<Option<authentic::AuthenticationStep<Self::Request>>, AuthenticError> {
        match self {
            AuthScheme::None(scheme) => scheme.step(),
            AuthScheme::AccessToken(scheme) => scheme.step(),
            AuthScheme::Basic(scheme) => scheme.step(),
            AuthScheme::JWT(scheme) => scheme.step(),
        }
    }

    fn respond(&mut self, response: Result<Self::Response, Self::Error>) {
        match self {
            AuthScheme::None(scheme) => scheme.respond(response),
            AuthScheme::AccessToken(scheme) => scheme.respond(response),
            AuthScheme::Basic(scheme) => scheme.respond(response),
            AuthScheme::JWT(scheme) => scheme.respond(response),
        }
    }

    fn has_completed(
        &mut self,
        response: &Self::Response,
    ) -> Result<bool, AuthenticError> {
        match self {
            AuthScheme::None(scheme) => scheme.has_completed(response),
            AuthScheme::AccessToken(scheme) => scheme.has_completed(response),
            AuthScheme::Basic(scheme) => scheme.has_completed(response),
            AuthScheme::JWT(scheme) => scheme.has_completed(response),
        }
    }
}

impl AuthenticationProtocolConfigure<reqwest::Request> for AuthScheme
{
    fn configure(
        &self,
        builder: reqwest::Request,
    ) -> Result<reqwest::Request, AuthenticError> {

        match self {
            AuthScheme::None(scheme) => scheme.configure(builder),
            AuthScheme::AccessToken(scheme) => scheme.configure(builder),
            AuthScheme::Basic(scheme) => scheme.configure(builder),
            AuthScheme::JWT(scheme) => scheme.configure(builder),
        }
    }
}

pub struct Caller<Sleep, SleepFut, SleepOut>
where
    Sleep: Fn(Duration) -> SleepFut,
    SleepFut: Future<Output = SleepOut>
{
    client: ::reqwest::Client,
    config: Configuration,
    sleep: Sleep
}

impl<Sleep, SleepFut, SleepOut> Caller<Sleep, SleepFut, SleepOut>
where
    Sleep: Fn(Duration) -> SleepFut,
    SleepFut: Future<Output = SleepOut>
{
    pub fn new(
        client: ::reqwest::Client,
        config: Configuration,
        sleep: Sleep,
    ) -> Caller<Sleep, SleepFut, SleepOut> {
        Caller {
            client,
            config,
            sleep,
        }
    }

    /// GitHub API Root
    ///
    /// Get Hypermedia links to resources accessible in GitHub's REST API
    ///
    /// [API method documentation](https://docs.github.com/rest/overview/resources-in-the-rest-api#root-endpoint)
    pub async fn meta_root(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::meta_root::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::meta_root::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get the authenticated app
    ///
    /// Returns the GitHub App associated with the authentication credentials used. To see how many app installations are associated with this GitHub App, see the `installations_count` in the response. For more details about your app's installations, see the "[List installations for the authenticated app](https://docs.github.com/rest/reference/apps#list-installations-for-the-authenticated-app)" endpoint.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#get-the-authenticated-app)
    pub async fn apps_get_authenticated(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_authenticated::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_authenticated::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a GitHub App from a manifest
    ///
    /// Use this endpoint to complete the handshake necessary when implementing the [GitHub App Manifest flow](https://docs.github.com/apps/building-github-apps/creating-github-apps-from-a-manifest/). When you create a GitHub App with the manifest flow, you receive a temporary `code` used to retrieve the GitHub App's `id`, `pem` (private key), and `webhook_secret`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#create-a-github-app-from-a-manifest)
    pub async fn apps_create_from_manifest(
        &self,
        code: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_create_from_manifest::reqwest_builder(
                self.config.base_url.as_ref(),
                code,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_create_from_manifest::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a webhook configuration for an app
    ///
    /// Returns the webhook configuration for a GitHub App. For more information about configuring a webhook for your app, see "[Creating a GitHub App](/developers/apps/creating-a-github-app)."
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#get-a-webhook-configuration-for-an-app)
    pub async fn apps_get_webhook_config_for_app(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_webhook_config_for_app::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_webhook_config_for_app::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a webhook configuration for an app
    ///
    /// Updates the webhook configuration for a GitHub App. For more information about configuring a webhook for your app, see "[Creating a GitHub App](/developers/apps/creating-a-github-app)."
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#update-a-webhook-configuration-for-an-app)
    ///
    /// # Content
    ///
    /// - [`&request::apps_update_webhook_config_for_app::body::Json`](crate::v1_1_4::request::apps_update_webhook_config_for_app::body::Json)
    pub async fn apps_update_webhook_config_for_app<Content>(
        &self,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::apps_update_webhook_config_for_app::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::apps_update_webhook_config_for_app::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_update_webhook_config_for_app::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::apps_update_webhook_config_for_app::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List deliveries for an app webhook
    ///
    /// Returns a list of webhook deliveries for the webhook configured for a GitHub App.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#list-deliveries-for-an-app-webhook)
    pub async fn apps_list_webhook_deliveries(
        &self,
        per_page: ::std::option::Option<i64>,
        cursor: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_list_webhook_deliveries::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                cursor,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_list_webhook_deliveries::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a delivery for an app webhook
    ///
    /// Returns a delivery for the webhook configured for a GitHub App.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#get-a-delivery-for-an-app-webhook)
    pub async fn apps_get_webhook_delivery(
        &self,
        delivery_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_webhook_delivery::reqwest_builder(
                self.config.base_url.as_ref(),
                delivery_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_webhook_delivery::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Redeliver a delivery for an app webhook
    ///
    /// Redeliver a delivery for the webhook configured for a GitHub App.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#redeliver-a-delivery-for-an-app-webhook)
    pub async fn apps_redeliver_webhook_delivery(
        &self,
        delivery_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_redeliver_webhook_delivery::reqwest_builder(
                self.config.base_url.as_ref(),
                delivery_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_redeliver_webhook_delivery::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List installations for the authenticated app
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// The permissions the installation has are included under the `permissions` key.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#list-installations-for-the-authenticated-app)
    pub async fn apps_list_installations(
        &self,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        since: ::std::option::Option<&str>,
        outdated: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_list_installations::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                page,
                since,
                outdated,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_list_installations::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an installation for the authenticated app
    ///
    /// Enables an authenticated GitHub App to find an installation's information using the installation id.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#get-an-installation-for-the-authenticated-app)
    pub async fn apps_get_installation(
        &self,
        installation_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_installation::reqwest_builder(
                self.config.base_url.as_ref(),
                installation_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_installation::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an installation for the authenticated app
    ///
    /// Uninstalls a GitHub App on a user, organization, or business account. If you prefer to temporarily suspend an app's access to your account's resources, then we recommend the "[Suspend an app installation](https://docs.github.com/rest/reference/apps/#suspend-an-app-installation)" endpoint.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#delete-an-installation-for-the-authenticated-app)
    pub async fn apps_delete_installation(
        &self,
        installation_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_delete_installation::reqwest_builder(
                self.config.base_url.as_ref(),
                installation_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_delete_installation::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create an installation access token for an app
    ///
    /// Creates an installation access token that enables a GitHub App to make authenticated API requests for the app's installation on an organization or individual account. Installation tokens expire one hour from the time you create them. Using an expired token produces a status code of `401 - Unauthorized`, and requires creating a new installation token. By default the installation token has access to all repositories that the installation can access. To restrict the access to specific repositories, you can provide the `repository_ids` when creating the token. When you omit `repository_ids`, the response does not contain the `repositories` key.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps/#create-an-installation-access-token-for-an-app)
    ///
    /// # Content
    ///
    /// - [`&request::apps_create_installation_access_token::body::Json`](crate::v1_1_4::request::apps_create_installation_access_token::body::Json)
    pub async fn apps_create_installation_access_token<Content>(
        &self,
        installation_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::apps_create_installation_access_token::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::apps_create_installation_access_token::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_create_installation_access_token::reqwest_builder(
                self.config.base_url.as_ref(),
                installation_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::apps_create_installation_access_token::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Suspend an app installation
    ///
    /// Suspends a GitHub App on a user, organization, or business account, which blocks the app from accessing the account's resources. When a GitHub App is suspended, the app's access to the GitHub API or webhook events is blocked for that account.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#suspend-an-app-installation)
    pub async fn apps_suspend_installation(
        &self,
        installation_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_suspend_installation::reqwest_builder(
                self.config.base_url.as_ref(),
                installation_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_suspend_installation::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Unsuspend an app installation
    ///
    /// Removes a GitHub App installation suspension.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#unsuspend-an-app-installation)
    pub async fn apps_unsuspend_installation(
        &self,
        installation_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_unsuspend_installation::reqwest_builder(
                self.config.base_url.as_ref(),
                installation_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_unsuspend_installation::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List your grants
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations/), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/developers/apps/authorizing-oauth-apps#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// You can use this API to list the set of OAuth applications that have been granted access to your account. Unlike the [list your authorizations](https://docs.github.com/rest/reference/oauth-authorizations#list-your-authorizations) API, this API does not manage individual tokens. This API will return one entry for each OAuth application that has been granted access to your account, regardless of the number of tokens an application has generated for your user. The list of OAuth applications returned matches what is shown on [the application authorizations settings screen within GitHub](https://github.com/settings/applications#authorized). The `scopes` returned are the union of scopes authorized for the application. For example, if an application has one token with `repo` scope and another token with `user` scope, the grant will return `["repo", "user"]`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#list-your-grants)
    pub async fn oauth_authorizations_list_grants(
        &self,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        client_id: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_list_grants::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                page,
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::oauth_authorizations_list_grants::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a single grant
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/apps/building-oauth-apps/authorizing-oauth-apps/#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#get-a-single-grant)
    pub async fn oauth_authorizations_get_grant(
        &self,
        grant_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_get_grant::reqwest_builder(
                self.config.base_url.as_ref(),
                grant_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::oauth_authorizations_get_grant::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a grant
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations/), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/developers/apps/authorizing-oauth-apps#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations/) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// Deleting an OAuth application's grant will also delete all OAuth tokens associated with the application for your user. Once deleted, the application has no access to your account and is no longer listed on [the application authorizations settings screen within GitHub](https://github.com/settings/applications#authorized).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#delete-a-grant)
    pub async fn oauth_authorizations_delete_grant(
        &self,
        grant_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_delete_grant::reqwest_builder(
                self.config.base_url.as_ref(),
                grant_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::oauth_authorizations_delete_grant::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an app authorization
    ///
    /// OAuth application owners can revoke a grant for their OAuth application and a specific user. You must use [Basic Authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) when accessing this endpoint, using the OAuth application's `client_id` and `client_secret` as the username and password. You must also provide a valid OAuth `access_token` as an input parameter and the grant for the token's owner will be deleted.
    /// Deleting an OAuth application's grant will also delete all OAuth tokens associated with the application for the user. Once deleted, the application will have no access to the user's account and will no longer be listed on [the application authorizations settings screen within GitHub](https://github.com/settings/applications#authorized).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#delete-an-app-authorization)
    ///
    /// # Content
    ///
    /// - [`&request::apps_delete_authorization::body::Json`](crate::v1_1_4::request::apps_delete_authorization::body::Json)
    pub async fn apps_delete_authorization<Content>(
        &self,
        client_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::apps_delete_authorization::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::apps_delete_authorization::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_delete_authorization::reqwest_builder(
                self.config.base_url.as_ref(),
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::apps_delete_authorization::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check a token
    ///
    /// OAuth applications can use a special API method for checking OAuth token validity without exceeding the normal rate limits for failed login attempts. Authentication works differently with this particular endpoint. You must use [Basic Authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) to use this endpoint, where the username is the OAuth application `client_id` and the password is its `client_secret`. Invalid tokens will return `404 NOT FOUND`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#check-a-token)
    ///
    /// # Content
    ///
    /// - [`&request::apps_check_token::body::Json`](crate::v1_1_4::request::apps_check_token::body::Json)
    pub async fn apps_check_token<Content>(
        &self,
        client_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::apps_check_token::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::apps_check_token::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_check_token::reqwest_builder(
                self.config.base_url.as_ref(),
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::apps_check_token::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an app token
    ///
    /// OAuth application owners can revoke a single token for an OAuth application. You must use [Basic Authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) when accessing this endpoint, using the OAuth application's `client_id` and `client_secret` as the username and password.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#delete-an-app-token)
    ///
    /// # Content
    ///
    /// - [`&request::apps_delete_token::body::Json`](crate::v1_1_4::request::apps_delete_token::body::Json)
    pub async fn apps_delete_token<Content>(
        &self,
        client_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::apps_delete_token::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::apps_delete_token::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_delete_token::reqwest_builder(
                self.config.base_url.as_ref(),
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::apps_delete_token::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Reset a token
    ///
    /// OAuth applications can use this API method to reset a valid OAuth token without end-user involvement. Applications must save the "token" property in the response because changes take effect immediately. You must use [Basic Authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) when accessing this endpoint, using the OAuth application's `client_id` and `client_secret` as the username and password. Invalid tokens will return `404 NOT FOUND`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#reset-a-token)
    ///
    /// # Content
    ///
    /// - [`&request::apps_reset_token::body::Json`](crate::v1_1_4::request::apps_reset_token::body::Json)
    pub async fn apps_reset_token<Content>(
        &self,
        client_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::apps_reset_token::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::apps_reset_token::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_reset_token::reqwest_builder(
                self.config.base_url.as_ref(),
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::apps_reset_token::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a scoped access token
    ///
    /// Use a non-scoped user-to-server OAuth access token to create a repository scoped and/or permission scoped user-to-server OAuth access token. You can specify which repositories the token can access and which permissions are granted to the token. You must use [Basic Authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) when accessing this endpoint, using the OAuth application's `client_id` and `client_secret` as the username and password. Invalid tokens will return `404 NOT FOUND`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#create-a-scoped-access-token)
    ///
    /// # Content
    ///
    /// - [`&request::apps_scope_token::body::Json`](crate::v1_1_4::request::apps_scope_token::body::Json)
    pub async fn apps_scope_token<Content>(
        &self,
        client_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::apps_scope_token::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::apps_scope_token::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_scope_token::reqwest_builder(
                self.config.base_url.as_ref(),
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::apps_scope_token::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an app
    ///
    /// **Note**: The `:app_slug` is just the URL-friendly name of your GitHub App. You can find this on the settings page for your GitHub App (e.g., `https://github.com/settings/apps/:app_slug`).
    ///
    /// If the GitHub App you specify is public, you can access this endpoint without authenticating. If the GitHub App you specify is private, you must authenticate with a [personal access token](https://docs.github.com/articles/creating-a-personal-access-token-for-the-command-line/) or an [installation access token](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-an-installation) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps/#get-an-app)
    pub async fn apps_get_by_slug(
        &self,
        app_slug: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_by_slug::reqwest_builder(
                self.config.base_url.as_ref(),
                app_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_by_slug::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List your authorizations
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/apps/building-oauth-apps/authorizing-oauth-apps/#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#list-your-authorizations)
    pub async fn oauth_authorizations_list_authorizations(
        &self,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        client_id: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_list_authorizations::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                page,
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::oauth_authorizations_list_authorizations::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a new authorization
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/developers/apps/authorizing-oauth-apps#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// **Warning:** Apps must use the [web application flow](https://docs.github.com/apps/building-oauth-apps/authorizing-oauth-apps/#web-application-flow) to obtain OAuth tokens that work with GitHub SAML organizations. OAuth tokens created using the Authorizations API will be unable to access GitHub SAML organizations. For more information, see the [blog post](https://developer.github.com/changes/2019-11-05-deprecated-passwords-and-authorizations-api).
    ///
    /// Creates OAuth tokens using [Basic Authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication). If you have two-factor authentication setup, Basic Authentication for this endpoint requires that you use a one-time password (OTP) and your username and password instead of tokens. For more information, see "[Working with two-factor authentication](https://docs.github.com/rest/overview/other-authentication-methods#working-with-two-factor-authentication)."
    ///
    /// To create tokens for a particular OAuth application using this endpoint, you must authenticate as the user you want to create an authorization for and provide the app's client ID and secret, found on your OAuth application's settings page. If your OAuth application intends to create multiple tokens for one user, use `fingerprint` to differentiate between them.
    ///
    /// You can also create tokens on GitHub from the [personal access tokens settings](https://github.com/settings/tokens) page. Read more about these tokens in [the GitHub Help documentation](https://docs.github.com/articles/creating-an-access-token-for-command-line-use).
    ///
    /// Organizations that enforce SAML SSO require personal access tokens to be allowed. Read more about allowing tokens in [the GitHub Help documentation](https://docs.github.com/articles/about-identity-and-access-management-with-saml-single-sign-on).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#create-a-new-authorization)
    ///
    /// # Content
    ///
    /// - [`&request::oauth_authorizations_create_authorization::body::Json`](crate::v1_1_4::request::oauth_authorizations_create_authorization::body::Json)
    pub async fn oauth_authorizations_create_authorization<Content>(
        &self,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::oauth_authorizations_create_authorization::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::oauth_authorizations_create_authorization::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_create_authorization::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::oauth_authorizations_create_authorization::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get-or-create an authorization for a specific app
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations/), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/developers/apps/authorizing-oauth-apps#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// **Warning:** Apps must use the [web application flow](https://docs.github.com/apps/building-oauth-apps/authorizing-oauth-apps/#web-application-flow) to obtain OAuth tokens that work with GitHub SAML organizations. OAuth tokens created using the Authorizations API will be unable to access GitHub SAML organizations. For more information, see the [blog post](https://developer.github.com/changes/2019-11-05-deprecated-passwords-and-authorizations-api).
    ///
    /// Creates a new authorization for the specified OAuth application, only if an authorization for that application doesn't already exist for the user. The URL includes the 20 character client ID for the OAuth app that is requesting the token. It returns the user's existing authorization for the application if one is present. Otherwise, it creates and returns a new one.
    ///
    /// If you have two-factor authentication setup, Basic Authentication for this endpoint requires that you use a one-time password (OTP) and your username and password instead of tokens. For more information, see "[Working with two-factor authentication](https://docs.github.com/rest/overview/other-authentication-methods#working-with-two-factor-authentication)."
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations/), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/developers/apps/authorizing-oauth-apps#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#get-or-create-an-authorization-for-a-specific-app)
    ///
    /// # Content
    ///
    /// - [`&request::oauth_authorizations_get_or_create_authorization_for_app::body::Json`](crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app::body::Json)
    pub async fn oauth_authorizations_get_or_create_authorization_for_app<Content>(
        &self,
        client_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app::reqwest_builder(
                self.config.base_url.as_ref(),
                client_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get-or-create an authorization for a specific app and fingerprint
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations/), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/developers/apps/authorizing-oauth-apps#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// **Warning:** Apps must use the [web application flow](https://docs.github.com/apps/building-oauth-apps/authorizing-oauth-apps/#web-application-flow) to obtain OAuth tokens that work with GitHub SAML organizations. OAuth tokens created using the Authorizations API will be unable to access GitHub SAML organizations. For more information, see the [blog post](https://developer.github.com/changes/2019-11-05-deprecated-passwords-and-authorizations-api).
    ///
    /// This method will create a new authorization for the specified OAuth application, only if an authorization for that application and fingerprint do not already exist for the user. The URL includes the 20 character client ID for the OAuth app that is requesting the token. `fingerprint` is a unique string to distinguish an authorization from others created for the same client ID and user. It returns the user's existing authorization for the application if one is present. Otherwise, it creates and returns a new one.
    ///
    /// If you have two-factor authentication setup, Basic Authentication for this endpoint requires that you use a one-time password (OTP) and your username and password instead of tokens. For more information, see "[Working with two-factor authentication](https://docs.github.com/rest/overview/other-authentication-methods#working-with-two-factor-authentication)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#get-or-create-an-authorization-for-a-specific-app-and-fingerprint)
    ///
    /// # Content
    ///
    /// - [`&request::oauth_authorizations_get_or_create_authorization_for_app_and_fingerprint::body::Json`](crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app_and_fingerprint::body::Json)
    pub async fn oauth_authorizations_get_or_create_authorization_for_app_and_fingerprint<Content>(
        &self,
        client_id: &str,
        fingerprint: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app_and_fingerprint::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app_and_fingerprint::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app_and_fingerprint::reqwest_builder(
                self.config.base_url.as_ref(),
                client_id,
                fingerprint,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::oauth_authorizations_get_or_create_authorization_for_app_and_fingerprint::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a single authorization
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/apps/building-oauth-apps/authorizing-oauth-apps/#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#get-a-single-authorization)
    pub async fn oauth_authorizations_get_authorization(
        &self,
        authorization_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_get_authorization::reqwest_builder(
                self.config.base_url.as_ref(),
                authorization_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::oauth_authorizations_get_authorization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an authorization
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/apps/building-oauth-apps/authorizing-oauth-apps/#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#delete-an-authorization)
    pub async fn oauth_authorizations_delete_authorization(
        &self,
        authorization_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_delete_authorization::reqwest_builder(
                self.config.base_url.as_ref(),
                authorization_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::oauth_authorizations_delete_authorization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update an existing authorization
    ///
    /// **Deprecation Notice:** GitHub will discontinue the [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations/), which is used by integrations to create personal access tokens and OAuth tokens, and you must now create these tokens using our [web application flow](https://docs.github.com/developers/apps/authorizing-oauth-apps#web-application-flow). The [OAuth Authorizations API](https://docs.github.com/rest/reference/oauth-authorizations) will be removed on November, 13, 2020. For more information, including scheduled brownouts, see the [blog post](https://developer.github.com/changes/2020-02-14-deprecating-oauth-auth-endpoint/).
    ///
    /// If you have two-factor authentication setup, Basic Authentication for this endpoint requires that you use a one-time password (OTP) and your username and password instead of tokens. For more information, see "[Working with two-factor authentication](https://docs.github.com/rest/overview/other-authentication-methods#working-with-two-factor-authentication)."
    ///
    /// You can only send one of these scope keys at a time.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/oauth-authorizations#update-an-existing-authorization)
    ///
    /// # Content
    ///
    /// - [`&request::oauth_authorizations_update_authorization::body::Json`](crate::v1_1_4::request::oauth_authorizations_update_authorization::body::Json)
    pub async fn oauth_authorizations_update_authorization<Content>(
        &self,
        authorization_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::oauth_authorizations_update_authorization::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::oauth_authorizations_update_authorization::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::oauth_authorizations_update_authorization::reqwest_builder(
                self.config.base_url.as_ref(),
                authorization_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::oauth_authorizations_update_authorization::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get all codes of conduct
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/codes-of-conduct#get-all-codes-of-conduct)
    pub async fn codes_of_conduct_get_all_codes_of_conduct(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::codes_of_conduct_get_all_codes_of_conduct::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::codes_of_conduct_get_all_codes_of_conduct::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a code of conduct
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/codes-of-conduct#get-a-code-of-conduct)
    pub async fn codes_of_conduct_get_conduct_code(
        &self,
        key: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::codes_of_conduct_get_conduct_code::reqwest_builder(
                self.config.base_url.as_ref(),
                key,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::codes_of_conduct_get_conduct_code::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get emojis
    ///
    /// Lists all the emojis available to use on GitHub.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/emojis#get-emojis)
    pub async fn emojis_get(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::emojis_get::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::emojis_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions cache usage for an enterprise
    ///
    /// Gets the total GitHub Actions cache usage for an enterprise.
    /// The data fetched using this API is refreshed approximately every 5 minutes, so values returned from this endpoint may take at least 5 minutes to get updated.
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-github-actions-cache-usage-for-an-enterprise)
    pub async fn actions_get_actions_cache_usage_for_enterprise(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_actions_cache_usage_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_actions_cache_usage_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions permissions for an enterprise
    ///
    /// Gets the GitHub Actions permissions policy for organizations and allowed actions and reusable workflows in an enterprise.
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-github-actions-permissions-for-an-enterprise)
    pub async fn enterprise_admin_get_github_actions_permissions_enterprise(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_get_github_actions_permissions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_get_github_actions_permissions_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set GitHub Actions permissions for an enterprise
    ///
    /// Sets the GitHub Actions permissions policy for organizations and allowed actions and reusable workflows in an enterprise.
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-github-actions-permissions-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_set_github_actions_permissions_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_set_github_actions_permissions_enterprise::body::Json)
    pub async fn enterprise_admin_set_github_actions_permissions_enterprise<Content>(
        &self,
        enterprise: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_set_github_actions_permissions_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_set_github_actions_permissions_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_set_github_actions_permissions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_set_github_actions_permissions_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List selected organizations enabled for GitHub Actions in an enterprise
    ///
    /// Lists the organizations that are selected to have GitHub Actions enabled in an enterprise. To use this endpoint, the enterprise permission policy for `enabled_organizations` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an enterprise](#set-github-actions-permissions-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-selected-organizations-enabled-for-github-actions-in-an-enterprise)
    pub async fn enterprise_admin_list_selected_organizations_enabled_github_actions_enterprise(
        &self,
        enterprise: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_list_selected_organizations_enabled_github_actions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_list_selected_organizations_enabled_github_actions_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set selected organizations enabled for GitHub Actions in an enterprise
    ///
    /// Replaces the list of selected organizations that are enabled for GitHub Actions in an enterprise. To use this endpoint, the enterprise permission policy for `enabled_organizations` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an enterprise](#set-github-actions-permissions-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-selected-organizations-enabled-for-github-actions-in-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_set_selected_organizations_enabled_github_actions_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_set_selected_organizations_enabled_github_actions_enterprise::body::Json)
    pub async fn enterprise_admin_set_selected_organizations_enabled_github_actions_enterprise<Content>(
        &self,
        enterprise: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_set_selected_organizations_enabled_github_actions_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_set_selected_organizations_enabled_github_actions_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_set_selected_organizations_enabled_github_actions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_set_selected_organizations_enabled_github_actions_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Enable a selected organization for GitHub Actions in an enterprise
    ///
    /// Adds an organization to the list of selected organizations that are enabled for GitHub Actions in an enterprise. To use this endpoint, the enterprise permission policy for `enabled_organizations` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an enterprise](#set-github-actions-permissions-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#enable-a-selected-organization-for-github-actions-in-an-enterprise)
    pub async fn enterprise_admin_enable_selected_organization_github_actions_enterprise(
        &self,
        enterprise: &str,
        org_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_enable_selected_organization_github_actions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                org_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_enable_selected_organization_github_actions_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Disable a selected organization for GitHub Actions in an enterprise
    ///
    /// Removes an organization from the list of selected organizations that are enabled for GitHub Actions in an enterprise. To use this endpoint, the enterprise permission policy for `enabled_organizations` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an enterprise](#set-github-actions-permissions-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#disable-a-selected-organization-for-github-actions-in-an-enterprise)
    pub async fn enterprise_admin_disable_selected_organization_github_actions_enterprise(
        &self,
        enterprise: &str,
        org_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_disable_selected_organization_github_actions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                org_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_disable_selected_organization_github_actions_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get allowed actions and reusable workflows for an enterprise
    ///
    /// Gets the selected actions and reusable workflows that are allowed in an enterprise. To use this endpoint, the enterprise permission policy for `allowed_actions` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an enterprise](#set-github-actions-permissions-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-allowed-actions-for-an-enterprise)
    pub async fn enterprise_admin_get_allowed_actions_enterprise(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_get_allowed_actions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_get_allowed_actions_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set allowed actions and reusable workflows for an enterprise
    ///
    /// Sets the actions and reusable workflows that are allowed in an enterprise. To use this endpoint, the enterprise permission policy for `allowed_actions` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an enterprise](#set-github-actions-permissions-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-allowed-actions-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&schema::SelectedActions`](crate::v1_1_4::schema::SelectedActions)
    pub async fn enterprise_admin_set_allowed_actions_enterprise<Content>(
        &self,
        enterprise: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_set_allowed_actions_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_set_allowed_actions_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_set_allowed_actions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_set_allowed_actions_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get default workflow permissions for an enterprise
    ///
    /// Gets the default workflow permissions granted to the `GITHUB_TOKEN` when running workflows in an enterprise,
    /// as well as whether GitHub Actions can submit approving pull request reviews. For more information, see
    /// "[Enforcing a policy for workflow permissions in your enterprise](https://docs.github.com/enterprise-cloud@latest/admin/policies/enforcing-policies-for-your-enterprise/enforcing-policies-for-github-actions-in-your-enterprise#enforcing-a-policy-for-workflow-permissions-in-your-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    /// GitHub Apps must have the `enterprise_administration:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-default-workflow-permissions-for-an-enterprise)
    pub async fn actions_get_github_actions_default_workflow_permissions_enterprise(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_github_actions_default_workflow_permissions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_github_actions_default_workflow_permissions_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set default workflow permissions for an enterprise
    ///
    /// Sets the default workflow permissions granted to the `GITHUB_TOKEN` when running workflows in an enterprise, and sets
    /// whether GitHub Actions can submit approving pull request reviews. For more information, see
    /// "[Enforcing a policy for workflow permissions in your enterprise](https://docs.github.com/enterprise-cloud@latest/admin/policies/enforcing-policies-for-your-enterprise/enforcing-policies-for-github-actions-in-your-enterprise#enforcing-a-policy-for-workflow-permissions-in-your-enterprise)."
    ///
    /// You must authenticate using an access token with the `admin:enterprise` scope to use this endpoint.
    /// GitHub Apps must have the `enterprise_administration:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-default-workflow-permissions-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&schema::ActionsSetDefaultWorkflowPermissions`](crate::v1_1_4::schema::ActionsSetDefaultWorkflowPermissions)
    pub async fn actions_set_github_actions_default_workflow_permissions_enterprise<Content>(
        &self,
        enterprise: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List self-hosted runner groups for an enterprise
    ///
    /// Lists all self-hosted runner groups for an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-self-hosted-runner-groups-for-an-enterprise)
    pub async fn enterprise_admin_list_self_hosted_runner_groups_for_enterprise(
        &self,
        enterprise: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        visible_to_organization: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_list_self_hosted_runner_groups_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                per_page,
                page,
                visible_to_organization,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_list_self_hosted_runner_groups_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a self-hosted runner group for an enterprise
    ///
    /// Creates a new self-hosted runner group for an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-self-hosted-runner-group-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_create_self_hosted_runner_group_for_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_create_self_hosted_runner_group_for_enterprise::body::Json)
    pub async fn enterprise_admin_create_self_hosted_runner_group_for_enterprise<Content>(
        &self,
        enterprise: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_create_self_hosted_runner_group_for_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_create_self_hosted_runner_group_for_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_create_self_hosted_runner_group_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_create_self_hosted_runner_group_for_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a self-hosted runner group for an enterprise
    ///
    /// Gets a specific self-hosted runner group for an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-self-hosted-runner-group-for-an-enterprise)
    pub async fn enterprise_admin_get_self_hosted_runner_group_for_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_get_self_hosted_runner_group_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_get_self_hosted_runner_group_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a self-hosted runner group from an enterprise
    ///
    /// Deletes a self-hosted runner group for an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-a-self-hosted-runner-group-from-an-enterprise)
    pub async fn enterprise_admin_delete_self_hosted_runner_group_from_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_delete_self_hosted_runner_group_from_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_delete_self_hosted_runner_group_from_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a self-hosted runner group for an enterprise
    ///
    /// Updates the `name` and `visibility` of a self-hosted runner group in an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#update-a-self-hosted-runner-group-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_update_self_hosted_runner_group_for_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_update_self_hosted_runner_group_for_enterprise::body::Json)
    pub async fn enterprise_admin_update_self_hosted_runner_group_for_enterprise<Content>(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_update_self_hosted_runner_group_for_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_update_self_hosted_runner_group_for_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_update_self_hosted_runner_group_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_update_self_hosted_runner_group_for_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization access to a self-hosted runner group in an enterprise
    ///
    /// Lists the organizations with access to a self-hosted runner group.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-organization-access-to-a-self-hosted-runner-group-in-a-enterprise)
    pub async fn enterprise_admin_list_org_access_to_self_hosted_runner_group_in_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_list_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_list_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set organization access for a self-hosted runner group in an enterprise
    ///
    /// Replaces the list of organizations that have access to a self-hosted runner configured in an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-organization-access-to-a-self-hosted-runner-group-in-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_set_org_access_to_self_hosted_runner_group_in_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_set_org_access_to_self_hosted_runner_group_in_enterprise::body::Json)
    pub async fn enterprise_admin_set_org_access_to_self_hosted_runner_group_in_enterprise<Content>(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_set_org_access_to_self_hosted_runner_group_in_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_set_org_access_to_self_hosted_runner_group_in_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_set_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_set_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add organization access to a self-hosted runner group in an enterprise
    ///
    /// Adds an organization to the list of selected organizations that can access a self-hosted runner group. The runner group must have `visibility` set to `selected`. For more information, see "[Create a self-hosted runner group for an enterprise](#create-a-self-hosted-runner-group-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-organization-access-to-a-self-hosted-runner-group-in-an-enterprise)
    pub async fn enterprise_admin_add_org_access_to_self_hosted_runner_group_in_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        org_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_add_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                org_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_add_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove organization access to a self-hosted runner group in an enterprise
    ///
    /// Removes an organization from the list of selected organizations that can access a self-hosted runner group. The runner group must have `visibility` set to `selected`. For more information, see "[Create a self-hosted runner group for an enterprise](#create-a-self-hosted-runner-group-for-an-enterprise)."
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-organization-access-to-a-self-hosted-runner-group-in-an-enterprise)
    pub async fn enterprise_admin_remove_org_access_to_self_hosted_runner_group_in_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        org_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_remove_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                org_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_remove_org_access_to_self_hosted_runner_group_in_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List self-hosted runners in a group for an enterprise
    ///
    /// Lists the self-hosted runners that are in a specific enterprise group.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-self-hosted-runners-in-a-group-for-an-enterprise)
    pub async fn enterprise_admin_list_self_hosted_runners_in_group_for_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_list_self_hosted_runners_in_group_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_list_self_hosted_runners_in_group_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set self-hosted runners in a group for an enterprise
    ///
    /// Replaces the list of self-hosted runners that are part of an enterprise runner group.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-self-hosted-runners-in-a-group-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_set_self_hosted_runners_in_group_for_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_set_self_hosted_runners_in_group_for_enterprise::body::Json)
    pub async fn enterprise_admin_set_self_hosted_runners_in_group_for_enterprise<Content>(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_set_self_hosted_runners_in_group_for_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_set_self_hosted_runners_in_group_for_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_set_self_hosted_runners_in_group_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_set_self_hosted_runners_in_group_for_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add a self-hosted runner to a group for an enterprise
    ///
    /// Adds a self-hosted runner to a runner group configured in an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise`
    /// scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-a-self-hosted-runner-to-a-group-for-an-enterprise)
    pub async fn enterprise_admin_add_self_hosted_runner_to_group_for_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_add_self_hosted_runner_to_group_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_add_self_hosted_runner_to_group_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a self-hosted runner from a group for an enterprise
    ///
    /// Removes a self-hosted runner from a group configured in an enterprise. The runner is then returned to the default group.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-a-self-hosted-runner-from-a-group-for-an-enterprise)
    pub async fn enterprise_admin_remove_self_hosted_runner_from_group_for_enterprise(
        &self,
        enterprise: &str,
        runner_group_id: i64,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_remove_self_hosted_runner_from_group_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_group_id,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_remove_self_hosted_runner_from_group_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List self-hosted runners for an enterprise
    ///
    /// Lists all self-hosted runners configured for an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-self-hosted-runners-for-an-enterprise)
    pub async fn enterprise_admin_list_self_hosted_runners_for_enterprise(
        &self,
        enterprise: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_list_self_hosted_runners_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_list_self_hosted_runners_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List runner applications for an enterprise
    ///
    /// Lists binaries for the runner application that you can download and run.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-runner-applications-for-an-enterprise)
    pub async fn enterprise_admin_list_runner_applications_for_enterprise(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_list_runner_applications_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_list_runner_applications_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a registration token for an enterprise
    ///
    /// Returns a token that you can pass to the `config` script. The token expires after one hour.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// #### Example using registration token
    ///
    /// Configure your self-hosted runner, replacing `TOKEN` with the registration token provided by this endpoint.
    ///
    /// ```text
    /// ./config.sh --url https://github.com/enterprises/octo-enterprise --token TOKEN
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-registration-token-for-an-enterprise)
    pub async fn enterprise_admin_create_registration_token_for_enterprise(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_create_registration_token_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_create_registration_token_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a remove token for an enterprise
    ///
    /// Returns a token that you can pass to the `config` script to remove a self-hosted runner from an enterprise. The token expires after one hour.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// #### Example using remove token
    ///
    /// To remove your self-hosted runner from an enterprise, replace `TOKEN` with the remove token provided by this
    /// endpoint.
    ///
    /// ```text
    /// ./config.sh remove --token TOKEN
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-remove-token-for-an-enterprise)
    pub async fn enterprise_admin_create_remove_token_for_enterprise(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_create_remove_token_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_create_remove_token_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a self-hosted runner for an enterprise
    ///
    /// Gets a specific self-hosted runner configured in an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-self-hosted-runner-for-an-enterprise)
    pub async fn enterprise_admin_get_self_hosted_runner_for_enterprise(
        &self,
        enterprise: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_get_self_hosted_runner_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_get_self_hosted_runner_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a self-hosted runner from an enterprise
    ///
    /// Forces the removal of a self-hosted runner from an enterprise. You can use this endpoint to completely remove the runner when the machine you were using no longer exists.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-self-hosted-runner-from-an-enterprise)
    pub async fn enterprise_admin_delete_self_hosted_runner_from_enterprise(
        &self,
        enterprise: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_delete_self_hosted_runner_from_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_delete_self_hosted_runner_from_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List labels for a self-hosted runner for an enterprise
    ///
    /// Lists all labels for a self-hosted runner configured in an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-labels-for-a-self-hosted-runner-for-an-enterprise)
    pub async fn enterprise_admin_list_labels_for_self_hosted_runner_for_enterprise(
        &self,
        enterprise: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_list_labels_for_self_hosted_runner_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_list_labels_for_self_hosted_runner_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set custom labels for a self-hosted runner for an enterprise
    ///
    /// Remove all previous custom labels and set the new custom labels for a specific
    /// self-hosted runner configured in an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-custom-labels-for-a-self-hosted-runner-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_set_custom_labels_for_self_hosted_runner_for_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_set_custom_labels_for_self_hosted_runner_for_enterprise::body::Json)
    pub async fn enterprise_admin_set_custom_labels_for_self_hosted_runner_for_enterprise<Content>(
        &self,
        enterprise: &str,
        runner_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_set_custom_labels_for_self_hosted_runner_for_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_set_custom_labels_for_self_hosted_runner_for_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_set_custom_labels_for_self_hosted_runner_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_set_custom_labels_for_self_hosted_runner_for_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add custom labels to a self-hosted runner for an enterprise
    ///
    /// Add custom labels to a self-hosted runner configured in an enterprise.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-custom-labels-to-a-self-hosted-runner-for-an-enterprise)
    ///
    /// # Content
    ///
    /// - [`&request::enterprise_admin_add_custom_labels_to_self_hosted_runner_for_enterprise::body::Json`](crate::v1_1_4::request::enterprise_admin_add_custom_labels_to_self_hosted_runner_for_enterprise::body::Json)
    pub async fn enterprise_admin_add_custom_labels_to_self_hosted_runner_for_enterprise<Content>(
        &self,
        enterprise: &str,
        runner_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::enterprise_admin_add_custom_labels_to_self_hosted_runner_for_enterprise::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::enterprise_admin_add_custom_labels_to_self_hosted_runner_for_enterprise::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_add_custom_labels_to_self_hosted_runner_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::enterprise_admin_add_custom_labels_to_self_hosted_runner_for_enterprise::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove all custom labels from a self-hosted runner for an enterprise
    ///
    /// Remove all custom labels from a self-hosted runner configured in an
    /// enterprise. Returns the remaining read-only labels from the runner.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-all-custom-labels-from-a-self-hosted-runner-for-an-enterprise)
    pub async fn enterprise_admin_remove_all_custom_labels_from_self_hosted_runner_for_enterprise(
        &self,
        enterprise: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_remove_all_custom_labels_from_self_hosted_runner_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_remove_all_custom_labels_from_self_hosted_runner_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a custom label from a self-hosted runner for an enterprise
    ///
    /// Remove a custom label from a self-hosted runner configured
    /// in an enterprise. Returns the remaining labels from the runner.
    ///
    /// This endpoint returns a `404 Not Found` status if the custom label is not
    /// present on the runner.
    ///
    /// You must authenticate using an access token with the `manage_runners:enterprise` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-a-custom-label-from-a-self-hosted-runner-for-an-enterprise)
    pub async fn enterprise_admin_remove_custom_label_from_self_hosted_runner_for_enterprise(
        &self,
        enterprise: &str,
        runner_id: i64,
        name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_remove_custom_label_from_self_hosted_runner_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                runner_id,
                name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_remove_custom_label_from_self_hosted_runner_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get the audit log for an enterprise
    ///
    /// Gets the audit log for an enterprise. To use this endpoint, you must be an enterprise admin, and you must use an access token with the `admin:enterprise` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/enterprise-admin#get-the-audit-log-for-an-enterprise)
    #[allow(clippy::too_many_arguments)]
    pub async fn enterprise_admin_get_audit_log(
        &self,
        enterprise: &str,
        phrase: ::std::option::Option<&str>,
        include: ::std::option::Option<&str>,
        after: ::std::option::Option<&str>,
        before: ::std::option::Option<&str>,
        order: ::std::option::Option<&str>,
        page: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::enterprise_admin_get_audit_log::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                phrase,
                include,
                after,
                before,
                order,
                page,
                per_page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::enterprise_admin_get_audit_log::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List secret scanning alerts for an enterprise
    ///
    /// Lists secret scanning alerts for eligible repositories in an enterprise, from newest to oldest.
    /// To use this endpoint, you must be a member of the enterprise, and you must use an access token with the `repo` scope or `security_events` scope. Alerts are only returned for organizations in the enterprise for which you are an organization owner or a [security manager](https://docs.github.com/organizations/managing-peoples-access-to-your-organization-with-roles/managing-security-managers-in-your-organization).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/secret-scanning#list-secret-scanning-alerts-for-an-enterprise)
    #[allow(clippy::too_many_arguments)]
    pub async fn secret_scanning_list_alerts_for_enterprise(
        &self,
        enterprise: &str,
        state: ::std::option::Option<&str>,
        secret_type: ::std::option::Option<&str>,
        resolution: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        before: ::std::option::Option<&str>,
        after: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::secret_scanning_list_alerts_for_enterprise::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                state,
                secret_type,
                resolution,
                per_page,
                before,
                after,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::secret_scanning_list_alerts_for_enterprise::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions billing for an enterprise
    ///
    /// Gets the summary of the free and paid GitHub Actions minutes used.
    ///
    /// Paid minutes only apply to workflows in private repositories that use GitHub-hosted runners. Minutes used is listed for each GitHub-hosted runner operating system. Any job re-runs are also included in the usage. The usage does not include the multiplier for macOS and Windows runners and is not rounded up to the nearest whole minute. For more information, see "[Managing billing for GitHub Actions](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-actions)".
    ///
    /// The authenticated user must be an enterprise admin.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#get-github-actions-billing-for-an-enterprise)
    pub async fn billing_get_github_actions_billing_ghe(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_github_actions_billing_ghe::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_github_actions_billing_ghe::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Advanced Security active committers for an enterprise
    ///
    /// Gets the GitHub Advanced Security active committers for an enterprise per repository.
    /// Each distinct user login across all repositories is counted as a single Advanced Security seat, so the total_advanced_security_committers is not the sum of active_users for each repository.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#export-advanced-security-active-committers-data-for-enterprise)
    pub async fn billing_get_github_advanced_security_billing_ghe(
        &self,
        enterprise: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_github_advanced_security_billing_ghe::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_github_advanced_security_billing_ghe::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Packages billing for an enterprise
    ///
    /// Gets the free and paid storage used for GitHub Packages in gigabytes.
    ///
    /// Paid minutes only apply to packages stored for private repositories. For more information, see "[Managing billing for GitHub Packages](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-packages)."
    ///
    /// The authenticated user must be an enterprise admin.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#get-github-packages-billing-for-an-enterprise)
    pub async fn billing_get_github_packages_billing_ghe(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_github_packages_billing_ghe::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_github_packages_billing_ghe::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get shared storage billing for an enterprise
    ///
    /// Gets the estimated paid and estimated total storage used for GitHub Actions and GitHub Packages.
    ///
    /// Paid minutes only apply to packages stored for private repositories. For more information, see "[Managing billing for GitHub Packages](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-packages)."
    ///
    /// The authenticated user must be an enterprise admin.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#get-shared-storage-billing-for-an-enterprise)
    pub async fn billing_get_shared_storage_billing_ghe(
        &self,
        enterprise: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_shared_storage_billing_ghe::reqwest_builder(
                self.config.base_url.as_ref(),
                enterprise,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_shared_storage_billing_ghe::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List public events
    ///
    /// We delay the public events feed by five minutes, which means the most recent event returned by the public events API actually occurred at least five minutes ago.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#list-public-events)
    pub async fn activity_list_public_events(
        &self,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_list_public_events::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_list_public_events::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get feeds
    ///
    /// GitHub provides several timeline resources in [Atom](http://en.wikipedia.org/wiki/Atom_(standard)) format. The Feeds API lists all the feeds available to the authenticated user:
    ///
    /// *   **Timeline**: The GitHub global public timeline
    /// *   **User**: The public timeline for any user, using [URI template](https://docs.github.com/rest/overview/resources-in-the-rest-api#hypermedia)
    /// *   **Current user public**: The public timeline for the authenticated user
    /// *   **Current user**: The private timeline for the authenticated user
    /// *   **Current user actor**: The private timeline for activity created by the authenticated user
    /// *   **Current user organizations**: The private timeline for the organizations the authenticated user is a member of.
    /// *   **Security advisories**: A collection of public announcements that provide information about security-related vulnerabilities in software on GitHub.
    ///
    /// **Note**: Private feeds are only returned when [authenticating via Basic Auth](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) since current feed URIs use the older, non revocable auth tokens.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#get-feeds)
    pub async fn activity_get_feeds(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_get_feeds::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_get_feeds::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List gists for the authenticated user
    ///
    /// Lists the authenticated user's gists or if called anonymously, this endpoint returns all public gists:
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#list-gists-for-the-authenticated-user)
    pub async fn gists_list(
        &self,
        since: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_list::reqwest_builder(
                self.config.base_url.as_ref(),
                since,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_list::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a gist
    ///
    /// Allows you to add a new gist with one or more files.
    ///
    /// **Note:** Don't name your files "gistfile" with a numerical suffix. This is the format of the automatic naming scheme that Gist uses internally.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#create-a-gist)
    ///
    /// # Content
    ///
    /// - [`&request::gists_create::body::Json`](crate::v1_1_4::request::gists_create::body::Json)
    pub async fn gists_create<Content>(
        &self,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::gists_create::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::gists_create::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_create::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::gists_create::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List public gists
    ///
    /// List public gists sorted by most recently updated to least recently updated.
    ///
    /// Note: With [pagination](https://docs.github.com/rest/overview/resources-in-the-rest-api#pagination), you can fetch up to 3000 gists. For example, you can fetch 100 pages with 30 gists per page or 30 pages with 100 gists per page.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#list-public-gists)
    pub async fn gists_list_public(
        &self,
        since: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_list_public::reqwest_builder(
                self.config.base_url.as_ref(),
                since,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_list_public::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List starred gists
    ///
    /// List the authenticated user's starred gists:
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#list-starred-gists)
    pub async fn gists_list_starred(
        &self,
        since: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_list_starred::reqwest_builder(
                self.config.base_url.as_ref(),
                since,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_list_starred::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a gist
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#get-a-gist)
    pub async fn gists_get(
        &self,
        gist_id: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_get::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a gist
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#delete-a-gist)
    pub async fn gists_delete(
        &self,
        gist_id: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_delete::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_delete::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a gist
    ///
    /// Allows you to update or delete a gist file and rename gist files. Files from the previous version of the gist that aren't explicitly changed during an edit are unchanged.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists/#update-a-gist)
    ///
    /// # Content
    ///
    /// - [`&::std::option::Option<request::gists_update::body::Json>`](crate::v1_1_4::request::gists_update::body::Json)
    pub async fn gists_update<Content>(
        &self,
        gist_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::gists_update::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::gists_update::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_update::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::gists_update::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List gist comments
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#list-gist-comments)
    pub async fn gists_list_comments(
        &self,
        gist_id: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_list_comments::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_list_comments::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a gist comment
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#create-a-gist-comment)
    ///
    /// # Content
    ///
    /// - [`&request::gists_create_comment::body::Json`](crate::v1_1_4::request::gists_create_comment::body::Json)
    pub async fn gists_create_comment<Content>(
        &self,
        gist_id: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::gists_create_comment::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::gists_create_comment::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_create_comment::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::gists_create_comment::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a gist comment
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#get-a-gist-comment)
    pub async fn gists_get_comment(
        &self,
        gist_id: &str,
        comment_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_get_comment::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                comment_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_get_comment::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a gist comment
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#delete-a-gist-comment)
    pub async fn gists_delete_comment(
        &self,
        gist_id: &str,
        comment_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_delete_comment::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                comment_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_delete_comment::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a gist comment
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#update-a-gist-comment)
    ///
    /// # Content
    ///
    /// - [`&request::gists_update_comment::body::Json`](crate::v1_1_4::request::gists_update_comment::body::Json)
    pub async fn gists_update_comment<Content>(
        &self,
        gist_id: &str,
        comment_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::gists_update_comment::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::gists_update_comment::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_update_comment::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                comment_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::gists_update_comment::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List gist commits
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#list-gist-commits)
    pub async fn gists_list_commits(
        &self,
        gist_id: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_list_commits::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_list_commits::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List gist forks
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#list-gist-forks)
    pub async fn gists_list_forks(
        &self,
        gist_id: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_list_forks::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_list_forks::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Fork a gist
    ///
    /// **Note**: This was previously `/gists/:gist_id/fork`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#fork-a-gist)
    pub async fn gists_fork(
        &self,
        gist_id: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_fork::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_fork::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check if a gist is starred
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#check-if-a-gist-is-starred)
    pub async fn gists_check_is_starred(
        &self,
        gist_id: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_check_is_starred::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_check_is_starred::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Star a gist
    ///
    /// Note that you'll need to set `Content-Length` to zero when calling out to this endpoint. For more information, see "[HTTP verbs](https://docs.github.com/rest/overview/resources-in-the-rest-api#http-verbs)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#star-a-gist)
    pub async fn gists_star(
        &self,
        gist_id: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_star::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_star::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Unstar a gist
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#unstar-a-gist)
    pub async fn gists_unstar(
        &self,
        gist_id: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_unstar::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_unstar::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a gist revision
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gists#get-a-gist-revision)
    pub async fn gists_get_revision(
        &self,
        gist_id: &str,
        sha: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gists_get_revision::reqwest_builder(
                self.config.base_url.as_ref(),
                gist_id,
                sha,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gists_get_revision::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get all gitignore templates
    ///
    /// List all templates available to pass as an option when [creating a repository](https://docs.github.com/rest/reference/repos#create-a-repository-for-the-authenticated-user).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gitignore#get-all-gitignore-templates)
    pub async fn gitignore_get_all_templates(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gitignore_get_all_templates::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gitignore_get_all_templates::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a gitignore template
    ///
    /// The API also allows fetching the source of a single template.
    /// Use the raw [media type](https://docs.github.com/rest/overview/media-types/) to get the raw contents.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/gitignore#get-a-gitignore-template)
    pub async fn gitignore_get_template(
        &self,
        name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::gitignore_get_template::reqwest_builder(
                self.config.base_url.as_ref(),
                name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::gitignore_get_template::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List repositories accessible to the app installation
    ///
    /// List repositories that an app installation can access.
    ///
    /// You must use an [installation access token](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-an-installation) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#list-repositories-accessible-to-the-app-installation)
    pub async fn apps_list_repos_accessible_to_installation(
        &self,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_list_repos_accessible_to_installation::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_list_repos_accessible_to_installation::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Revoke an installation access token
    ///
    /// Revokes the installation token you're using to authenticate as an installation and access this endpoint.
    ///
    /// Once an installation token is revoked, the token is invalidated and cannot be used. Other endpoints that require the revoked installation token must have a new installation token to work. You can create a new token using the "[Create an installation access token for an app](https://docs.github.com/rest/reference/apps#create-an-installation-access-token-for-an-app)" endpoint.
    ///
    /// You must use an [installation access token](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-an-installation) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#revoke-an-installation-access-token)
    pub async fn apps_revoke_installation_access_token(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_revoke_installation_access_token::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_revoke_installation_access_token::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List issues assigned to the authenticated user
    ///
    /// List issues assigned to the authenticated user across all visible repositories including owned repositories, member
    /// repositories, and organization repositories. You can use the `filter` query parameter to fetch issues that are not
    /// necessarily assigned to you.
    ///
    ///
    /// **Note**: GitHub's REST API v3 considers every pull request an issue, but not every issue is a pull request. For this
    /// reason, "Issues" endpoints may return both issues and pull requests in the response. You can identify pull requests by
    /// the `pull_request` key. Be aware that the `id` of a pull request returned from "Issues" endpoints will be an _issue id_. To find out the pull
    /// request id, use the "[List pull requests](https://docs.github.com/rest/reference/pulls#list-pull-requests)" endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/issues#list-issues-assigned-to-the-authenticated-user)
    #[allow(clippy::too_many_arguments)]
    pub async fn issues_list(
        &self,
        filter: &crate::types::IssueFilter<'_>,
        sort: &crate::types::Sort<'_>,
        collab: ::std::option::Option<bool>,
        orgs: ::std::option::Option<bool>,
        owned: ::std::option::Option<bool>,
        pulls: ::std::option::Option<bool>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let (sort, direction) = sort.extract();
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::issues_list::reqwest_builder(
                self.config.base_url.as_ref(),
                filter.filter,
                filter.state,
                filter.labels,
                sort,
                direction,
                filter.since,
                collab,
                orgs,
                owned,
                pulls,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::issues_list::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get all commonly used licenses
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/licenses#get-all-commonly-used-licenses)
    pub async fn licenses_get_all_commonly_used(
        &self,
        featured: ::std::option::Option<bool>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::licenses_get_all_commonly_used::reqwest_builder(
                self.config.base_url.as_ref(),
                featured,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::licenses_get_all_commonly_used::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a license
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/licenses#get-a-license)
    pub async fn licenses_get(
        &self,
        license: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::licenses_get::reqwest_builder(
                self.config.base_url.as_ref(),
                license,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::licenses_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Render a Markdown document
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/markdown#render-a-markdown-document)
    ///
    /// # Content
    ///
    /// - [`&request::markdown_render::body::Json`](crate::v1_1_4::request::markdown_render::body::Json)
    pub async fn markdown_render<Content>(
        &self,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::markdown_render::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::markdown_render::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::markdown_render::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::markdown_render::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Render a Markdown document in raw mode
    ///
    /// You must send Markdown as plain text (using a `Content-Type` header of `text/plain` or `text/x-markdown`) to this endpoint, rather than using JSON format. In raw mode, [GitHub Flavored Markdown](https://github.github.com/gfm/) is not supported and Markdown will be rendered in plain format like a README.md file. Markdown content must be 400 KB or less.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/markdown#render-a-markdown-document-in-raw-mode)
    ///
    /// # Content
    ///
    /// - [`&Cow<'static, str>`](::std::borrow::Cow)
    pub async fn markdown_render_raw<Content>(
        &self,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::markdown_render_raw::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::markdown_render_raw::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::markdown_render_raw::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::markdown_render_raw::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a subscription plan for an account
    ///
    /// Shows whether the user or organization account actively subscribes to a plan listed by the authenticated GitHub App. When someone submits a plan change that won't be processed until the end of their billing cycle, you will also see the upcoming pending change.
    ///
    /// GitHub Apps must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint. OAuth Apps must use [basic authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) with their client ID and client secret to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#get-a-subscription-plan-for-an-account)
    pub async fn apps_get_subscription_plan_for_account(
        &self,
        account_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_subscription_plan_for_account::reqwest_builder(
                self.config.base_url.as_ref(),
                account_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_subscription_plan_for_account::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List plans
    ///
    /// Lists all plans that are part of your GitHub Marketplace listing.
    ///
    /// GitHub Apps must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint. OAuth Apps must use [basic authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) with their client ID and client secret to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#list-plans)
    pub async fn apps_list_plans(
        &self,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_list_plans::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_list_plans::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List accounts for a plan
    ///
    /// Returns user and organization accounts associated with the specified plan, including free plans. For per-seat pricing, you see the list of accounts that have purchased the plan, including the number of seats purchased. When someone submits a plan change that won't be processed until the end of their billing cycle, you will also see the upcoming pending change.
    ///
    /// GitHub Apps must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint. OAuth Apps must use [basic authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) with their client ID and client secret to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#list-accounts-for-a-plan)
    pub async fn apps_list_accounts_for_plan(
        &self,
        plan_id: i64,
        sort: &crate::types::Sort<'_>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let (sort, direction) = sort.extract();
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_list_accounts_for_plan::reqwest_builder(
                self.config.base_url.as_ref(),
                plan_id,
                sort,
                direction,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_list_accounts_for_plan::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a subscription plan for an account (stubbed)
    ///
    /// Shows whether the user or organization account actively subscribes to a plan listed by the authenticated GitHub App. When someone submits a plan change that won't be processed until the end of their billing cycle, you will also see the upcoming pending change.
    ///
    /// GitHub Apps must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint. OAuth Apps must use [basic authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) with their client ID and client secret to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#get-a-subscription-plan-for-an-account-stubbed)
    pub async fn apps_get_subscription_plan_for_account_stubbed(
        &self,
        account_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_subscription_plan_for_account_stubbed::reqwest_builder(
                self.config.base_url.as_ref(),
                account_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_subscription_plan_for_account_stubbed::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List plans (stubbed)
    ///
    /// Lists all plans that are part of your GitHub Marketplace listing.
    ///
    /// GitHub Apps must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint. OAuth Apps must use [basic authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) with their client ID and client secret to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#list-plans-stubbed)
    pub async fn apps_list_plans_stubbed(
        &self,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_list_plans_stubbed::reqwest_builder(
                self.config.base_url.as_ref(),
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_list_plans_stubbed::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List accounts for a plan (stubbed)
    ///
    /// Returns repository and organization accounts associated with the specified plan, including free plans. For per-seat pricing, you see the list of accounts that have purchased the plan, including the number of seats purchased. When someone submits a plan change that won't be processed until the end of their billing cycle, you will also see the upcoming pending change.
    ///
    /// GitHub Apps must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint. OAuth Apps must use [basic authentication](https://docs.github.com/rest/overview/other-authentication-methods#basic-authentication) with their client ID and client secret to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#list-accounts-for-a-plan-stubbed)
    pub async fn apps_list_accounts_for_plan_stubbed(
        &self,
        plan_id: i64,
        sort: &crate::types::Sort<'_>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let (sort, direction) = sort.extract();
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_list_accounts_for_plan_stubbed::reqwest_builder(
                self.config.base_url.as_ref(),
                plan_id,
                sort,
                direction,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_list_accounts_for_plan_stubbed::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub meta information
    ///
    /// Returns meta information about GitHub, including a list of GitHub's IP addresses. For more information, see "[About GitHub's IP addresses](https://docs.github.com/articles/about-github-s-ip-addresses/)."
    ///
    /// **Note:** The IP addresses shown in the documentation's response are only example values. You must always query the API directly to get the latest list of IP addresses.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/meta#get-github-meta-information)
    pub async fn meta_get(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::meta_get::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::meta_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List public events for a network of repositories
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#list-public-events-for-a-network-of-repositories)
    pub async fn activity_list_public_events_for_repo_network(
        &self,
        owner: &str,
        repo: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_list_public_events_for_repo_network::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_list_public_events_for_repo_network::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List notifications for the authenticated user
    ///
    /// List all notifications for the current user, sorted by most recently updated.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#list-notifications-for-the-authenticated-user)
    pub async fn activity_list_notifications_for_authenticated_user(
        &self,
        all: ::std::option::Option<bool>,
        participating: ::std::option::Option<bool>,
        since: ::std::option::Option<&str>,
        before: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_list_notifications_for_authenticated_user::reqwest_builder(
                self.config.base_url.as_ref(),
                all,
                participating,
                since,
                before,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_list_notifications_for_authenticated_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Mark notifications as read
    ///
    /// Marks all notifications as "read" removes it from the [default view on GitHub](https://github.com/notifications). If the number of notifications is too large to complete in one request, you will receive a `202 Accepted` status and GitHub will run an asynchronous process to mark notifications as "read." To check whether any "unread" notifications remain, you can use the [List notifications for the authenticated user](https://docs.github.com/rest/reference/activity#list-notifications-for-the-authenticated-user) endpoint and pass the query parameter `all=false`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#mark-notifications-as-read)
    ///
    /// # Content
    ///
    /// - [`&request::activity_mark_notifications_as_read::body::Json`](crate::v1_1_4::request::activity_mark_notifications_as_read::body::Json)
    pub async fn activity_mark_notifications_as_read<Content>(
        &self,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::activity_mark_notifications_as_read::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::activity_mark_notifications_as_read::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_mark_notifications_as_read::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::activity_mark_notifications_as_read::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a thread
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#get-a-thread)
    pub async fn activity_get_thread(
        &self,
        thread_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_get_thread::reqwest_builder(
                self.config.base_url.as_ref(),
                thread_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_get_thread::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Mark a thread as read
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#mark-a-thread-as-read)
    pub async fn activity_mark_thread_as_read(
        &self,
        thread_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_mark_thread_as_read::reqwest_builder(
                self.config.base_url.as_ref(),
                thread_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_mark_thread_as_read::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a thread subscription for the authenticated user
    ///
    /// This checks to see if the current user is subscribed to a thread. You can also [get a repository subscription](https://docs.github.com/rest/reference/activity#get-a-repository-subscription).
    ///
    /// Note that subscriptions are only generated if a user is participating in a conversation--for example, they've replied to the thread, were **@mentioned**, or manually subscribe to a thread.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#get-a-thread-subscription-for-the-authenticated-user)
    pub async fn activity_get_thread_subscription_for_authenticated_user(
        &self,
        thread_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_get_thread_subscription_for_authenticated_user::reqwest_builder(
                self.config.base_url.as_ref(),
                thread_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_get_thread_subscription_for_authenticated_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set a thread subscription
    ///
    /// If you are watching a repository, you receive notifications for all threads by default. Use this endpoint to ignore future notifications for threads until you comment on the thread or get an **@mention**.
    ///
    /// You can also use this endpoint to subscribe to threads that you are currently not receiving notifications for or to subscribed to threads that you have previously ignored.
    ///
    /// Unsubscribing from a conversation in a repository that you are not watching is functionally equivalent to the [Delete a thread subscription](https://docs.github.com/rest/reference/activity#delete-a-thread-subscription) endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#set-a-thread-subscription)
    ///
    /// # Content
    ///
    /// - [`&request::activity_set_thread_subscription::body::Json`](crate::v1_1_4::request::activity_set_thread_subscription::body::Json)
    pub async fn activity_set_thread_subscription<Content>(
        &self,
        thread_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::activity_set_thread_subscription::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::activity_set_thread_subscription::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_set_thread_subscription::reqwest_builder(
                self.config.base_url.as_ref(),
                thread_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::activity_set_thread_subscription::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a thread subscription
    ///
    /// Mutes all future notifications for a conversation until you comment on the thread or get an **@mention**. If you are watching the repository of the thread, you will still receive notifications. To ignore future notifications for a repository you are watching, use the [Set a thread subscription](https://docs.github.com/rest/reference/activity#set-a-thread-subscription) endpoint and set `ignore` to `true`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#delete-a-thread-subscription)
    pub async fn activity_delete_thread_subscription(
        &self,
        thread_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_delete_thread_subscription::reqwest_builder(
                self.config.base_url.as_ref(),
                thread_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_delete_thread_subscription::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get Octocat
    ///
    /// Get the octocat as ASCII art
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/meta#get-octocat)
    pub async fn meta_get_octocat(
        &self,
        s: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::meta_get_octocat::reqwest_builder(
                self.config.base_url.as_ref(),
                s,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::meta_get_octocat::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organizations
    ///
    /// Lists all organizations, in the order that they were created on GitHub.
    ///
    /// **Note:** Pagination is powered exclusively by the `since` parameter. Use the [Link header](https://docs.github.com/rest/overview/resources-in-the-rest-api#link-header) to get the URL for the next page of organizations.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-organizations)
    pub async fn orgs_list(
        &self,
        since: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list::reqwest_builder(
                self.config.base_url.as_ref(),
                since,
                per_page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List custom repository roles in an organization
    ///
    /// List the custom repository roles available in this organization. In order to see custom
    /// repository roles in an organization, the authenticated user must be an organization owner.
    ///
    /// For more information on custom repository roles, see "[Managing custom repository roles for an organization](https://docs.github.com/organizations/managing-peoples-access-to-your-organization-with-roles/managing-custom-repository-roles-for-an-organization)".
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-custom-repository-roles-in-an-organization)
    pub async fn orgs_list_custom_roles(
        &self,
        organization_id: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_custom_roles::reqwest_builder(
                self.config.base_url.as_ref(),
                organization_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_custom_roles::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization
    ///
    /// To see many of the organization response values, you need to be an authenticated organization owner with the `admin:org` scope. When the value of `two_factor_requirement_enabled` is `true`, the organization requires all members, billing managers, and outside collaborators to enable [two-factor authentication](https://docs.github.com/articles/securing-your-account-with-two-factor-authentication-2fa/).
    ///
    /// GitHub Apps with the `Organization plan` permission can use this endpoint to retrieve information about an organization's GitHub plan. See "[Authenticating with GitHub Apps](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/)" for details. For an example response, see 'Response with GitHub plan information' below."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#get-an-organization)
    pub async fn orgs_get(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_get::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update an organization
    ///
    /// **Parameter Deprecation Notice:** GitHub will replace and discontinue `members_allowed_repository_creation_type` in favor of more granular permissions. The new input parameters are `members_can_create_public_repositories`, `members_can_create_private_repositories` for all organizations and `members_can_create_internal_repositories` for organizations associated with an enterprise account using GitHub Enterprise Cloud or GitHub Enterprise Server 2.20+. For more information, see the [blog post](https://developer.github.com/changes/2019-12-03-internal-visibility-changes).
    ///
    /// Enables an authenticated organization owner with the `admin:org` scope to update the organization's profile and member privileges.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs/#update-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::orgs_update::body::Json`](crate::v1_1_4::request::orgs_update::body::Json)
    pub async fn orgs_update<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::orgs_update::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::orgs_update::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_update::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::orgs_update::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions cache usage for an organization
    ///
    /// Gets the total GitHub Actions cache usage for an organization.
    /// The data fetched using this API is refreshed approximately every 5 minutes, so values returned from this endpoint may take at least 5 minutes to get updated.
    /// You must authenticate using an access token with the `read:org` scope to use this endpoint. GitHub Apps must have the `organization_admistration:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-github-actions-cache-usage-for-an-organization)
    pub async fn actions_get_actions_cache_usage_for_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_actions_cache_usage_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_actions_cache_usage_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List repositories with GitHub Actions cache usage for an organization
    ///
    /// Lists repositories and their GitHub Actions cache usage for an organization.
    /// The data fetched using this API is refreshed approximately every 5 minutes, so values returned from this endpoint may take at least 5 minutes to get updated.
    /// You must authenticate using an access token with the `read:org` scope to use this endpoint. GitHub Apps must have the `organization_admistration:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-repositories-with-github-actions-cache-usage-for-an-organization)
    pub async fn actions_get_actions_cache_usage_by_repo_for_org(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_actions_cache_usage_by_repo_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_actions_cache_usage_by_repo_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions permissions for an organization
    ///
    /// Gets the GitHub Actions permissions policy for repositories and allowed actions and reusable workflows in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-github-actions-permissions-for-an-organization)
    pub async fn actions_get_github_actions_permissions_organization(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_github_actions_permissions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_github_actions_permissions_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set GitHub Actions permissions for an organization
    ///
    /// Sets the GitHub Actions permissions policy for repositories and allowed actions and reusable workflows in an organization.
    ///
    /// If the organization belongs to an enterprise that has set restrictive permissions at the enterprise level, such as `allowed_actions` to `selected` actions and reusable workflows, then you cannot override them for the organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-github-actions-permissions-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_github_actions_permissions_organization::body::Json`](crate::v1_1_4::request::actions_set_github_actions_permissions_organization::body::Json)
    pub async fn actions_set_github_actions_permissions_organization<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_github_actions_permissions_organization::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_github_actions_permissions_organization::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_github_actions_permissions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_github_actions_permissions_organization::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List selected repositories enabled for GitHub Actions in an organization
    ///
    /// Lists the selected repositories that are enabled for GitHub Actions in an organization. To use this endpoint, the organization permission policy for `enabled_repositories` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an organization](#set-github-actions-permissions-for-an-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-selected-repositories-enabled-for-github-actions-in-an-organization)
    pub async fn actions_list_selected_repositories_enabled_github_actions_organization(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_selected_repositories_enabled_github_actions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_selected_repositories_enabled_github_actions_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set selected repositories enabled for GitHub Actions in an organization
    ///
    /// Replaces the list of selected repositories that are enabled for GitHub Actions in an organization. To use this endpoint, the organization permission policy for `enabled_repositories` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an organization](#set-github-actions-permissions-for-an-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-selected-repositories-enabled-for-github-actions-in-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_selected_repositories_enabled_github_actions_organization::body::Json`](crate::v1_1_4::request::actions_set_selected_repositories_enabled_github_actions_organization::body::Json)
    pub async fn actions_set_selected_repositories_enabled_github_actions_organization<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_selected_repositories_enabled_github_actions_organization::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_selected_repositories_enabled_github_actions_organization::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_selected_repositories_enabled_github_actions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_selected_repositories_enabled_github_actions_organization::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Enable a selected repository for GitHub Actions in an organization
    ///
    /// Adds a repository to the list of selected repositories that are enabled for GitHub Actions in an organization. To use this endpoint, the organization permission policy for `enabled_repositories` must be must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an organization](#set-github-actions-permissions-for-an-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#enable-a-selected-repository-for-github-actions-in-an-organization)
    pub async fn actions_enable_selected_repository_github_actions_organization(
        &self,
        org: &str,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_enable_selected_repository_github_actions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_enable_selected_repository_github_actions_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Disable a selected repository for GitHub Actions in an organization
    ///
    /// Removes a repository from the list of selected repositories that are enabled for GitHub Actions in an organization. To use this endpoint, the organization permission policy for `enabled_repositories` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an organization](#set-github-actions-permissions-for-an-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#disable-a-selected-repository-for-github-actions-in-an-organization)
    pub async fn actions_disable_selected_repository_github_actions_organization(
        &self,
        org: &str,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_disable_selected_repository_github_actions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_disable_selected_repository_github_actions_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get allowed actions and reusable workflows for an organization
    ///
    /// Gets the selected actions and reusable workflows that are allowed in an organization. To use this endpoint, the organization permission policy for `allowed_actions` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an organization](#set-github-actions-permissions-for-an-organization).""
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-allowed-actions-for-an-organization)
    pub async fn actions_get_allowed_actions_organization(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_allowed_actions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_allowed_actions_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set allowed actions and reusable workflows for an organization
    ///
    /// Sets the actions and reusable workflows that are allowed in an organization. To use this endpoint, the organization permission policy for `allowed_actions` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for an organization](#set-github-actions-permissions-for-an-organization)."
    ///
    /// If the organization belongs to an enterprise that has `selected` actions and reusable workflows set at the enterprise level, then you cannot override any of the enterprise's allowed actions and reusable workflows settings.
    ///
    /// To use the `patterns_allowed` setting for private repositories, the organization must belong to an enterprise. If the organization does not belong to an enterprise, then the `patterns_allowed` setting only applies to public repositories in the organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-allowed-actions-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&schema::SelectedActions`](crate::v1_1_4::schema::SelectedActions)
    pub async fn actions_set_allowed_actions_organization<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_allowed_actions_organization::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_allowed_actions_organization::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_allowed_actions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_allowed_actions_organization::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get default workflow permissions for an organization
    ///
    /// Gets the default workflow permissions granted to the `GITHUB_TOKEN` when running workflows in an organization,
    /// as well as whether GitHub Actions can submit approving pull request reviews. For more information, see
    /// "[Setting the permissions of the GITHUB_TOKEN for your organization](https://docs.github.com/organizations/managing-organization-settings/disabling-or-limiting-github-actions-for-your-organization#setting-the-permissions-of-the-github_token-for-your-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-default-workflow-permissions)
    pub async fn actions_get_github_actions_default_workflow_permissions_organization(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_github_actions_default_workflow_permissions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_github_actions_default_workflow_permissions_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set default workflow permissions for an organization
    ///
    /// Sets the default workflow permissions granted to the `GITHUB_TOKEN` when running workflows in an organization, and sets if GitHub Actions
    /// can submit approving pull request reviews. For more information, see
    /// "[Setting the permissions of the GITHUB_TOKEN for your organization](https://docs.github.com/organizations/managing-organization-settings/disabling-or-limiting-github-actions-for-your-organization#setting-the-permissions-of-the-github_token-for-your-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `administration` organization permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-default-workflow-permissions)
    ///
    /// # Content
    ///
    /// - [`&schema::ActionsSetDefaultWorkflowPermissions`](crate::v1_1_4::schema::ActionsSetDefaultWorkflowPermissions)
    pub async fn actions_set_github_actions_default_workflow_permissions_organization<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_organization::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_organization::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_organization::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List self-hosted runner groups for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Lists all self-hosted runner groups configured in an organization and inherited from an enterprise.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-self-hosted-runner-groups-for-an-organization)
    pub async fn actions_list_self_hosted_runner_groups_for_org(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        visible_to_repository: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_self_hosted_runner_groups_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                visible_to_repository,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_self_hosted_runner_groups_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a self-hosted runner group for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud and GitHub Enterprise Server. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Creates a new self-hosted runner group for an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-self-hosted-runner-group-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_create_self_hosted_runner_group_for_org::body::Json`](crate::v1_1_4::request::actions_create_self_hosted_runner_group_for_org::body::Json)
    pub async fn actions_create_self_hosted_runner_group_for_org<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_create_self_hosted_runner_group_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_create_self_hosted_runner_group_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_self_hosted_runner_group_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_create_self_hosted_runner_group_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a self-hosted runner group for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Gets a specific self-hosted runner group for an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-self-hosted-runner-group-for-an-organization)
    pub async fn actions_get_self_hosted_runner_group_for_org(
        &self,
        org: &str,
        runner_group_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_self_hosted_runner_group_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_self_hosted_runner_group_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a self-hosted runner group from an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Deletes a self-hosted runner group for an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-a-self-hosted-runner-group-from-an-organization)
    pub async fn actions_delete_self_hosted_runner_group_from_org(
        &self,
        org: &str,
        runner_group_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_self_hosted_runner_group_from_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_self_hosted_runner_group_from_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a self-hosted runner group for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Updates the `name` and `visibility` of a self-hosted runner group in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#update-a-self-hosted-runner-group-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_update_self_hosted_runner_group_for_org::body::Json`](crate::v1_1_4::request::actions_update_self_hosted_runner_group_for_org::body::Json)
    pub async fn actions_update_self_hosted_runner_group_for_org<Content>(
        &self,
        org: &str,
        runner_group_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_update_self_hosted_runner_group_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_update_self_hosted_runner_group_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_update_self_hosted_runner_group_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_update_self_hosted_runner_group_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List repository access to a self-hosted runner group in an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud and GitHub Enterprise Server. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Lists the repositories with access to a self-hosted runner group configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-repository-access-to-a-self-hosted-runner-group-in-an-organization)
    pub async fn actions_list_repo_access_to_self_hosted_runner_group_in_org(
        &self,
        org: &str,
        runner_group_id: i64,
        page: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_repo_access_to_self_hosted_runner_group_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                page,
                per_page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_repo_access_to_self_hosted_runner_group_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set repository access for a self-hosted runner group in an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Replaces the list of repositories that have access to a self-hosted runner group configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-repository-access-to-a-self-hosted-runner-group-in-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_repo_access_to_self_hosted_runner_group_in_org::body::Json`](crate::v1_1_4::request::actions_set_repo_access_to_self_hosted_runner_group_in_org::body::Json)
    pub async fn actions_set_repo_access_to_self_hosted_runner_group_in_org<Content>(
        &self,
        org: &str,
        runner_group_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_repo_access_to_self_hosted_runner_group_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_repo_access_to_self_hosted_runner_group_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_repo_access_to_self_hosted_runner_group_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_repo_access_to_self_hosted_runner_group_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add repository access to a self-hosted runner group in an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    ///
    /// Adds a repository to the list of selected repositories that can access a self-hosted runner group. The runner group must have `visibility` set to `selected`. For more information, see "[Create a self-hosted runner group for an organization](#create-a-self-hosted-runner-group-for-an-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org`
    /// scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-repository-acess-to-a-self-hosted-runner-group-in-an-organization)
    pub async fn actions_add_repo_access_to_self_hosted_runner_group_in_org(
        &self,
        org: &str,
        runner_group_id: i64,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_add_repo_access_to_self_hosted_runner_group_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_add_repo_access_to_self_hosted_runner_group_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove repository access to a self-hosted runner group in an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    ///
    /// Removes a repository from the list of selected repositories that can access a self-hosted runner group. The runner group must have `visibility` set to `selected`. For more information, see "[Create a self-hosted runner group for an organization](#create-a-self-hosted-runner-group-for-an-organization)."
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-repository-access-to-a-self-hosted-runner-group-in-an-organization)
    pub async fn actions_remove_repo_access_to_self_hosted_runner_group_in_org(
        &self,
        org: &str,
        runner_group_id: i64,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_remove_repo_access_to_self_hosted_runner_group_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_remove_repo_access_to_self_hosted_runner_group_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List self-hosted runners in a group for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Lists self-hosted runners that are in a specific organization group.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-self-hosted-runners-in-a-group-for-an-organization)
    pub async fn actions_list_self_hosted_runners_in_group_for_org(
        &self,
        org: &str,
        runner_group_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_self_hosted_runners_in_group_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_self_hosted_runners_in_group_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set self-hosted runners in a group for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    /// Replaces the list of self-hosted runners that are part of an organization runner group.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-self-hosted-runners-in-a-group-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_self_hosted_runners_in_group_for_org::body::Json`](crate::v1_1_4::request::actions_set_self_hosted_runners_in_group_for_org::body::Json)
    pub async fn actions_set_self_hosted_runners_in_group_for_org<Content>(
        &self,
        org: &str,
        runner_group_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_self_hosted_runners_in_group_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_self_hosted_runners_in_group_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_self_hosted_runners_in_group_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_self_hosted_runners_in_group_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add a self-hosted runner to a group for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    ///
    /// Adds a self-hosted runner to a runner group configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org`
    /// scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-a-self-hosted-runner-to-a-group-for-an-organization)
    pub async fn actions_add_self_hosted_runner_to_group_for_org(
        &self,
        org: &str,
        runner_group_id: i64,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_add_self_hosted_runner_to_group_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_add_self_hosted_runner_to_group_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a self-hosted runner from a group for an organization
    ///
    /// The self-hosted runner groups REST API is available with GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)."
    ///
    ///
    /// Removes a self-hosted runner from a group configured in an organization. The runner is then returned to the default group.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-a-self-hosted-runner-from-a-group-for-an-organization)
    pub async fn actions_remove_self_hosted_runner_from_group_for_org(
        &self,
        org: &str,
        runner_group_id: i64,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_remove_self_hosted_runner_from_group_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_group_id,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_remove_self_hosted_runner_from_group_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List self-hosted runners for an organization
    ///
    /// Lists all self-hosted runners configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-self-hosted-runners-for-an-organization)
    pub async fn actions_list_self_hosted_runners_for_org(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_self_hosted_runners_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_self_hosted_runners_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List runner applications for an organization
    ///
    /// Lists binaries for the runner application that you can download and run.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-runner-applications-for-an-organization)
    pub async fn actions_list_runner_applications_for_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_runner_applications_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_runner_applications_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a registration token for an organization
    ///
    /// Returns a token that you can pass to the `config` script. The token expires after one hour.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// #### Example using registration token
    ///
    /// Configure your self-hosted runner, replacing `TOKEN` with the registration token provided by this endpoint.
    ///
    /// ```text
    /// ./config.sh --url https://github.com/octo-org --token TOKEN
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-registration-token-for-an-organization)
    pub async fn actions_create_registration_token_for_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_registration_token_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_create_registration_token_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a remove token for an organization
    ///
    /// Returns a token that you can pass to the `config` script to remove a self-hosted runner from an organization. The token expires after one hour.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// #### Example using remove token
    ///
    /// To remove your self-hosted runner from an organization, replace `TOKEN` with the remove token provided by this
    /// endpoint.
    ///
    /// ```text
    /// ./config.sh remove --token TOKEN
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-remove-token-for-an-organization)
    pub async fn actions_create_remove_token_for_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_remove_token_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_create_remove_token_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a self-hosted runner for an organization
    ///
    /// Gets a specific self-hosted runner configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-self-hosted-runner-for-an-organization)
    pub async fn actions_get_self_hosted_runner_for_org(
        &self,
        org: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_self_hosted_runner_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_self_hosted_runner_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a self-hosted runner from an organization
    ///
    /// Forces the removal of a self-hosted runner from an organization. You can use this endpoint to completely remove the runner when the machine you were using no longer exists.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-a-self-hosted-runner-from-an-organization)
    pub async fn actions_delete_self_hosted_runner_from_org(
        &self,
        org: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_self_hosted_runner_from_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_self_hosted_runner_from_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List labels for a self-hosted runner for an organization
    ///
    /// Lists all labels for a self-hosted runner configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-labels-for-a-self-hosted-runner-for-an-organization)
    pub async fn actions_list_labels_for_self_hosted_runner_for_org(
        &self,
        org: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_labels_for_self_hosted_runner_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_labels_for_self_hosted_runner_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set custom labels for a self-hosted runner for an organization
    ///
    /// Remove all previous custom labels and set the new custom labels for a specific
    /// self-hosted runner configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-custom-labels-for-a-self-hosted-runner-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_custom_labels_for_self_hosted_runner_for_org::body::Json`](crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_org::body::Json)
    pub async fn actions_set_custom_labels_for_self_hosted_runner_for_org<Content>(
        &self,
        org: &str,
        runner_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add custom labels to a self-hosted runner for an organization
    ///
    /// Add custom labels to a self-hosted runner configured in an organization.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-custom-labels-to-a-self-hosted-runner-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::actions_add_custom_labels_to_self_hosted_runner_for_org::body::Json`](crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_org::body::Json)
    pub async fn actions_add_custom_labels_to_self_hosted_runner_for_org<Content>(
        &self,
        org: &str,
        runner_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove all custom labels from a self-hosted runner for an organization
    ///
    /// Remove all custom labels from a self-hosted runner configured in an
    /// organization. Returns the remaining read-only labels from the runner.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-all-custom-labels-from-a-self-hosted-runner-for-an-organization)
    pub async fn actions_remove_all_custom_labels_from_self_hosted_runner_for_org(
        &self,
        org: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_remove_all_custom_labels_from_self_hosted_runner_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_remove_all_custom_labels_from_self_hosted_runner_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a custom label from a self-hosted runner for an organization
    ///
    /// Remove a custom label from a self-hosted runner configured
    /// in an organization. Returns the remaining labels from the runner.
    ///
    /// This endpoint returns a `404 Not Found` status if the custom label is not
    /// present on the runner.
    ///
    /// You must authenticate using an access token with the `admin:org` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-a-custom-label-from-a-self-hosted-runner-for-an-organization)
    pub async fn actions_remove_custom_label_from_self_hosted_runner_for_org(
        &self,
        org: &str,
        runner_id: i64,
        name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_remove_custom_label_from_self_hosted_runner_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                runner_id,
                name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_remove_custom_label_from_self_hosted_runner_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization secrets
    ///
    /// Lists all secrets available in an organization without revealing their encrypted values. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-organization-secrets)
    pub async fn actions_list_org_secrets(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_org_secrets::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_org_secrets::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization public key
    ///
    /// Gets your public key, which you need to encrypt secrets. You need to encrypt a secret before you can create or update secrets. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-an-organization-public-key)
    pub async fn actions_get_org_public_key(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_org_public_key::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_org_public_key::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization secret
    ///
    /// Gets a single organization secret without revealing its encrypted value. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-an-organization-secret)
    pub async fn actions_get_org_secret(
        &self,
        org: &str,
        secret_name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create or update an organization secret
    ///
    /// Creates or updates an organization secret with an encrypted value. Encrypt your secret using
    /// [LibSodium](https://libsodium.gitbook.io/doc/bindings_for_other_languages). You must authenticate using an access
    /// token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to
    /// use this endpoint.
    ///
    /// #### Example encrypting a secret using Node.js
    ///
    /// Encrypt your secret using the [tweetsodium](https://github.com/github/tweetsodium) library.
    ///
    /// ```text
    /// const sodium = require('tweetsodium');
    ///
    /// const key = "base64-encoded-public-key";
    /// const value = "plain-text-secret";
    ///
    /// // Convert the message and key to Uint8Array's (Buffer implements that interface)
    /// const messageBytes = Buffer.from(value);
    /// const keyBytes = Buffer.from(key, 'base64');
    ///
    /// // Encrypt using LibSodium.
    /// const encryptedBytes = sodium.seal(messageBytes, keyBytes);
    ///
    /// // Base64 the encrypted secret
    /// const encrypted = Buffer.from(encryptedBytes).toString('base64');
    ///
    /// console.log(encrypted);
    /// ```
    ///
    ///
    /// #### Example encrypting a secret using Python
    ///
    /// Encrypt your secret using [pynacl](https://pynacl.readthedocs.io/en/latest/public/#nacl-public-sealedbox) with Python 3.
    ///
    /// ```text
    /// from base64 import b64encode
    /// from nacl import encoding, public
    ///
    /// def encrypt(public_key: str, secret_value: str) -> str:
    ///   """Encrypt a Unicode string using the public key."""
    ///   public_key = public.PublicKey(public_key.encode("utf-8"), encoding.Base64Encoder())
    ///   sealed_box = public.SealedBox(public_key)
    ///   encrypted = sealed_box.encrypt(secret_value.encode("utf-8"))
    ///   return b64encode(encrypted).decode("utf-8")
    /// ```
    ///
    /// #### Example encrypting a secret using C#
    ///
    /// Encrypt your secret using the [Sodium.Core](https://www.nuget.org/packages/Sodium.Core/) package.
    ///
    /// ```text
    /// var secretValue = System.Text.Encoding.UTF8.GetBytes("mySecret");
    /// var publicKey = Convert.FromBase64String("2Sg8iYjAxxmI2LvUXpJjkYrMxURPc8r+dB7TJyvvcCU=");
    ///
    /// var sealedPublicKeyBox = Sodium.SealedPublicKeyBox.Create(secretValue, publicKey);
    ///
    /// Console.WriteLine(Convert.ToBase64String(sealedPublicKeyBox));
    /// ```
    ///
    /// #### Example encrypting a secret using Ruby
    ///
    /// Encrypt your secret using the [rbnacl](https://github.com/RubyCrypto/rbnacl) gem.
    ///
    /// ```ruby
    /// require "rbnacl"
    /// require "base64"
    ///
    /// key = Base64.decode64("+ZYvJDZMHUfBkJdyq5Zm9SKqeuBQ4sj+6sfjlH4CgG0=")
    /// public_key = RbNaCl::PublicKey.new(key)
    ///
    /// box = RbNaCl::Boxes::Sealed.from_public_key(public_key)
    /// encrypted_secret = box.encrypt("my_secret")
    ///
    /// # Print the base64 encoded secret
    /// puts Base64.strict_encode64(encrypted_secret)
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-or-update-an-organization-secret)
    ///
    /// # Content
    ///
    /// - [`&request::actions_create_or_update_org_secret::body::Json`](crate::v1_1_4::request::actions_create_or_update_org_secret::body::Json)
    pub async fn actions_create_or_update_org_secret<Content>(
        &self,
        org: &str,
        secret_name: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_create_or_update_org_secret::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_create_or_update_org_secret::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_or_update_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_create_or_update_org_secret::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an organization secret
    ///
    /// Deletes a secret in an organization using the secret name. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-an-organization-secret)
    pub async fn actions_delete_org_secret(
        &self,
        org: &str,
        secret_name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List selected repositories for an organization secret
    ///
    /// Lists all repositories that have been selected when the `visibility` for repository access to a secret is set to `selected`. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-selected-repositories-for-an-organization-secret)
    pub async fn actions_list_selected_repos_for_org_secret(
        &self,
        org: &str,
        secret_name: &str,
        page: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_selected_repos_for_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                page,
                per_page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_selected_repos_for_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set selected repositories for an organization secret
    ///
    /// Replaces all repositories for an organization secret when the `visibility` for repository access is set to `selected`. The visibility is set when you [Create or update an organization secret](https://docs.github.com/rest/reference/actions#create-or-update-an-organization-secret). You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-selected-repositories-for-an-organization-secret)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_selected_repos_for_org_secret::body::Json`](crate::v1_1_4::request::actions_set_selected_repos_for_org_secret::body::Json)
    pub async fn actions_set_selected_repos_for_org_secret<Content>(
        &self,
        org: &str,
        secret_name: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_selected_repos_for_org_secret::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_selected_repos_for_org_secret::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_selected_repos_for_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_selected_repos_for_org_secret::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add selected repository to an organization secret
    ///
    /// Adds a repository to an organization secret when the `visibility` for repository access is set to `selected`. The visibility is set when you [Create or update an organization secret](https://docs.github.com/rest/reference/actions#create-or-update-an-organization-secret). You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-selected-repository-to-an-organization-secret)
    pub async fn actions_add_selected_repo_to_org_secret(
        &self,
        org: &str,
        secret_name: &str,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_add_selected_repo_to_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_add_selected_repo_to_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove selected repository from an organization secret
    ///
    /// Removes a repository from an organization secret when the `visibility` for repository access is set to `selected`. The visibility is set when you [Create or update an organization secret](https://docs.github.com/rest/reference/actions#create-or-update-an-organization-secret). You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-selected-repository-from-an-organization-secret)
    pub async fn actions_remove_selected_repo_from_org_secret(
        &self,
        org: &str,
        secret_name: &str,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_remove_selected_repo_from_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_remove_selected_repo_from_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get the audit log for an organization
    ///
    /// Gets the audit log for an organization. For more information, see "[Reviewing the audit log for your organization](https://docs.github.com/github/setting-up-and-managing-organizations-and-teams/reviewing-the-audit-log-for-your-organization)."
    ///
    /// This endpoint is available for organizations on GitHub Enterprise Cloud. To use this endpoint, you must be an organization owner, and you must use an access token with the `admin:org` scope. GitHub Apps must have the `organization_administration` read permission to use this endpoint.
    ///
    /// By default, the response includes up to 30 events from the past three months. Use the `phrase` parameter to filter results and retrieve older events. For example, use the `phrase` parameter with the `created` qualifier to filter events based on when the events occurred. For more information, see "[Reviewing the audit log for your organization](https://docs.github.com/organizations/keeping-your-organization-secure/managing-security-settings-for-your-organization/reviewing-the-audit-log-for-your-organization#searching-the-audit-log)."
    ///
    /// Use pagination to retrieve fewer or more than 30 events. For more information, see "[Resources in the REST API](https://docs.github.com/rest/overview/resources-in-the-rest-api#pagination)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#get-audit-log)
    #[allow(clippy::too_many_arguments)]
    pub async fn orgs_get_audit_log(
        &self,
        org: &str,
        phrase: ::std::option::Option<&str>,
        include: ::std::option::Option<&str>,
        after: ::std::option::Option<&str>,
        before: ::std::option::Option<&str>,
        order: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_get_audit_log::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                phrase,
                include,
                after,
                before,
                order,
                per_page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_get_audit_log::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List users blocked by an organization
    ///
    /// List the users blocked by an organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-users-blocked-by-an-organization)
    pub async fn orgs_list_blocked_users(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_blocked_users::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_blocked_users::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check if a user is blocked by an organization
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#check-if-a-user-is-blocked-by-an-organization)
    pub async fn orgs_check_blocked_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_check_blocked_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_check_blocked_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Block a user from an organization
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#block-a-user-from-an-organization)
    pub async fn orgs_block_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_block_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_block_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Unblock a user from an organization
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#unblock-a-user-from-an-organization)
    pub async fn orgs_unblock_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_unblock_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_unblock_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List code scanning alerts for an organization
    ///
    /// Lists all code scanning alerts for the default branch (usually `main`
    /// or `master`) for all eligible repositories in an organization.
    /// To use this endpoint, you must be an administrator or security manager for the organization, and you must use an access token with the `repo` scope or `security_events` scope.
    ///
    /// GitHub Apps must have the `security_events` read permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/code-scanning#list-code-scanning-alerts-by-organization)
    #[allow(clippy::too_many_arguments)]
    pub async fn code_scanning_list_alerts_for_org(
        &self,
        org: &str,
        tool_name: ::std::option::Option<&str>,
        tool_guid: ::std::option::Option<::std::option::Option<&str>>,
        before: ::std::option::Option<&str>,
        after: ::std::option::Option<&str>,
        page: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
        state: ::std::option::Option<&str>,
        sort: &crate::types::Sort<'_>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let (sort, direction) = sort.extract();
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::code_scanning_list_alerts_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                tool_name,
                tool_guid,
                before,
                after,
                page,
                per_page,
                direction,
                state,
                sort,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::code_scanning_list_alerts_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List SAML SSO authorizations for an organization
    ///
    /// Listing and deleting credential authorizations is available to organizations with GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products).
    ///
    /// An authenticated organization owner with the `read:org` scope can list all credential authorizations for an organization that uses SAML single sign-on (SSO). The credentials are either personal access tokens or SSH keys that organization members have authorized for the organization. For more information, see [About authentication with SAML single sign-on](https://docs.github.com/en/articles/about-authentication-with-saml-single-sign-on).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-saml-sso-authorizations-for-an-organization)
    pub async fn orgs_list_saml_sso_authorizations(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        login: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_saml_sso_authorizations::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                login,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_saml_sso_authorizations::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a SAML SSO authorization for an organization
    ///
    /// Listing and deleting credential authorizations is available to organizations with GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products).
    ///
    /// An authenticated organization owner with the `admin:org` scope can remove a credential authorization for an organization that uses SAML SSO. Once you remove someone's credential authorization, they will need to create a new personal access token or SSH key and authorize it for the organization they want to access.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#remove-a-saml-sso-authorization-for-an-organization)
    pub async fn orgs_remove_saml_sso_authorization(
        &self,
        org: &str,
        credential_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_remove_saml_sso_authorization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                credential_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_remove_saml_sso_authorization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization secrets
    ///
    /// Lists all secrets available in an organization without revealing their encrypted values. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#list-organization-secrets)
    pub async fn dependabot_list_org_secrets(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_list_org_secrets::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::dependabot_list_org_secrets::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization public key
    ///
    /// Gets your public key, which you need to encrypt secrets. You need to encrypt a secret before you can create or update secrets. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#get-an-organization-public-key)
    pub async fn dependabot_get_org_public_key(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_get_org_public_key::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::dependabot_get_org_public_key::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization secret
    ///
    /// Gets a single organization secret without revealing its encrypted value. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#get-an-organization-secret)
    pub async fn dependabot_get_org_secret(
        &self,
        org: &str,
        secret_name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_get_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::dependabot_get_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create or update an organization secret
    ///
    /// Creates or updates an organization secret with an encrypted value. Encrypt your secret using
    /// [LibSodium](https://libsodium.gitbook.io/doc/bindings_for_other_languages). You must authenticate using an access
    /// token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization
    /// permission to use this endpoint.
    ///
    /// #### Example encrypting a secret using Node.js
    ///
    /// Encrypt your secret using the [tweetsodium](https://github.com/github/tweetsodium) library.
    ///
    /// ```text
    /// const sodium = require('tweetsodium');
    ///
    /// const key = "base64-encoded-public-key";
    /// const value = "plain-text-secret";
    ///
    /// // Convert the message and key to Uint8Array's (Buffer implements that interface)
    /// const messageBytes = Buffer.from(value);
    /// const keyBytes = Buffer.from(key, 'base64');
    ///
    /// // Encrypt using LibSodium.
    /// const encryptedBytes = sodium.seal(messageBytes, keyBytes);
    ///
    /// // Base64 the encrypted secret
    /// const encrypted = Buffer.from(encryptedBytes).toString('base64');
    ///
    /// console.log(encrypted);
    /// ```
    ///
    ///
    /// #### Example encrypting a secret using Python
    ///
    /// Encrypt your secret using [pynacl](https://pynacl.readthedocs.io/en/latest/public/#nacl-public-sealedbox) with Python 3.
    ///
    /// ```text
    /// from base64 import b64encode
    /// from nacl import encoding, public
    ///
    /// def encrypt(public_key: str, secret_value: str) -> str:
    ///   """Encrypt a Unicode string using the public key."""
    ///   public_key = public.PublicKey(public_key.encode("utf-8"), encoding.Base64Encoder())
    ///   sealed_box = public.SealedBox(public_key)
    ///   encrypted = sealed_box.encrypt(secret_value.encode("utf-8"))
    ///   return b64encode(encrypted).decode("utf-8")
    /// ```
    ///
    /// #### Example encrypting a secret using C#
    ///
    /// Encrypt your secret using the [Sodium.Core](https://www.nuget.org/packages/Sodium.Core/) package.
    ///
    /// ```text
    /// var secretValue = System.Text.Encoding.UTF8.GetBytes("mySecret");
    /// var publicKey = Convert.FromBase64String("2Sg8iYjAxxmI2LvUXpJjkYrMxURPc8r+dB7TJyvvcCU=");
    ///
    /// var sealedPublicKeyBox = Sodium.SealedPublicKeyBox.Create(secretValue, publicKey);
    ///
    /// Console.WriteLine(Convert.ToBase64String(sealedPublicKeyBox));
    /// ```
    ///
    /// #### Example encrypting a secret using Ruby
    ///
    /// Encrypt your secret using the [rbnacl](https://github.com/RubyCrypto/rbnacl) gem.
    ///
    /// ```ruby
    /// require "rbnacl"
    /// require "base64"
    ///
    /// key = Base64.decode64("+ZYvJDZMHUfBkJdyq5Zm9SKqeuBQ4sj+6sfjlH4CgG0=")
    /// public_key = RbNaCl::PublicKey.new(key)
    ///
    /// box = RbNaCl::Boxes::Sealed.from_public_key(public_key)
    /// encrypted_secret = box.encrypt("my_secret")
    ///
    /// # Print the base64 encoded secret
    /// puts Base64.strict_encode64(encrypted_secret)
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#create-or-update-an-organization-secret)
    ///
    /// # Content
    ///
    /// - [`&request::dependabot_create_or_update_org_secret::body::Json`](crate::v1_1_4::request::dependabot_create_or_update_org_secret::body::Json)
    pub async fn dependabot_create_or_update_org_secret<Content>(
        &self,
        org: &str,
        secret_name: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::dependabot_create_or_update_org_secret::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::dependabot_create_or_update_org_secret::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_create_or_update_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::dependabot_create_or_update_org_secret::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an organization secret
    ///
    /// Deletes a secret in an organization using the secret name. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#delete-an-organization-secret)
    pub async fn dependabot_delete_org_secret(
        &self,
        org: &str,
        secret_name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_delete_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::dependabot_delete_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List selected repositories for an organization secret
    ///
    /// Lists all repositories that have been selected when the `visibility` for repository access to a secret is set to `selected`. You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#list-selected-repositories-for-an-organization-secret)
    pub async fn dependabot_list_selected_repos_for_org_secret(
        &self,
        org: &str,
        secret_name: &str,
        page: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_list_selected_repos_for_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                page,
                per_page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::dependabot_list_selected_repos_for_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set selected repositories for an organization secret
    ///
    /// Replaces all repositories for an organization secret when the `visibility` for repository access is set to `selected`. The visibility is set when you [Create or update an organization secret](https://docs.github.com/rest/reference/dependabot#create-or-update-an-organization-secret). You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#set-selected-repositories-for-an-organization-secret)
    ///
    /// # Content
    ///
    /// - [`&request::dependabot_set_selected_repos_for_org_secret::body::Json`](crate::v1_1_4::request::dependabot_set_selected_repos_for_org_secret::body::Json)
    pub async fn dependabot_set_selected_repos_for_org_secret<Content>(
        &self,
        org: &str,
        secret_name: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::dependabot_set_selected_repos_for_org_secret::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::dependabot_set_selected_repos_for_org_secret::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_set_selected_repos_for_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::dependabot_set_selected_repos_for_org_secret::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add selected repository to an organization secret
    ///
    /// Adds a repository to an organization secret when the `visibility` for repository access is set to `selected`. The visibility is set when you [Create or update an organization secret](https://docs.github.com/rest/reference/dependabot#create-or-update-an-organization-secret). You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#add-selected-repository-to-an-organization-secret)
    pub async fn dependabot_add_selected_repo_to_org_secret(
        &self,
        org: &str,
        secret_name: &str,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_add_selected_repo_to_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::dependabot_add_selected_repo_to_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove selected repository from an organization secret
    ///
    /// Removes a repository from an organization secret when the `visibility` for repository access is set to `selected`. The visibility is set when you [Create or update an organization secret](https://docs.github.com/rest/reference/dependabot#create-or-update-an-organization-secret). You must authenticate using an access token with the `admin:org` scope to use this endpoint. GitHub Apps must have the `dependabot_secrets` organization permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/dependabot#remove-selected-repository-from-an-organization-secret)
    pub async fn dependabot_remove_selected_repo_from_org_secret(
        &self,
        org: &str,
        secret_name: &str,
        repository_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::dependabot_remove_selected_repo_from_org_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                secret_name,
                repository_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::dependabot_remove_selected_repo_from_org_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List public organization events
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/activity#list-public-organization-events)
    pub async fn activity_list_public_org_events(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::activity_list_public_org_events::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::activity_list_public_org_events::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an external group
    ///
    /// Displays information about the specific group's usage.  Provides a list of the group's external members as well as a list of teams that this group is connected to.
    ///
    /// You can manage team membership with your identity provider using Enterprise Managed Users for GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)" in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#external-idp-group-info-for-an-organization)
    pub async fn teams_external_idp_group_info_for_org(
        &self,
        org: &str,
        group_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_external_idp_group_info_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                group_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_external_idp_group_info_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List external groups in an organization
    ///
    /// Lists external groups available in an organization. You can query the groups using the `display_name` parameter, only groups with a `group_name` containing the text provided in the `display_name` parameter will be returned.  You can also limit your page results using the `per_page` parameter. GitHub generates a url-encoded `page` token using a cursor value for where the next page begins. For more information on cursor pagination, see "[Offset and Cursor Pagination explained](https://dev.to/jackmarchant/offset-and-cursor-pagination-explained-b89)."
    ///
    /// You can manage team membership with your identity provider using Enterprise Managed Users for GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)" in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-external-idp-groups-for-an-organization)
    pub async fn teams_list_external_idp_groups_for_org(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        display_name: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_external_idp_groups_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                display_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_external_idp_groups_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List failed organization invitations
    ///
    /// The return hash contains `failed_at` and `failed_reason` fields which represent the time at which the invitation failed and the reason for the failure.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-failed-organization-invitations)
    pub async fn orgs_list_failed_invitations(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_failed_invitations::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_failed_invitations::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization webhooks
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-organization-webhooks)
    pub async fn orgs_list_webhooks(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_webhooks::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_webhooks::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create an organization webhook
    ///
    /// Here's how you can create a hook that posts payloads in JSON format:
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#create-an-organization-webhook)
    ///
    /// # Content
    ///
    /// - [`&request::orgs_create_webhook::body::Json`](crate::v1_1_4::request::orgs_create_webhook::body::Json)
    pub async fn orgs_create_webhook<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::orgs_create_webhook::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::orgs_create_webhook::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_create_webhook::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::orgs_create_webhook::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization webhook
    ///
    /// Returns a webhook configured in an organization. To get only the webhook `config` properties, see "[Get a webhook configuration for an organization](/rest/reference/orgs#get-a-webhook-configuration-for-an-organization)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#get-an-organization-webhook)
    pub async fn orgs_get_webhook(
        &self,
        org: &str,
        hook_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_get_webhook::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_get_webhook::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an organization webhook
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#delete-an-organization-webhook)
    pub async fn orgs_delete_webhook(
        &self,
        org: &str,
        hook_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_delete_webhook::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_delete_webhook::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update an organization webhook
    ///
    /// Updates a webhook configured in an organization. When you update a webhook, the `secret` will be overwritten. If you previously had a `secret` set, you must provide the same `secret` or set a new `secret` or the secret will be removed. If you are only updating individual webhook `config` properties, use "[Update a webhook configuration for an organization](/rest/reference/orgs#update-a-webhook-configuration-for-an-organization)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#update-an-organization-webhook)
    ///
    /// # Content
    ///
    /// - [`&request::orgs_update_webhook::body::Json`](crate::v1_1_4::request::orgs_update_webhook::body::Json)
    pub async fn orgs_update_webhook<Content>(
        &self,
        org: &str,
        hook_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::orgs_update_webhook::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::orgs_update_webhook::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_update_webhook::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::orgs_update_webhook::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a webhook configuration for an organization
    ///
    /// Returns the webhook configuration for an organization. To get more information about the webhook, including the `active` state and `events`, use "[Get an organization webhook ](/rest/reference/orgs#get-an-organization-webhook)."
    ///
    /// Access tokens must have the `admin:org_hook` scope, and GitHub Apps must have the `organization_hooks:read` permission.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#get-a-webhook-configuration-for-an-organization)
    pub async fn orgs_get_webhook_config_for_org(
        &self,
        org: &str,
        hook_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_get_webhook_config_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_get_webhook_config_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a webhook configuration for an organization
    ///
    /// Updates the webhook configuration for an organization. To update more information about the webhook, including the `active` state and `events`, use "[Update an organization webhook ](/rest/reference/orgs#update-an-organization-webhook)."
    ///
    /// Access tokens must have the `admin:org_hook` scope, and GitHub Apps must have the `organization_hooks:write` permission.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#update-a-webhook-configuration-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&request::orgs_update_webhook_config_for_org::body::Json`](crate::v1_1_4::request::orgs_update_webhook_config_for_org::body::Json)
    pub async fn orgs_update_webhook_config_for_org<Content>(
        &self,
        org: &str,
        hook_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::orgs_update_webhook_config_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::orgs_update_webhook_config_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_update_webhook_config_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::orgs_update_webhook_config_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List deliveries for an organization webhook
    ///
    /// Returns a list of webhook deliveries for a webhook configured in an organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-deliveries-for-an-organization-webhook)
    pub async fn orgs_list_webhook_deliveries(
        &self,
        org: &str,
        hook_id: i64,
        per_page: ::std::option::Option<i64>,
        cursor: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_webhook_deliveries::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                per_page,
                cursor,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_webhook_deliveries::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a webhook delivery for an organization webhook
    ///
    /// Returns a delivery for a webhook configured in an organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#get-a-webhook-delivery-for-an-organization-webhook)
    pub async fn orgs_get_webhook_delivery(
        &self,
        org: &str,
        hook_id: i64,
        delivery_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_get_webhook_delivery::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                delivery_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_get_webhook_delivery::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Redeliver a delivery for an organization webhook
    ///
    /// Redeliver a delivery for a webhook configured in an organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#redeliver-a-delivery-for-an-organization-webhook)
    pub async fn orgs_redeliver_webhook_delivery(
        &self,
        org: &str,
        hook_id: i64,
        delivery_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_redeliver_webhook_delivery::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                delivery_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_redeliver_webhook_delivery::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Ping an organization webhook
    ///
    /// This will trigger a [ping event](https://docs.github.com/webhooks/#ping-event) to be sent to the hook.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#ping-an-organization-webhook)
    pub async fn orgs_ping_webhook(
        &self,
        org: &str,
        hook_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_ping_webhook::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                hook_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_ping_webhook::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization installation for the authenticated app
    ///
    /// Enables an authenticated GitHub App to find the organization's installation information.
    ///
    /// You must use a [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app) to access this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/apps#get-an-organization-installation-for-the-authenticated-app)
    pub async fn apps_get_org_installation(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::apps_get_org_installation::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::apps_get_org_installation::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List app installations for an organization
    ///
    /// Lists all GitHub Apps in an organization. The installation count includes all GitHub Apps installed on repositories in the organization. You must be an organization owner with `admin:read` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-app-installations-for-an-organization)
    pub async fn orgs_list_app_installations(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_app_installations::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_app_installations::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get interaction restrictions for an organization
    ///
    /// Shows which type of GitHub user can interact with this organization and when the restriction expires. If there is no restrictions, you will see an empty response.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/interactions#get-interaction-restrictions-for-an-organization)
    pub async fn interactions_get_restrictions_for_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::interactions_get_restrictions_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::interactions_get_restrictions_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set interaction restrictions for an organization
    ///
    /// Temporarily restricts interactions to a certain type of GitHub user in any public repository in the given organization. You must be an organization owner to set these restrictions. Setting the interaction limit at the organization level will overwrite any interaction limits that are set for individual repositories owned by the organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/interactions#set-interaction-restrictions-for-an-organization)
    ///
    /// # Content
    ///
    /// - [`&schema::InteractionLimit`](crate::v1_1_4::schema::InteractionLimit)
    pub async fn interactions_set_restrictions_for_org<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::interactions_set_restrictions_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::interactions_set_restrictions_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::interactions_set_restrictions_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::interactions_set_restrictions_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove interaction restrictions for an organization
    ///
    /// Removes all interaction restrictions from public repositories in the given organization. You must be an organization owner to remove restrictions.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/interactions#remove-interaction-restrictions-for-an-organization)
    pub async fn interactions_remove_restrictions_for_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::interactions_remove_restrictions_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::interactions_remove_restrictions_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List pending organization invitations
    ///
    /// The return hash contains a `role` field which refers to the Organization Invitation role and will be one of the following values: `direct_member`, `admin`, `billing_manager`, `hiring_manager`, or `reinstate`. If the invitee is not a GitHub member, the `login` field in the return hash will be `null`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-pending-organization-invitations)
    pub async fn orgs_list_pending_invitations(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_pending_invitations::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_pending_invitations::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create an organization invitation
    ///
    /// Invite people to an organization by using their GitHub user ID or their email address. In order to create invitations in an organization, the authenticated user must be an organization owner.
    ///
    /// This endpoint triggers [notifications](https://docs.github.com/en/github/managing-subscriptions-and-notifications-on-github/about-notifications). Creating content too quickly using this endpoint may result in secondary rate limiting. See "[Secondary rate limits](https://docs.github.com/rest/overview/resources-in-the-rest-api#secondary-rate-limits)" and "[Dealing with secondary rate limits](https://docs.github.com/rest/guides/best-practices-for-integrators#dealing-with-secondary-rate-limits)" for details.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#create-an-organization-invitation)
    ///
    /// # Content
    ///
    /// - [`&request::orgs_create_invitation::body::Json`](crate::v1_1_4::request::orgs_create_invitation::body::Json)
    pub async fn orgs_create_invitation<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::orgs_create_invitation::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::orgs_create_invitation::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_create_invitation::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::orgs_create_invitation::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Cancel an organization invitation
    ///
    /// Cancel an organization invitation. In order to cancel an organization invitation, the authenticated user must be an organization owner.
    ///
    /// This endpoint triggers [notifications](https://docs.github.com/en/github/managing-subscriptions-and-notifications-on-github/about-notifications).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#cancel-an-organization-invitation)
    pub async fn orgs_cancel_invitation(
        &self,
        org: &str,
        invitation_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_cancel_invitation::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                invitation_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_cancel_invitation::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization invitation teams
    ///
    /// List all teams associated with an invitation. In order to see invitations in an organization, the authenticated user must be an organization owner.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-organization-invitation-teams)
    pub async fn orgs_list_invitation_teams(
        &self,
        org: &str,
        invitation_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_invitation_teams::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                invitation_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_invitation_teams::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization issues assigned to the authenticated user
    ///
    /// List issues in an organization assigned to the authenticated user.
    ///
    /// **Note**: GitHub's REST API v3 considers every pull request an issue, but not every issue is a pull request. For this
    /// reason, "Issues" endpoints may return both issues and pull requests in the response. You can identify pull requests by
    /// the `pull_request` key. Be aware that the `id` of a pull request returned from "Issues" endpoints will be an _issue id_. To find out the pull
    /// request id, use the "[List pull requests](https://docs.github.com/rest/reference/pulls#list-pull-requests)" endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/issues#list-organization-issues-assigned-to-the-authenticated-user)
    #[allow(clippy::too_many_arguments)]
    pub async fn issues_list_for_org(
        &self,
        org: &str,
        filter: ::std::option::Option<&str>,
        state: ::std::option::Option<&str>,
        labels: ::std::option::Option<&str>,
        sort: &crate::types::Sort<'_>,
        since: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let (sort, direction) = sort.extract();
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::issues_list_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                filter,
                state,
                labels,
                sort,
                direction,
                since,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::issues_list_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization members
    ///
    /// List all users who are members of an organization. If the authenticated user is also a member of this organization then both concealed and public members will be returned.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-organization-members)
    pub async fn orgs_list_members(
        &self,
        org: &str,
        filter: ::std::option::Option<&str>,
        role: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_members::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                filter,
                role,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_members::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check organization membership for a user
    ///
    /// Check if a user is, publicly or privately, a member of the organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#check-organization-membership-for-a-user)
    pub async fn orgs_check_membership_for_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_check_membership_for_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_check_membership_for_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove an organization member
    ///
    /// Removing a user from this list will remove them from all teams and they will no longer have any access to the organization's repositories.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#remove-an-organization-member)
    pub async fn orgs_remove_member(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_remove_member::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_remove_member::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get organization membership for a user
    ///
    /// In order to get a user's membership with an organization, the authenticated user must be an organization member. The `state` parameter in the response can be used to identify the user's membership status.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#get-organization-membership-for-a-user)
    pub async fn orgs_get_membership_for_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_get_membership_for_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_get_membership_for_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set organization membership for a user
    ///
    /// Only authenticated organization owners can add a member to the organization or update the member's role.
    ///
    /// *   If the authenticated user is _adding_ a member to the organization, the invited user will receive an email inviting them to the organization. The user's [membership status](https://docs.github.com/rest/reference/orgs#get-organization-membership-for-a-user) will be `pending` until they accept the invitation.
    ///
    /// *   Authenticated users can _update_ a user's membership by passing the `role` parameter. If the authenticated user changes a member's role to `admin`, the affected user will receive an email notifying them that they've been made an organization owner. If the authenticated user changes an owner's role to `member`, no email will be sent.
    ///
    /// **Rate limits**
    ///
    /// To prevent abuse, the authenticated user is limited to 50 organization invitations per 24 hour period. If the organization is more than one month old or on a paid plan, the limit is 500 invitations per 24 hour period.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#set-organization-membership-for-a-user)
    ///
    /// # Content
    ///
    /// - [`&request::orgs_set_membership_for_user::body::Json`](crate::v1_1_4::request::orgs_set_membership_for_user::body::Json)
    pub async fn orgs_set_membership_for_user<Content>(
        &self,
        org: &str,
        username: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::orgs_set_membership_for_user::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::orgs_set_membership_for_user::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_set_membership_for_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::orgs_set_membership_for_user::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove organization membership for a user
    ///
    /// In order to remove a user's membership with an organization, the authenticated user must be an organization owner.
    ///
    /// If the specified user is an active member of the organization, this will remove them from the organization. If the specified user has been invited to the organization, this will cancel their invitation. The specified user will receive an email notification in both cases.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#remove-organization-membership-for-a-user)
    pub async fn orgs_remove_membership_for_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_remove_membership_for_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_remove_membership_for_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization migrations
    ///
    /// Lists the most recent migrations.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/migrations#list-organization-migrations)
    pub async fn migrations_list_for_org(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        exclude: ::std::option::Option<&[::std::borrow::Cow<'_, str>]>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::migrations_list_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                exclude,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::migrations_list_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Start an organization migration
    ///
    /// Initiates the generation of a migration archive.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/migrations#start-an-organization-migration)
    ///
    /// # Content
    ///
    /// - [`&request::migrations_start_for_org::body::Json`](crate::v1_1_4::request::migrations_start_for_org::body::Json)
    pub async fn migrations_start_for_org<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::migrations_start_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::migrations_start_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::migrations_start_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::migrations_start_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an organization migration status
    ///
    /// Fetches the status of a migration.
    ///
    /// The `state` of a migration can be one of the following values:
    ///
    /// *   `pending`, which means the migration hasn't started yet.
    /// *   `exporting`, which means the migration is in progress.
    /// *   `exported`, which means the migration finished successfully.
    /// *   `failed`, which means the migration failed.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/migrations#get-an-organization-migration-status)
    pub async fn migrations_get_status_for_org(
        &self,
        org: &str,
        migration_id: i64,
        exclude: ::std::option::Option<&[::std::borrow::Cow<'_, str>]>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::migrations_get_status_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                migration_id,
                exclude,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::migrations_get_status_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Download an organization migration archive
    ///
    /// Fetches the URL to a migration archive.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/migrations#download-an-organization-migration-archive)
    pub async fn migrations_download_archive_for_org(
        &self,
        org: &str,
        migration_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::migrations_download_archive_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                migration_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::migrations_download_archive_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an organization migration archive
    ///
    /// Deletes a previous migration archive. Migration archives are automatically deleted after seven days.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/migrations#delete-an-organization-migration-archive)
    pub async fn migrations_delete_archive_for_org(
        &self,
        org: &str,
        migration_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::migrations_delete_archive_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                migration_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::migrations_delete_archive_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Unlock an organization repository
    ///
    /// Unlocks a repository that was locked for migration. You should unlock each migrated repository and [delete them](https://docs.github.com/rest/reference/repos#delete-a-repository) when the migration is complete and you no longer need the source data.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/migrations#unlock-an-organization-repository)
    pub async fn migrations_unlock_repo_for_org(
        &self,
        org: &str,
        migration_id: i64,
        repo_name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::migrations_unlock_repo_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                migration_id,
                repo_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::migrations_unlock_repo_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List repositories in an organization migration
    ///
    /// List all the repositories for this organization migration.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/migrations#list-repositories-in-an-organization-migration)
    pub async fn migrations_list_repos_for_org(
        &self,
        org: &str,
        migration_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::migrations_list_repos_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                migration_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::migrations_list_repos_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List outside collaborators for an organization
    ///
    /// List all users who are outside collaborators of an organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-outside-collaborators-for-an-organization)
    pub async fn orgs_list_outside_collaborators(
        &self,
        org: &str,
        filter: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_outside_collaborators::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                filter,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_outside_collaborators::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Convert an organization member to outside collaborator
    ///
    /// When an organization member is converted to an outside collaborator, they'll only have access to the repositories that their current team membership allows. The user will no longer be a member of the organization. For more information, see "[Converting an organization member to an outside collaborator](https://docs.github.com/articles/converting-an-organization-member-to-an-outside-collaborator/)". Converting an organization member to an outside collaborator may be restricted by enterprise administrators. For more information, see "[Enforcing repository management policies in your enterprise](https://docs.github.com/enterprise-cloud@latest/admin/policies/enforcing-policies-for-your-enterprise/enforcing-repository-management-policies-in-your-enterprise#enforcing-a-policy-for-inviting-outside-collaborators-to-repositories)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#convert-an-organization-member-to-outside-collaborator)
    pub async fn orgs_convert_member_to_outside_collaborator(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_convert_member_to_outside_collaborator::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_convert_member_to_outside_collaborator::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove outside collaborator from an organization
    ///
    /// Removing a user from this list will remove them from all the organization's repositories.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#remove-outside-collaborator-from-an-organization)
    pub async fn orgs_remove_outside_collaborator(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_remove_outside_collaborator::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_remove_outside_collaborator::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List packages for an organization
    ///
    /// Lists all packages in an organization readable by the user.
    ///
    /// To use this endpoint, you must authenticate using an access token with the `packages:read` scope.
    /// If `package_type` is not `container`, your token must also include the `repo` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#list-packages-for-an-organization)
    pub async fn packages_list_packages_for_organization(
        &self,
        package_type: &str,
        org: &str,
        visibility: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_list_packages_for_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                package_type,
                visibility,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_list_packages_for_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a package for an organization
    ///
    /// Gets a specific package in an organization.
    ///
    /// To use this endpoint, you must authenticate using an access token with the `packages:read` scope.
    /// If `package_type` is not `container`, your token must also include the `repo` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#get-a-package-for-an-organization)
    pub async fn packages_get_package_for_organization(
        &self,
        package_type: &str,
        package_name: &str,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_get_package_for_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                package_type,
                package_name,
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_get_package_for_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a package for an organization
    ///
    /// Deletes an entire package in an organization. You cannot delete a public package if any version of the package has more than 5,000 downloads. In this scenario, contact GitHub support for further assistance.
    ///
    /// To use this endpoint, you must have admin permissions in the organization and authenticate using an access token with the `packages:read` and `packages:delete` scopes. In addition:
    /// - If `package_type` is not `container`, your token must also include the `repo` scope.
    /// - If `package_type` is `container`, you must also have admin permissions to the container you want to delete.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#delete-a-package-for-an-organization)
    pub async fn packages_delete_package_for_org(
        &self,
        package_type: &str,
        package_name: &str,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_delete_package_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                package_type,
                package_name,
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_delete_package_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Restore a package for an organization
    ///
    /// Restores an entire package in an organization.
    ///
    /// You can restore a deleted package under the following conditions:
    ///   - The package was deleted within the last 30 days.
    ///   - The same package namespace and version is still available and not reused for a new package. If the same package namespace is not available, you will not be able to restore your package. In this scenario, to restore the deleted package, you must delete the new package that uses the deleted package's namespace first.
    ///
    /// To use this endpoint, you must have admin permissions in the organization and authenticate using an access token with the `packages:read` and `packages:write` scopes. In addition:
    /// - If `package_type` is not `container`, your token must also include the `repo` scope.
    /// - If `package_type` is `container`, you must also have admin permissions to the container that you want to restore.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#restore-a-package-for-an-organization)
    pub async fn packages_restore_package_for_org(
        &self,
        package_type: &str,
        package_name: &str,
        org: &str,
        token: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_restore_package_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                package_type,
                package_name,
                org,
                token,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_restore_package_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get all package versions for a package owned by an organization
    ///
    /// Returns all package versions for a package owned by an organization.
    ///
    /// To use this endpoint, you must authenticate using an access token with the `packages:read` scope.
    /// If `package_type` is not `container`, your token must also include the `repo` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#get-all-package-versions-for-a-package-owned-by-an-organization)
    pub async fn packages_get_all_package_versions_for_package_owned_by_org(
        &self,
        package_type: &str,
        package_name: &str,
        org: &str,
        page: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
        state: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_get_all_package_versions_for_package_owned_by_org::reqwest_builder(
                self.config.base_url.as_ref(),
                package_type,
                package_name,
                org,
                page,
                per_page,
                state,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_get_all_package_versions_for_package_owned_by_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a package version for an organization
    ///
    /// Gets a specific package version in an organization.
    ///
    /// You must authenticate using an access token with the `packages:read` scope.
    /// If `package_type` is not `container`, your token must also include the `repo` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#get-a-package-version-for-an-organization)
    pub async fn packages_get_package_version_for_organization(
        &self,
        package_type: &str,
        package_name: &str,
        org: &str,
        package_version_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_get_package_version_for_organization::reqwest_builder(
                self.config.base_url.as_ref(),
                package_type,
                package_name,
                org,
                package_version_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_get_package_version_for_organization::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete package version for an organization
    ///
    /// Deletes a specific package version in an organization. If the package is public and the package version has more than 5,000 downloads, you cannot delete the package version. In this scenario, contact GitHub support for further assistance.
    ///
    /// To use this endpoint, you must have admin permissions in the organization and authenticate using an access token with the `packages:read` and `packages:delete` scopes. In addition:
    /// - If `package_type` is not `container`, your token must also include the `repo` scope.
    /// - If `package_type` is `container`, you must also have admin permissions to the container you want to delete.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#delete-a-package-version-for-an-organization)
    pub async fn packages_delete_package_version_for_org(
        &self,
        package_type: &str,
        package_name: &str,
        org: &str,
        package_version_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_delete_package_version_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                package_type,
                package_name,
                org,
                package_version_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_delete_package_version_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Restore package version for an organization
    ///
    /// Restores a specific package version in an organization.
    ///
    /// You can restore a deleted package under the following conditions:
    ///   - The package was deleted within the last 30 days.
    ///   - The same package namespace and version is still available and not reused for a new package. If the same package namespace is not available, you will not be able to restore your package. In this scenario, to restore the deleted package, you must delete the new package that uses the deleted package's namespace first.
    ///
    /// To use this endpoint, you must have admin permissions in the organization and authenticate using an access token with the `packages:read` and `packages:write` scopes. In addition:
    /// - If `package_type` is not `container`, your token must also include the `repo` scope.
    /// - If `package_type` is `container`, you must also have admin permissions to the container that you want to restore.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/packages#restore-a-package-version-for-an-organization)
    pub async fn packages_restore_package_version_for_org(
        &self,
        package_type: &str,
        package_name: &str,
        org: &str,
        package_version_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::packages_restore_package_version_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                package_type,
                package_name,
                org,
                package_version_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::packages_restore_package_version_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization projects
    ///
    /// Lists the projects in an organization. Returns a `404 Not Found` status if projects are disabled in the organization. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#list-organization-projects)
    pub async fn projects_list_for_org(
        &self,
        org: &str,
        state: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_list_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                state,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_list_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create an organization project
    ///
    /// Creates an organization project board. Returns a `404 Not Found` status if projects are disabled in the organization. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#create-an-organization-project)
    ///
    /// # Content
    ///
    /// - [`&request::projects_create_for_org::body::Json`](crate::v1_1_4::request::projects_create_for_org::body::Json)
    pub async fn projects_create_for_org<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_create_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_create_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_create_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_create_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List public organization members
    ///
    /// Members of an organization can choose to have their membership publicized or not.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#list-public-organization-members)
    pub async fn orgs_list_public_members(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_list_public_members::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_list_public_members::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check public organization membership for a user
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#check-public-organization-membership-for-a-user)
    pub async fn orgs_check_public_membership_for_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_check_public_membership_for_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_check_public_membership_for_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set public organization membership for the authenticated user
    ///
    /// The user can publicize their own membership. (A user cannot publicize the membership for another user.)
    ///
    /// Note that you'll need to set `Content-Length` to zero when calling out to this endpoint. For more information, see "[HTTP verbs](https://docs.github.com/rest/overview/resources-in-the-rest-api#http-verbs)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#set-public-organization-membership-for-the-authenticated-user)
    pub async fn orgs_set_public_membership_for_authenticated_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_set_public_membership_for_authenticated_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_set_public_membership_for_authenticated_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove public organization membership for the authenticated user
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/orgs#remove-public-organization-membership-for-the-authenticated-user)
    pub async fn orgs_remove_public_membership_for_authenticated_user(
        &self,
        org: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::orgs_remove_public_membership_for_authenticated_user::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::orgs_remove_public_membership_for_authenticated_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List organization repositories
    ///
    /// Lists repositories for the specified organization.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#list-organization-repositories)
    pub async fn repos_list_for_org(
        &self,
        org: &str,
        r#type: ::std::option::Option<&str>,
        sort: &crate::types::Sort<'_>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let (sort, direction) = sort.extract();
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_list_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                r#type,
                sort,
                direction,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_list_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create an organization repository
    ///
    /// Creates a new repository in the specified organization. The authenticated user must be a member of the organization.
    ///
    /// **OAuth scope requirements**
    ///
    /// When using [OAuth](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/), authorizations must include:
    ///
    /// *   `public_repo` scope or `repo` scope to create a public repository. Note: For GitHub AE, use `repo` scope to create an internal repository.
    /// *   `repo` scope to create a private repository
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#create-an-organization-repository)
    ///
    /// # Content
    ///
    /// - [`&request::repos_create_in_org::body::Json`](crate::v1_1_4::request::repos_create_in_org::body::Json)
    pub async fn repos_create_in_org<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_create_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_create_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_create_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_create_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List secret scanning alerts for an organization
    ///
    /// Lists secret scanning alerts for eligible repositories in an organization, from newest to oldest.
    /// To use this endpoint, you must be an administrator or security manager for the organization, and you must use an access token with the `repo` scope or `security_events` scope.
    ///
    /// GitHub Apps must have the `secret_scanning_alerts` read permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/secret-scanning#list-secret-scanning-alerts-for-an-organization)
    pub async fn secret_scanning_list_alerts_for_org(
        &self,
        org: &str,
        state: ::std::option::Option<&str>,
        secret_type: ::std::option::Option<&str>,
        resolution: ::std::option::Option<&str>,
        page: ::std::option::Option<i64>,
        per_page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::secret_scanning_list_alerts_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                state,
                secret_type,
                resolution,
                page,
                per_page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::secret_scanning_list_alerts_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions billing for an organization
    ///
    /// Gets the summary of the free and paid GitHub Actions minutes used.
    ///
    /// Paid minutes only apply to workflows in private repositories that use GitHub-hosted runners. Minutes used is listed for each GitHub-hosted runner operating system. Any job re-runs are also included in the usage. The usage returned includes any minute multipliers for macOS and Windows runners, and is rounded up to the nearest whole minute. For more information, see "[Managing billing for GitHub Actions](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-actions)".
    ///
    /// Access tokens must have the `repo` or `admin:org` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#get-github-actions-billing-for-an-organization)
    pub async fn billing_get_github_actions_billing_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_github_actions_billing_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_github_actions_billing_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Advanced Security active committers for an organization
    ///
    /// Gets the GitHub Advanced Security active committers for an organization per repository.
    /// Each distinct user login across all repositories is counted as a single Advanced Security seat, so the total_advanced_security_committers is not the sum of advanced_security_committers for each repository.
    /// If this organization defers to an enterprise for billing, the total_advanced_security_committers returned from the organization API may include some users that are in more than one organization, so they will only consume a single Advanced Security seat at the enterprise level.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#get-github-advanced-security-active-committers-for-an-organization)
    pub async fn billing_get_github_advanced_security_billing_org(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_github_advanced_security_billing_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_github_advanced_security_billing_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Packages billing for an organization
    ///
    /// Gets the free and paid storage used for GitHub Packages in gigabytes.
    ///
    /// Paid minutes only apply to packages stored for private repositories. For more information, see "[Managing billing for GitHub Packages](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-packages)."
    ///
    /// Access tokens must have the `repo` or `admin:org` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#get-github-packages-billing-for-an-organization)
    pub async fn billing_get_github_packages_billing_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_github_packages_billing_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_github_packages_billing_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get shared storage billing for an organization
    ///
    /// Gets the estimated paid and estimated total storage used for GitHub Actions and GitHub Packages.
    ///
    /// Paid minutes only apply to packages stored for private repositories. For more information, see "[Managing billing for GitHub Packages](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-packages)."
    ///
    /// Access tokens must have the `repo` or `admin:org` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/billing#get-shared-storage-billing-for-an-organization)
    pub async fn billing_get_shared_storage_billing_org(
        &self,
        org: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::billing_get_shared_storage_billing_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::billing_get_shared_storage_billing_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List IdP groups for an organization
    ///
    /// Team synchronization is available for organizations using GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// List IdP groups available in an organization. You can limit your page results using the `per_page` parameter. GitHub generates a url-encoded `page` token using a cursor value for where the next page begins. For more information on cursor pagination, see "[Offset and Cursor Pagination explained](https://dev.to/jackmarchant/offset-and-cursor-pagination-explained-b89)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-idp-groups-for-an-organization)
    pub async fn teams_list_idp_groups_for_org(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_idp_groups_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_idp_groups_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List teams
    ///
    /// Lists all teams in an organization that are visible to the authenticated user.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-teams)
    pub async fn teams_list(
        &self,
        org: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a team
    ///
    /// To create a team, the authenticated user must be a member or owner of `{org}`. By default, organization members can create teams. Organization owners can limit team creation to organization owners. For more information, see "[Setting team creation permissions](https://docs.github.com/en/articles/setting-team-creation-permissions-in-your-organization)."
    ///
    /// When you create a new team, you automatically become a team maintainer without explicitly adding yourself to the optional array of `maintainers`. For more information, see "[About teams](https://docs.github.com/en/github/setting-up-and-managing-organizations-and-teams/about-teams)".
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#create-a-team)
    ///
    /// # Content
    ///
    /// - [`&request::teams_create::body::Json`](crate::v1_1_4::request::teams_create::body::Json)
    pub async fn teams_create<Content>(
        &self,
        org: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_create::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_create::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_create::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_create::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a team by name
    ///
    /// Gets a team using the team's `slug`. GitHub generates the `slug` from the team `name`.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#get-a-team-by-name)
    pub async fn teams_get_by_name(
        &self,
        org: &str,
        team_slug: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_get_by_name::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_get_by_name::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a team
    ///
    /// To delete a team, the authenticated user must be an organization owner or team maintainer.
    ///
    /// If you are an organization owner, deleting a parent team will delete all of its child teams as well.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `DELETE /organizations/{org_id}/team/{team_id}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#delete-a-team)
    pub async fn teams_delete_in_org(
        &self,
        org: &str,
        team_slug: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_delete_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_delete_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a team
    ///
    /// To edit a team, the authenticated user must either be an organization owner or a team maintainer.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `PATCH /organizations/{org_id}/team/{team_id}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#update-a-team)
    ///
    /// # Content
    ///
    /// - [`&request::teams_update_in_org::body::Json`](crate::v1_1_4::request::teams_update_in_org::body::Json)
    pub async fn teams_update_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_update_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_update_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_update_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_update_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List discussions
    ///
    /// List all discussions on a team's page. OAuth access tokens require the `read:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/discussions`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-discussions)
    pub async fn teams_list_discussions_in_org(
        &self,
        org: &str,
        team_slug: &str,
        direction: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        pinned: ::std::option::Option<&str>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_discussions_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                direction,
                per_page,
                page,
                pinned,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_discussions_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a discussion
    ///
    /// Creates a new discussion post on a team's page. OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// This endpoint triggers [notifications](https://docs.github.com/en/github/managing-subscriptions-and-notifications-on-github/about-notifications). Creating content too quickly using this endpoint may result in secondary rate limiting. See "[Secondary rate limits](https://docs.github.com/rest/overview/resources-in-the-rest-api#secondary-rate-limits)" and "[Dealing with secondary rate limits](https://docs.github.com/rest/guides/best-practices-for-integrators#dealing-with-secondary-rate-limits)" for details.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `POST /organizations/{org_id}/team/{team_id}/discussions`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#create-a-discussion)
    ///
    /// # Content
    ///
    /// - [`&request::teams_create_discussion_in_org::body::Json`](crate::v1_1_4::request::teams_create_discussion_in_org::body::Json)
    pub async fn teams_create_discussion_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_create_discussion_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_create_discussion_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_create_discussion_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_create_discussion_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a discussion
    ///
    /// Get a specific discussion on a team's page. OAuth access tokens require the `read:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#get-a-discussion)
    pub async fn teams_get_discussion_in_org(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_get_discussion_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_get_discussion_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a discussion
    ///
    /// Delete a discussion from a team's page. OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `DELETE /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#delete-a-discussion)
    pub async fn teams_delete_discussion_in_org(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_delete_discussion_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_delete_discussion_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a discussion
    ///
    /// Edits the title and body text of a discussion post. Only the parameters you provide are updated. OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `PATCH /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#update-a-discussion)
    ///
    /// # Content
    ///
    /// - [`&request::teams_update_discussion_in_org::body::Json`](crate::v1_1_4::request::teams_update_discussion_in_org::body::Json)
    pub async fn teams_update_discussion_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_update_discussion_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_update_discussion_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_update_discussion_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_update_discussion_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List discussion comments
    ///
    /// List all comments on a team discussion. OAuth access tokens require the `read:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}/comments`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-discussion-comments)
    pub async fn teams_list_discussion_comments_in_org(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        direction: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_discussion_comments_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                direction,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_discussion_comments_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a discussion comment
    ///
    /// Creates a new comment on a team discussion. OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// This endpoint triggers [notifications](https://docs.github.com/en/github/managing-subscriptions-and-notifications-on-github/about-notifications). Creating content too quickly using this endpoint may result in secondary rate limiting. See "[Secondary rate limits](https://docs.github.com/rest/overview/resources-in-the-rest-api#secondary-rate-limits)" and "[Dealing with secondary rate limits](https://docs.github.com/rest/guides/best-practices-for-integrators#dealing-with-secondary-rate-limits)" for details.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `POST /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}/comments`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#create-a-discussion-comment)
    ///
    /// # Content
    ///
    /// - [`&request::teams_create_discussion_comment_in_org::body::Json`](crate::v1_1_4::request::teams_create_discussion_comment_in_org::body::Json)
    pub async fn teams_create_discussion_comment_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_create_discussion_comment_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_create_discussion_comment_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_create_discussion_comment_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_create_discussion_comment_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a discussion comment
    ///
    /// Get a specific comment on a team discussion. OAuth access tokens require the `read:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}/comments/{comment_number}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#get-a-discussion-comment)
    pub async fn teams_get_discussion_comment_in_org(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        comment_number: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_get_discussion_comment_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                comment_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_get_discussion_comment_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a discussion comment
    ///
    /// Deletes a comment on a team discussion. OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `DELETE /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}/comments/{comment_number}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#delete-a-discussion-comment)
    pub async fn teams_delete_discussion_comment_in_org(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        comment_number: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_delete_discussion_comment_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                comment_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_delete_discussion_comment_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a discussion comment
    ///
    /// Edits the body text of a discussion comment. OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `PATCH /organizations/{org_id}/team/{team_id}/discussions/{discussion_number}/comments/{comment_number}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#update-a-discussion-comment)
    ///
    /// # Content
    ///
    /// - [`&request::teams_update_discussion_comment_in_org::body::Json`](crate::v1_1_4::request::teams_update_discussion_comment_in_org::body::Json)
    pub async fn teams_update_discussion_comment_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        comment_number: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_update_discussion_comment_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_update_discussion_comment_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_update_discussion_comment_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                comment_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_update_discussion_comment_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List reactions for a team discussion comment
    ///
    /// List the reactions to a [team discussion comment](https://docs.github.com/rest/reference/teams#discussion-comments/). OAuth access tokens require the `read:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/:org_id/team/:team_id/discussions/:discussion_number/comments/:comment_number/reactions`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/reactions#list-reactions-for-a-team-discussion-comment)
    #[allow(clippy::too_many_arguments)]
    pub async fn reactions_list_for_team_discussion_comment_in_org(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        comment_number: i64,
        content: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::reactions_list_for_team_discussion_comment_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                comment_number,
                content,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::reactions_list_for_team_discussion_comment_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create reaction for a team discussion comment
    ///
    /// Create a reaction to a [team discussion comment](https://docs.github.com/rest/reference/teams#discussion-comments). OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/). A response with an HTTP `200` status means that you already added the reaction type to this team discussion comment.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `POST /organizations/:org_id/team/:team_id/discussions/:discussion_number/comments/:comment_number/reactions`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/reactions#create-reaction-for-a-team-discussion-comment)
    ///
    /// # Content
    ///
    /// - [`&request::reactions_create_for_team_discussion_comment_in_org::body::Json`](crate::v1_1_4::request::reactions_create_for_team_discussion_comment_in_org::body::Json)
    pub async fn reactions_create_for_team_discussion_comment_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        comment_number: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::reactions_create_for_team_discussion_comment_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::reactions_create_for_team_discussion_comment_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::reactions_create_for_team_discussion_comment_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                comment_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::reactions_create_for_team_discussion_comment_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete team discussion comment reaction
    ///
    /// **Note:** You can also specify a team or organization with `team_id` and `org_id` using the route `DELETE /organizations/:org_id/team/:team_id/discussions/:discussion_number/comments/:comment_number/reactions/:reaction_id`.
    ///
    /// Delete a reaction to a [team discussion comment](https://docs.github.com/rest/reference/teams#discussion-comments). OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/reactions#delete-team-discussion-comment-reaction)
    pub async fn reactions_delete_for_team_discussion_comment(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        comment_number: i64,
        reaction_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::reactions_delete_for_team_discussion_comment::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                comment_number,
                reaction_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::reactions_delete_for_team_discussion_comment::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List reactions for a team discussion
    ///
    /// List the reactions to a [team discussion](https://docs.github.com/rest/reference/teams#discussions). OAuth access tokens require the `read:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/:org_id/team/:team_id/discussions/:discussion_number/reactions`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/reactions#list-reactions-for-a-team-discussion)
    pub async fn reactions_list_for_team_discussion_in_org(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        content: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::reactions_list_for_team_discussion_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                content,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::reactions_list_for_team_discussion_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create reaction for a team discussion
    ///
    /// Create a reaction to a [team discussion](https://docs.github.com/rest/reference/teams#discussions). OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/). A response with an HTTP `200` status means that you already added the reaction type to this team discussion.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `POST /organizations/:org_id/team/:team_id/discussions/:discussion_number/reactions`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/reactions#create-reaction-for-a-team-discussion)
    ///
    /// # Content
    ///
    /// - [`&request::reactions_create_for_team_discussion_in_org::body::Json`](crate::v1_1_4::request::reactions_create_for_team_discussion_in_org::body::Json)
    pub async fn reactions_create_for_team_discussion_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::reactions_create_for_team_discussion_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::reactions_create_for_team_discussion_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::reactions_create_for_team_discussion_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::reactions_create_for_team_discussion_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete team discussion reaction
    ///
    /// **Note:** You can also specify a team or organization with `team_id` and `org_id` using the route `DELETE /organizations/:org_id/team/:team_id/discussions/:discussion_number/reactions/:reaction_id`.
    ///
    /// Delete a reaction to a [team discussion](https://docs.github.com/rest/reference/teams#discussions). OAuth access tokens require the `write:discussion` [scope](https://docs.github.com/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/reactions#delete-team-discussion-reaction)
    pub async fn reactions_delete_for_team_discussion(
        &self,
        org: &str,
        team_slug: &str,
        discussion_number: i64,
        reaction_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::reactions_delete_for_team_discussion::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                discussion_number,
                reaction_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::reactions_delete_for_team_discussion::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List a connection between an external group and a team
    ///
    /// Lists a connection between a team and an external group.
    ///
    /// You can manage team membership with your identity provider using Enterprise Managed Users for GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)" in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-external-idp-group-team-connection)
    pub async fn teams_list_linked_external_idp_groups_to_team_for_org(
        &self,
        org: &str,
        team_slug: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_linked_external_idp_groups_to_team_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_linked_external_idp_groups_to_team_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove the connection between an external group and a team
    ///
    /// Deletes a connection between a team and an external group.
    ///
    /// You can manage team membership with your IdP using Enterprise Managed Users for GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#unlink-external-idp-group-team-connection)
    pub async fn teams_unlink_external_idp_group_from_team_for_org(
        &self,
        org: &str,
        team_slug: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_unlink_external_idp_group_from_team_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_unlink_external_idp_group_from_team_for_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update the connection between an external group and a team
    ///
    /// Creates a connection between a team and an external group.  Only one external group can be linked to a team.
    ///
    /// You can manage team membership with your identity provider using Enterprise Managed Users for GitHub Enterprise Cloud. For more information, see "[GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products)" in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#link-external-idp-group-team-connection)
    ///
    /// # Content
    ///
    /// - [`&request::teams_link_external_idp_group_to_team_for_org::body::Json`](crate::v1_1_4::request::teams_link_external_idp_group_to_team_for_org::body::Json)
    pub async fn teams_link_external_idp_group_to_team_for_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_link_external_idp_group_to_team_for_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_link_external_idp_group_to_team_for_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_link_external_idp_group_to_team_for_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_link_external_idp_group_to_team_for_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List pending team invitations
    ///
    /// The return hash contains a `role` field which refers to the Organization Invitation role and will be one of the following values: `direct_member`, `admin`, `billing_manager`, `hiring_manager`, or `reinstate`. If the invitee is not a GitHub member, the `login` field in the return hash will be `null`.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/invitations`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-pending-team-invitations)
    pub async fn teams_list_pending_invitations_in_org(
        &self,
        org: &str,
        team_slug: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_pending_invitations_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_pending_invitations_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List team members
    ///
    /// Team members will include the members of child teams.
    ///
    /// To list members in a team, the team must be visible to the authenticated user.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-team-members)
    pub async fn teams_list_members_in_org(
        &self,
        org: &str,
        team_slug: &str,
        role: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_members_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                role,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_members_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get team membership for a user
    ///
    /// Team members will include the members of child teams.
    ///
    /// To get a user's membership with a team, the team must be visible to the authenticated user.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/memberships/{username}`.
    ///
    /// **Note:**
    /// The response contains the `state` of the membership and the member's `role`.
    ///
    /// The `role` for organization owners is set to `maintainer`. For more information about `maintainer` roles, see see [Create a team](https://docs.github.com/rest/reference/teams#create-a-team).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#get-team-membership-for-a-user)
    pub async fn teams_get_membership_for_user_in_org(
        &self,
        org: &str,
        team_slug: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_get_membership_for_user_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_get_membership_for_user_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add or update team membership for a user
    ///
    /// Team synchronization is available for organizations using GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Adds an organization member to a team. An authenticated organization owner or team maintainer can add organization members to a team.
    ///
    /// **Note:** When you have team synchronization set up for a team with your organization's identity provider (IdP), you will see an error if you attempt to use the API for making changes to the team's membership. If you have access to manage group membership in your IdP, you can manage GitHub team membership through your identity provider, which automatically adds and removes team members in an organization. For more information, see "[Synchronizing teams between your identity provider and GitHub](https://docs.github.com/articles/synchronizing-teams-between-your-identity-provider-and-github/)."
    ///
    /// An organization owner can add someone who is not part of the team's organization to a team. When an organization owner adds someone to a team who is not an organization member, this endpoint will send an invitation to the person via email. This newly-created membership will be in the "pending" state until the person accepts the invitation, at which point the membership will transition to the "active" state and the user will be added as a member of the team.
    ///
    /// If the user is already a member of the team, this endpoint will update the role of the team member's role. To update the membership of a team member, the authenticated user must be an organization owner or a team maintainer.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `PUT /organizations/{org_id}/team/{team_id}/memberships/{username}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#add-or-update-team-membership-for-a-user)
    ///
    /// # Content
    ///
    /// - [`&request::teams_add_or_update_membership_for_user_in_org::body::Json`](crate::v1_1_4::request::teams_add_or_update_membership_for_user_in_org::body::Json)
    pub async fn teams_add_or_update_membership_for_user_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        username: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_add_or_update_membership_for_user_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_add_or_update_membership_for_user_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_add_or_update_membership_for_user_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_add_or_update_membership_for_user_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove team membership for a user
    ///
    /// Team synchronization is available for organizations using GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// To remove a membership between a user and a team, the authenticated user must have 'admin' permissions to the team or be an owner of the organization that the team is associated with. Removing team membership does not delete the user, it just removes their membership from the team.
    ///
    /// **Note:** When you have team synchronization set up for a team with your organization's identity provider (IdP), you will see an error if you attempt to use the API for making changes to the team's membership. If you have access to manage group membership in your IdP, you can manage GitHub team membership through your identity provider, which automatically adds and removes team members in an organization. For more information, see "[Synchronizing teams between your identity provider and GitHub](https://docs.github.com/articles/synchronizing-teams-between-your-identity-provider-and-github/)."
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `DELETE /organizations/{org_id}/team/{team_id}/memberships/{username}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#remove-team-membership-for-a-user)
    pub async fn teams_remove_membership_for_user_in_org(
        &self,
        org: &str,
        team_slug: &str,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_remove_membership_for_user_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_remove_membership_for_user_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List team projects
    ///
    /// Lists the organization projects for a team.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/projects`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-team-projects)
    pub async fn teams_list_projects_in_org(
        &self,
        org: &str,
        team_slug: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_projects_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_projects_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check team permissions for a project
    ///
    /// Checks whether a team has `read`, `write`, or `admin` permissions for an organization project. The response includes projects inherited from a parent team.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/projects/{project_id}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#check-team-permissions-for-a-project)
    pub async fn teams_check_permissions_for_project_in_org(
        &self,
        org: &str,
        team_slug: &str,
        project_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_check_permissions_for_project_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                project_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_check_permissions_for_project_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add or update team project permissions
    ///
    /// Adds an organization project to a team. To add a project to a team or update the team's permission on a project, the authenticated user must have `admin` permissions for the project. The project and team must be part of the same organization.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `PUT /organizations/{org_id}/team/{team_id}/projects/{project_id}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#add-or-update-team-project-permissions)
    ///
    /// # Content
    ///
    /// - [`&::std::option::Option<request::teams_add_or_update_project_permissions_in_org::body::Json>`](crate::v1_1_4::request::teams_add_or_update_project_permissions_in_org::body::Json)
    pub async fn teams_add_or_update_project_permissions_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        project_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_add_or_update_project_permissions_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_add_or_update_project_permissions_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_add_or_update_project_permissions_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                project_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_add_or_update_project_permissions_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a project from a team
    ///
    /// Removes an organization project from a team. An organization owner or a team maintainer can remove any project from the team. To remove a project from a team as an organization member, the authenticated user must have `read` access to both the team and project, or `admin` access to the team or project. This endpoint removes the project from the team, but does not delete the project.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `DELETE /organizations/{org_id}/team/{team_id}/projects/{project_id}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#remove-a-project-from-a-team)
    pub async fn teams_remove_project_in_org(
        &self,
        org: &str,
        team_slug: &str,
        project_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_remove_project_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                project_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_remove_project_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List team repositories
    ///
    /// Lists a team's repositories visible to the authenticated user.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/repos`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-team-repositories)
    pub async fn teams_list_repos_in_org(
        &self,
        org: &str,
        team_slug: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_repos_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_repos_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check team permissions for a repository
    ///
    /// Checks whether a team has `admin`, `push`, `maintain`, `triage`, or `pull` permission for a repository. Repositories inherited through a parent team will also be checked.
    ///
    /// You can also get information about the specified repository, including what permissions the team grants on it, by passing the following custom [media type](https://docs.github.com/rest/overview/media-types/) via the `application/vnd.github.v3.repository+json` accept header.
    ///
    /// If a team doesn't have permission for the repository, you will receive a `404 Not Found` response status.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/repos/{owner}/{repo}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams/#check-team-permissions-for-a-repository)
    pub async fn teams_check_permissions_for_repo_in_org(
        &self,
        org: &str,
        team_slug: &str,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_check_permissions_for_repo_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_check_permissions_for_repo_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add or update team repository permissions
    ///
    /// To add a repository to a team or update the team's permission on a repository, the authenticated user must have admin access to the repository, and must be able to see the team. The repository must be owned by the organization, or a direct fork of a repository owned by the organization. You will get a `422 Unprocessable Entity` status if you attempt to add a repository to a team that is not owned by the organization. Note that, if you choose not to pass any parameters, you'll need to set `Content-Length` to zero when calling out to this endpoint. For more information, see "[HTTP verbs](https://docs.github.com/rest/overview/resources-in-the-rest-api#http-verbs)."
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `PUT /organizations/{org_id}/team/{team_id}/repos/{owner}/{repo}`.
    ///
    /// For more information about the permission levels, see "[Repository permission levels for an organization](https://docs.github.com/en/github/setting-up-and-managing-organizations-and-teams/repository-permission-levels-for-an-organization#permission-levels-for-repositories-owned-by-an-organization)".
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams/#add-or-update-team-repository-permissions)
    ///
    /// # Content
    ///
    /// - [`&request::teams_add_or_update_repo_permissions_in_org::body::Json`](crate::v1_1_4::request::teams_add_or_update_repo_permissions_in_org::body::Json)
    pub async fn teams_add_or_update_repo_permissions_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_add_or_update_repo_permissions_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_add_or_update_repo_permissions_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_add_or_update_repo_permissions_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_add_or_update_repo_permissions_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a repository from a team
    ///
    /// If the authenticated user is an organization owner or a team maintainer, they can remove any repositories from the team. To remove a repository from a team as an organization member, the authenticated user must have admin access to the repository and must be able to see the team. This does not delete the repository, it just removes it from the team.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `DELETE /organizations/{org_id}/team/{team_id}/repos/{owner}/{repo}`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams/#remove-a-repository-from-a-team)
    pub async fn teams_remove_repo_in_org(
        &self,
        org: &str,
        team_slug: &str,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_remove_repo_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_remove_repo_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List IdP groups for a team
    ///
    /// Team synchronization is available for organizations using GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// List IdP groups connected to a team on GitHub.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/team-sync/group-mappings`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-idp-groups-for-a-team)
    pub async fn teams_list_idp_groups_in_org(
        &self,
        org: &str,
        team_slug: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_idp_groups_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_idp_groups_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create or update IdP group connections
    ///
    /// Team synchronization is available for organizations using GitHub Enterprise Cloud. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Creates, updates, or removes a connection between a team and an IdP group. When adding groups to a team, you must include all new and existing groups to avoid replacing existing groups with the new ones. Specifying an empty `groups` array will remove all connections for a team.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `PATCH /organizations/{org_id}/team/{team_id}/team-sync/group-mappings`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#create-or-update-idp-group-connections)
    ///
    /// # Content
    ///
    /// - [`&request::teams_create_or_update_idp_group_connections_in_org::body::Json`](crate::v1_1_4::request::teams_create_or_update_idp_group_connections_in_org::body::Json)
    pub async fn teams_create_or_update_idp_group_connections_in_org<Content>(
        &self,
        org: &str,
        team_slug: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::teams_create_or_update_idp_group_connections_in_org::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::teams_create_or_update_idp_group_connections_in_org::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_create_or_update_idp_group_connections_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::teams_create_or_update_idp_group_connections_in_org::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List child teams
    ///
    /// Lists the child teams of the team specified by `{team_slug}`.
    ///
    /// **Note:** You can also specify a team by `org_id` and `team_id` using the route `GET /organizations/{org_id}/team/{team_id}/teams`.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/teams#list-child-teams)
    pub async fn teams_list_child_in_org(
        &self,
        org: &str,
        team_slug: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::teams_list_child_in_org::reqwest_builder(
                self.config.base_url.as_ref(),
                org,
                team_slug,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::teams_list_child_in_org::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a project card
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#get-a-project-card)
    pub async fn projects_get_card(
        &self,
        card_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_get_card::reqwest_builder(
                self.config.base_url.as_ref(),
                card_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_get_card::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a project card
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#delete-a-project-card)
    pub async fn projects_delete_card(
        &self,
        card_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_delete_card::reqwest_builder(
                self.config.base_url.as_ref(),
                card_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_delete_card::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update an existing project card
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#update-a-project-card)
    ///
    /// # Content
    ///
    /// - [`&request::projects_update_card::body::Json`](crate::v1_1_4::request::projects_update_card::body::Json)
    pub async fn projects_update_card<Content>(
        &self,
        card_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_update_card::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_update_card::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_update_card::reqwest_builder(
                self.config.base_url.as_ref(),
                card_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_update_card::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Move a project card
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#move-a-project-card)
    ///
    /// # Content
    ///
    /// - [`&request::projects_move_card::body::Json`](crate::v1_1_4::request::projects_move_card::body::Json)
    pub async fn projects_move_card<Content>(
        &self,
        card_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_move_card::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_move_card::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_move_card::reqwest_builder(
                self.config.base_url.as_ref(),
                card_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_move_card::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a project column
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#get-a-project-column)
    pub async fn projects_get_column(
        &self,
        column_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_get_column::reqwest_builder(
                self.config.base_url.as_ref(),
                column_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_get_column::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a project column
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#delete-a-project-column)
    pub async fn projects_delete_column(
        &self,
        column_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_delete_column::reqwest_builder(
                self.config.base_url.as_ref(),
                column_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_delete_column::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update an existing project column
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#update-a-project-column)
    ///
    /// # Content
    ///
    /// - [`&request::projects_update_column::body::Json`](crate::v1_1_4::request::projects_update_column::body::Json)
    pub async fn projects_update_column<Content>(
        &self,
        column_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_update_column::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_update_column::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_update_column::reqwest_builder(
                self.config.base_url.as_ref(),
                column_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_update_column::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List project cards
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#list-project-cards)
    pub async fn projects_list_cards(
        &self,
        column_id: i64,
        archived_state: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_list_cards::reqwest_builder(
                self.config.base_url.as_ref(),
                column_id,
                archived_state,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_list_cards::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a project card
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#create-a-project-card)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn projects_create_card<Content>(
        &self,
        column_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_create_card::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_create_card::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_create_card::reqwest_builder(
                self.config.base_url.as_ref(),
                column_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_create_card::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Move a project column
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#move-a-project-column)
    ///
    /// # Content
    ///
    /// - [`&request::projects_move_column::body::Json`](crate::v1_1_4::request::projects_move_column::body::Json)
    pub async fn projects_move_column<Content>(
        &self,
        column_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_move_column::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_move_column::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_move_column::reqwest_builder(
                self.config.base_url.as_ref(),
                column_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_move_column::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a project
    ///
    /// Gets a project by its `id`. Returns a `404 Not Found` status if projects are disabled. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#get-a-project)
    pub async fn projects_get(
        &self,
        project_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_get::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a project
    ///
    /// Deletes a project board. Returns a `404 Not Found` status if projects are disabled.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#delete-a-project)
    pub async fn projects_delete(
        &self,
        project_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_delete::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_delete::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a project
    ///
    /// Updates a project board's information. Returns a `404 Not Found` status if projects are disabled. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#update-a-project)
    ///
    /// # Content
    ///
    /// - [`&request::projects_update::body::Json`](crate::v1_1_4::request::projects_update::body::Json)
    pub async fn projects_update<Content>(
        &self,
        project_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_update::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_update::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_update::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_update::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List project collaborators
    ///
    /// Lists the collaborators for an organization project. For a project, the list of collaborators includes outside collaborators, organization members that are direct collaborators, organization members with access through team memberships, organization members with access through default organization permissions, and organization owners. You must be an organization owner or a project `admin` to list collaborators.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#list-project-collaborators)
    pub async fn projects_list_collaborators(
        &self,
        project_id: i64,
        affiliation: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_list_collaborators::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                affiliation,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_list_collaborators::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add project collaborator
    ///
    /// Adds a collaborator to an organization project and sets their permission level. You must be an organization owner or a project `admin` to add a collaborator.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#add-project-collaborator)
    ///
    /// # Content
    ///
    /// - [`&::std::option::Option<request::projects_add_collaborator::body::Json>`](crate::v1_1_4::request::projects_add_collaborator::body::Json)
    pub async fn projects_add_collaborator<Content>(
        &self,
        project_id: i64,
        username: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_add_collaborator::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_add_collaborator::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_add_collaborator::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_add_collaborator::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove user as a collaborator
    ///
    /// Removes a collaborator from an organization project. You must be an organization owner or a project `admin` to remove a collaborator.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#remove-project-collaborator)
    pub async fn projects_remove_collaborator(
        &self,
        project_id: i64,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_remove_collaborator::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_remove_collaborator::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get project permission for a user
    ///
    /// Returns the collaborator's permission level for an organization project. Possible values for the `permission` key: `admin`, `write`, `read`, `none`. You must be an organization owner or a project `admin` to review a user's permission level.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#get-project-permission-for-a-user)
    pub async fn projects_get_permission_for_user(
        &self,
        project_id: i64,
        username: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_get_permission_for_user::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                username,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_get_permission_for_user::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List project columns
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#list-project-columns)
    pub async fn projects_list_columns(
        &self,
        project_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_list_columns::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::projects_list_columns::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a project column
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/projects#create-a-project-column)
    ///
    /// # Content
    ///
    /// - [`&request::projects_create_column::body::Json`](crate::v1_1_4::request::projects_create_column::body::Json)
    pub async fn projects_create_column<Content>(
        &self,
        project_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::projects_create_column::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::projects_create_column::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::projects_create_column::reqwest_builder(
                self.config.base_url.as_ref(),
                project_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::projects_create_column::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get rate limit status for the authenticated user
    ///
    /// **Note:** Accessing this endpoint does not count against your REST API rate limit.
    ///
    /// **Note:** The `rate` object is deprecated. If you're writing new API client code or updating existing code, you should use the `core` object instead of the `rate` object. The `core` object contains the same information that is present in the `rate` object.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/rate-limit#get-rate-limit-status-for-the-authenticated-user)
    pub async fn rate_limit_get(
        &self,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::rate_limit_get::reqwest_builder(
                self.config.base_url.as_ref(),
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::rate_limit_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a repository
    ///
    /// The `parent` and `source` objects are present when the repository is a fork. `parent` is the repository this repository was forked from, `source` is the ultimate source for the network.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-a-repository)
    pub async fn repos_get(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a repository
    ///
    /// Deleting a repository requires admin access. If OAuth is used, the `delete_repo` scope is required.
    ///
    /// If an organization owner has configured the organization to prevent members from deleting organization-owned
    /// repositories, you will get a `403 Forbidden` response.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#delete-a-repository)
    pub async fn repos_delete(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_delete::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_delete::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a repository
    ///
    /// **Note**: To edit a repository's topics, use the [Replace all repository topics](https://docs.github.com/rest/reference/repos#replace-all-repository-topics) endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos/#update-a-repository)
    ///
    /// # Content
    ///
    /// - [`&request::repos_update::body::Json`](crate::v1_1_4::request::repos_update::body::Json)
    pub async fn repos_update<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_update::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_update::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_update::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_update::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List artifacts for a repository
    ///
    /// Lists all artifacts for a repository. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-artifacts-for-a-repository)
    pub async fn actions_list_artifacts_for_repo(
        &self,
        owner: &str,
        repo: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_artifacts_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_artifacts_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an artifact
    ///
    /// Gets a specific artifact for a workflow run. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-an-artifact)
    pub async fn actions_get_artifact(
        &self,
        owner: &str,
        repo: &str,
        artifact_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_artifact::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                artifact_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_artifact::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an artifact
    ///
    /// Deletes an artifact for a workflow run. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-an-artifact)
    pub async fn actions_delete_artifact(
        &self,
        owner: &str,
        repo: &str,
        artifact_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_artifact::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                artifact_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_artifact::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Download an artifact
    ///
    /// Gets a redirect URL to download an archive for a repository. This URL expires after 1 minute. Look for `Location:` in
    /// the response header to find the URL for the download. The `:archive_format` must be `zip`. Anyone with read access to
    /// the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope.
    /// GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#download-an-artifact)
    pub async fn actions_download_artifact(
        &self,
        owner: &str,
        repo: &str,
        artifact_id: i64,
        archive_format: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_download_artifact::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                artifact_id,
                archive_format,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_download_artifact::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions cache usage for a repository
    ///
    /// Gets GitHub Actions cache usage for a repository.
    /// The data fetched using this API is refreshed approximately every 5 minutes, so values returned from this endpoint may take at least 5 minutes to get updated.
    /// Anyone with read access to the repository can use this endpoint. If the repository is private, you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-github-actions-cache-usage-for-a-repository)
    pub async fn actions_get_actions_cache_usage(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_actions_cache_usage::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_actions_cache_usage::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a job for a workflow run
    ///
    /// Gets a specific job in a workflow run. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-job-for-a-workflow-run)
    pub async fn actions_get_job_for_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        job_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_job_for_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                job_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_job_for_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Download job logs for a workflow run
    ///
    /// Gets a redirect URL to download a plain text file of logs for a workflow job. This link expires after 1 minute. Look
    /// for `Location:` in the response header to find the URL for the download. Anyone with read access to the repository can
    /// use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must
    /// have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#download-job-logs-for-a-workflow-run)
    pub async fn actions_download_job_logs_for_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        job_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_download_job_logs_for_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                job_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_download_job_logs_for_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Re-run a job from a workflow run
    ///
    /// Re-run a job and its dependent jobs in a workflow run. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#re-run-job-for-workflow-run)
    pub async fn actions_re_run_job_for_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        job_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_re_run_job_for_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                job_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_re_run_job_for_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get GitHub Actions permissions for a repository
    ///
    /// Gets the GitHub Actions permissions policy for a repository, including whether GitHub Actions is enabled and the actions and reusable workflows allowed to run in the repository.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `administration` repository permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-github-actions-permissions-for-a-repository)
    pub async fn actions_get_github_actions_permissions_repository(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_github_actions_permissions_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_github_actions_permissions_repository::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set GitHub Actions permissions for a repository
    ///
    /// Sets the GitHub Actions permissions policy for enabling GitHub Actions and allowed actions and reusable workflows in the repository.
    ///
    /// If the repository belongs to an organization or enterprise that has set restrictive permissions at the organization or enterprise levels, such as `allowed_actions` to `selected` actions and reusable workflows, then you cannot override them for the repository.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `administration` repository permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-github-actions-permissions-for-a-repository)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_github_actions_permissions_repository::body::Json`](crate::v1_1_4::request::actions_set_github_actions_permissions_repository::body::Json)
    pub async fn actions_set_github_actions_permissions_repository<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_github_actions_permissions_repository::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_github_actions_permissions_repository::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_github_actions_permissions_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_github_actions_permissions_repository::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get the level of access for workflows outside of the repository
    ///
    /// Gets the level of access that workflows outside of the repository have to actions and reusable workflows in the repository.
    /// This endpoint only applies to internal repositories. For more information, see "[Managing GitHub Actions settings for a repository](https://docs.github.com/repositories/managing-your-repositorys-settings-and-features/enabling-features-for-your-repository/managing-github-actions-settings-for-a-repository#allowing-access-to-components-in-an-internal-repository)."
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the
    /// repository `administration` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-workflow-access-level-to-a-repository)
    pub async fn actions_get_workflow_access_to_repository(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_workflow_access_to_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_workflow_access_to_repository::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set the level of access for workflows outside of the repository
    ///
    /// Sets the level of access that workflows outside of the repository have to actions and reusable workflows in the repository.
    /// This endpoint only applies to internal repositories. For more information, see "[Managing GitHub Actions settings for a repository](https://docs.github.com/repositories/managing-your-repositorys-settings-and-features/enabling-features-for-your-repository/managing-github-actions-settings-for-a-repository#allowing-access-to-components-in-an-internal-repository)."
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the
    /// repository `administration` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-workflow-access-to-a-repository)
    ///
    /// # Content
    ///
    /// - [`&schema::ActionsWorkflowAccessToRepository`](crate::v1_1_4::schema::ActionsWorkflowAccessToRepository)
    pub async fn actions_set_workflow_access_to_repository<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_workflow_access_to_repository::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_workflow_access_to_repository::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_workflow_access_to_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_workflow_access_to_repository::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get allowed actions and reusable workflows for a repository
    ///
    /// Gets the settings for selected actions and reusable workflows that are allowed in a repository. To use this endpoint, the repository policy for `allowed_actions` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for a repository](#set-github-actions-permissions-for-a-repository)."
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `administration` repository permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-allowed-actions-for-a-repository)
    pub async fn actions_get_allowed_actions_repository(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_allowed_actions_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_allowed_actions_repository::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set allowed actions and reusable workflows for a repository
    ///
    /// Sets the actions and reusable workflows that are allowed in a repository. To use this endpoint, the repository permission policy for `allowed_actions` must be configured to `selected`. For more information, see "[Set GitHub Actions permissions for a repository](#set-github-actions-permissions-for-a-repository)."
    ///
    /// If the repository belongs to an organization or enterprise that has `selected` actions and reusable workflows set at the organization or enterprise levels, then you cannot override any of the allowed actions and reusable workflows settings.
    ///
    /// To use the `patterns_allowed` setting for private repositories, the repository must belong to an enterprise. If the repository does not belong to an enterprise, then the `patterns_allowed` setting only applies to public repositories.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `administration` repository permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-allowed-actions-for-a-repository)
    ///
    /// # Content
    ///
    /// - [`&schema::SelectedActions`](crate::v1_1_4::schema::SelectedActions)
    pub async fn actions_set_allowed_actions_repository<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_allowed_actions_repository::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_allowed_actions_repository::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_allowed_actions_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_allowed_actions_repository::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get default workflow permissions for a repository
    ///
    /// Gets the default workflow permissions granted to the `GITHUB_TOKEN` when running workflows in a repository,
    /// as well as if GitHub Actions can submit approving pull request reviews.
    /// For more information, see "[Setting the permissions of the GITHUB_TOKEN for your repository](https://docs.github.com/repositories/managing-your-repositorys-settings-and-features/enabling-features-for-your-repository/managing-github-actions-settings-for-a-repository#setting-the-permissions-of-the-github_token-for-your-repository)."
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the repository `administration` permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-default-workflow-permissions-for-a-repository)
    pub async fn actions_get_github_actions_default_workflow_permissions_repository(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_github_actions_default_workflow_permissions_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_github_actions_default_workflow_permissions_repository::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set default workflow permissions for a repository
    ///
    /// Sets the default workflow permissions granted to the `GITHUB_TOKEN` when running workflows in a repository, and sets if GitHub Actions
    /// can submit approving pull request reviews.
    /// For more information, see "[Setting the permissions of the GITHUB_TOKEN for your repository](https://docs.github.com/repositories/managing-your-repositorys-settings-and-features/enabling-features-for-your-repository/managing-github-actions-settings-for-a-repository#setting-the-permissions-of-the-github_token-for-your-repository)."
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the repository `administration` permission to use this API.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-default-workflow-permissions-for-a-repository)
    ///
    /// # Content
    ///
    /// - [`&schema::ActionsSetDefaultWorkflowPermissions`](crate::v1_1_4::schema::ActionsSetDefaultWorkflowPermissions)
    pub async fn actions_set_github_actions_default_workflow_permissions_repository<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_repository::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_repository::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_repository::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_github_actions_default_workflow_permissions_repository::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List self-hosted runners for a repository
    ///
    /// Lists all self-hosted runners configured in a repository. You must authenticate using an access token with the `repo` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-self-hosted-runners-for-a-repository)
    pub async fn actions_list_self_hosted_runners_for_repo(
        &self,
        owner: &str,
        repo: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_self_hosted_runners_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_self_hosted_runners_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List runner applications for a repository
    ///
    /// Lists binaries for the runner application that you can download and run.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-runner-applications-for-a-repository)
    pub async fn actions_list_runner_applications_for_repo(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_runner_applications_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_runner_applications_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a registration token for a repository
    ///
    /// Returns a token that you can pass to the `config` script. The token expires after one hour. You must authenticate
    /// using an access token with the `repo` scope to use this endpoint.
    ///
    /// #### Example using registration token
    ///
    /// Configure your self-hosted runner, replacing `TOKEN` with the registration token provided by this endpoint.
    ///
    /// ```text
    /// ./config.sh --url https://github.com/octo-org/octo-repo-artifacts --token TOKEN
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-registration-token-for-a-repository)
    pub async fn actions_create_registration_token_for_repo(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_registration_token_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_create_registration_token_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a remove token for a repository
    ///
    /// Returns a token that you can pass to remove a self-hosted runner from a repository. The token expires after one hour.
    /// You must authenticate using an access token with the `repo` scope to use this endpoint.
    ///
    /// #### Example using remove token
    ///
    /// To remove your self-hosted runner from a repository, replace TOKEN with the remove token provided by this endpoint.
    ///
    /// ```text
    /// ./config.sh remove --token TOKEN
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-remove-token-for-a-repository)
    pub async fn actions_create_remove_token_for_repo(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_remove_token_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_create_remove_token_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a self-hosted runner for a repository
    ///
    /// Gets a specific self-hosted runner configured in a repository.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this
    /// endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-self-hosted-runner-for-a-repository)
    pub async fn actions_get_self_hosted_runner_for_repo(
        &self,
        owner: &str,
        repo: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_self_hosted_runner_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_self_hosted_runner_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a self-hosted runner from a repository
    ///
    /// Forces the removal of a self-hosted runner from a repository. You can use this endpoint to completely remove the runner when the machine you were using no longer exists.
    ///
    /// You must authenticate using an access token with the `repo`
    /// scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-a-self-hosted-runner-from-a-repository)
    pub async fn actions_delete_self_hosted_runner_from_repo(
        &self,
        owner: &str,
        repo: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_self_hosted_runner_from_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_self_hosted_runner_from_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List labels for a self-hosted runner for a repository
    ///
    /// Lists all labels for a self-hosted runner configured in a repository.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this
    /// endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-labels-for-a-self-hosted-runner-for-a-repository)
    pub async fn actions_list_labels_for_self_hosted_runner_for_repo(
        &self,
        owner: &str,
        repo: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_labels_for_self_hosted_runner_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_labels_for_self_hosted_runner_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set custom labels for a self-hosted runner for a repository
    ///
    /// Remove all previous custom labels and set the new custom labels for a specific
    /// self-hosted runner configured in a repository.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this
    /// endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#set-custom-labels-for-a-self-hosted-runner-for-a-repository)
    ///
    /// # Content
    ///
    /// - [`&request::actions_set_custom_labels_for_self_hosted_runner_for_repo::body::Json`](crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_repo::body::Json)
    pub async fn actions_set_custom_labels_for_self_hosted_runner_for_repo<Content>(
        &self,
        owner: &str,
        repo: &str,
        runner_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_repo::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_repo::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_set_custom_labels_for_self_hosted_runner_for_repo::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add custom labels to a self-hosted runner for a repository
    ///
    /// Add custom labels to a self-hosted runner configured in a repository.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this
    /// endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#add-custom-labels-to-a-self-hosted-runner-for-a-repository)
    ///
    /// # Content
    ///
    /// - [`&request::actions_add_custom_labels_to_self_hosted_runner_for_repo::body::Json`](crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_repo::body::Json)
    pub async fn actions_add_custom_labels_to_self_hosted_runner_for_repo<Content>(
        &self,
        owner: &str,
        repo: &str,
        runner_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_repo::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_repo::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_add_custom_labels_to_self_hosted_runner_for_repo::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove all custom labels from a self-hosted runner for a repository
    ///
    /// Remove all custom labels from a self-hosted runner configured in a
    /// repository. Returns the remaining read-only labels from the runner.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this
    /// endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-all-custom-labels-from-a-self-hosted-runner-for-a-repository)
    pub async fn actions_remove_all_custom_labels_from_self_hosted_runner_for_repo(
        &self,
        owner: &str,
        repo: &str,
        runner_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_remove_all_custom_labels_from_self_hosted_runner_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                runner_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_remove_all_custom_labels_from_self_hosted_runner_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove a custom label from a self-hosted runner for a repository
    ///
    /// Remove a custom label from a self-hosted runner configured
    /// in a repository. Returns the remaining labels from the runner.
    ///
    /// This endpoint returns a `404 Not Found` status if the custom label is not
    /// present on the runner.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this
    /// endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#remove-a-custom-label-from-a-self-hosted-runner-for-a-repository)
    pub async fn actions_remove_custom_label_from_self_hosted_runner_for_repo(
        &self,
        owner: &str,
        repo: &str,
        runner_id: i64,
        name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_remove_custom_label_from_self_hosted_runner_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                runner_id,
                name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_remove_custom_label_from_self_hosted_runner_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List workflow runs for a repository
    ///
    /// Lists all workflow runs for a repository. You can use parameters to narrow the list of results. For more information about using parameters, see [Parameters](https://docs.github.com/rest/overview/resources-in-the-rest-api#parameters).
    ///
    /// Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-workflow-runs-for-a-repository)
    #[allow(clippy::too_many_arguments)]
    pub async fn actions_list_workflow_runs_for_repo(
        &self,
        owner: &str,
        repo: &str,
        actor: ::std::option::Option<&str>,
        branch: ::std::option::Option<&str>,
        event: ::std::option::Option<&str>,
        status: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        created: ::std::option::Option<&str>,
        exclude_pull_requests: ::std::option::Option<bool>,
        check_suite_id: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_workflow_runs_for_repo::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                actor,
                branch,
                event,
                status,
                per_page,
                page,
                created,
                exclude_pull_requests,
                check_suite_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_workflow_runs_for_repo::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a workflow run
    ///
    /// Gets a specific workflow run. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-workflow-run)
    pub async fn actions_get_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
        exclude_pull_requests: ::std::option::Option<bool>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                exclude_pull_requests,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a workflow run
    ///
    /// Delete a specific workflow run. Anyone with write access to the repository can use this endpoint. If the repository is
    /// private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:write` permission to use
    /// this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-a-workflow-run)
    pub async fn actions_delete_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get the review history for a workflow run
    ///
    /// Anyone with read access to the repository can use this endpoint. If the repository is private, you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-the-review-history-for-a-workflow-run)
    pub async fn actions_get_reviews_for_run(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_reviews_for_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_reviews_for_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Approve a workflow run for a fork pull request
    ///
    /// Approves a workflow run for a pull request from a public fork of a first time contributor. For more information, see ["Approving workflow runs from public forks](https://docs.github.com/actions/managing-workflow-runs/approving-workflow-runs-from-public-forks)."
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#approve-a-workflow-run-for-a-fork-pull-request)
    pub async fn actions_approve_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_approve_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_approve_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List workflow run artifacts
    ///
    /// Lists artifacts for a workflow run. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-workflow-run-artifacts)
    pub async fn actions_list_workflow_run_artifacts(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_workflow_run_artifacts::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_workflow_run_artifacts::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a workflow run attempt
    ///
    /// Gets a specific workflow run attempt. Anyone with read access to the repository
    /// can use this endpoint. If the repository is private you must use an access token
    /// with the `repo` scope. GitHub Apps must have the `actions:read` permission to
    /// use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-workflow-run-attempt)
    pub async fn actions_get_workflow_run_attempt(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
        attempt_number: i64,
        exclude_pull_requests: ::std::option::Option<bool>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_workflow_run_attempt::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                attempt_number,
                exclude_pull_requests,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_workflow_run_attempt::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List jobs for a workflow run attempt
    ///
    /// Lists jobs for a specific workflow run attempt. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint. You can use parameters to narrow the list of results. For more information about using parameters, see [Parameters](https://docs.github.com/rest/overview/resources-in-the-rest-api#parameters).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-jobs-for-a-workflow-run-attempt)
    pub async fn actions_list_jobs_for_workflow_run_attempt(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
        attempt_number: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_jobs_for_workflow_run_attempt::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                attempt_number,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_jobs_for_workflow_run_attempt::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Download workflow run attempt logs
    ///
    /// Gets a redirect URL to download an archive of log files for a specific workflow run attempt. This link expires after
    /// 1 minute. Look for `Location:` in the response header to find the URL for the download. Anyone with read access to
    /// the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope.
    /// GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#download-workflow-run-attempt-logs)
    pub async fn actions_download_workflow_run_attempt_logs(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
        attempt_number: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_download_workflow_run_attempt_logs::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                attempt_number,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_download_workflow_run_attempt_logs::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Cancel a workflow run
    ///
    /// Cancels a workflow run using its `id`. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#cancel-a-workflow-run)
    pub async fn actions_cancel_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_cancel_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_cancel_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List jobs for a workflow run
    ///
    /// Lists jobs for a workflow run. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint. You can use parameters to narrow the list of results. For more information about using parameters, see [Parameters](https://docs.github.com/rest/overview/resources-in-the-rest-api#parameters).
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-jobs-for-a-workflow-run)
    pub async fn actions_list_jobs_for_workflow_run(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
        filter: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_jobs_for_workflow_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                filter,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_jobs_for_workflow_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Download workflow run logs
    ///
    /// Gets a redirect URL to download an archive of log files for a workflow run. This link expires after 1 minute. Look for
    /// `Location:` in the response header to find the URL for the download. Anyone with read access to the repository can use
    /// this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have
    /// the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#download-workflow-run-logs)
    pub async fn actions_download_workflow_run_logs(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_download_workflow_run_logs::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_download_workflow_run_logs::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete workflow run logs
    ///
    /// Deletes all logs for a workflow run. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-workflow-run-logs)
    pub async fn actions_delete_workflow_run_logs(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_workflow_run_logs::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_workflow_run_logs::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get pending deployments for a workflow run
    ///
    /// Get all deployment environments for a workflow run that are waiting for protection rules to pass.
    ///
    /// Anyone with read access to the repository can use this endpoint. If the repository is private, you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-pending-deployments-for-a-workflow-run)
    pub async fn actions_get_pending_deployments_for_run(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_pending_deployments_for_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_pending_deployments_for_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Review pending deployments for a workflow run
    ///
    /// Approve or reject pending deployments that are waiting on approval by a required reviewer.
    ///
    /// Anyone with read access to the repository contents and deployments can use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#review-pending-deployments-for-a-workflow-run)
    ///
    /// # Content
    ///
    /// - [`&request::actions_review_pending_deployments_for_run::body::Json`](crate::v1_1_4::request::actions_review_pending_deployments_for_run::body::Json)
    pub async fn actions_review_pending_deployments_for_run<Content>(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_review_pending_deployments_for_run::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_review_pending_deployments_for_run::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_review_pending_deployments_for_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_review_pending_deployments_for_run::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Re-run a workflow
    ///
    /// Re-runs your workflow run using its `id`. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#re-run-a-workflow)
    pub async fn actions_re_run_workflow(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_re_run_workflow::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_re_run_workflow::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Re-run failed jobs from a workflow run
    ///
    /// Re-run all of the failed jobs and their dependent jobs in a workflow run using the `id` of the workflow run. You must authenticate using an access token with the `repo` scope to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#re-run-workflow-failed-jobs)
    pub async fn actions_re_run_workflow_failed_jobs(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_re_run_workflow_failed_jobs::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_re_run_workflow_failed_jobs::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get workflow run usage
    ///
    /// Gets the number of billable minutes and total run time for a specific workflow run. Billable minutes only apply to workflows in private repositories that use GitHub-hosted runners. Usage is listed for each GitHub-hosted runner operating system in milliseconds. Any job re-runs are also included in the usage. The usage does not include the multiplier for macOS and Windows runners and is not rounded up to the nearest whole minute. For more information, see "[Managing billing for GitHub Actions](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-actions)".
    ///
    /// Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-workflow-run-usage)
    pub async fn actions_get_workflow_run_usage(
        &self,
        owner: &str,
        repo: &str,
        run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_workflow_run_usage::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_workflow_run_usage::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List repository secrets
    ///
    /// Lists all secrets available in a repository without revealing their encrypted values. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `secrets` repository permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-repository-secrets)
    pub async fn actions_list_repo_secrets(
        &self,
        owner: &str,
        repo: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_repo_secrets::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_repo_secrets::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a repository public key
    ///
    /// Gets your public key, which you need to encrypt secrets. You need to encrypt a secret before you can create or update secrets. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `secrets` repository permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-repository-public-key)
    pub async fn actions_get_repo_public_key(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_repo_public_key::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_repo_public_key::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a repository secret
    ///
    /// Gets a single repository secret without revealing its encrypted value. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `secrets` repository permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-repository-secret)
    pub async fn actions_get_repo_secret(
        &self,
        owner: &str,
        repo: &str,
        secret_name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_repo_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_repo_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create or update a repository secret
    ///
    /// Creates or updates a repository secret with an encrypted value. Encrypt your secret using
    /// [LibSodium](https://libsodium.gitbook.io/doc/bindings_for_other_languages). You must authenticate using an access
    /// token with the `repo` scope to use this endpoint. GitHub Apps must have the `secrets` repository permission to use
    /// this endpoint.
    ///
    /// #### Example encrypting a secret using Node.js
    ///
    /// Encrypt your secret using the [tweetsodium](https://github.com/github/tweetsodium) library.
    ///
    /// ```text
    /// const sodium = require('tweetsodium');
    ///
    /// const key = "base64-encoded-public-key";
    /// const value = "plain-text-secret";
    ///
    /// // Convert the message and key to Uint8Array's (Buffer implements that interface)
    /// const messageBytes = Buffer.from(value);
    /// const keyBytes = Buffer.from(key, 'base64');
    ///
    /// // Encrypt using LibSodium.
    /// const encryptedBytes = sodium.seal(messageBytes, keyBytes);
    ///
    /// // Base64 the encrypted secret
    /// const encrypted = Buffer.from(encryptedBytes).toString('base64');
    ///
    /// console.log(encrypted);
    /// ```
    ///
    ///
    /// #### Example encrypting a secret using Python
    ///
    /// Encrypt your secret using [pynacl](https://pynacl.readthedocs.io/en/latest/public/#nacl-public-sealedbox) with Python 3.
    ///
    /// ```text
    /// from base64 import b64encode
    /// from nacl import encoding, public
    ///
    /// def encrypt(public_key: str, secret_value: str) -> str:
    ///   """Encrypt a Unicode string using the public key."""
    ///   public_key = public.PublicKey(public_key.encode("utf-8"), encoding.Base64Encoder())
    ///   sealed_box = public.SealedBox(public_key)
    ///   encrypted = sealed_box.encrypt(secret_value.encode("utf-8"))
    ///   return b64encode(encrypted).decode("utf-8")
    /// ```
    ///
    /// #### Example encrypting a secret using C#
    ///
    /// Encrypt your secret using the [Sodium.Core](https://www.nuget.org/packages/Sodium.Core/) package.
    ///
    /// ```text
    /// var secretValue = System.Text.Encoding.UTF8.GetBytes("mySecret");
    /// var publicKey = Convert.FromBase64String("2Sg8iYjAxxmI2LvUXpJjkYrMxURPc8r+dB7TJyvvcCU=");
    ///
    /// var sealedPublicKeyBox = Sodium.SealedPublicKeyBox.Create(secretValue, publicKey);
    ///
    /// Console.WriteLine(Convert.ToBase64String(sealedPublicKeyBox));
    /// ```
    ///
    /// #### Example encrypting a secret using Ruby
    ///
    /// Encrypt your secret using the [rbnacl](https://github.com/RubyCrypto/rbnacl) gem.
    ///
    /// ```ruby
    /// require "rbnacl"
    /// require "base64"
    ///
    /// key = Base64.decode64("+ZYvJDZMHUfBkJdyq5Zm9SKqeuBQ4sj+6sfjlH4CgG0=")
    /// public_key = RbNaCl::PublicKey.new(key)
    ///
    /// box = RbNaCl::Boxes::Sealed.from_public_key(public_key)
    /// encrypted_secret = box.encrypt("my_secret")
    ///
    /// # Print the base64 encoded secret
    /// puts Base64.strict_encode64(encrypted_secret)
    /// ```
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-or-update-a-repository-secret)
    ///
    /// # Content
    ///
    /// - [`&request::actions_create_or_update_repo_secret::body::Json`](crate::v1_1_4::request::actions_create_or_update_repo_secret::body::Json)
    pub async fn actions_create_or_update_repo_secret<Content>(
        &self,
        owner: &str,
        repo: &str,
        secret_name: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_create_or_update_repo_secret::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_create_or_update_repo_secret::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_or_update_repo_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_create_or_update_repo_secret::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete a repository secret
    ///
    /// Deletes a secret in a repository using the secret name. You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `secrets` repository permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#delete-a-repository-secret)
    pub async fn actions_delete_repo_secret(
        &self,
        owner: &str,
        repo: &str,
        secret_name: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_delete_repo_secret::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                secret_name,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_delete_repo_secret::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List repository workflows
    ///
    /// Lists the workflows in a repository. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-repository-workflows)
    pub async fn actions_list_repo_workflows(
        &self,
        owner: &str,
        repo: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_repo_workflows::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_repo_workflows::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a workflow
    ///
    /// Gets a specific workflow. You can replace `workflow_id` with the workflow file name. For example, you could use `main.yaml`. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-a-workflow)
    pub async fn actions_get_workflow(
        &self,
        owner: &str,
        repo: &str,
        workflow_id: &::serde_json::value::Value,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_workflow::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                workflow_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_workflow::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Disable a workflow
    ///
    /// Disables a workflow and sets the `state` of the workflow to `disabled_manually`. You can replace `workflow_id` with the workflow file name. For example, you could use `main.yaml`.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#disable-a-workflow)
    pub async fn actions_disable_workflow(
        &self,
        owner: &str,
        repo: &str,
        workflow_id: &::serde_json::value::Value,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_disable_workflow::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                workflow_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_disable_workflow::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a workflow dispatch event
    ///
    /// You can use this endpoint to manually trigger a GitHub Actions workflow run. You can replace `workflow_id` with the workflow file name. For example, you could use `main.yaml`.
    ///
    /// You must configure your GitHub Actions workflow to run when the [`workflow_dispatch` webhook](/developers/webhooks-and-events/webhook-events-and-payloads#workflow_dispatch) event occurs. The `inputs` are configured in the workflow file. For more information about how to configure the `workflow_dispatch` event in the workflow file, see "[Events that trigger workflows](/actions/reference/events-that-trigger-workflows#workflow_dispatch)."
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint. For more information, see "[Creating a personal access token for the command line](https://docs.github.com/articles/creating-a-personal-access-token-for-the-command-line)."
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#create-a-workflow-dispatch-event)
    ///
    /// # Content
    ///
    /// - [`&request::actions_create_workflow_dispatch::body::Json`](crate::v1_1_4::request::actions_create_workflow_dispatch::body::Json)
    pub async fn actions_create_workflow_dispatch<Content>(
        &self,
        owner: &str,
        repo: &str,
        workflow_id: &::serde_json::value::Value,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::actions_create_workflow_dispatch::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::actions_create_workflow_dispatch::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_create_workflow_dispatch::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                workflow_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::actions_create_workflow_dispatch::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Enable a workflow
    ///
    /// Enables a workflow and sets the `state` of the workflow to `active`. You can replace `workflow_id` with the workflow file name. For example, you could use `main.yaml`.
    ///
    /// You must authenticate using an access token with the `repo` scope to use this endpoint. GitHub Apps must have the `actions:write` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#enable-a-workflow)
    pub async fn actions_enable_workflow(
        &self,
        owner: &str,
        repo: &str,
        workflow_id: &::serde_json::value::Value,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_enable_workflow::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                workflow_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_enable_workflow::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List workflow runs
    ///
    /// List all workflow runs for a workflow. You can replace `workflow_id` with the workflow file name. For example, you could use `main.yaml`. You can use parameters to narrow the list of results. For more information about using parameters, see [Parameters](https://docs.github.com/rest/overview/resources-in-the-rest-api#parameters).
    ///
    /// Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#list-workflow-runs)
    #[allow(clippy::too_many_arguments)]
    pub async fn actions_list_workflow_runs(
        &self,
        owner: &str,
        repo: &str,
        workflow_id: &::serde_json::value::Value,
        actor: ::std::option::Option<&str>,
        branch: ::std::option::Option<&str>,
        event: ::std::option::Option<&str>,
        status: ::std::option::Option<&str>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
        created: ::std::option::Option<&str>,
        exclude_pull_requests: ::std::option::Option<bool>,
        check_suite_id: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_list_workflow_runs::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                workflow_id,
                actor,
                branch,
                event,
                status,
                per_page,
                page,
                created,
                exclude_pull_requests,
                check_suite_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_list_workflow_runs::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get workflow usage
    ///
    /// Gets the number of billable minutes used by a specific workflow during the current billing cycle. Billable minutes only apply to workflows in private repositories that use GitHub-hosted runners. Usage is listed for each GitHub-hosted runner operating system in milliseconds. Any job re-runs are also included in the usage. The usage does not include the multiplier for macOS and Windows runners and is not rounded up to the nearest whole minute. For more information, see "[Managing billing for GitHub Actions](https://docs.github.com/github/setting-up-and-managing-billing-and-payments-on-github/managing-billing-for-github-actions)".
    ///
    /// You can replace `workflow_id` with the workflow file name. For example, you could use `main.yaml`. Anyone with read access to the repository can use this endpoint. If the repository is private you must use an access token with the `repo` scope. GitHub Apps must have the `actions:read` permission to use this endpoint.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/actions#get-workflow-usage)
    pub async fn actions_get_workflow_usage(
        &self,
        owner: &str,
        repo: &str,
        workflow_id: &::serde_json::value::Value,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::actions_get_workflow_usage::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                workflow_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::actions_get_workflow_usage::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List assignees
    ///
    /// Lists the [available assignees](https://docs.github.com/articles/assigning-issues-and-pull-requests-to-other-github-users/) for issues in a repository.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/issues#list-assignees)
    pub async fn issues_list_assignees(
        &self,
        owner: &str,
        repo: &str,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::issues_list_assignees::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::issues_list_assignees::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Check if a user can be assigned
    ///
    /// Checks if a user has permission to be assigned to an issue in this repository.
    ///
    /// If the `assignee` can be assigned to issues in the repository, a `204` header with no content is returned.
    ///
    /// Otherwise a `404` status code is returned.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/issues#check-if-a-user-can-be-assigned)
    pub async fn issues_check_user_can_be_assigned(
        &self,
        owner: &str,
        repo: &str,
        assignee: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::issues_check_user_can_be_assigned::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                assignee,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::issues_check_user_can_be_assigned::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List all autolinks of a repository
    ///
    /// This returns a list of autolinks configured for the given repository.
    ///
    /// Information about autolinks are only available to repository administrators.
    ///
    /// [API method documentation](https://docs.github.com/v3/repos#list-autolinks)
    pub async fn repos_list_autolinks(
        &self,
        owner: &str,
        repo: &str,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_list_autolinks::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_list_autolinks::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create an autolink reference for a repository
    ///
    /// Users with admin access to the repository can create an autolink.
    ///
    /// [API method documentation](https://docs.github.com/v3/repos#create-an-autolink)
    ///
    /// # Content
    ///
    /// - [`&request::repos_create_autolink::body::Json`](crate::v1_1_4::request::repos_create_autolink::body::Json)
    pub async fn repos_create_autolink<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_create_autolink::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_create_autolink::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_create_autolink::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_create_autolink::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get an autolink reference of a repository
    ///
    /// This returns a single autolink reference by ID that was configured for the given repository.
    ///
    /// Information about autolinks are only available to repository administrators.
    ///
    /// [API method documentation](https://docs.github.com/v3/repos#get-autolink)
    pub async fn repos_get_autolink(
        &self,
        owner: &str,
        repo: &str,
        autolink_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_autolink::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                autolink_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_autolink::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete an autolink reference from a repository
    ///
    /// This deletes a single autolink reference by ID that was configured for the given repository.
    ///
    /// Information about autolinks are only available to repository administrators.
    ///
    /// [API method documentation](https://docs.github.com/v3/repos#delete-autolink)
    pub async fn repos_delete_autolink(
        &self,
        owner: &str,
        repo: &str,
        autolink_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_delete_autolink::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                autolink_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_delete_autolink::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Enable automated security fixes
    ///
    /// Enables automated security fixes for a repository. The authenticated user must have admin access to the repository. For more information, see "[Configuring automated security fixes](https://docs.github.com/en/articles/configuring-automated-security-fixes)".
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#enable-automated-security-fixes)
    pub async fn repos_enable_automated_security_fixes(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_enable_automated_security_fixes::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_enable_automated_security_fixes::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Disable automated security fixes
    ///
    /// Disables automated security fixes for a repository. The authenticated user must have admin access to the repository. For more information, see "[Configuring automated security fixes](https://docs.github.com/en/articles/configuring-automated-security-fixes)".
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#disable-automated-security-fixes)
    pub async fn repos_disable_automated_security_fixes(
        &self,
        owner: &str,
        repo: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_disable_automated_security_fixes::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_disable_automated_security_fixes::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List branches
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#list-branches)
    pub async fn repos_list_branches(
        &self,
        owner: &str,
        repo: &str,
        protected: ::std::option::Option<bool>,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_list_branches::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                protected,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_list_branches::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a branch
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-a-branch)
    pub async fn repos_get_branch(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_branch::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_branch::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get branch protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-branch-protection)
    pub async fn repos_get_branch_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_branch_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_branch_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update branch protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Protecting a branch requires admin or owner permissions to the repository.
    ///
    /// **Note**: Passing new arrays of `users` and `teams` replaces their previous values.
    ///
    /// **Note**: The list of users, apps, and teams in total is limited to 100 items.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#update-branch-protection)
    ///
    /// # Content
    ///
    /// - [`&request::repos_update_branch_protection::body::Json`](crate::v1_1_4::request::repos_update_branch_protection::body::Json)
    pub async fn repos_update_branch_protection<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_update_branch_protection::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_update_branch_protection::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_update_branch_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_update_branch_protection::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete branch protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#delete-branch-protection)
    pub async fn repos_delete_branch_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_delete_branch_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_delete_branch_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get admin branch protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-admin-branch-protection)
    pub async fn repos_get_admin_branch_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_admin_branch_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_admin_branch_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set admin branch protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Adding admin enforcement requires admin or owner permissions to the repository and branch protection to be enabled.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#set-admin-branch-protection)
    pub async fn repos_set_admin_branch_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_set_admin_branch_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_set_admin_branch_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete admin branch protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Removing admin enforcement requires admin or owner permissions to the repository and branch protection to be enabled.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#delete-admin-branch-protection)
    pub async fn repos_delete_admin_branch_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_delete_admin_branch_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_delete_admin_branch_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get pull request review protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-pull-request-review-protection)
    pub async fn repos_get_pull_request_review_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_pull_request_review_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_pull_request_review_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete pull request review protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#delete-pull-request-review-protection)
    pub async fn repos_delete_pull_request_review_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_delete_pull_request_review_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_delete_pull_request_review_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update pull request review protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Updating pull request review enforcement requires admin or owner permissions to the repository and branch protection to be enabled.
    ///
    /// **Note**: Passing new arrays of `users` and `teams` replaces their previous values.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#update-pull-request-review-protection)
    ///
    /// # Content
    ///
    /// - [`&request::repos_update_pull_request_review_protection::body::Json`](crate::v1_1_4::request::repos_update_pull_request_review_protection::body::Json)
    pub async fn repos_update_pull_request_review_protection<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_update_pull_request_review_protection::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_update_pull_request_review_protection::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_update_pull_request_review_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_update_pull_request_review_protection::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get commit signature protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// When authenticated with admin or owner permissions to the repository, you can use this endpoint to check whether a branch requires signed commits. An enabled status of `true` indicates you must sign commits on this branch. For more information, see [Signing commits with GPG](https://docs.github.com/articles/signing-commits-with-gpg) in GitHub Help.
    ///
    /// **Note**: You must enable branch protection to require signed commits.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-commit-signature-protection)
    pub async fn repos_get_commit_signature_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_commit_signature_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_commit_signature_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create commit signature protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// When authenticated with admin or owner permissions to the repository, you can use this endpoint to require signed commits on a branch. You must enable branch protection to require signed commits.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#create-commit-signature-protection)
    pub async fn repos_create_commit_signature_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_create_commit_signature_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_create_commit_signature_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete commit signature protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// When authenticated with admin or owner permissions to the repository, you can use this endpoint to disable required signed commits on a branch. You must enable branch protection to require signed commits.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#delete-commit-signature-protection)
    pub async fn repos_delete_commit_signature_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_delete_commit_signature_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_delete_commit_signature_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get status checks protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-status-checks-protection)
    pub async fn repos_get_status_checks_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_status_checks_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_status_checks_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove status check protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#remove-status-check-protection)
    pub async fn repos_remove_status_check_protection(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_remove_status_check_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_remove_status_check_protection::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update status check protection
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Updating required status checks requires admin or owner permissions to the repository and branch protection to be enabled.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#update-status-check-protection)
    ///
    /// # Content
    ///
    /// - [`&request::repos_update_status_check_protection::body::Json`](crate::v1_1_4::request::repos_update_status_check_protection::body::Json)
    pub async fn repos_update_status_check_protection<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_update_status_check_protection::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_update_status_check_protection::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_update_status_check_protection::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_update_status_check_protection::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get all status check contexts
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-all-status-check-contexts)
    pub async fn repos_get_all_status_check_contexts(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_all_status_check_contexts::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_all_status_check_contexts::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set status check contexts
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#set-status-check-contexts)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_set_status_check_contexts<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_set_status_check_contexts::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_set_status_check_contexts::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_set_status_check_contexts::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_set_status_check_contexts::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add status check contexts
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#add-status-check-contexts)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_add_status_check_contexts<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_add_status_check_contexts::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_add_status_check_contexts::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_add_status_check_contexts::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_add_status_check_contexts::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove status check contexts
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#remove-status-check-contexts)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_remove_status_check_contexts<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_remove_status_check_contexts::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_remove_status_check_contexts::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_remove_status_check_contexts::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_remove_status_check_contexts::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Lists who has access to this protected branch.
    ///
    /// **Note**: Users, apps, and teams `restrictions` are only available for organization-owned repositories.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#get-access-restrictions)
    pub async fn repos_get_access_restrictions(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_access_restrictions::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Delete access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Disables the ability to restrict who can push to this branch.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#delete-access-restrictions)
    pub async fn repos_delete_access_restrictions(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_delete_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_delete_access_restrictions::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get apps with access to the protected branch
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Lists the GitHub Apps that have push access to this branch. Only installed GitHub Apps with `write` access to the `contents` permission can be added as authorized actors on a protected branch.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#list-apps-with-access-to-the-protected-branch)
    pub async fn repos_get_apps_with_access_to_protected_branch(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_apps_with_access_to_protected_branch::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_apps_with_access_to_protected_branch::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set app access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Replaces the list of apps that have push access to this branch. This removes all apps that previously had push access and grants push access to the new list of apps. Only installed GitHub Apps with `write` access to the `contents` permission can be added as authorized actors on a protected branch.
    ///
    /// | Type    | Description                                                                                                                                                |
    /// | ------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------- |
    /// | `array` | The GitHub Apps that have push access to this branch. Use the app's `slug`. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#set-app-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_set_app_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_set_app_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_set_app_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_set_app_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_set_app_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add app access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Grants the specified apps push access for this branch. Only installed GitHub Apps with `write` access to the `contents` permission can be added as authorized actors on a protected branch.
    ///
    /// | Type    | Description                                                                                                                                                |
    /// | ------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------- |
    /// | `array` | The GitHub Apps that have push access to this branch. Use the app's `slug`. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#add-app-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_add_app_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_add_app_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_add_app_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_add_app_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_add_app_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove app access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Removes the ability of an app to push to this branch. Only installed GitHub Apps with `write` access to the `contents` permission can be added as authorized actors on a protected branch.
    ///
    /// | Type    | Description                                                                                                                                                |
    /// | ------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------- |
    /// | `array` | The GitHub Apps that have push access to this branch. Use the app's `slug`. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#remove-app-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_remove_app_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_remove_app_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_remove_app_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_remove_app_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_remove_app_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get teams with access to the protected branch
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Lists the teams who have push access to this branch. The list includes child teams.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#list-teams-with-access-to-the-protected-branch)
    pub async fn repos_get_teams_with_access_to_protected_branch(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_teams_with_access_to_protected_branch::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_teams_with_access_to_protected_branch::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set team access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Replaces the list of teams that have push access to this branch. This removes all teams that previously had push access and grants push access to the new list of teams. Team restrictions include child teams.
    ///
    /// | Type    | Description                                                                                                                                |
    /// | ------- | ------------------------------------------------------------------------------------------------------------------------------------------ |
    /// | `array` | The teams that can have push access. Use the team's `slug`. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#set-team-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_set_team_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_set_team_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_set_team_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_set_team_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_set_team_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add team access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Grants the specified teams push access for this branch. You can also give push access to child teams.
    ///
    /// | Type    | Description                                                                                                                                |
    /// | ------- | ------------------------------------------------------------------------------------------------------------------------------------------ |
    /// | `array` | The teams that can have push access. Use the team's `slug`. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#add-team-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_add_team_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_add_team_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_add_team_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_add_team_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_add_team_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove team access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Removes the ability of a team to push to this branch. You can also remove push access for child teams.
    ///
    /// | Type    | Description                                                                                                                                         |
    /// | ------- | --------------------------------------------------------------------------------------------------------------------------------------------------- |
    /// | `array` | Teams that should no longer have push access. Use the team's `slug`. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#remove-team-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_remove_team_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_remove_team_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_remove_team_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_remove_team_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_remove_team_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get users with access to the protected branch
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Lists the people who have push access to this branch.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#list-users-with-access-to-the-protected-branch)
    pub async fn repos_get_users_with_access_to_protected_branch(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_get_users_with_access_to_protected_branch::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::repos_get_users_with_access_to_protected_branch::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Set user access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Replaces the list of people that have push access to this branch. This removes all people that previously had push access and grants push access to the new list of people.
    ///
    /// | Type    | Description                                                                                                                   |
    /// | ------- | ----------------------------------------------------------------------------------------------------------------------------- |
    /// | `array` | Usernames for people who can have push access. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#set-user-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_set_user_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_set_user_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_set_user_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_set_user_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_set_user_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Add user access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Grants the specified people push access for this branch.
    ///
    /// | Type    | Description                                                                                                                   |
    /// | ------- | ----------------------------------------------------------------------------------------------------------------------------- |
    /// | `array` | Usernames for people who can have push access. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#add-user-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_add_user_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_add_user_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_add_user_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_add_user_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_add_user_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Remove user access restrictions
    ///
    /// Protected branches are available in public repositories with GitHub Free and GitHub Free for organizations, and in public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. For more information, see [GitHub's products](https://docs.github.com/github/getting-started-with-github/githubs-products) in the GitHub Help documentation.
    ///
    /// Removes the ability of a user to push to this branch.
    ///
    /// | Type    | Description                                                                                                                                   |
    /// | ------- | --------------------------------------------------------------------------------------------------------------------------------------------- |
    /// | `array` | Usernames of the people who should no longer have push access. **Note**: The list of users, apps, and teams in total is limited to 100 items. |
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#remove-user-access-restrictions)
    ///
    /// # Content
    ///
    /// - [`&::serde_json::value::Value`](::serde_json::value::Value)
    pub async fn repos_remove_user_access_restrictions<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_remove_user_access_restrictions::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_remove_user_access_restrictions::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_remove_user_access_restrictions::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_remove_user_access_restrictions::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Rename a branch
    ///
    /// Renames a branch in a repository.
    ///
    /// **Note:** Although the API responds immediately, the branch rename process might take some extra time to complete in the background. You won't be able to push to the old branch name while the rename process is in progress. For more information, see "[Renaming a branch](https://docs.github.com/github/administering-a-repository/renaming-a-branch)".
    ///
    /// The permissions required to use this endpoint depends on whether you are renaming the default branch.
    ///
    /// To rename a non-default branch:
    ///
    /// * Users must have push access.
    /// * GitHub Apps must have the `contents:write` repository permission.
    ///
    /// To rename the default branch:
    ///
    /// * Users must have admin or owner permissions.
    /// * GitHub Apps must have the `administration:write` repository permission.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/repos#rename-a-branch)
    ///
    /// # Content
    ///
    /// - [`&request::repos_rename_branch::body::Json`](crate::v1_1_4::request::repos_rename_branch::body::Json)
    pub async fn repos_rename_branch<Content>(
        &self,
        owner: &str,
        repo: &str,
        branch: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::repos_rename_branch::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::repos_rename_branch::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::repos_rename_branch::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                branch,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::repos_rename_branch::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a check run
    ///
    /// **Note:** The Checks API only looks for pushes in the repository where the check suite or check run were created. Pushes to a branch in a forked repository are not detected and return an empty `pull_requests` array.
    ///
    /// Creates a new check run for a specific commit in a repository. Your GitHub App must have the `checks:write` permission to create check runs.
    ///
    /// In a check suite, GitHub limits the number of check runs with the same name to 1000. Once these check runs exceed 1000, GitHub will start to automatically delete older check runs.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/checks#create-a-check-run)
    ///
    /// # Content
    ///
    /// - [`&request::checks_create::body::Json`](crate::v1_1_4::request::checks_create::body::Json)
    pub async fn checks_create<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::checks_create::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::checks_create::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::checks_create::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::checks_create::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Get a check run
    ///
    /// **Note:** The Checks API only looks for pushes in the repository where the check suite or check run were created. Pushes to a branch in a forked repository are not detected and return an empty `pull_requests` array.
    ///
    /// Gets a single check run using its `id`. GitHub Apps must have the `checks:read` permission on a private repository or pull access to a public repository to get check runs. OAuth Apps and authenticated users must have the `repo` scope to get check runs in a private repository.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/checks#get-a-check-run)
    pub async fn checks_get(
        &self,
        owner: &str,
        repo: &str,
        check_run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::checks_get::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                check_run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::checks_get::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update a check run
    ///
    /// **Note:** The Checks API only looks for pushes in the repository where the check suite or check run were created. Pushes to a branch in a forked repository are not detected and return an empty `pull_requests` array.
    ///
    /// Updates a check run for a specific commit in a repository. Your GitHub App must have the `checks:write` permission to edit check runs.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/checks#update-a-check-run)
    ///
    /// # Content
    ///
    /// - [`&request::checks_update::body::Json`](crate::v1_1_4::request::checks_update::body::Json)
    pub async fn checks_update<Content>(
        &self,
        owner: &str,
        repo: &str,
        check_run_id: i64,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::checks_update::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::checks_update::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::checks_update::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                check_run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::checks_update::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// List check run annotations
    ///
    /// Lists annotations for a check run using the annotation `id`. GitHub Apps must have the `checks:read` permission on a private repository or pull access to a public repository to get annotations for a check run. OAuth Apps and authenticated users must have the `repo` scope to get annotations for a check run in a private repository.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/checks#list-check-run-annotations)
    pub async fn checks_list_annotations(
        &self,
        owner: &str,
        repo: &str,
        check_run_id: i64,
        per_page: ::std::option::Option<i64>,
        page: ::std::option::Option<i64>,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::checks_list_annotations::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                check_run_id,
                per_page,
                page,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::checks_list_annotations::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Rerequest a check run
    ///
    /// Triggers GitHub to rerequest an existing check run, without pushing new code to a repository. This endpoint will trigger the [`check_run` webhook](https://docs.github.com/webhooks/event-payloads/#check_run) event with the action `rerequested`. When a check run is `rerequested`, its `status` is reset to `queued` and the `conclusion` is cleared.
    ///
    /// To rerequest a check run, your GitHub App must have the `checks:read` permission on a private repository or pull access to a public repository.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/checks#rerequest-a-check-run)
    pub async fn checks_rerequest_run(
        &self,
        owner: &str,
        repo: &str,
        check_run_id: i64,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError> {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::checks_rerequest_run::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                check_run_id,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest =
                crate::v1_1_4::request::checks_rerequest_run::reqwest_request(theBuilder)?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Create a check suite
    ///
    /// **Note:** The Checks API only looks for pushes in the repository where the check suite or check run were created. Pushes to a branch in a forked repository are not detected and return an empty `pull_requests` array and a `null` value for `head_branch`.
    ///
    /// By default, check suites are automatically created when you create a [check run](https://docs.github.com/rest/reference/checks#check-runs). You only need to use this endpoint for manually creating check suites when you've disabled automatic creation using "[Update repository preferences for check suites](https://docs.github.com/rest/reference/checks#update-repository-preferences-for-check-suites)". Your GitHub App must have the `checks:write` permission to create check suites.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/checks#create-a-check-suite)
    ///
    /// # Content
    ///
    /// - [`&request::checks_create_suite::body::Json`](crate::v1_1_4::request::checks_create_suite::body::Json)
    pub async fn checks_create_suite<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::checks_create_suite::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::checks_create_suite::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
            while let Some(auth_step) = theScheme.step()? {
                match auth_step {
                    ::authentic::AuthenticationStep::Request(auth_request) => {
                        theScheme.respond(self.client.execute(auth_request).await);
                    }
                    ::authentic::AuthenticationStep::WaitFor(duration) => {
                        (self.sleep)(duration).await;
                    }
                }
            }
            let theBuilder = crate::v1_1_4::request::checks_create_suite::reqwest_builder(
                self.config.base_url.as_ref(),
                owner,
                repo,
                self.config.user_agent.as_ref(),
                self.config.accept.as_deref(),
            )?
            .with_authentication(&theScheme)?;

            let theRequest = crate::v1_1_4::request::checks_create_suite::reqwest_request(
                theBuilder,
                theContent.try_into()?,
            )?;

            ::log::debug!("HTTP request: {:?}", &theRequest);

            let theResponse = self.client.execute(theRequest).await?;

            ::log::debug!("HTTP response: {:?}", &theResponse);

            if theScheme.has_completed(&theResponse)? {
                break theResponse;
            }
        };
        Ok(theResponse)
    }

    /// Update repository preferences for check suites
    ///
    /// Changes the default automatic flow when creating check suites. By default, a check suite is automatically created each time code is pushed to a repository. When you disable the automatic creation of check suites, you can manually [Create a check suite](https://docs.github.com/rest/reference/checks#create-a-check-suite). You must have admin permissions in the repository to set preferences for check suites.
    ///
    /// [API method documentation](https://docs.github.com/rest/reference/checks#update-repository-preferences-for-check-suites)
    ///
    /// # Content
    ///
    /// - [`&request::checks_set_suites_preferences::body::Json`](crate::v1_1_4::request::checks_set_suites_preferences::body::Json)
    pub async fn checks_set_suites_preferences<Content>(
        &self,
        owner: &str,
        repo: &str,
        theContent: Content,
    ) -> Result<::reqwest::Response, crate::v1_1_4::ApiError>
    where
        Content: Copy + TryInto<crate::v1_1_4::request::checks_set_suites_preferences::Content<::reqwest::Body>>,
        crate::v1_1_4::ApiError: From<<Content as TryInto<crate::v1_1_4::request::checks_set_suites_preferences::Content<::reqwest::Body>>>::Error>
    {
        let mut theScheme = AuthScheme::from(&self.config.authentication);

        let theResponse = loop {
