use std::io;
use tracing::subscriber::SetGlobalDefaultError;
pub use tracing::{debug, error, info, instrument, trace, warn, Level};
use tracing_subscriber::{EnvFilter, FmtSubscriber};

/// Basic invocation. Simply call `jacklog::init(Some("mybin=debug"))` to
/// automatically set a default log level, that can be overridden via the usual
/// RUST_LOG env var.
pub fn init<T: AsRef<str>>(default_level: Option<&T>) -> Result<(), SetGlobalDefaultError> {
    // Create a subscriber writing to stderr with the filter set up above.
    let subscriber = FmtSubscriber::builder()
        .with_writer(io::stderr)
        .with_env_filter(parse_from_env(default_level))
        .finish();

    // Register the subscriber as the default.
    Ok(tracing::subscriber::set_global_default(subscriber)?)
}

/// Set the log level using 0-indexed integers, starting from 0=off. Especially
/// useful if writing a CLI and accepting verbosity settings via number of
/// flags (-v, -vv, -vvv, etc.).  Will be overridden via the usual RUST_LOG
/// env var, if it's present.
///
/// In what is probably the most common use case for this method, where a
/// repeating flag is used to increase verbosity, do the following math for the
/// behavior you want:
///
/// Default of warn, with increasing verbosity when number of -v arguments
/// is one or more:
///
/// 2 + (number of -v flags)
///
/// Default of no logging, with increasing verbosity when number of -v arguments
/// is one or more:
///
/// 0 + (number of -v flags)
///
/// Default of info, with increasing verbosity when number of -v arguments
/// is one or more:
///
/// 3 + (number of -v flags)
///
/// You probably get the idea.
pub fn from_level(level: usize) -> Result<(), SetGlobalDefaultError> {
    // Create a subscriber writing to stderr with the filter set up above.
    let subscriber = FmtSubscriber::builder()
        .with_writer(io::stderr)
        .with_env_filter(parse_from_level(level))
        .finish();

    // Register the subscriber as the default.
    Ok(tracing::subscriber::set_global_default(subscriber)?)
}

/// Figure out the level based on a 0-indexed hierarchy, with 0 being "off".
fn parse_from_level(level: usize) -> EnvFilter {
    // Set up the filter to use, defaulting to the level passed in.
    EnvFilter::try_from_default_env().unwrap_or(EnvFilter::new(match level {
        0 => "off",
        1 => "error",
        2 => "warn",
        3 => "info",
        4 => "debug",
        _ => "trace",
    }))
}

fn parse_from_env<T: AsRef<str>>(default_level: Option<&T>) -> EnvFilter {
    // Set up the filter to use, defaulting to just the warn log level for all crates/modules.
    EnvFilter::try_from_default_env().unwrap_or(EnvFilter::new(
        default_level.map(|s| s.as_ref()).unwrap_or("warn"),
    ))
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_init() {
        init(Some(&"warn")).unwrap();

        debug!("you should not see this");
        warn!("this is a warning");
        error!("this is an error");
    }

    #[test]
    fn test_from_level() {
        // TODO: Refactor things so we can run tests in closure using
        // https://docs.rs/tracing/0.1.26/tracing/subscriber/fn.with_default.html
        //from_level(2).unwrap();

        debug!("you should not see this");
        warn!("this is a warning");
        error!("this is an error");
    }

    #[test]
    fn test_parse_from_env() {
        // Passing the default value should equal the default value.
        assert_eq!(
            parse_from_env(Some(&"warn")).to_string(),
            EnvFilter::new("warn").to_string()
        );

        // Passing a non-default value should override the default.
        assert_eq!(
            parse_from_env(Some(&"trace")).to_string(),
            EnvFilter::new("trace").to_string()
        );
    }

    #[test]
    fn test_parse_from_level() {
        assert_eq!(
            parse_from_level(0).to_string(),
            EnvFilter::new("off").to_string()
        );

        assert_eq!(
            parse_from_level(1).to_string(),
            EnvFilter::new("error").to_string()
        );

        assert_eq!(
            parse_from_level(2).to_string(),
            EnvFilter::new("warn").to_string()
        );

        assert_eq!(
            parse_from_level(3).to_string(),
            EnvFilter::new("info").to_string()
        );

        assert_eq!(
            parse_from_level(4).to_string(),
            EnvFilter::new("debug").to_string()
        );

        assert_eq!(
            parse_from_level(5).to_string(),
            EnvFilter::new("trace").to_string()
        );

        assert_eq!(
            parse_from_level(6).to_string(),
            EnvFilter::new("trace").to_string()
        );

        assert_eq!(
            parse_from_level(999).to_string(),
            EnvFilter::new("trace").to_string()
        );
    }
}
