use ivy_base::Color;
use ivy_base::DrawGizmos;
use ivy_base::Gizmos;
use slotmap::new_key_type;
use slotmap::Key;
use std::fmt::Debug;
use std::ops::DerefMut;

use crate::CollisionTreeNode;

use super::Nodes;

new_key_type!(
    pub struct NodeIndex;
);

impl NodeIndex {
    /// Creates a new handle that is always invalid and distinct from any non-null
    /// handle. A null key can only be created through this method (or default
    /// initialization of handles made with `new_key_type!`, which calls this
    /// method).
    ///
    /// A null handle is always invalid, but an invalid key (that is, a key that
    /// has been removed from the slot map) does not become a null handle. A null
    /// is safe to use with any safe method of any slot map instance.
    pub fn null() -> Self {
        Key::null()
    }

    /// Checks if a handle is null. There is only a single null key, that is
    /// `a.is_null() && b.is_null()` implies `a == b`.
    pub fn is_null(&self) -> bool {
        Key::is_null(self)
    }
}

impl NodeIndex {
    pub fn draw_gizmos_recursive<N: CollisionTreeNode + DrawGizmos>(
        self,
        nodes: &Nodes<N>,
        mut gizmos: &mut Gizmos,
        color: Color,
    ) {
        nodes[self].draw_gizmos(gizmos.deref_mut(), color);

        for val in nodes[self].children() {
            val.draw_gizmos_recursive(nodes, gizmos, color)
        }
    }
}

pub(crate) struct DebugNode<'a, N> {
    index: NodeIndex,
    nodes: &'a Nodes<N>,
}

impl<'a, N> DebugNode<'a, N> {
    pub(crate) fn new(index: NodeIndex, nodes: &'a Nodes<N>) -> Self {
        Self { index, nodes }
    }
}

impl<'a, N: CollisionTreeNode> Debug for DebugNode<'a, N> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let node = &self.nodes[self.index];

        let mut children = f.debug_list();
        children.entries(node.children().iter().map(|val| {
            DebugNode::new(*val, self.nodes);
        }));

        let children = children.finish();
        let mut dbg = f.debug_struct("Node");
        dbg.field("children: ", &children);

        dbg.finish()
    }
}
