use std::ops::{Bound, RangeBounds, RangeInclusive};

#[derive(Debug, Clone, PartialEq, Eq, Hash)]
#[cfg_attr(feature = "serde_derive", derive(serde::Deserialize, serde::Serialize))]
pub struct Interval<N>(RangeInclusive<N>);

impl<N: Ord + Clone> From<RangeInclusive<N>> for Interval<N> {
    fn from(range: RangeInclusive<N>) -> Self {
        if range.end() < range.start() {
            panic!("interval end bound must be >= start bound")
        }
        Self(range)
    }
}

impl<N: Ord + Clone> From<&'_ Interval<N>> for Interval<N> {
    fn from(interval: &Interval<N>) -> Self {
        interval.clone()
    }
}

impl<N: Ord + Clone> From<N> for Interval<N> {
    fn from(point: N) -> Self {
        Self::from(point.clone()..=point)
    }
}

impl<N> RangeBounds<N> for Interval<N> {
    fn start_bound(&self) -> Bound<&N> {
        self.0.start_bound()
    }

    fn end_bound(&self) -> Bound<&N> {
        self.0.end_bound()
    }
}

impl<N> Interval<N> {
    pub fn start(&self) -> &N {
        self.0.start()
    }

    pub fn end(&self) -> &N {
        self.0.end()
    }

    pub fn into_inner(self) -> (N, N) {
        self.0.into_inner()
    }
}
