use super::*;

use embedded_hal_mock::pin::{Mock as PinMock, State as PinState, Transaction as PinTransaction};
use embedded_hal_mock::serial::{Mock as SerialMock, Transaction as SerialTransaction};

// No RM-provided examples, so create our own

#[test]
fn command_serialization_read() {
    // Read-only
    std::assert_eq!(
        registers::DevTyp::read().encode(true),
        [0x10, 0x01, 0x00, 0x00]
    );
    // Read-write
    std::assert_eq!(
        registers::Status::read().encode(true),
        [0x00, 0x00, 0x00, 0x00]
    );
    // Extras
    std::assert_eq!(
        registers::Ftf::read().encode(false),
        [0xC8, 0x62, 0x00, 0x00]
    );
    std::assert_eq!(
        registers::DitherR::read().encode(false),
        [0x78, 0x5A, 0x00, 0x00]
    );
    std::assert_eq!(
        registers::Lfh2::read().encode(true),
        [0x00, 0x55, 0x00, 0x00]
    );
    std::assert_eq!(
        registers::ppcl200::CleanJumpSled::read().encode(true),
        [0x20, 0xEC, 0x00, 0x00]
    );
}

#[test]
fn command_serialization_write() {
    // No write-only
    // Read-write
    std::assert_eq!(
        registers::Status::write(0x55AA).encode(true),
        [0x11, 0x00, 0x55, 0xAA]
    );
    // Extras
    std::assert_eq!(
        registers::Ftf::write(0x1234).encode(false),
        [0x99, 0x62, 0x12, 0x34]
    );
    std::assert_eq!(
        registers::DitherR::write(0xE920).encode(false),
        [0x39, 0x5A, 0xE9, 0x20]
    );
    std::assert_eq!(
        registers::FAgeTh::write(0x2010).encode(true),
        [0x81, 0x5F, 0x20, 0x10]
    );
    std::assert_eq!(
        registers::ppcl200::CleanJumpSled::write(0x0000).encode(true),
        [0x31, 0xEC, 0x00, 0x00]
    );
}

#[test]
fn command_deserialization() {
    std::assert_eq!(
        Response::decode([0x44, 0x00, 0x00, 0x00]).unwrap(),
        Response {
            ce: false,
            status: CommandStatus::Ok,
            register: 0x00,
            data: [0x00, 0x00]
        }
    );
    std::assert_eq!(
        Response::decode([0xFC, 0x21, 0xFF, 0xAA]).unwrap(),
        Response {
            ce: true,
            status: CommandStatus::Ok,
            register: 0x21,
            data: [0xFF, 0xAA]
        }
    );
    std::assert_eq!(
        Response::decode([0x05, 0x21, 0x4F, 0xE3]).unwrap(),
        Response {
            ce: false,
            status: CommandStatus::ExecutionError,
            register: 0x21,
            data: [0x4F, 0xE3]
        }
    );
    std::assert_eq!(
        Response::decode([0x2E, 0xAB, 0x29, 0xE8]).unwrap(),
        Response {
            ce: true,
            status: CommandStatus::ExtendedAddressing,
            register: 0xAB,
            data: [0x29, 0xE8]
        }
    );
    std::assert_eq!(
        Response::decode([0xB7, 0x81, 0x60, 0xED]).unwrap(),
        Response {
            ce: false,
            status: CommandStatus::CommandPending,
            register: 0x81,
            data: [0x60, 0xED]
        }
    );
}

#[test]
fn command_deserialization_failure() {
    // Random values
    std::assert_eq!(Response::decode([0xA7, 0x81, 0x60, 0xED]), None,);
    std::assert_eq!(Response::decode([0x83, 0xBD, 0xE0, 0x23]), None,);
    std::assert_eq!(Response::decode([0x85, 0xD6, 0x89, 0xAB]), None,);
    std::assert_eq!(Response::decode([0xD1, 0x66, 0xD9, 0xD0]), None,);
    // Bit 3 is off
    std::assert_eq!(Response::decode([0xF3, 0x81, 0x60, 0xED]), None,);
}

#[test]
fn new_laser_resets() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[]);
    let serial = SerialMock::new(&[SerialTransaction::read_error(nb::Error::WouldBlock)]);

    let laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn start_command_does_not_read() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[]);
    let serial = SerialMock::new(&[SerialTransaction::read_error(nb::Error::WouldBlock)]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn get_srq() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[
        PinTransaction::get(PinState::Low),
        PinTransaction::get(PinState::High),
    ]);
    let serial = SerialMock::new(&[SerialTransaction::read_error(nb::Error::WouldBlock)]);

    let laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    assert!(laser.srq().unwrap());
    assert!(!laser.srq().unwrap());
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn enable() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[
        PinTransaction::set(PinState::High),
        PinTransaction::set(PinState::Low),
    ]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[]);
    let serial = SerialMock::new(&[SerialTransaction::read_error(nb::Error::WouldBlock)]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.enable().unwrap();
    laser.disable().unwrap();
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn reset() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read_error(nb::Error::WouldBlock),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.reset().unwrap();
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn reset_comms() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[
        PinTransaction::set(PinState::High),
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let srq = PinMock::new(&[]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read_error(nb::Error::WouldBlock),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.reset_comms().unwrap();
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn send_read() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x00, 0x00]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::Ok,
            register: 0x00,
            data: [0x00, 0x00]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn send_read_retries_bit3() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x33, 0x00, 0x00, 0x00]),
        SerialTransaction::write_many([0x88, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x00, 0x00]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::Ok,
            register: 0x00,
            data: [0x00, 0x00]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn send_read_retries_ce() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0xCC, 0x00, 0x00, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x00, 0x00]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::Ok,
            register: 0x00,
            data: [0x00, 0x00]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn send_read_retries_checksum() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x43, 0x00, 0x00, 0x00]),
        SerialTransaction::write_many([0x88, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x00, 0x00]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::Ok,
            register: 0x00,
            data: [0x00, 0x00]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn send_error_reads_back_status() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x55, 0x00, 0x00, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x00, 0x00]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap_err();
    assert!(matches!(
        got,
        Error::CommandError(
            r,
            Response {
                ce: false,
                status: CommandStatus::Ok,
                register: 0x00,
                data: [0x00, 0x00]
            }
        ) if r == registers::Status::read()
    ));
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn send_error_on_srq() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::Low)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x00, 0x00]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap_err();
    assert!(matches!(
        got,
        Error::Srq(Response {
            ce: false,
            status: CommandStatus::Ok,
            register: 0x00,
            data: [0x00, 0x00]
        })
    ));
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn reset_comms_resets_buffer() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[
        PinTransaction::set(PinState::High),
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x43, 0x00, 0x00]),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x55, 0x00, 0x00]),
        SerialTransaction::read_many([0x66, 0x55, 0xAB, 0xCD]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    for _ in 0..1 {
        assert!(matches!(laser.step(), Err(nb::Error::WouldBlock)));
    }
    laser.reset_comms().unwrap();
    laser.send(registers::Lfh2::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::ExtendedAddressing,
            register: 0x55,
            data: [0xAB, 0xCD]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn reset_resets_buffer() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x43, 0x00, 0x00]),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x55, 0x00, 0x00]),
        SerialTransaction::read_many([0x66, 0x55, 0xAB, 0xCD]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    for _ in 0..1 {
        assert!(matches!(laser.step(), Err(nb::Error::WouldBlock)));
    }
    laser.reset().unwrap();
    laser.send(registers::Lfh2::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::ExtendedAddressing,
            register: 0x55,
            data: [0xAB, 0xCD]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn read_one_byte_at_a_time() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[PinTransaction::get(PinState::High)]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write(0x00),
        SerialTransaction::write_error(0x00, nb::Error::WouldBlock),
        SerialTransaction::write(0x00),
        SerialTransaction::write_error(0x00, nb::Error::WouldBlock),
        SerialTransaction::write(0x00),
        SerialTransaction::write_error(0x00, nb::Error::WouldBlock),
        SerialTransaction::write(0x00),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read(0x44),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read(0x00),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read(0x00),
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::read(0x00),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::Ok,
            register: 0x00,
            data: [0x00, 0x00]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}

#[test]
fn command_pending_waits() {
    let _ = pretty_env_logger::try_init();

    let n_reset = PinMock::new(&[
        PinTransaction::set(PinState::Low),
        PinTransaction::set(PinState::High),
    ]);
    let n_dis = PinMock::new(&[]);
    let n_ms = PinMock::new(&[PinTransaction::set(PinState::High)]);
    let srq = PinMock::new(&[
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
        PinTransaction::get(PinState::High),
    ]);
    let serial = SerialMock::new(&[
        SerialTransaction::read_error(nb::Error::WouldBlock),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x67, 0x00, 0x00, 0x10]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x01, 0x10]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x64, 0x00, 0x02, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x74, 0x00, 0x03, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x04, 0x00, 0x04, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x14, 0x00, 0x05, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x24, 0x00, 0x06, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x34, 0x00, 0x07, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x34, 0x00, 0x07, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x44, 0x00, 0x00, 0x00]),
        SerialTransaction::write_many([0x00, 0x00, 0x00, 0x00]),
        SerialTransaction::read_many([0x64, 0x00, 0x00, 0x13]),
    ]);

    let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
    laser.send(registers::Status::read());
    let got = nb::block!(laser.step()).unwrap();
    assert_eq!(
        got,
        Response {
            ce: false,
            status: CommandStatus::CommandPending,
            register: 0x00,
            data: [0x00, 0x10]
        }
    );
    let (mut n_reset, mut n_dis, mut n_ms, mut srq, mut serial) = laser.release();

    n_reset.done();
    n_dis.done();
    n_ms.done();
    srq.done();
    serial.done();
}
