# ITLA-rs
`ITLA-rs` is a library to control ITLA (micro lasers) through a serial interface in a non-blocking fashion.

For details about ITLA, please read https://www.oiforum.com/wp-content/uploads/2019/01/OIF-ITLA-MSA-01.3.pdf.

This library uses embedded-hal and, as such, can be used on microcontrollers as well as linux systems.

For now, the only vendor-specific registers implemented are the ones on the PPCL200. You can send a merge request if you wish to add more registers.

# Example

Here is an example of usage for the library. `n_reset`, `n_dis` and `n_ms` are `OutputPin`s, either physical or mocked (in case you don't want to control those wires). `serial` needs to implement `Read` and `Write` from `embedded-hal`.

```
let mut laser = Laser::new(n_reset, n_dis, n_ms, srq, serial).unwrap();
laser.enable().unwrap();
laser.send(itla::registers::Status::read()); // This only starts the command
let out = nb::block!(laser.step()).unwrap(); // You need to poll it until completion
// You can either call `nb::block!` to wait in a busy loop, or wait for interrupts on your microcontroller
let value: u16 = out.val(); // Get output value
```

When the laser returns a "command pending" response, the library will wait for that command to complete. By default, it will busy loop and wait until completion, but if you prefer to add a delay and/or bound the number of retries, you can use the `new_with_handler_between_retries` constructor with a closure as last paremeter. This will be called between each request for pending command. You can return `AfterRequest::Exit` when you want to abort instead.

```
let mut laser = Laser::new_with_handler_between_retries(n_reset, n_dis, n_ms, srq, serial, Some(|retry_nb| {
    cortex_m::asm::delay(10_000_000 * retry_nb);
    if retry_nb > 200 {
        AfterRequest::Exit
    } else {
        AfterRequest::Continue
    }
})).unwrap();
```

# License

Licensed under Lesser GNU Public License, Version 3.0 or later (https://www.gnu.org/licenses/lgpl-3.0.en.html) 

# Contribution

Unless you explicitly state otherwise, any contribution intentionally submitted for inclusion in the work by you, as defined in the LGPL-3.0 license, shall be licensed as above, without any additional terms or conditions.
