# iter-comprehensions

[![rs-graph crate](https://img.shields.io/crates/v/iter-comprehensions.svg)](https://crates.io/crates/iter-comprehensions)
[![iter-comprehensions docs](https://docs.rs/iter-comprehensions/badge.svg)](https://docs.rs/iter-comprehensions)

## Introduction

`iter-comprehensions` provides a few macros implementing iterator and vector
comprehensions for `Rust`.

1. `comprehension!` for generating a sequence of index tuples
2. `map!` for generating a sequence of expressions
3. `vec!` for constructing vectors
4. `sum!` for computing the sum of some values
5. `product!` for computing the product of some values

The macro `comprehension!` can be used to generate a sequence of elements using
generating sequences and conditional filters.

    comprehension!(i1 in RANGE1, COND1, ..., ik in RANGEk)

where RANGE* are iterators (in fact, everything implementing `IntoIterator`)
and each COND* is a boolean condition. Each `RANGE` and `COND` term can use
the variables declared in preceeding range expressions.

The macro `map!` adds an additional expression that computes a value
depending on the indices:

    map!(i1 in RANGE1, COND1, ..., ik in RANGEk, EXPR)
    map!(EXPR; i1 in RANGE1, COND1, ..., ik in RANGEk)

### Example
The expression $\\{ 5i + j : i \\in \\{0..4\\}, j \\in \\{0..4\\}, i < j \\}$ is
equivalent to the following form

    use iter_comprehensions::map;
    assert_eq!(map!(4*i + j; i in 0..5, j in 0..5, i < j).collect::<Vec<_>>(),
               vec![1, 2, 3, 4, 6, 7, 8, 11, 12, 16]);

The analogous syntax can be used to create vectors:

    use iter_comprehensions::vec;
    assert_eq!(vec![i; i in 0..5], vec![0,1,2,3,4]);
    assert_eq!(vec![(i,j); i in 0..3, j in 0..3, i < j],
               vec![(0,1), (0,2), (1,2)]);

Computing a sum of values:

    use iter_comprehensions::{sum, vec};
    assert_eq!(sum!(i; i in 1..10, i % 2 == 1), 25);
    let S = vec![i; i in 1..10];
    assert_eq!(sum!(i in S, i % 2 == 1, i), 25);

Computing a product of values:

    use iter_comprehensions::product;
    assert_eq!(product!(i; i in 1..=5), 120);
    assert_eq!(product!(i in 1..=5, i), 120);

## Author

Frank Fischer <frank-fischer@shadow-soft.de>

## Installation

Put the requirement `iter-comprehensions = "^0.4.2"` into the `Cargo.toml` of
your project.

## Documentation

See [docs.rs](https://docs.rs/iter-comprehensions).

## Example

## Download

Source code of latest tagged version: [iter-comprehensions-v0.4.2.tar.gz](/tarball/iter-comprehensions-v0.4.2.tar.gz?name=iter-comprehensions)

Source code of trunk: [iter-comprehensions-trunk.tar.gz](/tarball/iter-comprehensions-trunk.tar.gz?name=iter-comprehensions)
