/*
 * Project: Approximation and Finite Elements in Isogeometric Problems
 * Author:  Luca Carlon
 * Date:    2022.02.06
 *
 * Copyright (c) 2022 Luca Carlon. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

use crate::bezier::BezierSurf;
use crate::bezier::BezierFactory;

pub const TEACUP_PACTHES: [[usize; 16]; 26] = [
    [   1,   2,   3,4,5,6,7,8,9,10,11,12,13,14,15,16 ],
    [   4,  17,  18,19,8,20,21,22,12,23,24,25,16,26,27,28 ],
    [  19,  29,  30,31,22,32,33,34,25,35,36,37,28,38,39,40 ],
    [  31,  41,  42,1,34,43,44,5,37,45,46,9,40,47,48,13 ],
    [  13,  14,  15,16,49,50,51,52,53,54,55,56,57,58,59,60 ],
    [  16,  26,  27,28,52,61,62,63,56,64,65,66,60,67,68,69 ],
    [  28,  38,  39,40,63,70,71,72,66,73,74,75,69,76,77,78 ],
    [  40,  47,  48,13,72,79,80,49,75,81,82,53,78,83,84,57 ],
    [ 193, 194, 195,196,197,198,199,200,201,202,203,204,1,2,3,4 ],
    [ 196, 205, 206,207,200,208,209,210,204,211,212,213,4,17,18,19 ],
    [ 207, 214, 215,216,210,217,218,219,213,220,221,222,19,29,30,31 ],
    [ 216, 223, 224,193,219,225,226,197,222,227,228,201,31,41,42,1 ],
    [ 229, 230, 231,28,232,233,234,235,236,237,238,239,240,241,242,243 ],
    [  28, 244, 245,229,235,246,247,232,239,248,249,236,243,250,251,240 ],
    [  57,  58, 59,60,85,86,87,88,89,90,91,92,93,94,95,96 ],
    [  60,  67, 68,69,88,97,98,99,92,100,101,102,96,103,104,105 ],
    [  69,  76, 77,78,99,106,107,108,102,109,110,111,105,112,113,114 ],
    [  78,  83, 84,57,108,115,116,85,111,117,118,89,114,119,120,93 ],
    [  93,  94, 95,96,121,122,123,124,125,126,127,128,129,130,131,132 ],
    [  96, 103, 104,105,124,133,134,135,128,136,137,138,132,139,140,141 ],
    [ 105, 112, 113,114,135,142,143,144,138,145,146,147,141,148,149,150 ],
    [ 114, 119, 120,93,144,151,152,121,147,153,154,125,150,155,156,129 ],
    [ 129, 130, 131,132,157,158,159,160,161,162,163,164,165,166,167,168 ],
    [ 132, 139, 140,141,160,169,170,171,164,172,173,174,168,175,176,177 ],
    [ 141, 148, 149,150,171,178,179,180,174,181,182,183,177,184,185,186 ],
    [ 150, 155, 156,129,180,187,188,157,183,189,190,161,186,191,192,165 ]
];

pub const TEACUP_VERTICES: [[f64; 3]; 251] = [
    [  0.409091000,  0.772727000,  0.000000000 ],
    [  0.409091000,  0.772727000, -0.229091000 ],
    [  0.229091000,  0.772727000, -0.409091000 ],
    [  0.000000000,  0.772727000, -0.409091000 ],
    [  0.409091000,  0.886364000,  0.000000000 ],
    [  0.409091000,  0.886364000, -0.229091000 ],
    [  0.229091000,  0.886364000, -0.409091000 ],
    [  0.000000000,  0.886364000, -0.409091000 ],
    [  0.454545000,  0.886364000,  0.000000000 ],
    [  0.454545000,  0.886364000, -0.254545000 ],
    [  0.254545000,  0.886364000, -0.454545000 ],
    [  0.000000000,  0.886364000, -0.454545000 ],
    [  0.454545000,  0.772727000,  0.000000000 ],
    [  0.454545000,  0.772727000, -0.254545000 ],
    [  0.254545000,  0.772727000, -0.454545000 ],
    [  0.000000000,  0.772727000, -0.454545000 ],
    [ -0.229091000,  0.772727000, -0.409091000 ],
    [ -0.409091000,  0.772727000, -0.229091000 ],
    [ -0.409091000,  0.772727000,  0.000000000 ],
    [ -0.229091000,  0.886364000, -0.409091000 ],
    [ -0.409091000,  0.886364000, -0.229091000 ],
    [ -0.409091000,  0.886364000,  0.000000000 ],
    [ -0.254545000,  0.886364000, -0.454545000 ],
    [ -0.454545000,  0.886364000, -0.254545000 ],
    [ -0.454545000,  0.886364000,  0.000000000 ],
    [ -0.254545000,  0.772727000, -0.454545000 ],
    [ -0.454545000,  0.772727000, -0.254545000 ],
    [ -0.454545000,  0.772727000,  0.000000000 ],
    [ -0.409091000,  0.772727000,  0.229091000 ],
    [ -0.229091000,  0.772727000,  0.409091000 ],
    [  0.000000000,  0.772727000,  0.409091000 ],
    [ -0.409091000,  0.886364000,  0.229091000 ],
    [ -0.229091000,  0.886364000,  0.409091000 ],
    [  0.000000000,  0.886364000,  0.409091000 ],
    [ -0.454545000,  0.886364000,  0.254545000 ],
    [ -0.254545000,  0.886364000,  0.454545000 ],
    [  0.000000000,  0.886364000,  0.454545000 ],
    [ -0.454545000,  0.772727000,  0.254545000 ],
    [ -0.254545000,  0.772727000,  0.454545000 ],
    [  0.000000000,  0.772727000,  0.454545000 ],
    [  0.229091000,  0.772727000,  0.409091000 ],
    [  0.409091000,  0.772727000,  0.229091000 ],
    [  0.229091000,  0.886364000,  0.409091000 ],
    [  0.409091000,  0.886364000,  0.229091000 ],
    [  0.254545000,  0.886364000,  0.454545000 ],
    [  0.454545000,  0.886364000,  0.254545000 ],
    [  0.254545000,  0.772727000,  0.454545000 ],
    [  0.454545000,  0.772727000,  0.254545000 ],
    [  0.454545000,  0.545455000,  0.000000000 ],
    [  0.454545000,  0.545455000, -0.254545000 ],
    [  0.254545000,  0.545455000, -0.454545000 ],
    [  0.000000000,  0.545455000, -0.454545000 ],
    [  0.454545000,  0.272727000,  0.000000000 ],
    [  0.454545000,  0.272727000, -0.254545000 ],
    [  0.254545000,  0.272727000, -0.454545000 ],
    [  0.000000000,  0.272727000, -0.454545000 ],
    [  0.318182000,  0.045454500,  0.000000000 ],
    [  0.318182000,  0.045454500, -0.178182000 ],
    [  0.178182000,  0.045454500, -0.318182000 ],
    [  0.000000000,  0.045454500, -0.318182000 ],
    [ -0.254545000,  0.545455000, -0.454545000 ],
    [ -0.454545000,  0.545455000, -0.254545000 ],
    [ -0.454545000,  0.545455000,  0.000000000 ],
    [ -0.254545000,  0.272727000, -0.454545000 ],
    [ -0.454545000,  0.272727000, -0.254545000 ],
    [ -0.454545000,  0.272727000,  0.000000000 ],
    [ -0.178182000,  0.045454500, -0.318182000 ],
    [ -0.318182000,  0.045454500, -0.178182000 ],
    [ -0.318182000,  0.045454500,  0.000000000 ],
    [ -0.454545000,  0.545455000,  0.254545000 ],
    [ -0.254545000,  0.545455000,  0.454545000 ],
    [  0.000000000,  0.545455000,  0.454545000 ],
    [ -0.454545000,  0.272727000,  0.254545000 ],
    [ -0.254545000,  0.272727000,  0.454545000 ],
    [  0.000000000,  0.272727000,  0.454545000 ],
    [ -0.318182000,  0.045454500,  0.178182000 ],
    [ -0.178182000,  0.045454500,  0.318182000 ],
    [  0.000000000,  0.045454500,  0.318182000 ],
    [  0.254545000,  0.545455000,  0.454545000 ],
    [  0.454545000,  0.545455000,  0.254545000 ],
    [  0.254545000,  0.272727000,  0.454545000 ],
    [  0.454545000,  0.272727000,  0.254545000 ],
    [  0.178182000,  0.045454500,  0.318182000 ],
    [  0.318182000,  0.045454500,  0.178182000 ],
    [  0.545455000,  0.045454500,  0.000000000 ],
    [  0.545455000,  0.045454500, -0.305455000 ],
    [  0.305455000,  0.045454500, -0.545455000 ],
    [  0.000000000,  0.045454500, -0.545455000 ],
    [  0.727273000,  0.136364000,  0.000000000 ],
    [  0.727273000,  0.136364000, -0.407273000 ],
    [  0.407273000,  0.136364000, -0.727273000 ],
    [  0.000000000,  0.136364000, -0.727273000 ],
    [  0.909091000,  0.136364000,  0.000000000 ],
    [  0.909091000,  0.136364000, -0.509091000 ],
    [  0.509091000,  0.136364000, -0.909091000 ],
    [  0.000000000,  0.136364000, -0.909091000 ],
    [ -0.305455000,  0.045454500, -0.545455000 ],
    [ -0.545455000,  0.045454500, -0.305455000 ],
    [ -0.545455000,  0.045454500,  0.000000000 ],
    [ -0.407273000,  0.136364000, -0.727273000 ],
    [ -0.727273000,  0.136364000, -0.407273000 ],
    [ -0.727273000,  0.136364000,  0.000000000 ],
    [ -0.509091000,  0.136364000, -0.909091000 ],
    [ -0.909091000,  0.136364000, -0.509091000 ],
    [ -0.909091000,  0.136364000,  0.000000000 ],
    [ -0.545455000,  0.045454500,  0.305455000 ],
    [ -0.305455000,  0.045454500,  0.545455000 ],
    [  0.000000000,  0.045454500,  0.545455000 ],
    [ -0.727273000,  0.136364000,  0.407273000 ],
    [ -0.407273000,  0.136364000,  0.727273000 ],
    [  0.000000000,  0.136364000,  0.727273000 ],
    [ -0.909091000,  0.136364000,  0.509091000 ],
    [ -0.509091000,  0.136364000,  0.909091000 ],
    [  0.000000000,  0.136364000,  0.909091000 ],
    [  0.305455000,  0.045454500,  0.545455000 ],
    [  0.545455000,  0.045454500,  0.305455000 ],
    [  0.407273000,  0.136364000,  0.727273000 ],
    [  0.727273000,  0.136364000,  0.407273000 ],
    [  0.509091000,  0.136364000,  0.909091000 ],
    [  0.909091000,  0.136364000,  0.509091000 ],
    [  1.000000000,  0.136364000,  0.000000000 ],
    [  1.000000000,  0.136364000, -0.560000000 ],
    [  0.560000000,  0.136364000, -1.000000000 ],
    [  0.000000000,  0.136364000, -1.000000000 ],
    [  1.000000000,  0.090909100,  0.000000000 ],
    [  1.000000000,  0.090909100, -0.560000000 ],
    [  0.560000000,  0.090909100, -1.000000000 ],
    [  0.000000000,  0.090909100, -1.000000000 ],
    [  0.909091000,  0.090909100,  0.000000000 ],
    [  0.909091000,  0.090909100, -0.509091000 ],
    [  0.509091000,  0.090909100, -0.909091000 ],
    [  0.000000000,  0.090909100, -0.909091000 ],
    [ -0.560000000,  0.136364000, -1.000000000 ],
    [ -1.000000000,  0.136364000, -0.560000000 ],
    [ -1.000000000,  0.136364000,  0.000000000 ],
    [ -0.560000000,  0.090909100, -1.000000000 ],
    [ -1.000000000,  0.090909100, -0.560000000 ],
    [ -1.000000000,  0.090909100,  0.000000000 ],
    [ -0.509091000,  0.090909100, -0.909091000 ],
    [ -0.909091000,  0.090909100, -0.509091000 ],
    [ -0.909091000,  0.090909100,  0.000000000 ],
    [ -1.000000000,  0.136364000,  0.560000000 ],
    [ -0.560000000,  0.136364000,  1.000000000 ],
    [  0.000000000,  0.136364000,  1.000000000 ],
    [ -1.000000000,  0.090909100,  0.560000000 ],
    [ -0.560000000,  0.090909100,  1.000000000 ],
    [  0.000000000,  0.090909100,  1.000000000 ],
    [ -0.909091000,  0.090909100,  0.509091000 ],
    [ -0.509091000,  0.090909100,  0.909091000 ],
    [  0.000000000,  0.090909100,  0.909091000 ],
    [  0.560000000,  0.136364000,  1.000000000 ],
    [  1.000000000,  0.136364000,  0.560000000 ],
    [  0.560000000,  0.090909100,  1.000000000 ],
    [  1.000000000,  0.090909100,  0.560000000 ],
    [  0.509091000,  0.090909100,  0.909091000 ],
    [  0.909091000,  0.090909100,  0.509091000 ],
    [  0.727273000,  0.090909100,  0.000000000 ],
    [  0.727273000,  0.090909100, -0.407273000 ],
    [  0.407273000,  0.090909100, -0.727273000 ],
    [  0.000000000,  0.090909100, -0.727273000 ],
    [  0.545455000,  0.000000000,  0.000000000 ],
    [  0.545455000,  0.000000000, -0.305455000 ],
    [  0.305455000,  0.000000000, -0.545455000 ],
    [  0.000000000,  0.000000000, -0.545455000 ],
    [  0.318182000,  0.000000000,  0.000000000 ],
    [  0.318182000,  0.000000000, -0.178182000 ],
    [  0.178182000,  0.000000000, -0.318182000 ],
    [  0.000000000,  0.000000000, -0.318182000 ],
    [ -0.407273000,  0.090909100, -0.727273000 ],
    [ -0.727273000,  0.090909100, -0.407273000 ],
    [ -0.727273000,  0.090909100,  0.000000000 ],
    [ -0.305455000,  0.000000000, -0.545455000 ],
    [ -0.545455000,  0.000000000, -0.305455000 ],
    [ -0.545455000,  0.000000000,  0.000000000 ],
    [ -0.178182000,  0.000000000, -0.318182000 ],
    [ -0.318182000,  0.000000000, -0.178182000 ],
    [ -0.318182000,  0.000000000,  0.000000000 ],
    [ -0.727273000,  0.090909100,  0.407273000 ],
    [ -0.407273000,  0.090909100,  0.727273000 ],
    [  0.000000000,  0.090909100,  0.727273000 ],
    [ -0.545455000,  0.000000000,  0.305455000 ],
    [ -0.305455000,  0.000000000,  0.545455000 ],
    [  0.000000000,  0.000000000,  0.545455000 ],
    [ -0.318182000,  0.000000000,  0.178182000 ],
    [ -0.178182000,  0.000000000,  0.318182000 ],
    [  0.000000000,  0.000000000,  0.318182000 ],
    [  0.407273000,  0.090909100,  0.727273000 ],
    [  0.727273000,  0.090909100,  0.407273000 ],
    [  0.305455000,  0.000000000,  0.545455000 ],
    [  0.545455000,  0.000000000,  0.305455000 ],
    [  0.178182000,  0.000000000,  0.318182000 ],
    [  0.318182000,  0.000000000,  0.178182000 ],
    [  0.272727000,  0.045454500,  0.000000000 ],
    [  0.272727000,  0.045454500, -0.152727000 ],
    [  0.152727000,  0.045454500, -0.272727000 ],
    [  0.000000000,  0.045454500, -0.272727000 ],
    [  0.409091000,  0.272727000,  0.000000000 ],
    [  0.409091000,  0.272727000, -0.229091000 ],
    [  0.229091000,  0.272727000, -0.409091000 ],
    [  0.000000000,  0.272727000, -0.409091000 ],
    [  0.409091000,  0.545455000,  0.000000000 ],
    [  0.409091000,  0.545455000, -0.229091000 ],
    [  0.229091000,  0.545455000, -0.409091000 ],
    [  0.000000000,  0.545455000, -0.409091000 ],
    [ -0.152727000,  0.045454500, -0.272727000 ],
    [ -0.272727000,  0.045454500, -0.152727000 ],
    [ -0.272727000,  0.045454500,  0.000000000 ],
    [ -0.229091000,  0.272727000, -0.409091000 ],
    [ -0.409091000,  0.272727000, -0.229091000 ],
    [ -0.409091000,  0.272727000,  0.000000000 ],
    [ -0.229091000,  0.545455000, -0.409091000 ],
    [ -0.409091000,  0.545455000, -0.229091000 ],
    [ -0.409091000,  0.545455000,  0.000000000 ],
    [ -0.272727000,  0.045454500,  0.152727000 ],
    [ -0.152727000,  0.045454500,  0.272727000 ],
    [  0.000000000,  0.045454500,  0.272727000 ],
    [ -0.409091000,  0.272727000,  0.229091000 ],
    [ -0.229091000,  0.272727000,  0.409091000 ],
    [  0.000000000,  0.272727000,  0.409091000 ],
    [ -0.409091000,  0.545455000,  0.229091000 ],
    [ -0.229091000,  0.545455000,  0.409091000 ],
    [  0.000000000,  0.545455000,  0.409091000 ],
    [  0.152727000,  0.045454500,  0.272727000 ],
    [  0.272727000,  0.045454500,  0.152727000 ],
    [  0.229091000,  0.272727000,  0.409091000 ],
    [  0.409091000,  0.272727000,  0.229091000 ],
    [  0.229091000,  0.545455000,  0.409091000 ],
    [  0.409091000,  0.545455000,  0.229091000 ],
    [ -0.454545000,  0.704545000,  0.000000000 ],
    [ -0.454545000,  0.704545000, -0.045454500 ],
    [ -0.454545000,  0.772727000, -0.045454500 ],
    [ -0.772727000,  0.863636000,  0.000000000 ],
    [ -0.772727000,  0.863636000, -0.045454500 ],
    [ -0.818182000,  0.954545000, -0.045454500 ],
    [ -0.818182000,  0.954545000,  0.000000000 ],
    [ -0.772727000,  0.522727000,  0.000000000 ],
    [ -0.772727000,  0.522727000, -0.045454500 ],
    [ -0.909091000,  0.477273000, -0.045454500 ],
    [ -0.909091000,  0.477273000,  0.000000000 ],
    [ -0.409091000,  0.363636000,  0.000000000 ],
    [ -0.409091000,  0.363636000, -0.045454500 ],
    [ -0.409091000,  0.295455000, -0.045454500 ],
    [ -0.409091000,  0.295455000,  0.000000000 ],
    [ -0.454545000,  0.772727000,  0.045454500 ],
    [ -0.454545000,  0.704545000,  0.045454500 ],
    [ -0.818182000,  0.954545000,  0.045454500 ],
    [ -0.772727000,  0.863636000,  0.045454500 ],
    [ -0.909091000,  0.477273000,  0.045454500 ],
    [ -0.772727000,  0.522727000,  0.045454500 ],
    [ -0.409091000,  0.295455000,  0.045454500 ],
    [ -0.409091000,  0.363636000,  0.045454500 ]
];

///
/// Builds the pacthes needed to draw the teacup.
/// 
/// ```rust
/// use isogeometric_analysis::bezier::BezierTeacup;
/// use isogeometric_analysis::core::RealRange;
/// use isogeometric_analysis::core::Evaluator;
/// let patches = BezierTeacup::build_patches();
/// for patch in patches {
///     let r = RealRange { a: 0f64, b: 1f64 };
///     let (_xpoints, ypoints) = Evaluator::<2, 3, 100>::evaluate_parametric_range2d(&patch, &r, &r);
///     let (xvalues, yvalues, zvalues) = Evaluator::<2, 3, 0>::split_coords(0, &ypoints, 1, &ypoints, 2, &ypoints);
/// }
/// ```
/// 
pub struct BezierTeacup {}

impl BezierTeacup {
	///
	/// Build the patches from raw data.
	/// 
	pub fn build_patches() -> Vec<BezierSurf<3>> {
		BezierFactory::from_indexed_vertices(TEACUP_PACTHES.to_vec(), TEACUP_VERTICES.to_vec())
	}
}
