/// Represents the languages defined in ISO-639
/// ```
/// use iso_639::Language;
///
/// // get a language from any ISO639 code
/// let eng = Language::from_str("eng");
///
/// // get a language from an ISO639 alpha-2 code
/// let eng = Language::from_alpha3_t_str("en");
///
/// // get a language from an ISO639 alpha-3b code
/// let eng = Language::from_alpha3_b_str("eng");
///
/// // get a language from an ISO639 alpha-3t code
/// let fra = Language::from_alpha3_t_str("fra");
///
/// // get ISO630 codes
/// Language::French.alpha2();
/// Language::French.alpha3_b();
/// Language::French.alpha3_t();
///
/// // get English name
/// Language::French.english_name();
/// ```
#[derive(Debug, PartialEq)]
pub enum Language {
    /// Abkhazian
    /// - alpha2: ab
    /// - alpha3-b: abk
    Abkhazian,
    /// Achinese
    /// - alpha3-b: ace
    Achinese,
    /// Acoli
    /// - alpha3-b: ach
    Acoli,
    /// Adangme
    /// - alpha3-b: ada
    Adangme,
    /// Adyghe; Adygei
    /// - alpha3-b: ady
    Adyghe,
    /// Afar
    /// - alpha2: aa
    /// - alpha3-b: aar
    Afar,
    /// Afrihili
    /// - alpha3-b: afh
    Afrihili,
    /// Afrikaans
    /// - alpha2: af
    /// - alpha3-b: afr
    Afrikaans,
    /// Afro-Asiatic languages
    /// - alpha3-b: afa
    AfroAsiaticLanguages,
    /// Ainu
    /// - alpha3-b: ain
    Ainu,
    /// Akan
    /// - alpha2: ak
    /// - alpha3-b: aka
    Akan,
    /// Akkadian
    /// - alpha3-b: akk
    Akkadian,
    /// Albanian
    /// - alpha2: sq
    /// - alpha3-b: alb
    /// - alpha3-t: sqi
    Albanian,
    /// Aleut
    /// - alpha3-b: ale
    Aleut,
    /// Algonquian languages
    /// - alpha3-b: alg
    AlgonquianLanguages,
    /// Altaic languages
    /// - alpha3-b: tut
    AltaicLanguages,
    /// Amharic
    /// - alpha2: am
    /// - alpha3-b: amh
    Amharic,
    /// Angika
    /// - alpha3-b: anp
    Angika,
    /// Apache languages
    /// - alpha3-b: apa
    ApacheLanguages,
    /// Arabic
    /// - alpha2: ar
    /// - alpha3-b: ara
    Arabic,
    /// Aragonese
    /// - alpha2: an
    /// - alpha3-b: arg
    Aragonese,
    /// Arapaho
    /// - alpha3-b: arp
    Arapaho,
    /// Arawak
    /// - alpha3-b: arw
    Arawak,
    /// Armenian
    /// - alpha2: hy
    /// - alpha3-b: arm
    /// - alpha3-t: hye
    Armenian,
    /// Aromanian; Arumanian; Macedo-Romanian
    /// - alpha3-b: rup
    Aromanian,
    /// Artificial languages
    /// - alpha3-b: art
    ArtificialLanguages,
    /// Assamese
    /// - alpha2: as
    /// - alpha3-b: asm
    Assamese,
    /// Asturian; Bable; Leonese; Asturleonese
    /// - alpha3-b: ast
    Asturian,
    /// Athapascan languages
    /// - alpha3-b: ath
    AthapascanLanguages,
    /// Australian languages
    /// - alpha3-b: aus
    AustralianLanguages,
    /// Austronesian languages
    /// - alpha3-b: map
    AustronesianLanguages,
    /// Avaric
    /// - alpha2: av
    /// - alpha3-b: ava
    Avaric,
    /// Avestan
    /// - alpha2: ae
    /// - alpha3-b: ave
    Avestan,
    /// Awadhi
    /// - alpha3-b: awa
    Awadhi,
    /// Aymara
    /// - alpha2: ay
    /// - alpha3-b: aym
    Aymara,
    /// Azerbaijani
    /// - alpha2: az
    /// - alpha3-b: aze
    Azerbaijani,
    /// Balinese
    /// - alpha3-b: ban
    Balinese,
    /// Baltic languages
    /// - alpha3-b: bat
    BalticLanguages,
    /// Baluchi
    /// - alpha3-b: bal
    Baluchi,
    /// Bambara
    /// - alpha2: bm
    /// - alpha3-b: bam
    Bambara,
    /// Bamileke languages
    /// - alpha3-b: bai
    BamilekeLanguages,
    /// Banda languages
    /// - alpha3-b: bad
    BandaLanguages,
    /// Bantu languages
    /// - alpha3-b: bnt
    BantuLanguages,
    /// Basa
    /// - alpha3-b: bas
    Basa,
    /// Bashkir
    /// - alpha2: ba
    /// - alpha3-b: bak
    Bashkir,
    /// Basque
    /// - alpha2: eu
    /// - alpha3-b: baq
    /// - alpha3-t: eus
    Basque,
    /// Batak languages
    /// - alpha3-b: btk
    BatakLanguages,
    /// Beja; Bedawiyet
    /// - alpha3-b: bej
    Beja,
    /// Belarusian
    /// - alpha2: be
    /// - alpha3-b: bel
    Belarusian,
    /// Bemba
    /// - alpha3-b: bem
    Bemba,
    /// Bengali
    /// - alpha2: bn
    /// - alpha3-b: ben
    Bengali,
    /// Berber languages
    /// - alpha3-b: ber
    BerberLanguages,
    /// Bhojpuri
    /// - alpha3-b: bho
    Bhojpuri,
    /// Bihari languages
    /// - alpha2: bh
    /// - alpha3-b: bih
    BihariLanguages,
    /// Bikol
    /// - alpha3-b: bik
    Bikol,
    /// Bini; Edo
    /// - alpha3-b: bin
    Bini,
    /// Bislama
    /// - alpha2: bi
    /// - alpha3-b: bis
    Bislama,
    /// Blin; Bilin
    /// - alpha3-b: byn
    Blin,
    /// Blissymbols; Blissymbolics; Bliss
    /// - alpha3-b: zbl
    Blissymbols,
    /// Bokmål, Norwegian; Norwegian Bokmål
    /// - alpha2: nb
    /// - alpha3-b: nob
    BokmålNorwegian,
    /// Bosnian
    /// - alpha2: bs
    /// - alpha3-b: bos
    Bosnian,
    /// Braj
    /// - alpha3-b: bra
    Braj,
    /// Breton
    /// - alpha2: br
    /// - alpha3-b: bre
    Breton,
    /// Buginese
    /// - alpha3-b: bug
    Buginese,
    /// Bulgarian
    /// - alpha2: bg
    /// - alpha3-b: bul
    Bulgarian,
    /// Buriat
    /// - alpha3-b: bua
    Buriat,
    /// Burmese
    /// - alpha2: my
    /// - alpha3-b: bur
    /// - alpha3-t: mya
    Burmese,
    /// Caddo
    /// - alpha3-b: cad
    Caddo,
    /// Catalan; Valencian
    /// - alpha2: ca
    /// - alpha3-b: cat
    Catalan,
    /// Caucasian languages
    /// - alpha3-b: cau
    CaucasianLanguages,
    /// Cebuano
    /// - alpha3-b: ceb
    Cebuano,
    /// Celtic languages
    /// - alpha3-b: cel
    CelticLanguages,
    /// Central American Indian languages
    /// - alpha3-b: cai
    CentralAmericanIndianLanguages,
    /// Central Khmer
    /// - alpha2: km
    /// - alpha3-b: khm
    CentralKhmer,
    /// Chagatai
    /// - alpha3-b: chg
    Chagatai,
    /// Chamic languages
    /// - alpha3-b: cmc
    ChamicLanguages,
    /// Chamorro
    /// - alpha2: ch
    /// - alpha3-b: cha
    Chamorro,
    /// Chechen
    /// - alpha2: ce
    /// - alpha3-b: che
    Chechen,
    /// Cherokee
    /// - alpha3-b: chr
    Cherokee,
    /// Cheyenne
    /// - alpha3-b: chy
    Cheyenne,
    /// Chibcha
    /// - alpha3-b: chb
    Chibcha,
    /// Chichewa; Chewa; Nyanja
    /// - alpha2: ny
    /// - alpha3-b: nya
    Chichewa,
    /// Chinese
    /// - alpha2: zh
    /// - alpha3-b: chi
    /// - alpha3-t: zho
    Chinese,
    /// Chinook jargon
    /// - alpha3-b: chn
    Chinook,
    /// Chipewyan; Dene Suline
    /// - alpha3-b: chp
    Chipewyan,
    /// Choctaw
    /// - alpha3-b: cho
    Choctaw,
    /// Church Slavic; Old Slavonic; Church Slavonic; Old Bulgarian; Old Church Slavonic
    /// - alpha2: cu
    /// - alpha3-b: chu
    ChurchSlavic,
    /// Chuukese
    /// - alpha3-b: chk
    Chuukese,
    /// Chuvash
    /// - alpha2: cv
    /// - alpha3-b: chv
    Chuvash,
    /// Classical Newari; Old Newari; Classical Nepal Bhasa
    /// - alpha3-b: nwc
    ClassicalNewari,
    /// Classical Syriac
    /// - alpha3-b: syc
    ClassicalSyriac,
    /// Coptic
    /// - alpha3-b: cop
    Coptic,
    /// Cornish
    /// - alpha2: kw
    /// - alpha3-b: cor
    Cornish,
    /// Corsican
    /// - alpha2: co
    /// - alpha3-b: cos
    Corsican,
    /// Cree
    /// - alpha2: cr
    /// - alpha3-b: cre
    Cree,
    /// Creek
    /// - alpha3-b: mus
    Creek,
    /// Creoles and pidgins
    /// - alpha3-b: crp
    CreolesAndPidgins,
    /// Creoles and pidgins, English based
    /// - alpha3-b: cpe
    CreolesAndPidginsEnglishBased,
    /// Creoles and pidgins, French-based
    /// - alpha3-b: cpf
    CreolesAndPidginsFrenchBased,
    /// Creoles and pidgins, Portuguese-based
    /// - alpha3-b: cpp
    CreolesAndPidginsPortugueseBased,
    /// Crimean Tatar; Crimean Turkish
    /// - alpha3-b: crh
    CrimeanTatar,
    /// Croatian
    /// - alpha2: hr
    /// - alpha3-b: hrv
    Croatian,
    /// Cushitic languages
    /// - alpha3-b: cus
    CushiticLanguages,
    /// Czech
    /// - alpha2: cs
    /// - alpha3-b: cze
    /// - alpha3-t: ces
    Czech,
    /// Dakota
    /// - alpha3-b: dak
    Dakota,
    /// Danish
    /// - alpha2: da
    /// - alpha3-b: dan
    Danish,
    /// Dargwa
    /// - alpha3-b: dar
    Dargwa,
    /// Delaware
    /// - alpha3-b: del
    Delaware,
    /// Dinka
    /// - alpha3-b: din
    Dinka,
    /// Divehi; Dhivehi; Maldivian
    /// - alpha2: dv
    /// - alpha3-b: div
    Divehi,
    /// Dogri
    /// - alpha3-b: doi
    Dogri,
    /// Dogrib
    /// - alpha3-b: dgr
    Dogrib,
    /// Dravidian languages
    /// - alpha3-b: dra
    DravidianLanguages,
    /// Duala
    /// - alpha3-b: dua
    Duala,
    /// Dutch; Flemish
    /// - alpha2: nl
    /// - alpha3-b: dut
    /// - alpha3-t: nld
    Dutch,
    /// Dutch, Middle (ca.1050-1350)
    /// - alpha3-b: dum
    DutchMiddle,
    /// Dyula
    /// - alpha3-b: dyu
    Dyula,
    /// Dzongkha
    /// - alpha2: dz
    /// - alpha3-b: dzo
    Dzongkha,
    /// Eastern Frisian
    /// - alpha3-b: frs
    EasternFrisian,
    /// Efik
    /// - alpha3-b: efi
    Efik,
    /// Egyptian (Ancient)
    /// - alpha3-b: egy
    EgyptianAncient,
    /// Ekajuk
    /// - alpha3-b: eka
    Ekajuk,
    /// Elamite
    /// - alpha3-b: elx
    Elamite,
    /// English
    /// - alpha2: en
    /// - alpha3-b: eng
    English,
    /// English, Middle (1100-1500)
    /// - alpha3-b: enm
    EnglishMiddle,
    /// English, Old (ca.450-1100)
    /// - alpha3-b: ang
    EnglishOld,
    /// Erzya
    /// - alpha3-b: myv
    Erzya,
    /// Esperanto
    /// - alpha2: eo
    /// - alpha3-b: epo
    Esperanto,
    /// Estonian
    /// - alpha2: et
    /// - alpha3-b: est
    Estonian,
    /// Ewe
    /// - alpha2: ee
    /// - alpha3-b: ewe
    Ewe,
    /// Ewondo
    /// - alpha3-b: ewo
    Ewondo,
    /// Fang
    /// - alpha3-b: fan
    Fang,
    /// Fanti
    /// - alpha3-b: fat
    Fanti,
    /// Faroese
    /// - alpha2: fo
    /// - alpha3-b: fao
    Faroese,
    /// Fijian
    /// - alpha2: fj
    /// - alpha3-b: fij
    Fijian,
    /// Filipino; Pilipino
    /// - alpha3-b: fil
    Filipino,
    /// Finnish
    /// - alpha2: fi
    /// - alpha3-b: fin
    Finnish,
    /// Finno-Ugrian languages
    /// - alpha3-b: fiu
    FinnoUgrianLanguages,
    /// Fon
    /// - alpha3-b: fon
    Fon,
    /// French
    /// - alpha2: fr
    /// - alpha3-b: fre
    /// - alpha3-t: fra
    French,
    /// French, Middle (ca.1400-1600)
    /// - alpha3-b: frm
    FrenchMiddle,
    /// French, Old (842-ca.1400)
    /// - alpha3-b: fro
    FrenchOld,
    /// Friulian
    /// - alpha3-b: fur
    Friulian,
    /// Fulah
    /// - alpha2: ff
    /// - alpha3-b: ful
    Fulah,
    /// Ga
    /// - alpha3-b: gaa
    Ga,
    /// Gaelic; Scottish Gaelic
    /// - alpha2: gd
    /// - alpha3-b: gla
    Gaelic,
    /// Galibi Carib
    /// - alpha3-b: car
    GalibiCarib,
    /// Galician
    /// - alpha2: gl
    /// - alpha3-b: glg
    Galician,
    /// Ganda
    /// - alpha2: lg
    /// - alpha3-b: lug
    Ganda,
    /// Gayo
    /// - alpha3-b: gay
    Gayo,
    /// Gbaya
    /// - alpha3-b: gba
    Gbaya,
    /// Geez
    /// - alpha3-b: gez
    Geez,
    /// Georgian
    /// - alpha2: ka
    /// - alpha3-b: geo
    /// - alpha3-t: kat
    Georgian,
    /// German
    /// - alpha2: de
    /// - alpha3-b: ger
    /// - alpha3-t: deu
    German,
    /// German, Middle High (ca.1050-1500)
    /// - alpha3-b: gmh
    GermanMiddleHigh,
    /// German, Old High (ca.750-1050)
    /// - alpha3-b: goh
    GermanOldHigh,
    /// Germanic languages
    /// - alpha3-b: gem
    GermanicLanguages,
    /// Gilbertese
    /// - alpha3-b: gil
    Gilbertese,
    /// Gondi
    /// - alpha3-b: gon
    Gondi,
    /// Gorontalo
    /// - alpha3-b: gor
    Gorontalo,
    /// Gothic
    /// - alpha3-b: got
    Gothic,
    /// Grebo
    /// - alpha3-b: grb
    Grebo,
    /// Greek, Ancient (to 1453)
    /// - alpha3-b: grc
    GreekAncient,
    /// Greek, Modern (1453-)
    /// - alpha2: el
    /// - alpha3-b: gre
    /// - alpha3-t: ell
    GreekModern,
    /// Guarani
    /// - alpha2: gn
    /// - alpha3-b: grn
    Guarani,
    /// Gujarati
    /// - alpha2: gu
    /// - alpha3-b: guj
    Gujarati,
    /// Gwich'in
    /// - alpha3-b: gwi
    GwichIn,
    /// Haida
    /// - alpha3-b: hai
    Haida,
    /// Haitian; Haitian Creole
    /// - alpha2: ht
    /// - alpha3-b: hat
    Haitian,
    /// Hausa
    /// - alpha2: ha
    /// - alpha3-b: hau
    Hausa,
    /// Hawaiian
    /// - alpha3-b: haw
    Hawaiian,
    /// Hebrew
    /// - alpha2: he
    /// - alpha3-b: heb
    Hebrew,
    /// Herero
    /// - alpha2: hz
    /// - alpha3-b: her
    Herero,
    /// Hiligaynon
    /// - alpha3-b: hil
    Hiligaynon,
    /// Himachali languages; Western Pahari languages
    /// - alpha3-b: him
    HimachaliLanguages,
    /// Hindi
    /// - alpha2: hi
    /// - alpha3-b: hin
    Hindi,
    /// Hiri Motu
    /// - alpha2: ho
    /// - alpha3-b: hmo
    HiriMotu,
    /// Hittite
    /// - alpha3-b: hit
    Hittite,
    /// Hmong; Mong
    /// - alpha3-b: hmn
    Hmong,
    /// Hungarian
    /// - alpha2: hu
    /// - alpha3-b: hun
    Hungarian,
    /// Hupa
    /// - alpha3-b: hup
    Hupa,
    /// Iban
    /// - alpha3-b: iba
    Iban,
    /// Icelandic
    /// - alpha2: is
    /// - alpha3-b: ice
    /// - alpha3-t: isl
    Icelandic,
    /// Ido
    /// - alpha2: io
    /// - alpha3-b: ido
    Ido,
    /// Igbo
    /// - alpha2: ig
    /// - alpha3-b: ibo
    Igbo,
    /// Ijo languages
    /// - alpha3-b: ijo
    IjoLanguages,
    /// Iloko
    /// - alpha3-b: ilo
    Iloko,
    /// Inari Sami
    /// - alpha3-b: smn
    InariSami,
    /// Indic languages
    /// - alpha3-b: inc
    IndicLanguages,
    /// Indo-European languages
    /// - alpha3-b: ine
    IndoEuropeanLanguages,
    /// Indonesian
    /// - alpha2: id
    /// - alpha3-b: ind
    Indonesian,
    /// Ingush
    /// - alpha3-b: inh
    Ingush,
    /// Interlingua (International Auxiliary Language Association)
    /// - alpha2: ia
    /// - alpha3-b: ina
    Interlingua,
    /// Interlingue; Occidental
    /// - alpha2: ie
    /// - alpha3-b: ile
    Interlingue,
    /// Inuktitut
    /// - alpha2: iu
    /// - alpha3-b: iku
    Inuktitut,
    /// Inupiaq
    /// - alpha2: ik
    /// - alpha3-b: ipk
    Inupiaq,
    /// Iranian languages
    /// - alpha3-b: ira
    IranianLanguages,
    /// Irish
    /// - alpha2: ga
    /// - alpha3-b: gle
    Irish,
    /// Irish, Middle (900-1200)
    /// - alpha3-b: mga
    IrishMiddle,
    /// Irish, Old (to 900)
    /// - alpha3-b: sga
    IrishOld,
    /// Iroquoian languages
    /// - alpha3-b: iro
    IroquoianLanguages,
    /// Italian
    /// - alpha2: it
    /// - alpha3-b: ita
    Italian,
    /// Japanese
    /// - alpha2: ja
    /// - alpha3-b: jpn
    Japanese,
    /// Javanese
    /// - alpha2: jv
    /// - alpha3-b: jav
    Javanese,
    /// Judeo-Arabic
    /// - alpha3-b: jrb
    JudeoArabic,
    /// Judeo-Persian
    /// - alpha3-b: jpr
    JudeoPersian,
    /// Kabardian
    /// - alpha3-b: kbd
    Kabardian,
    /// Kabyle
    /// - alpha3-b: kab
    Kabyle,
    /// Kachin; Jingpho
    /// - alpha3-b: kac
    Kachin,
    /// Kalaallisut; Greenlandic
    /// - alpha2: kl
    /// - alpha3-b: kal
    Kalaallisut,
    /// Kalmyk; Oirat
    /// - alpha3-b: xal
    Kalmyk,
    /// Kamba
    /// - alpha3-b: kam
    Kamba,
    /// Kannada
    /// - alpha2: kn
    /// - alpha3-b: kan
    Kannada,
    /// Kanuri
    /// - alpha2: kr
    /// - alpha3-b: kau
    Kanuri,
    /// Kara-Kalpak
    /// - alpha3-b: kaa
    KaraKalpak,
    /// Karachay-Balkar
    /// - alpha3-b: krc
    KarachayBalkar,
    /// Karelian
    /// - alpha3-b: krl
    Karelian,
    /// Karen languages
    /// - alpha3-b: kar
    KarenLanguages,
    /// Kashmiri
    /// - alpha2: ks
    /// - alpha3-b: kas
    Kashmiri,
    /// Kashubian
    /// - alpha3-b: csb
    Kashubian,
    /// Kawi
    /// - alpha3-b: kaw
    Kawi,
    /// Kazakh
    /// - alpha2: kk
    /// - alpha3-b: kaz
    Kazakh,
    /// Khasi
    /// - alpha3-b: kha
    Khasi,
    /// Khoisan languages
    /// - alpha3-b: khi
    KhoisanLanguages,
    /// Khotanese; Sakan
    /// - alpha3-b: kho
    Khotanese,
    /// Kikuyu; Gikuyu
    /// - alpha2: ki
    /// - alpha3-b: kik
    Kikuyu,
    /// Kimbundu
    /// - alpha3-b: kmb
    Kimbundu,
    /// Kinyarwanda
    /// - alpha2: rw
    /// - alpha3-b: kin
    Kinyarwanda,
    /// Kirghiz; Kyrgyz
    /// - alpha2: ky
    /// - alpha3-b: kir
    Kirghiz,
    /// Klingon; tlhIngan-Hol
    /// - alpha3-b: tlh
    Klingon,
    /// Komi
    /// - alpha2: kv
    /// - alpha3-b: kom
    Komi,
    /// Kongo
    /// - alpha2: kg
    /// - alpha3-b: kon
    Kongo,
    /// Konkani
    /// - alpha3-b: kok
    Konkani,
    /// Korean
    /// - alpha2: ko
    /// - alpha3-b: kor
    Korean,
    /// Kosraean
    /// - alpha3-b: kos
    Kosraean,
    /// Kpelle
    /// - alpha3-b: kpe
    Kpelle,
    /// Kru languages
    /// - alpha3-b: kro
    KruLanguages,
    /// Kuanyama; Kwanyama
    /// - alpha2: kj
    /// - alpha3-b: kua
    Kuanyama,
    /// Kumyk
    /// - alpha3-b: kum
    Kumyk,
    /// Kurdish
    /// - alpha2: ku
    /// - alpha3-b: kur
    Kurdish,
    /// Kurukh
    /// - alpha3-b: kru
    Kurukh,
    /// Kutenai
    /// - alpha3-b: kut
    Kutenai,
    /// Ladino
    /// - alpha3-b: lad
    Ladino,
    /// Lahnda
    /// - alpha3-b: lah
    Lahnda,
    /// Lamba
    /// - alpha3-b: lam
    Lamba,
    /// Land Dayak languages
    /// - alpha3-b: day
    LandDayakLanguages,
    /// Lao
    /// - alpha2: lo
    /// - alpha3-b: lao
    Lao,
    /// Latin
    /// - alpha2: la
    /// - alpha3-b: lat
    Latin,
    /// Latvian
    /// - alpha2: lv
    /// - alpha3-b: lav
    Latvian,
    /// Lezghian
    /// - alpha3-b: lez
    Lezghian,
    /// Limburgan; Limburger; Limburgish
    /// - alpha2: li
    /// - alpha3-b: lim
    Limburgan,
    /// Lingala
    /// - alpha2: ln
    /// - alpha3-b: lin
    Lingala,
    /// Lithuanian
    /// - alpha2: lt
    /// - alpha3-b: lit
    Lithuanian,
    /// Lojban
    /// - alpha3-b: jbo
    Lojban,
    /// Low German; Low Saxon; German, Low; Saxon, Low
    /// - alpha3-b: nds
    LowGerman,
    /// Lower Sorbian
    /// - alpha3-b: dsb
    LowerSorbian,
    /// Lozi
    /// - alpha3-b: loz
    Lozi,
    /// Luba-Katanga
    /// - alpha2: lu
    /// - alpha3-b: lub
    LubaKatanga,
    /// Luba-Lulua
    /// - alpha3-b: lua
    LubaLulua,
    /// Luiseno
    /// - alpha3-b: lui
    Luiseno,
    /// Lule Sami
    /// - alpha3-b: smj
    LuleSami,
    /// Lunda
    /// - alpha3-b: lun
    Lunda,
    /// Luo (Kenya and Tanzania)
    /// - alpha3-b: luo
    LuoKenyaAndTanzania,
    /// Lushai
    /// - alpha3-b: lus
    Lushai,
    /// Luxembourgish; Letzeburgesch
    /// - alpha2: lb
    /// - alpha3-b: ltz
    Luxembourgish,
    /// Macedonian
    /// - alpha2: mk
    /// - alpha3-b: mac
    /// - alpha3-t: mkd
    Macedonian,
    /// Madurese
    /// - alpha3-b: mad
    Madurese,
    /// Magahi
    /// - alpha3-b: mag
    Magahi,
    /// Maithili
    /// - alpha3-b: mai
    Maithili,
    /// Makasar
    /// - alpha3-b: mak
    Makasar,
    /// Malagasy
    /// - alpha2: mg
    /// - alpha3-b: mlg
    Malagasy,
    /// Malay
    /// - alpha2: ms
    /// - alpha3-b: may
    /// - alpha3-t: msa
    Malay,
    /// Malayalam
    /// - alpha2: ml
    /// - alpha3-b: mal
    Malayalam,
    /// Maltese
    /// - alpha2: mt
    /// - alpha3-b: mlt
    Maltese,
    /// Manchu
    /// - alpha3-b: mnc
    Manchu,
    /// Mandar
    /// - alpha3-b: mdr
    Mandar,
    /// Mandingo
    /// - alpha3-b: man
    Mandingo,
    /// Manipuri
    /// - alpha3-b: mni
    Manipuri,
    /// Manobo languages
    /// - alpha3-b: mno
    ManoboLanguages,
    /// Manx
    /// - alpha2: gv
    /// - alpha3-b: glv
    Manx,
    /// Maori
    /// - alpha2: mi
    /// - alpha3-b: mao
    /// - alpha3-t: mri
    Maori,
    /// Mapudungun; Mapuche
    /// - alpha3-b: arn
    Mapudungun,
    /// Marathi
    /// - alpha2: mr
    /// - alpha3-b: mar
    Marathi,
    /// Mari
    /// - alpha3-b: chm
    Mari,
    /// Marshallese
    /// - alpha2: mh
    /// - alpha3-b: mah
    Marshallese,
    /// Marwari
    /// - alpha3-b: mwr
    Marwari,
    /// Masai
    /// - alpha3-b: mas
    Masai,
    /// Mayan languages
    /// - alpha3-b: myn
    MayanLanguages,
    /// Mende
    /// - alpha3-b: men
    Mende,
    /// Mi'kmaq; Micmac
    /// - alpha3-b: mic
    MiKmaq,
    /// Minangkabau
    /// - alpha3-b: min
    Minangkabau,
    /// Mirandese
    /// - alpha3-b: mwl
    Mirandese,
    /// Mohawk
    /// - alpha3-b: moh
    Mohawk,
    /// Moksha
    /// - alpha3-b: mdf
    Moksha,
    /// Mon-Khmer languages
    /// - alpha3-b: mkh
    MonKhmerLanguages,
    /// Mongo
    /// - alpha3-b: lol
    Mongo,
    /// Mongolian
    /// - alpha2: mn
    /// - alpha3-b: mon
    Mongolian,
    /// Montenegrin
    /// - alpha3-b: cnr
    Montenegrin,
    /// Mossi
    /// - alpha3-b: mos
    Mossi,
    /// Multiple languages
    /// - alpha3-b: mul
    MultipleLanguages,
    /// Munda languages
    /// - alpha3-b: mun
    MundaLanguages,
    /// N'Ko
    /// - alpha3-b: nqo
    NKo,
    /// Nahuatl languages
    /// - alpha3-b: nah
    NahuatlLanguages,
    /// Nauru
    /// - alpha2: na
    /// - alpha3-b: nau
    Nauru,
    /// Navajo; Navaho
    /// - alpha2: nv
    /// - alpha3-b: nav
    Navajo,
    /// Ndebele, North; North Ndebele
    /// - alpha2: nd
    /// - alpha3-b: nde
    NdebeleNorth,
    /// Ndebele, South; South Ndebele
    /// - alpha2: nr
    /// - alpha3-b: nbl
    NdebeleSouth,
    /// Ndonga
    /// - alpha2: ng
    /// - alpha3-b: ndo
    Ndonga,
    /// Neapolitan
    /// - alpha3-b: nap
    Neapolitan,
    /// Nepal Bhasa; Newari
    /// - alpha3-b: new
    NepalBhasa,
    /// Nepali
    /// - alpha2: ne
    /// - alpha3-b: nep
    Nepali,
    /// Nias
    /// - alpha3-b: nia
    Nias,
    /// Niger-Kordofanian languages
    /// - alpha3-b: nic
    NigerKordofanianLanguages,
    /// Nilo-Saharan languages
    /// - alpha3-b: ssa
    NiloSaharanLanguages,
    /// Niuean
    /// - alpha3-b: niu
    Niuean,
    /// No linguistic content; Not applicable
    /// - alpha3-b: zxx
    NoLinguisticContent,
    /// Nogai
    /// - alpha3-b: nog
    Nogai,
    /// Norse, Old
    /// - alpha3-b: non
    NorseOld,
    /// North American Indian languages
    /// - alpha3-b: nai
    NorthAmericanIndianLanguages,
    /// Northern Frisian
    /// - alpha3-b: frr
    NorthernFrisian,
    /// Northern Sami
    /// - alpha2: se
    /// - alpha3-b: sme
    NorthernSami,
    /// Norwegian
    /// - alpha2: no
    /// - alpha3-b: nor
    Norwegian,
    /// Norwegian Nynorsk; Nynorsk, Norwegian
    /// - alpha2: nn
    /// - alpha3-b: nno
    NorwegianNynorsk,
    /// Nubian languages
    /// - alpha3-b: nub
    NubianLanguages,
    /// Nyamwezi
    /// - alpha3-b: nym
    Nyamwezi,
    /// Nyankole
    /// - alpha3-b: nyn
    Nyankole,
    /// Nyoro
    /// - alpha3-b: nyo
    Nyoro,
    /// Nzima
    /// - alpha3-b: nzi
    Nzima,
    /// Occitan (post 1500)
    /// - alpha2: oc
    /// - alpha3-b: oci
    Occitan,
    /// Official Aramaic (700-300 BCE); Imperial Aramaic (700-300 BCE)
    /// - alpha3-b: arc
    OfficialAramaic,
    /// Ojibwa
    /// - alpha2: oj
    /// - alpha3-b: oji
    Ojibwa,
    /// Oriya
    /// - alpha2: or
    /// - alpha3-b: ori
    Oriya,
    /// Oromo
    /// - alpha2: om
    /// - alpha3-b: orm
    Oromo,
    /// Osage
    /// - alpha3-b: osa
    Osage,
    /// Ossetian; Ossetic
    /// - alpha2: os
    /// - alpha3-b: oss
    Ossetian,
    /// Otomian languages
    /// - alpha3-b: oto
    OtomianLanguages,
    /// Pahlavi
    /// - alpha3-b: pal
    Pahlavi,
    /// Palauan
    /// - alpha3-b: pau
    Palauan,
    /// Pali
    /// - alpha2: pi
    /// - alpha3-b: pli
    Pali,
    /// Pampanga; Kapampangan
    /// - alpha3-b: pam
    Pampanga,
    /// Pangasinan
    /// - alpha3-b: pag
    Pangasinan,
    /// Panjabi; Punjabi
    /// - alpha2: pa
    /// - alpha3-b: pan
    Panjabi,
    /// Papiamento
    /// - alpha3-b: pap
    Papiamento,
    /// Papuan languages
    /// - alpha3-b: paa
    PapuanLanguages,
    /// Pedi; Sepedi; Northern Sotho
    /// - alpha3-b: nso
    Pedi,
    /// Persian
    /// - alpha2: fa
    /// - alpha3-b: per
    /// - alpha3-t: fas
    Persian,
    /// Persian, Old (ca.600-400 B.C.)
    /// - alpha3-b: peo
    PersianOld,
    /// Philippine languages
    /// - alpha3-b: phi
    PhilippineLanguages,
    /// Phoenician
    /// - alpha3-b: phn
    Phoenician,
    /// Pohnpeian
    /// - alpha3-b: pon
    Pohnpeian,
    /// Polish
    /// - alpha2: pl
    /// - alpha3-b: pol
    Polish,
    /// Portuguese
    /// - alpha2: pt
    /// - alpha3-b: por
    Portuguese,
    /// Prakrit languages
    /// - alpha3-b: pra
    PrakritLanguages,
    /// Provençal, Old (to 1500); Occitan, Old (to 1500)
    /// - alpha3-b: pro
    ProvencalOld,
    /// Pushto; Pashto
    /// - alpha2: ps
    /// - alpha3-b: pus
    Pushto,
    /// Quechua
    /// - alpha2: qu
    /// - alpha3-b: que
    Quechua,
    /// Rajasthani
    /// - alpha3-b: raj
    Rajasthani,
    /// Rapanui
    /// - alpha3-b: rap
    Rapanui,
    /// Rarotongan; Cook Islands Maori
    /// - alpha3-b: rar
    Rarotongan,
    /// Romance languages
    /// - alpha3-b: roa
    RomanceLanguages,
    /// Romanian; Moldavian; Moldovan
    /// - alpha2: ro
    /// - alpha3-b: rum
    /// - alpha3-t: ron
    Romanian,
    /// Romansh
    /// - alpha2: rm
    /// - alpha3-b: roh
    Romansh,
    /// Romany
    /// - alpha3-b: rom
    Romany,
    /// Rundi
    /// - alpha2: rn
    /// - alpha3-b: run
    Rundi,
    /// Russian
    /// - alpha2: ru
    /// - alpha3-b: rus
    Russian,
    /// Salishan languages
    /// - alpha3-b: sal
    SalishanLanguages,
    /// Samaritan Aramaic
    /// - alpha3-b: sam
    SamaritanAramaic,
    /// Sami languages
    /// - alpha3-b: smi
    SamiLanguages,
    /// Samoan
    /// - alpha2: sm
    /// - alpha3-b: smo
    Samoan,
    /// Sandawe
    /// - alpha3-b: sad
    Sandawe,
    /// Sango
    /// - alpha2: sg
    /// - alpha3-b: sag
    Sango,
    /// Sanskrit
    /// - alpha2: sa
    /// - alpha3-b: san
    Sanskrit,
    /// Santali
    /// - alpha3-b: sat
    Santali,
    /// Sardinian
    /// - alpha2: sc
    /// - alpha3-b: srd
    Sardinian,
    /// Sasak
    /// - alpha3-b: sas
    Sasak,
    /// Scots
    /// - alpha3-b: sco
    Scots,
    /// Selkup
    /// - alpha3-b: sel
    Selkup,
    /// Semitic languages
    /// - alpha3-b: sem
    SemiticLanguages,
    /// Serbian
    /// - alpha2: sr
    /// - alpha3-b: srp
    Serbian,
    /// Serer
    /// - alpha3-b: srr
    Serer,
    /// Shan
    /// - alpha3-b: shn
    Shan,
    /// Shona
    /// - alpha2: sn
    /// - alpha3-b: sna
    Shona,
    /// Sichuan Yi; Nuosu
    /// - alpha2: ii
    /// - alpha3-b: iii
    SichuanYi,
    /// Sicilian
    /// - alpha3-b: scn
    Sicilian,
    /// Sidamo
    /// - alpha3-b: sid
    Sidamo,
    /// Sign Languages
    /// - alpha3-b: sgn
    SignLanguages,
    /// Siksika
    /// - alpha3-b: bla
    Siksika,
    /// Sindhi
    /// - alpha2: sd
    /// - alpha3-b: snd
    Sindhi,
    /// Sinhala; Sinhalese
    /// - alpha2: si
    /// - alpha3-b: sin
    Sinhala,
    /// Sino-Tibetan languages
    /// - alpha3-b: sit
    SinoTibetanLanguages,
    /// Siouan languages
    /// - alpha3-b: sio
    SiouanLanguages,
    /// Skolt Sami
    /// - alpha3-b: sms
    SkoltSami,
    /// Slave (Athapascan)
    /// - alpha3-b: den
    SlaveAthapascan,
    /// Slavic languages
    /// - alpha3-b: sla
    SlavicLanguages,
    /// Slovak
    /// - alpha2: sk
    /// - alpha3-b: slo
    /// - alpha3-t: slk
    Slovak,
    /// Slovenian
    /// - alpha2: sl
    /// - alpha3-b: slv
    Slovenian,
    /// Sogdian
    /// - alpha3-b: sog
    Sogdian,
    /// Somali
    /// - alpha2: so
    /// - alpha3-b: som
    Somali,
    /// Songhai languages
    /// - alpha3-b: son
    SonghaiLanguages,
    /// Soninke
    /// - alpha3-b: snk
    Soninke,
    /// Sorbian languages
    /// - alpha3-b: wen
    SorbianLanguages,
    /// Sotho, Southern
    /// - alpha2: st
    /// - alpha3-b: sot
    SothoSouthern,
    /// South American Indian languages
    /// - alpha3-b: sai
    SouthAmericanIndianLanguages,
    /// Southern Altai
    /// - alpha3-b: alt
    SouthernAltai,
    /// Southern Sami
    /// - alpha3-b: sma
    SouthernSami,
    /// Spanish; Castilian
    /// - alpha2: es
    /// - alpha3-b: spa
    Spanish,
    /// Sranan Tongo
    /// - alpha3-b: srn
    SrananTongo,
    /// Standard Moroccan Tamazight
    /// - alpha3-b: zgh
    StandardMoroccanTamazight,
    /// Sukuma
    /// - alpha3-b: suk
    Sukuma,
    /// Sumerian
    /// - alpha3-b: sux
    Sumerian,
    /// Sundanese
    /// - alpha2: su
    /// - alpha3-b: sun
    Sundanese,
    /// Susu
    /// - alpha3-b: sus
    Susu,
    /// Swahili
    /// - alpha2: sw
    /// - alpha3-b: swa
    Swahili,
    /// Swati
    /// - alpha2: ss
    /// - alpha3-b: ssw
    Swati,
    /// Swedish
    /// - alpha2: sv
    /// - alpha3-b: swe
    Swedish,
    /// Swiss German; Alemannic; Alsatian
    /// - alpha3-b: gsw
    SwissGerman,
    /// Syriac
    /// - alpha3-b: syr
    Syriac,
    /// Tagalog
    /// - alpha2: tl
    /// - alpha3-b: tgl
    Tagalog,
    /// Tahitian
    /// - alpha2: ty
    /// - alpha3-b: tah
    Tahitian,
    /// Tai languages
    /// - alpha3-b: tai
    TaiLanguages,
    /// Tajik
    /// - alpha2: tg
    /// - alpha3-b: tgk
    Tajik,
    /// Tamashek
    /// - alpha3-b: tmh
    Tamashek,
    /// Tamil
    /// - alpha2: ta
    /// - alpha3-b: tam
    Tamil,
    /// Tatar
    /// - alpha2: tt
    /// - alpha3-b: tat
    Tatar,
    /// Telugu
    /// - alpha2: te
    /// - alpha3-b: tel
    Telugu,
    /// Tereno
    /// - alpha3-b: ter
    Tereno,
    /// Tetum
    /// - alpha3-b: tet
    Tetum,
    /// Thai
    /// - alpha2: th
    /// - alpha3-b: tha
    Thai,
    /// Tibetan
    /// - alpha2: bo
    /// - alpha3-b: tib
    /// - alpha3-t: bod
    Tibetan,
    /// Tigre
    /// - alpha3-b: tig
    Tigre,
    /// Tigrinya
    /// - alpha2: ti
    /// - alpha3-b: tir
    Tigrinya,
    /// Timne
    /// - alpha3-b: tem
    Timne,
    /// Tiv
    /// - alpha3-b: tiv
    Tiv,
    /// Tlingit
    /// - alpha3-b: tli
    Tlingit,
    /// Tok Pisin
    /// - alpha3-b: tpi
    TokPisin,
    /// Tokelau
    /// - alpha3-b: tkl
    Tokelau,
    /// Tonga (Nyasa)
    /// - alpha3-b: tog
    TongaNyasa,
    /// Tonga (Tonga Islands)
    /// - alpha2: to
    /// - alpha3-b: ton
    TongaTongaIslands,
    /// Tsimshian
    /// - alpha3-b: tsi
    Tsimshian,
    /// Tsonga
    /// - alpha2: ts
    /// - alpha3-b: tso
    Tsonga,
    /// Tswana
    /// - alpha2: tn
    /// - alpha3-b: tsn
    Tswana,
    /// Tumbuka
    /// - alpha3-b: tum
    Tumbuka,
    /// Tupi languages
    /// - alpha3-b: tup
    TupiLanguages,
    /// Turkish
    /// - alpha2: tr
    /// - alpha3-b: tur
    Turkish,
    /// Turkish, Ottoman (1500-1928)
    /// - alpha3-b: ota
    TurkishOttoman,
    /// Turkmen
    /// - alpha2: tk
    /// - alpha3-b: tuk
    Turkmen,
    /// Tuvalu
    /// - alpha3-b: tvl
    Tuvalu,
    /// Tuvinian
    /// - alpha3-b: tyv
    Tuvinian,
    /// Twi
    /// - alpha2: tw
    /// - alpha3-b: twi
    Twi,
    /// Udmurt
    /// - alpha3-b: udm
    Udmurt,
    /// Ugaritic
    /// - alpha3-b: uga
    Ugaritic,
    /// Uighur; Uyghur
    /// - alpha2: ug
    /// - alpha3-b: uig
    Uighur,
    /// Ukrainian
    /// - alpha2: uk
    /// - alpha3-b: ukr
    Ukrainian,
    /// Umbundu
    /// - alpha3-b: umb
    Umbundu,
    /// Uncoded languages
    /// - alpha3-b: mis
    UncodedLanguages,
    /// Undetermined
    /// - alpha3-b: und
    Undetermined,
    /// Upper Sorbian
    /// - alpha3-b: hsb
    UpperSorbian,
    /// Urdu
    /// - alpha2: ur
    /// - alpha3-b: urd
    Urdu,
    /// Uzbek
    /// - alpha2: uz
    /// - alpha3-b: uzb
    Uzbek,
    /// Vai
    /// - alpha3-b: vai
    Vai,
    /// Venda
    /// - alpha2: ve
    /// - alpha3-b: ven
    Venda,
    /// Vietnamese
    /// - alpha2: vi
    /// - alpha3-b: vie
    Vietnamese,
    /// Volapük
    /// - alpha2: vo
    /// - alpha3-b: vol
    Volapuk,
    /// Votic
    /// - alpha3-b: vot
    Votic,
    /// Wakashan languages
    /// - alpha3-b: wak
    WakashanLanguages,
    /// Walloon
    /// - alpha2: wa
    /// - alpha3-b: wln
    Walloon,
    /// Waray
    /// - alpha3-b: war
    Waray,
    /// Washo
    /// - alpha3-b: was
    Washo,
    /// Welsh
    /// - alpha2: cy
    /// - alpha3-b: wel
    /// - alpha3-t: cym
    Welsh,
    /// Western Frisian
    /// - alpha2: fy
    /// - alpha3-b: fry
    WesternFrisian,
    /// Wolaitta; Wolaytta
    /// - alpha3-b: wal
    Wolaitta,
    /// Wolof
    /// - alpha2: wo
    /// - alpha3-b: wol
    Wolof,
    /// Xhosa
    /// - alpha2: xh
    /// - alpha3-b: xho
    Xhosa,
    /// Yakut
    /// - alpha3-b: sah
    Yakut,
    /// Yao
    /// - alpha3-b: yao
    Yao,
    /// Yapese
    /// - alpha3-b: yap
    Yapese,
    /// Yiddish
    /// - alpha2: yi
    /// - alpha3-b: yid
    Yiddish,
    /// Yoruba
    /// - alpha2: yo
    /// - alpha3-b: yor
    Yoruba,
    /// Yupik languages
    /// - alpha3-b: ypk
    YupikLanguages,
    /// Zande languages
    /// - alpha3-b: znd
    ZandeLanguages,
    /// Zapotec
    /// - alpha3-b: zap
    Zapotec,
    /// Zaza; Dimili; Dimli; Kirdki; Kirmanjki; Zazaki
    /// - alpha3-b: zza
    Zaza,
    /// Zenaga
    /// - alpha3-b: zen
    Zenaga,
    /// Zhuang; Chuang
    /// - alpha2: za
    /// - alpha3-b: zha
    Zhuang,
    /// Zulu
    /// - alpha2: zu
    /// - alpha3-b: zul
    Zulu,
    /// Zuni
    /// - alpha3-b: zun
    Zuni,
}

impl Language {
/// Attempts to match an ISO-639 code
/// ```
/// use iso_639::Language;
///
/// let french = Language::from_str("fr").unwrap();
/// let french = Language::from_str("fra").unwrap();
/// let french = Language::from_str("fre").unwrap();
/// ```

    pub fn from_str(s: &str) -> Option<Language> {
        match s {
            "ab" | "abk" => Some(Language::Abkhazian),
            "ace" => Some(Language::Achinese),
            "ach" => Some(Language::Acoli),
            "ada" => Some(Language::Adangme),
            "ady" => Some(Language::Adyghe),
            "aa" | "aar" => Some(Language::Afar),
            "afh" => Some(Language::Afrihili),
            "af" | "afr" => Some(Language::Afrikaans),
            "afa" => Some(Language::AfroAsiaticLanguages),
            "ain" => Some(Language::Ainu),
            "ak" | "aka" => Some(Language::Akan),
            "akk" => Some(Language::Akkadian),
            "sq" | "alb" | "sqi" => Some(Language::Albanian),
            "ale" => Some(Language::Aleut),
            "alg" => Some(Language::AlgonquianLanguages),
            "tut" => Some(Language::AltaicLanguages),
            "am" | "amh" => Some(Language::Amharic),
            "anp" => Some(Language::Angika),
            "apa" => Some(Language::ApacheLanguages),
            "ar" | "ara" => Some(Language::Arabic),
            "an" | "arg" => Some(Language::Aragonese),
            "arp" => Some(Language::Arapaho),
            "arw" => Some(Language::Arawak),
            "hy" | "arm" | "hye" => Some(Language::Armenian),
            "rup" => Some(Language::Aromanian),
            "art" => Some(Language::ArtificialLanguages),
            "as" | "asm" => Some(Language::Assamese),
            "ast" => Some(Language::Asturian),
            "ath" => Some(Language::AthapascanLanguages),
            "aus" => Some(Language::AustralianLanguages),
            "map" => Some(Language::AustronesianLanguages),
            "av" | "ava" => Some(Language::Avaric),
            "ae" | "ave" => Some(Language::Avestan),
            "awa" => Some(Language::Awadhi),
            "ay" | "aym" => Some(Language::Aymara),
            "az" | "aze" => Some(Language::Azerbaijani),
            "ban" => Some(Language::Balinese),
            "bat" => Some(Language::BalticLanguages),
            "bal" => Some(Language::Baluchi),
            "bm" | "bam" => Some(Language::Bambara),
            "bai" => Some(Language::BamilekeLanguages),
            "bad" => Some(Language::BandaLanguages),
            "bnt" => Some(Language::BantuLanguages),
            "bas" => Some(Language::Basa),
            "ba" | "bak" => Some(Language::Bashkir),
            "eu" | "baq" | "eus" => Some(Language::Basque),
            "btk" => Some(Language::BatakLanguages),
            "bej" => Some(Language::Beja),
            "be" | "bel" => Some(Language::Belarusian),
            "bem" => Some(Language::Bemba),
            "bn" | "ben" => Some(Language::Bengali),
            "ber" => Some(Language::BerberLanguages),
            "bho" => Some(Language::Bhojpuri),
            "bh" | "bih" => Some(Language::BihariLanguages),
            "bik" => Some(Language::Bikol),
            "bin" => Some(Language::Bini),
            "bi" | "bis" => Some(Language::Bislama),
            "byn" => Some(Language::Blin),
            "zbl" => Some(Language::Blissymbols),
            "nb" | "nob" => Some(Language::BokmålNorwegian),
            "bs" | "bos" => Some(Language::Bosnian),
            "bra" => Some(Language::Braj),
            "br" | "bre" => Some(Language::Breton),
            "bug" => Some(Language::Buginese),
            "bg" | "bul" => Some(Language::Bulgarian),
            "bua" => Some(Language::Buriat),
            "my" | "bur" | "mya" => Some(Language::Burmese),
            "cad" => Some(Language::Caddo),
            "ca" | "cat" => Some(Language::Catalan),
            "cau" => Some(Language::CaucasianLanguages),
            "ceb" => Some(Language::Cebuano),
            "cel" => Some(Language::CelticLanguages),
            "cai" => Some(Language::CentralAmericanIndianLanguages),
            "km" | "khm" => Some(Language::CentralKhmer),
            "chg" => Some(Language::Chagatai),
            "cmc" => Some(Language::ChamicLanguages),
            "ch" | "cha" => Some(Language::Chamorro),
            "ce" | "che" => Some(Language::Chechen),
            "chr" => Some(Language::Cherokee),
            "chy" => Some(Language::Cheyenne),
            "chb" => Some(Language::Chibcha),
            "ny" | "nya" => Some(Language::Chichewa),
            "zh" | "chi" | "zho" => Some(Language::Chinese),
            "chn" => Some(Language::Chinook),
            "chp" => Some(Language::Chipewyan),
            "cho" => Some(Language::Choctaw),
            "cu" | "chu" => Some(Language::ChurchSlavic),
            "chk" => Some(Language::Chuukese),
            "cv" | "chv" => Some(Language::Chuvash),
            "nwc" => Some(Language::ClassicalNewari),
            "syc" => Some(Language::ClassicalSyriac),
            "cop" => Some(Language::Coptic),
            "kw" | "cor" => Some(Language::Cornish),
            "co" | "cos" => Some(Language::Corsican),
            "cr" | "cre" => Some(Language::Cree),
            "mus" => Some(Language::Creek),
            "crp" => Some(Language::CreolesAndPidgins),
            "cpe" => Some(Language::CreolesAndPidginsEnglishBased),
            "cpf" => Some(Language::CreolesAndPidginsFrenchBased),
            "cpp" => Some(Language::CreolesAndPidginsPortugueseBased),
            "crh" => Some(Language::CrimeanTatar),
            "hr" | "hrv" => Some(Language::Croatian),
            "cus" => Some(Language::CushiticLanguages),
            "cs" | "cze" | "ces" => Some(Language::Czech),
            "dak" => Some(Language::Dakota),
            "da" | "dan" => Some(Language::Danish),
            "dar" => Some(Language::Dargwa),
            "del" => Some(Language::Delaware),
            "din" => Some(Language::Dinka),
            "dv" | "div" => Some(Language::Divehi),
            "doi" => Some(Language::Dogri),
            "dgr" => Some(Language::Dogrib),
            "dra" => Some(Language::DravidianLanguages),
            "dua" => Some(Language::Duala),
            "nl" | "dut" | "nld" => Some(Language::Dutch),
            "dum" => Some(Language::DutchMiddle),
            "dyu" => Some(Language::Dyula),
            "dz" | "dzo" => Some(Language::Dzongkha),
            "frs" => Some(Language::EasternFrisian),
            "efi" => Some(Language::Efik),
            "egy" => Some(Language::EgyptianAncient),
            "eka" => Some(Language::Ekajuk),
            "elx" => Some(Language::Elamite),
            "en" | "eng" => Some(Language::English),
            "enm" => Some(Language::EnglishMiddle),
            "ang" => Some(Language::EnglishOld),
            "myv" => Some(Language::Erzya),
            "eo" | "epo" => Some(Language::Esperanto),
            "et" | "est" => Some(Language::Estonian),
            "ee" | "ewe" => Some(Language::Ewe),
            "ewo" => Some(Language::Ewondo),
            "fan" => Some(Language::Fang),
            "fat" => Some(Language::Fanti),
            "fo" | "fao" => Some(Language::Faroese),
            "fj" | "fij" => Some(Language::Fijian),
            "fil" => Some(Language::Filipino),
            "fi" | "fin" => Some(Language::Finnish),
            "fiu" => Some(Language::FinnoUgrianLanguages),
            "fon" => Some(Language::Fon),
            "fr" | "fre" | "fra" => Some(Language::French),
            "frm" => Some(Language::FrenchMiddle),
            "fro" => Some(Language::FrenchOld),
            "fur" => Some(Language::Friulian),
            "ff" | "ful" => Some(Language::Fulah),
            "gaa" => Some(Language::Ga),
            "gd" | "gla" => Some(Language::Gaelic),
            "car" => Some(Language::GalibiCarib),
            "gl" | "glg" => Some(Language::Galician),
            "lg" | "lug" => Some(Language::Ganda),
            "gay" => Some(Language::Gayo),
            "gba" => Some(Language::Gbaya),
            "gez" => Some(Language::Geez),
            "ka" | "geo" | "kat" => Some(Language::Georgian),
            "de" | "ger" | "deu" => Some(Language::German),
            "gmh" => Some(Language::GermanMiddleHigh),
            "goh" => Some(Language::GermanOldHigh),
            "gem" => Some(Language::GermanicLanguages),
            "gil" => Some(Language::Gilbertese),
            "gon" => Some(Language::Gondi),
            "gor" => Some(Language::Gorontalo),
            "got" => Some(Language::Gothic),
            "grb" => Some(Language::Grebo),
            "grc" => Some(Language::GreekAncient),
            "el" | "gre" | "ell" => Some(Language::GreekModern),
            "gn" | "grn" => Some(Language::Guarani),
            "gu" | "guj" => Some(Language::Gujarati),
            "gwi" => Some(Language::GwichIn),
            "hai" => Some(Language::Haida),
            "ht" | "hat" => Some(Language::Haitian),
            "ha" | "hau" => Some(Language::Hausa),
            "haw" => Some(Language::Hawaiian),
            "he" | "heb" => Some(Language::Hebrew),
            "hz" | "her" => Some(Language::Herero),
            "hil" => Some(Language::Hiligaynon),
            "him" => Some(Language::HimachaliLanguages),
            "hi" | "hin" => Some(Language::Hindi),
            "ho" | "hmo" => Some(Language::HiriMotu),
            "hit" => Some(Language::Hittite),
            "hmn" => Some(Language::Hmong),
            "hu" | "hun" => Some(Language::Hungarian),
            "hup" => Some(Language::Hupa),
            "iba" => Some(Language::Iban),
            "is" | "ice" | "isl" => Some(Language::Icelandic),
            "io" | "ido" => Some(Language::Ido),
            "ig" | "ibo" => Some(Language::Igbo),
            "ijo" => Some(Language::IjoLanguages),
            "ilo" => Some(Language::Iloko),
            "smn" => Some(Language::InariSami),
            "inc" => Some(Language::IndicLanguages),
            "ine" => Some(Language::IndoEuropeanLanguages),
            "id" | "ind" => Some(Language::Indonesian),
            "inh" => Some(Language::Ingush),
            "ia" | "ina" => Some(Language::Interlingua),
            "ie" | "ile" => Some(Language::Interlingue),
            "iu" | "iku" => Some(Language::Inuktitut),
            "ik" | "ipk" => Some(Language::Inupiaq),
            "ira" => Some(Language::IranianLanguages),
            "ga" | "gle" => Some(Language::Irish),
            "mga" => Some(Language::IrishMiddle),
            "sga" => Some(Language::IrishOld),
            "iro" => Some(Language::IroquoianLanguages),
            "it" | "ita" => Some(Language::Italian),
            "ja" | "jpn" => Some(Language::Japanese),
            "jv" | "jav" => Some(Language::Javanese),
            "jrb" => Some(Language::JudeoArabic),
            "jpr" => Some(Language::JudeoPersian),
            "kbd" => Some(Language::Kabardian),
            "kab" => Some(Language::Kabyle),
            "kac" => Some(Language::Kachin),
            "kl" | "kal" => Some(Language::Kalaallisut),
            "xal" => Some(Language::Kalmyk),
            "kam" => Some(Language::Kamba),
            "kn" | "kan" => Some(Language::Kannada),
            "kr" | "kau" => Some(Language::Kanuri),
            "kaa" => Some(Language::KaraKalpak),
            "krc" => Some(Language::KarachayBalkar),
            "krl" => Some(Language::Karelian),
            "kar" => Some(Language::KarenLanguages),
            "ks" | "kas" => Some(Language::Kashmiri),
            "csb" => Some(Language::Kashubian),
            "kaw" => Some(Language::Kawi),
            "kk" | "kaz" => Some(Language::Kazakh),
            "kha" => Some(Language::Khasi),
            "khi" => Some(Language::KhoisanLanguages),
            "kho" => Some(Language::Khotanese),
            "ki" | "kik" => Some(Language::Kikuyu),
            "kmb" => Some(Language::Kimbundu),
            "rw" | "kin" => Some(Language::Kinyarwanda),
            "ky" | "kir" => Some(Language::Kirghiz),
            "tlh" => Some(Language::Klingon),
            "kv" | "kom" => Some(Language::Komi),
            "kg" | "kon" => Some(Language::Kongo),
            "kok" => Some(Language::Konkani),
            "ko" | "kor" => Some(Language::Korean),
            "kos" => Some(Language::Kosraean),
            "kpe" => Some(Language::Kpelle),
            "kro" => Some(Language::KruLanguages),
            "kj" | "kua" => Some(Language::Kuanyama),
            "kum" => Some(Language::Kumyk),
            "ku" | "kur" => Some(Language::Kurdish),
            "kru" => Some(Language::Kurukh),
            "kut" => Some(Language::Kutenai),
            "lad" => Some(Language::Ladino),
            "lah" => Some(Language::Lahnda),
            "lam" => Some(Language::Lamba),
            "day" => Some(Language::LandDayakLanguages),
            "lo" | "lao" => Some(Language::Lao),
            "la" | "lat" => Some(Language::Latin),
            "lv" | "lav" => Some(Language::Latvian),
            "lez" => Some(Language::Lezghian),
            "li" | "lim" => Some(Language::Limburgan),
            "ln" | "lin" => Some(Language::Lingala),
            "lt" | "lit" => Some(Language::Lithuanian),
            "jbo" => Some(Language::Lojban),
            "nds" => Some(Language::LowGerman),
            "dsb" => Some(Language::LowerSorbian),
            "loz" => Some(Language::Lozi),
            "lu" | "lub" => Some(Language::LubaKatanga),
            "lua" => Some(Language::LubaLulua),
            "lui" => Some(Language::Luiseno),
            "smj" => Some(Language::LuleSami),
            "lun" => Some(Language::Lunda),
            "luo" => Some(Language::LuoKenyaAndTanzania),
            "lus" => Some(Language::Lushai),
            "lb" | "ltz" => Some(Language::Luxembourgish),
            "mk" | "mac" | "mkd" => Some(Language::Macedonian),
            "mad" => Some(Language::Madurese),
            "mag" => Some(Language::Magahi),
            "mai" => Some(Language::Maithili),
            "mak" => Some(Language::Makasar),
            "mg" | "mlg" => Some(Language::Malagasy),
            "ms" | "may" | "msa" => Some(Language::Malay),
            "ml" | "mal" => Some(Language::Malayalam),
            "mt" | "mlt" => Some(Language::Maltese),
            "mnc" => Some(Language::Manchu),
            "mdr" => Some(Language::Mandar),
            "man" => Some(Language::Mandingo),
            "mni" => Some(Language::Manipuri),
            "mno" => Some(Language::ManoboLanguages),
            "gv" | "glv" => Some(Language::Manx),
            "mi" | "mao" | "mri" => Some(Language::Maori),
            "arn" => Some(Language::Mapudungun),
            "mr" | "mar" => Some(Language::Marathi),
            "chm" => Some(Language::Mari),
            "mh" | "mah" => Some(Language::Marshallese),
            "mwr" => Some(Language::Marwari),
            "mas" => Some(Language::Masai),
            "myn" => Some(Language::MayanLanguages),
            "men" => Some(Language::Mende),
            "mic" => Some(Language::MiKmaq),
            "min" => Some(Language::Minangkabau),
            "mwl" => Some(Language::Mirandese),
            "moh" => Some(Language::Mohawk),
            "mdf" => Some(Language::Moksha),
            "mkh" => Some(Language::MonKhmerLanguages),
            "lol" => Some(Language::Mongo),
            "mn" | "mon" => Some(Language::Mongolian),
            "cnr" => Some(Language::Montenegrin),
            "mos" => Some(Language::Mossi),
            "mul" => Some(Language::MultipleLanguages),
            "mun" => Some(Language::MundaLanguages),
            "nqo" => Some(Language::NKo),
            "nah" => Some(Language::NahuatlLanguages),
            "na" | "nau" => Some(Language::Nauru),
            "nv" | "nav" => Some(Language::Navajo),
            "nd" | "nde" => Some(Language::NdebeleNorth),
            "nr" | "nbl" => Some(Language::NdebeleSouth),
            "ng" | "ndo" => Some(Language::Ndonga),
            "nap" => Some(Language::Neapolitan),
            "new" => Some(Language::NepalBhasa),
            "ne" | "nep" => Some(Language::Nepali),
            "nia" => Some(Language::Nias),
            "nic" => Some(Language::NigerKordofanianLanguages),
            "ssa" => Some(Language::NiloSaharanLanguages),
            "niu" => Some(Language::Niuean),
            "zxx" => Some(Language::NoLinguisticContent),
            "nog" => Some(Language::Nogai),
            "non" => Some(Language::NorseOld),
            "nai" => Some(Language::NorthAmericanIndianLanguages),
            "frr" => Some(Language::NorthernFrisian),
            "se" | "sme" => Some(Language::NorthernSami),
            "no" | "nor" => Some(Language::Norwegian),
            "nn" | "nno" => Some(Language::NorwegianNynorsk),
            "nub" => Some(Language::NubianLanguages),
            "nym" => Some(Language::Nyamwezi),
            "nyn" => Some(Language::Nyankole),
            "nyo" => Some(Language::Nyoro),
            "nzi" => Some(Language::Nzima),
            "oc" | "oci" => Some(Language::Occitan),
            "arc" => Some(Language::OfficialAramaic),
            "oj" | "oji" => Some(Language::Ojibwa),
            "or" | "ori" => Some(Language::Oriya),
            "om" | "orm" => Some(Language::Oromo),
            "osa" => Some(Language::Osage),
            "os" | "oss" => Some(Language::Ossetian),
            "oto" => Some(Language::OtomianLanguages),
            "pal" => Some(Language::Pahlavi),
            "pau" => Some(Language::Palauan),
            "pi" | "pli" => Some(Language::Pali),
            "pam" => Some(Language::Pampanga),
            "pag" => Some(Language::Pangasinan),
            "pa" | "pan" => Some(Language::Panjabi),
            "pap" => Some(Language::Papiamento),
            "paa" => Some(Language::PapuanLanguages),
            "nso" => Some(Language::Pedi),
            "fa" | "per" | "fas" => Some(Language::Persian),
            "peo" => Some(Language::PersianOld),
            "phi" => Some(Language::PhilippineLanguages),
            "phn" => Some(Language::Phoenician),
            "pon" => Some(Language::Pohnpeian),
            "pl" | "pol" => Some(Language::Polish),
            "pt" | "por" => Some(Language::Portuguese),
            "pra" => Some(Language::PrakritLanguages),
            "pro" => Some(Language::ProvencalOld),
            "ps" | "pus" => Some(Language::Pushto),
            "qu" | "que" => Some(Language::Quechua),
            "raj" => Some(Language::Rajasthani),
            "rap" => Some(Language::Rapanui),
            "rar" => Some(Language::Rarotongan),
            "roa" => Some(Language::RomanceLanguages),
            "ro" | "rum" | "ron" => Some(Language::Romanian),
            "rm" | "roh" => Some(Language::Romansh),
            "rom" => Some(Language::Romany),
            "rn" | "run" => Some(Language::Rundi),
            "ru" | "rus" => Some(Language::Russian),
            "sal" => Some(Language::SalishanLanguages),
            "sam" => Some(Language::SamaritanAramaic),
            "smi" => Some(Language::SamiLanguages),
            "sm" | "smo" => Some(Language::Samoan),
            "sad" => Some(Language::Sandawe),
            "sg" | "sag" => Some(Language::Sango),
            "sa" | "san" => Some(Language::Sanskrit),
            "sat" => Some(Language::Santali),
            "sc" | "srd" => Some(Language::Sardinian),
            "sas" => Some(Language::Sasak),
            "sco" => Some(Language::Scots),
            "sel" => Some(Language::Selkup),
            "sem" => Some(Language::SemiticLanguages),
            "sr" | "srp" => Some(Language::Serbian),
            "srr" => Some(Language::Serer),
            "shn" => Some(Language::Shan),
            "sn" | "sna" => Some(Language::Shona),
            "ii" | "iii" => Some(Language::SichuanYi),
            "scn" => Some(Language::Sicilian),
            "sid" => Some(Language::Sidamo),
            "sgn" => Some(Language::SignLanguages),
            "bla" => Some(Language::Siksika),
            "sd" | "snd" => Some(Language::Sindhi),
            "si" | "sin" => Some(Language::Sinhala),
            "sit" => Some(Language::SinoTibetanLanguages),
            "sio" => Some(Language::SiouanLanguages),
            "sms" => Some(Language::SkoltSami),
            "den" => Some(Language::SlaveAthapascan),
            "sla" => Some(Language::SlavicLanguages),
            "sk" | "slo" | "slk" => Some(Language::Slovak),
            "sl" | "slv" => Some(Language::Slovenian),
            "sog" => Some(Language::Sogdian),
            "so" | "som" => Some(Language::Somali),
            "son" => Some(Language::SonghaiLanguages),
            "snk" => Some(Language::Soninke),
            "wen" => Some(Language::SorbianLanguages),
            "st" | "sot" => Some(Language::SothoSouthern),
            "sai" => Some(Language::SouthAmericanIndianLanguages),
            "alt" => Some(Language::SouthernAltai),
            "sma" => Some(Language::SouthernSami),
            "es" | "spa" => Some(Language::Spanish),
            "srn" => Some(Language::SrananTongo),
            "zgh" => Some(Language::StandardMoroccanTamazight),
            "suk" => Some(Language::Sukuma),
            "sux" => Some(Language::Sumerian),
            "su" | "sun" => Some(Language::Sundanese),
            "sus" => Some(Language::Susu),
            "sw" | "swa" => Some(Language::Swahili),
            "ss" | "ssw" => Some(Language::Swati),
            "sv" | "swe" => Some(Language::Swedish),
            "gsw" => Some(Language::SwissGerman),
            "syr" => Some(Language::Syriac),
            "tl" | "tgl" => Some(Language::Tagalog),
            "ty" | "tah" => Some(Language::Tahitian),
            "tai" => Some(Language::TaiLanguages),
            "tg" | "tgk" => Some(Language::Tajik),
            "tmh" => Some(Language::Tamashek),
            "ta" | "tam" => Some(Language::Tamil),
            "tt" | "tat" => Some(Language::Tatar),
            "te" | "tel" => Some(Language::Telugu),
            "ter" => Some(Language::Tereno),
            "tet" => Some(Language::Tetum),
            "th" | "tha" => Some(Language::Thai),
            "bo" | "tib" | "bod" => Some(Language::Tibetan),
            "tig" => Some(Language::Tigre),
            "ti" | "tir" => Some(Language::Tigrinya),
            "tem" => Some(Language::Timne),
            "tiv" => Some(Language::Tiv),
            "tli" => Some(Language::Tlingit),
            "tpi" => Some(Language::TokPisin),
            "tkl" => Some(Language::Tokelau),
            "tog" => Some(Language::TongaNyasa),
            "to" | "ton" => Some(Language::TongaTongaIslands),
            "tsi" => Some(Language::Tsimshian),
            "ts" | "tso" => Some(Language::Tsonga),
            "tn" | "tsn" => Some(Language::Tswana),
            "tum" => Some(Language::Tumbuka),
            "tup" => Some(Language::TupiLanguages),
            "tr" | "tur" => Some(Language::Turkish),
            "ota" => Some(Language::TurkishOttoman),
            "tk" | "tuk" => Some(Language::Turkmen),
            "tvl" => Some(Language::Tuvalu),
            "tyv" => Some(Language::Tuvinian),
            "tw" | "twi" => Some(Language::Twi),
            "udm" => Some(Language::Udmurt),
            "uga" => Some(Language::Ugaritic),
            "ug" | "uig" => Some(Language::Uighur),
            "uk" | "ukr" => Some(Language::Ukrainian),
            "umb" => Some(Language::Umbundu),
            "mis" => Some(Language::UncodedLanguages),
            "und" => Some(Language::Undetermined),
            "hsb" => Some(Language::UpperSorbian),
            "ur" | "urd" => Some(Language::Urdu),
            "uz" | "uzb" => Some(Language::Uzbek),
            "vai" => Some(Language::Vai),
            "ve" | "ven" => Some(Language::Venda),
            "vi" | "vie" => Some(Language::Vietnamese),
            "vo" | "vol" => Some(Language::Volapuk),
            "vot" => Some(Language::Votic),
            "wak" => Some(Language::WakashanLanguages),
            "wa" | "wln" => Some(Language::Walloon),
            "war" => Some(Language::Waray),
            "was" => Some(Language::Washo),
            "cy" | "wel" | "cym" => Some(Language::Welsh),
            "fy" | "fry" => Some(Language::WesternFrisian),
            "wal" => Some(Language::Wolaitta),
            "wo" | "wol" => Some(Language::Wolof),
            "xh" | "xho" => Some(Language::Xhosa),
            "sah" => Some(Language::Yakut),
            "yao" => Some(Language::Yao),
            "yap" => Some(Language::Yapese),
            "yi" | "yid" => Some(Language::Yiddish),
            "yo" | "yor" => Some(Language::Yoruba),
            "ypk" => Some(Language::YupikLanguages),
            "znd" => Some(Language::ZandeLanguages),
            "zap" => Some(Language::Zapotec),
            "zza" => Some(Language::Zaza),
            "zen" => Some(Language::Zenaga),
            "za" | "zha" => Some(Language::Zhuang),
            "zu" | "zul" => Some(Language::Zulu),
            "zun" => Some(Language::Zuni),
            _ => None,
        }
    }

/// Attempts to match an ISO-639 alpha2 code
/// ```
/// use iso_639::Language;
///
/// let french = Language::from_alpha2_str("fr").unwrap();
/// ```

    pub fn from_alpha2_str(s: &str) -> Option<Language> {
        match s {
            "ab" => Some(Language::Abkhazian),
            "aa" => Some(Language::Afar),
            "af" => Some(Language::Afrikaans),
            "ak" => Some(Language::Akan),
            "sq" => Some(Language::Albanian),
            "am" => Some(Language::Amharic),
            "ar" => Some(Language::Arabic),
            "an" => Some(Language::Aragonese),
            "hy" => Some(Language::Armenian),
            "as" => Some(Language::Assamese),
            "av" => Some(Language::Avaric),
            "ae" => Some(Language::Avestan),
            "ay" => Some(Language::Aymara),
            "az" => Some(Language::Azerbaijani),
            "bm" => Some(Language::Bambara),
            "ba" => Some(Language::Bashkir),
            "eu" => Some(Language::Basque),
            "be" => Some(Language::Belarusian),
            "bn" => Some(Language::Bengali),
            "bh" => Some(Language::BihariLanguages),
            "bi" => Some(Language::Bislama),
            "nb" => Some(Language::BokmålNorwegian),
            "bs" => Some(Language::Bosnian),
            "br" => Some(Language::Breton),
            "bg" => Some(Language::Bulgarian),
            "my" => Some(Language::Burmese),
            "ca" => Some(Language::Catalan),
            "km" => Some(Language::CentralKhmer),
            "ch" => Some(Language::Chamorro),
            "ce" => Some(Language::Chechen),
            "ny" => Some(Language::Chichewa),
            "zh" => Some(Language::Chinese),
            "cu" => Some(Language::ChurchSlavic),
            "cv" => Some(Language::Chuvash),
            "kw" => Some(Language::Cornish),
            "co" => Some(Language::Corsican),
            "cr" => Some(Language::Cree),
            "hr" => Some(Language::Croatian),
            "cs" => Some(Language::Czech),
            "da" => Some(Language::Danish),
            "dv" => Some(Language::Divehi),
            "nl" => Some(Language::Dutch),
            "dz" => Some(Language::Dzongkha),
            "en" => Some(Language::English),
            "eo" => Some(Language::Esperanto),
            "et" => Some(Language::Estonian),
            "ee" => Some(Language::Ewe),
            "fo" => Some(Language::Faroese),
            "fj" => Some(Language::Fijian),
            "fi" => Some(Language::Finnish),
            "fr" => Some(Language::French),
            "ff" => Some(Language::Fulah),
            "gd" => Some(Language::Gaelic),
            "gl" => Some(Language::Galician),
            "lg" => Some(Language::Ganda),
            "ka" => Some(Language::Georgian),
            "de" => Some(Language::German),
            "el" => Some(Language::GreekModern),
            "gn" => Some(Language::Guarani),
            "gu" => Some(Language::Gujarati),
            "ht" => Some(Language::Haitian),
            "ha" => Some(Language::Hausa),
            "he" => Some(Language::Hebrew),
            "hz" => Some(Language::Herero),
            "hi" => Some(Language::Hindi),
            "ho" => Some(Language::HiriMotu),
            "hu" => Some(Language::Hungarian),
            "is" => Some(Language::Icelandic),
            "io" => Some(Language::Ido),
            "ig" => Some(Language::Igbo),
            "id" => Some(Language::Indonesian),
            "ia" => Some(Language::Interlingua),
            "ie" => Some(Language::Interlingue),
            "iu" => Some(Language::Inuktitut),
            "ik" => Some(Language::Inupiaq),
            "ga" => Some(Language::Irish),
            "it" => Some(Language::Italian),
            "ja" => Some(Language::Japanese),
            "jv" => Some(Language::Javanese),
            "kl" => Some(Language::Kalaallisut),
            "kn" => Some(Language::Kannada),
            "kr" => Some(Language::Kanuri),
            "ks" => Some(Language::Kashmiri),
            "kk" => Some(Language::Kazakh),
            "ki" => Some(Language::Kikuyu),
            "rw" => Some(Language::Kinyarwanda),
            "ky" => Some(Language::Kirghiz),
            "kv" => Some(Language::Komi),
            "kg" => Some(Language::Kongo),
            "ko" => Some(Language::Korean),
            "kj" => Some(Language::Kuanyama),
            "ku" => Some(Language::Kurdish),
            "lo" => Some(Language::Lao),
            "la" => Some(Language::Latin),
            "lv" => Some(Language::Latvian),
            "li" => Some(Language::Limburgan),
            "ln" => Some(Language::Lingala),
            "lt" => Some(Language::Lithuanian),
            "lu" => Some(Language::LubaKatanga),
            "lb" => Some(Language::Luxembourgish),
            "mk" => Some(Language::Macedonian),
            "mg" => Some(Language::Malagasy),
            "ms" => Some(Language::Malay),
            "ml" => Some(Language::Malayalam),
            "mt" => Some(Language::Maltese),
            "gv" => Some(Language::Manx),
            "mi" => Some(Language::Maori),
            "mr" => Some(Language::Marathi),
            "mh" => Some(Language::Marshallese),
            "mn" => Some(Language::Mongolian),
            "na" => Some(Language::Nauru),
            "nv" => Some(Language::Navajo),
            "nd" => Some(Language::NdebeleNorth),
            "nr" => Some(Language::NdebeleSouth),
            "ng" => Some(Language::Ndonga),
            "ne" => Some(Language::Nepali),
            "se" => Some(Language::NorthernSami),
            "no" => Some(Language::Norwegian),
            "nn" => Some(Language::NorwegianNynorsk),
            "oc" => Some(Language::Occitan),
            "oj" => Some(Language::Ojibwa),
            "or" => Some(Language::Oriya),
            "om" => Some(Language::Oromo),
            "os" => Some(Language::Ossetian),
            "pi" => Some(Language::Pali),
            "pa" => Some(Language::Panjabi),
            "fa" => Some(Language::Persian),
            "pl" => Some(Language::Polish),
            "pt" => Some(Language::Portuguese),
            "ps" => Some(Language::Pushto),
            "qu" => Some(Language::Quechua),
            "ro" => Some(Language::Romanian),
            "rm" => Some(Language::Romansh),
            "rn" => Some(Language::Rundi),
            "ru" => Some(Language::Russian),
            "sm" => Some(Language::Samoan),
            "sg" => Some(Language::Sango),
            "sa" => Some(Language::Sanskrit),
            "sc" => Some(Language::Sardinian),
            "sr" => Some(Language::Serbian),
            "sn" => Some(Language::Shona),
            "ii" => Some(Language::SichuanYi),
            "sd" => Some(Language::Sindhi),
            "si" => Some(Language::Sinhala),
            "sk" => Some(Language::Slovak),
            "sl" => Some(Language::Slovenian),
            "so" => Some(Language::Somali),
            "st" => Some(Language::SothoSouthern),
            "es" => Some(Language::Spanish),
            "su" => Some(Language::Sundanese),
            "sw" => Some(Language::Swahili),
            "ss" => Some(Language::Swati),
            "sv" => Some(Language::Swedish),
            "tl" => Some(Language::Tagalog),
            "ty" => Some(Language::Tahitian),
            "tg" => Some(Language::Tajik),
            "ta" => Some(Language::Tamil),
            "tt" => Some(Language::Tatar),
            "te" => Some(Language::Telugu),
            "th" => Some(Language::Thai),
            "bo" => Some(Language::Tibetan),
            "ti" => Some(Language::Tigrinya),
            "to" => Some(Language::TongaTongaIslands),
            "ts" => Some(Language::Tsonga),
            "tn" => Some(Language::Tswana),
            "tr" => Some(Language::Turkish),
            "tk" => Some(Language::Turkmen),
            "tw" => Some(Language::Twi),
            "ug" => Some(Language::Uighur),
            "uk" => Some(Language::Ukrainian),
            "ur" => Some(Language::Urdu),
            "uz" => Some(Language::Uzbek),
            "ve" => Some(Language::Venda),
            "vi" => Some(Language::Vietnamese),
            "vo" => Some(Language::Volapuk),
            "wa" => Some(Language::Walloon),
            "cy" => Some(Language::Welsh),
            "fy" => Some(Language::WesternFrisian),
            "wo" => Some(Language::Wolof),
            "xh" => Some(Language::Xhosa),
            "yi" => Some(Language::Yiddish),
            "yo" => Some(Language::Yoruba),
            "za" => Some(Language::Zhuang),
            "zu" => Some(Language::Zulu),
            _ => None,
        }
    }

/// Attempts to match an ISO-639 alpha3-b code
/// ```
/// use iso_639::Language;
///
/// let french = Language::from_alpha3_b_str("fre").unwrap();
/// ```

    pub fn from_alpha3_b_str(s: &str) -> Option<Language> {
        match s {
            "abk" => Some(Language::Abkhazian),
            "ace" => Some(Language::Achinese),
            "ach" => Some(Language::Acoli),
            "ada" => Some(Language::Adangme),
            "ady" => Some(Language::Adyghe),
            "aar" => Some(Language::Afar),
            "afh" => Some(Language::Afrihili),
            "afr" => Some(Language::Afrikaans),
            "afa" => Some(Language::AfroAsiaticLanguages),
            "ain" => Some(Language::Ainu),
            "aka" => Some(Language::Akan),
            "akk" => Some(Language::Akkadian),
            "alb" => Some(Language::Albanian),
            "ale" => Some(Language::Aleut),
            "alg" => Some(Language::AlgonquianLanguages),
            "tut" => Some(Language::AltaicLanguages),
            "amh" => Some(Language::Amharic),
            "anp" => Some(Language::Angika),
            "apa" => Some(Language::ApacheLanguages),
            "ara" => Some(Language::Arabic),
            "arg" => Some(Language::Aragonese),
            "arp" => Some(Language::Arapaho),
            "arw" => Some(Language::Arawak),
            "arm" => Some(Language::Armenian),
            "rup" => Some(Language::Aromanian),
            "art" => Some(Language::ArtificialLanguages),
            "asm" => Some(Language::Assamese),
            "ast" => Some(Language::Asturian),
            "ath" => Some(Language::AthapascanLanguages),
            "aus" => Some(Language::AustralianLanguages),
            "map" => Some(Language::AustronesianLanguages),
            "ava" => Some(Language::Avaric),
            "ave" => Some(Language::Avestan),
            "awa" => Some(Language::Awadhi),
            "aym" => Some(Language::Aymara),
            "aze" => Some(Language::Azerbaijani),
            "ban" => Some(Language::Balinese),
            "bat" => Some(Language::BalticLanguages),
            "bal" => Some(Language::Baluchi),
            "bam" => Some(Language::Bambara),
            "bai" => Some(Language::BamilekeLanguages),
            "bad" => Some(Language::BandaLanguages),
            "bnt" => Some(Language::BantuLanguages),
            "bas" => Some(Language::Basa),
            "bak" => Some(Language::Bashkir),
            "baq" => Some(Language::Basque),
            "btk" => Some(Language::BatakLanguages),
            "bej" => Some(Language::Beja),
            "bel" => Some(Language::Belarusian),
            "bem" => Some(Language::Bemba),
            "ben" => Some(Language::Bengali),
            "ber" => Some(Language::BerberLanguages),
            "bho" => Some(Language::Bhojpuri),
            "bih" => Some(Language::BihariLanguages),
            "bik" => Some(Language::Bikol),
            "bin" => Some(Language::Bini),
            "bis" => Some(Language::Bislama),
            "byn" => Some(Language::Blin),
            "zbl" => Some(Language::Blissymbols),
            "nob" => Some(Language::BokmålNorwegian),
            "bos" => Some(Language::Bosnian),
            "bra" => Some(Language::Braj),
            "bre" => Some(Language::Breton),
            "bug" => Some(Language::Buginese),
            "bul" => Some(Language::Bulgarian),
            "bua" => Some(Language::Buriat),
            "bur" => Some(Language::Burmese),
            "cad" => Some(Language::Caddo),
            "cat" => Some(Language::Catalan),
            "cau" => Some(Language::CaucasianLanguages),
            "ceb" => Some(Language::Cebuano),
            "cel" => Some(Language::CelticLanguages),
            "cai" => Some(Language::CentralAmericanIndianLanguages),
            "khm" => Some(Language::CentralKhmer),
            "chg" => Some(Language::Chagatai),
            "cmc" => Some(Language::ChamicLanguages),
            "cha" => Some(Language::Chamorro),
            "che" => Some(Language::Chechen),
            "chr" => Some(Language::Cherokee),
            "chy" => Some(Language::Cheyenne),
            "chb" => Some(Language::Chibcha),
            "nya" => Some(Language::Chichewa),
            "chi" => Some(Language::Chinese),
            "chn" => Some(Language::Chinook),
            "chp" => Some(Language::Chipewyan),
            "cho" => Some(Language::Choctaw),
            "chu" => Some(Language::ChurchSlavic),
            "chk" => Some(Language::Chuukese),
            "chv" => Some(Language::Chuvash),
            "nwc" => Some(Language::ClassicalNewari),
            "syc" => Some(Language::ClassicalSyriac),
            "cop" => Some(Language::Coptic),
            "cor" => Some(Language::Cornish),
            "cos" => Some(Language::Corsican),
            "cre" => Some(Language::Cree),
            "mus" => Some(Language::Creek),
            "crp" => Some(Language::CreolesAndPidgins),
            "cpe" => Some(Language::CreolesAndPidginsEnglishBased),
            "cpf" => Some(Language::CreolesAndPidginsFrenchBased),
            "cpp" => Some(Language::CreolesAndPidginsPortugueseBased),
            "crh" => Some(Language::CrimeanTatar),
            "hrv" => Some(Language::Croatian),
            "cus" => Some(Language::CushiticLanguages),
            "cze" => Some(Language::Czech),
            "dak" => Some(Language::Dakota),
            "dan" => Some(Language::Danish),
            "dar" => Some(Language::Dargwa),
            "del" => Some(Language::Delaware),
            "din" => Some(Language::Dinka),
            "div" => Some(Language::Divehi),
            "doi" => Some(Language::Dogri),
            "dgr" => Some(Language::Dogrib),
            "dra" => Some(Language::DravidianLanguages),
            "dua" => Some(Language::Duala),
            "dut" => Some(Language::Dutch),
            "dum" => Some(Language::DutchMiddle),
            "dyu" => Some(Language::Dyula),
            "dzo" => Some(Language::Dzongkha),
            "frs" => Some(Language::EasternFrisian),
            "efi" => Some(Language::Efik),
            "egy" => Some(Language::EgyptianAncient),
            "eka" => Some(Language::Ekajuk),
            "elx" => Some(Language::Elamite),
            "eng" => Some(Language::English),
            "enm" => Some(Language::EnglishMiddle),
            "ang" => Some(Language::EnglishOld),
            "myv" => Some(Language::Erzya),
            "epo" => Some(Language::Esperanto),
            "est" => Some(Language::Estonian),
            "ewe" => Some(Language::Ewe),
            "ewo" => Some(Language::Ewondo),
            "fan" => Some(Language::Fang),
            "fat" => Some(Language::Fanti),
            "fao" => Some(Language::Faroese),
            "fij" => Some(Language::Fijian),
            "fil" => Some(Language::Filipino),
            "fin" => Some(Language::Finnish),
            "fiu" => Some(Language::FinnoUgrianLanguages),
            "fon" => Some(Language::Fon),
            "fre" => Some(Language::French),
            "frm" => Some(Language::FrenchMiddle),
            "fro" => Some(Language::FrenchOld),
            "fur" => Some(Language::Friulian),
            "ful" => Some(Language::Fulah),
            "gaa" => Some(Language::Ga),
            "gla" => Some(Language::Gaelic),
            "car" => Some(Language::GalibiCarib),
            "glg" => Some(Language::Galician),
            "lug" => Some(Language::Ganda),
            "gay" => Some(Language::Gayo),
            "gba" => Some(Language::Gbaya),
            "gez" => Some(Language::Geez),
            "geo" => Some(Language::Georgian),
            "ger" => Some(Language::German),
            "gmh" => Some(Language::GermanMiddleHigh),
            "goh" => Some(Language::GermanOldHigh),
            "gem" => Some(Language::GermanicLanguages),
            "gil" => Some(Language::Gilbertese),
            "gon" => Some(Language::Gondi),
            "gor" => Some(Language::Gorontalo),
            "got" => Some(Language::Gothic),
            "grb" => Some(Language::Grebo),
            "grc" => Some(Language::GreekAncient),
            "gre" => Some(Language::GreekModern),
            "grn" => Some(Language::Guarani),
            "guj" => Some(Language::Gujarati),
            "gwi" => Some(Language::GwichIn),
            "hai" => Some(Language::Haida),
            "hat" => Some(Language::Haitian),
            "hau" => Some(Language::Hausa),
            "haw" => Some(Language::Hawaiian),
            "heb" => Some(Language::Hebrew),
            "her" => Some(Language::Herero),
            "hil" => Some(Language::Hiligaynon),
            "him" => Some(Language::HimachaliLanguages),
            "hin" => Some(Language::Hindi),
            "hmo" => Some(Language::HiriMotu),
            "hit" => Some(Language::Hittite),
            "hmn" => Some(Language::Hmong),
            "hun" => Some(Language::Hungarian),
            "hup" => Some(Language::Hupa),
            "iba" => Some(Language::Iban),
            "ice" => Some(Language::Icelandic),
            "ido" => Some(Language::Ido),
            "ibo" => Some(Language::Igbo),
            "ijo" => Some(Language::IjoLanguages),
            "ilo" => Some(Language::Iloko),
            "smn" => Some(Language::InariSami),
            "inc" => Some(Language::IndicLanguages),
            "ine" => Some(Language::IndoEuropeanLanguages),
            "ind" => Some(Language::Indonesian),
            "inh" => Some(Language::Ingush),
            "ina" => Some(Language::Interlingua),
            "ile" => Some(Language::Interlingue),
            "iku" => Some(Language::Inuktitut),
            "ipk" => Some(Language::Inupiaq),
            "ira" => Some(Language::IranianLanguages),
            "gle" => Some(Language::Irish),
            "mga" => Some(Language::IrishMiddle),
            "sga" => Some(Language::IrishOld),
            "iro" => Some(Language::IroquoianLanguages),
            "ita" => Some(Language::Italian),
            "jpn" => Some(Language::Japanese),
            "jav" => Some(Language::Javanese),
            "jrb" => Some(Language::JudeoArabic),
            "jpr" => Some(Language::JudeoPersian),
            "kbd" => Some(Language::Kabardian),
            "kab" => Some(Language::Kabyle),
            "kac" => Some(Language::Kachin),
            "kal" => Some(Language::Kalaallisut),
            "xal" => Some(Language::Kalmyk),
            "kam" => Some(Language::Kamba),
            "kan" => Some(Language::Kannada),
            "kau" => Some(Language::Kanuri),
            "kaa" => Some(Language::KaraKalpak),
            "krc" => Some(Language::KarachayBalkar),
            "krl" => Some(Language::Karelian),
            "kar" => Some(Language::KarenLanguages),
            "kas" => Some(Language::Kashmiri),
            "csb" => Some(Language::Kashubian),
            "kaw" => Some(Language::Kawi),
            "kaz" => Some(Language::Kazakh),
            "kha" => Some(Language::Khasi),
            "khi" => Some(Language::KhoisanLanguages),
            "kho" => Some(Language::Khotanese),
            "kik" => Some(Language::Kikuyu),
            "kmb" => Some(Language::Kimbundu),
            "kin" => Some(Language::Kinyarwanda),
            "kir" => Some(Language::Kirghiz),
            "tlh" => Some(Language::Klingon),
            "kom" => Some(Language::Komi),
            "kon" => Some(Language::Kongo),
            "kok" => Some(Language::Konkani),
            "kor" => Some(Language::Korean),
            "kos" => Some(Language::Kosraean),
            "kpe" => Some(Language::Kpelle),
            "kro" => Some(Language::KruLanguages),
            "kua" => Some(Language::Kuanyama),
            "kum" => Some(Language::Kumyk),
            "kur" => Some(Language::Kurdish),
            "kru" => Some(Language::Kurukh),
            "kut" => Some(Language::Kutenai),
            "lad" => Some(Language::Ladino),
            "lah" => Some(Language::Lahnda),
            "lam" => Some(Language::Lamba),
            "day" => Some(Language::LandDayakLanguages),
            "lao" => Some(Language::Lao),
            "lat" => Some(Language::Latin),
            "lav" => Some(Language::Latvian),
            "lez" => Some(Language::Lezghian),
            "lim" => Some(Language::Limburgan),
            "lin" => Some(Language::Lingala),
            "lit" => Some(Language::Lithuanian),
            "jbo" => Some(Language::Lojban),
            "nds" => Some(Language::LowGerman),
            "dsb" => Some(Language::LowerSorbian),
            "loz" => Some(Language::Lozi),
            "lub" => Some(Language::LubaKatanga),
            "lua" => Some(Language::LubaLulua),
            "lui" => Some(Language::Luiseno),
            "smj" => Some(Language::LuleSami),
            "lun" => Some(Language::Lunda),
            "luo" => Some(Language::LuoKenyaAndTanzania),
            "lus" => Some(Language::Lushai),
            "ltz" => Some(Language::Luxembourgish),
            "mac" => Some(Language::Macedonian),
            "mad" => Some(Language::Madurese),
            "mag" => Some(Language::Magahi),
            "mai" => Some(Language::Maithili),
            "mak" => Some(Language::Makasar),
            "mlg" => Some(Language::Malagasy),
            "may" => Some(Language::Malay),
            "mal" => Some(Language::Malayalam),
            "mlt" => Some(Language::Maltese),
            "mnc" => Some(Language::Manchu),
            "mdr" => Some(Language::Mandar),
            "man" => Some(Language::Mandingo),
            "mni" => Some(Language::Manipuri),
            "mno" => Some(Language::ManoboLanguages),
            "glv" => Some(Language::Manx),
            "mao" => Some(Language::Maori),
            "arn" => Some(Language::Mapudungun),
            "mar" => Some(Language::Marathi),
            "chm" => Some(Language::Mari),
            "mah" => Some(Language::Marshallese),
            "mwr" => Some(Language::Marwari),
            "mas" => Some(Language::Masai),
            "myn" => Some(Language::MayanLanguages),
            "men" => Some(Language::Mende),
            "mic" => Some(Language::MiKmaq),
            "min" => Some(Language::Minangkabau),
            "mwl" => Some(Language::Mirandese),
            "moh" => Some(Language::Mohawk),
            "mdf" => Some(Language::Moksha),
            "mkh" => Some(Language::MonKhmerLanguages),
            "lol" => Some(Language::Mongo),
            "mon" => Some(Language::Mongolian),
            "cnr" => Some(Language::Montenegrin),
            "mos" => Some(Language::Mossi),
            "mul" => Some(Language::MultipleLanguages),
            "mun" => Some(Language::MundaLanguages),
            "nqo" => Some(Language::NKo),
            "nah" => Some(Language::NahuatlLanguages),
            "nau" => Some(Language::Nauru),
            "nav" => Some(Language::Navajo),
            "nde" => Some(Language::NdebeleNorth),
            "nbl" => Some(Language::NdebeleSouth),
            "ndo" => Some(Language::Ndonga),
            "nap" => Some(Language::Neapolitan),
            "new" => Some(Language::NepalBhasa),
            "nep" => Some(Language::Nepali),
            "nia" => Some(Language::Nias),
            "nic" => Some(Language::NigerKordofanianLanguages),
            "ssa" => Some(Language::NiloSaharanLanguages),
            "niu" => Some(Language::Niuean),
            "zxx" => Some(Language::NoLinguisticContent),
            "nog" => Some(Language::Nogai),
            "non" => Some(Language::NorseOld),
            "nai" => Some(Language::NorthAmericanIndianLanguages),
            "frr" => Some(Language::NorthernFrisian),
            "sme" => Some(Language::NorthernSami),
            "nor" => Some(Language::Norwegian),
            "nno" => Some(Language::NorwegianNynorsk),
            "nub" => Some(Language::NubianLanguages),
            "nym" => Some(Language::Nyamwezi),
            "nyn" => Some(Language::Nyankole),
            "nyo" => Some(Language::Nyoro),
            "nzi" => Some(Language::Nzima),
            "oci" => Some(Language::Occitan),
            "arc" => Some(Language::OfficialAramaic),
            "oji" => Some(Language::Ojibwa),
            "ori" => Some(Language::Oriya),
            "orm" => Some(Language::Oromo),
            "osa" => Some(Language::Osage),
            "oss" => Some(Language::Ossetian),
            "oto" => Some(Language::OtomianLanguages),
            "pal" => Some(Language::Pahlavi),
            "pau" => Some(Language::Palauan),
            "pli" => Some(Language::Pali),
            "pam" => Some(Language::Pampanga),
            "pag" => Some(Language::Pangasinan),
            "pan" => Some(Language::Panjabi),
            "pap" => Some(Language::Papiamento),
            "paa" => Some(Language::PapuanLanguages),
            "nso" => Some(Language::Pedi),
            "per" => Some(Language::Persian),
            "peo" => Some(Language::PersianOld),
            "phi" => Some(Language::PhilippineLanguages),
            "phn" => Some(Language::Phoenician),
            "pon" => Some(Language::Pohnpeian),
            "pol" => Some(Language::Polish),
            "por" => Some(Language::Portuguese),
            "pra" => Some(Language::PrakritLanguages),
            "pro" => Some(Language::ProvencalOld),
            "pus" => Some(Language::Pushto),
            "que" => Some(Language::Quechua),
            "raj" => Some(Language::Rajasthani),
            "rap" => Some(Language::Rapanui),
            "rar" => Some(Language::Rarotongan),
            "roa" => Some(Language::RomanceLanguages),
            "rum" => Some(Language::Romanian),
            "roh" => Some(Language::Romansh),
            "rom" => Some(Language::Romany),
            "run" => Some(Language::Rundi),
            "rus" => Some(Language::Russian),
            "sal" => Some(Language::SalishanLanguages),
            "sam" => Some(Language::SamaritanAramaic),
            "smi" => Some(Language::SamiLanguages),
            "smo" => Some(Language::Samoan),
            "sad" => Some(Language::Sandawe),
            "sag" => Some(Language::Sango),
            "san" => Some(Language::Sanskrit),
            "sat" => Some(Language::Santali),
            "srd" => Some(Language::Sardinian),
            "sas" => Some(Language::Sasak),
            "sco" => Some(Language::Scots),
            "sel" => Some(Language::Selkup),
            "sem" => Some(Language::SemiticLanguages),
            "srp" => Some(Language::Serbian),
            "srr" => Some(Language::Serer),
            "shn" => Some(Language::Shan),
            "sna" => Some(Language::Shona),
            "iii" => Some(Language::SichuanYi),
            "scn" => Some(Language::Sicilian),
            "sid" => Some(Language::Sidamo),
            "sgn" => Some(Language::SignLanguages),
            "bla" => Some(Language::Siksika),
            "snd" => Some(Language::Sindhi),
            "sin" => Some(Language::Sinhala),
            "sit" => Some(Language::SinoTibetanLanguages),
            "sio" => Some(Language::SiouanLanguages),
            "sms" => Some(Language::SkoltSami),
            "den" => Some(Language::SlaveAthapascan),
            "sla" => Some(Language::SlavicLanguages),
            "slo" => Some(Language::Slovak),
            "slv" => Some(Language::Slovenian),
            "sog" => Some(Language::Sogdian),
            "som" => Some(Language::Somali),
            "son" => Some(Language::SonghaiLanguages),
            "snk" => Some(Language::Soninke),
            "wen" => Some(Language::SorbianLanguages),
            "sot" => Some(Language::SothoSouthern),
            "sai" => Some(Language::SouthAmericanIndianLanguages),
            "alt" => Some(Language::SouthernAltai),
            "sma" => Some(Language::SouthernSami),
            "spa" => Some(Language::Spanish),
            "srn" => Some(Language::SrananTongo),
            "zgh" => Some(Language::StandardMoroccanTamazight),
            "suk" => Some(Language::Sukuma),
            "sux" => Some(Language::Sumerian),
            "sun" => Some(Language::Sundanese),
            "sus" => Some(Language::Susu),
            "swa" => Some(Language::Swahili),
            "ssw" => Some(Language::Swati),
            "swe" => Some(Language::Swedish),
            "gsw" => Some(Language::SwissGerman),
            "syr" => Some(Language::Syriac),
            "tgl" => Some(Language::Tagalog),
            "tah" => Some(Language::Tahitian),
            "tai" => Some(Language::TaiLanguages),
            "tgk" => Some(Language::Tajik),
            "tmh" => Some(Language::Tamashek),
            "tam" => Some(Language::Tamil),
            "tat" => Some(Language::Tatar),
            "tel" => Some(Language::Telugu),
            "ter" => Some(Language::Tereno),
            "tet" => Some(Language::Tetum),
            "tha" => Some(Language::Thai),
            "tib" => Some(Language::Tibetan),
            "tig" => Some(Language::Tigre),
            "tir" => Some(Language::Tigrinya),
            "tem" => Some(Language::Timne),
            "tiv" => Some(Language::Tiv),
            "tli" => Some(Language::Tlingit),
            "tpi" => Some(Language::TokPisin),
            "tkl" => Some(Language::Tokelau),
            "tog" => Some(Language::TongaNyasa),
            "ton" => Some(Language::TongaTongaIslands),
            "tsi" => Some(Language::Tsimshian),
            "tso" => Some(Language::Tsonga),
            "tsn" => Some(Language::Tswana),
            "tum" => Some(Language::Tumbuka),
            "tup" => Some(Language::TupiLanguages),
            "tur" => Some(Language::Turkish),
            "ota" => Some(Language::TurkishOttoman),
            "tuk" => Some(Language::Turkmen),
            "tvl" => Some(Language::Tuvalu),
            "tyv" => Some(Language::Tuvinian),
            "twi" => Some(Language::Twi),
            "udm" => Some(Language::Udmurt),
            "uga" => Some(Language::Ugaritic),
            "uig" => Some(Language::Uighur),
            "ukr" => Some(Language::Ukrainian),
            "umb" => Some(Language::Umbundu),
            "mis" => Some(Language::UncodedLanguages),
            "und" => Some(Language::Undetermined),
            "hsb" => Some(Language::UpperSorbian),
            "urd" => Some(Language::Urdu),
            "uzb" => Some(Language::Uzbek),
            "vai" => Some(Language::Vai),
            "ven" => Some(Language::Venda),
            "vie" => Some(Language::Vietnamese),
            "vol" => Some(Language::Volapuk),
            "vot" => Some(Language::Votic),
            "wak" => Some(Language::WakashanLanguages),
            "wln" => Some(Language::Walloon),
            "war" => Some(Language::Waray),
            "was" => Some(Language::Washo),
            "wel" => Some(Language::Welsh),
            "fry" => Some(Language::WesternFrisian),
            "wal" => Some(Language::Wolaitta),
            "wol" => Some(Language::Wolof),
            "xho" => Some(Language::Xhosa),
            "sah" => Some(Language::Yakut),
            "yao" => Some(Language::Yao),
            "yap" => Some(Language::Yapese),
            "yid" => Some(Language::Yiddish),
            "yor" => Some(Language::Yoruba),
            "ypk" => Some(Language::YupikLanguages),
            "znd" => Some(Language::ZandeLanguages),
            "zap" => Some(Language::Zapotec),
            "zza" => Some(Language::Zaza),
            "zen" => Some(Language::Zenaga),
            "zha" => Some(Language::Zhuang),
            "zul" => Some(Language::Zulu),
            "zun" => Some(Language::Zuni),
            _ => None,
        }
    }

/// Attempts to match an ISO-639 alpha3-t code
/// ```
/// use iso_639::Language;
///
/// let french = Language::from_alpha3_t_str("fra").unwrap();
/// ```

    pub fn from_alpha3_t_str(s: &str) -> Option<Language> {
        match s {
            "sqi" => Some(Language::Albanian),
            "hye" => Some(Language::Armenian),
            "eus" => Some(Language::Basque),
            "mya" => Some(Language::Burmese),
            "zho" => Some(Language::Chinese),
            "ces" => Some(Language::Czech),
            "nld" => Some(Language::Dutch),
            "fra" => Some(Language::French),
            "kat" => Some(Language::Georgian),
            "deu" => Some(Language::German),
            "ell" => Some(Language::GreekModern),
            "isl" => Some(Language::Icelandic),
            "mkd" => Some(Language::Macedonian),
            "msa" => Some(Language::Malay),
            "mri" => Some(Language::Maori),
            "fas" => Some(Language::Persian),
            "ron" => Some(Language::Romanian),
            "slk" => Some(Language::Slovak),
            "bod" => Some(Language::Tibetan),
            "cym" => Some(Language::Welsh),
            _ => None,
        }
    }
/// Returns the ISO-639 alpha2 code
/// ```
/// use iso_639::Language;
///
/// let french = Language::French.alpha2().unwrap();
/// ```

    pub fn alpha2(&self) -> Option<&str> {
        match self {
            Language::Abkhazian => Some("ab"),
            Language::Afar => Some("aa"),
            Language::Afrikaans => Some("af"),
            Language::Akan => Some("ak"),
            Language::Albanian => Some("sq"),
            Language::Amharic => Some("am"),
            Language::Arabic => Some("ar"),
            Language::Aragonese => Some("an"),
            Language::Armenian => Some("hy"),
            Language::Assamese => Some("as"),
            Language::Avaric => Some("av"),
            Language::Avestan => Some("ae"),
            Language::Aymara => Some("ay"),
            Language::Azerbaijani => Some("az"),
            Language::Bambara => Some("bm"),
            Language::Bashkir => Some("ba"),
            Language::Basque => Some("eu"),
            Language::Belarusian => Some("be"),
            Language::Bengali => Some("bn"),
            Language::BihariLanguages => Some("bh"),
            Language::Bislama => Some("bi"),
            Language::BokmålNorwegian => Some("nb"),
            Language::Bosnian => Some("bs"),
            Language::Breton => Some("br"),
            Language::Bulgarian => Some("bg"),
            Language::Burmese => Some("my"),
            Language::Catalan => Some("ca"),
            Language::CentralKhmer => Some("km"),
            Language::Chamorro => Some("ch"),
            Language::Chechen => Some("ce"),
            Language::Chichewa => Some("ny"),
            Language::Chinese => Some("zh"),
            Language::ChurchSlavic => Some("cu"),
            Language::Chuvash => Some("cv"),
            Language::Cornish => Some("kw"),
            Language::Corsican => Some("co"),
            Language::Cree => Some("cr"),
            Language::Croatian => Some("hr"),
            Language::Czech => Some("cs"),
            Language::Danish => Some("da"),
            Language::Divehi => Some("dv"),
            Language::Dutch => Some("nl"),
            Language::Dzongkha => Some("dz"),
            Language::English => Some("en"),
            Language::Esperanto => Some("eo"),
            Language::Estonian => Some("et"),
            Language::Ewe => Some("ee"),
            Language::Faroese => Some("fo"),
            Language::Fijian => Some("fj"),
            Language::Finnish => Some("fi"),
            Language::French => Some("fr"),
            Language::Fulah => Some("ff"),
            Language::Gaelic => Some("gd"),
            Language::Galician => Some("gl"),
            Language::Ganda => Some("lg"),
            Language::Georgian => Some("ka"),
            Language::German => Some("de"),
            Language::GreekModern => Some("el"),
            Language::Guarani => Some("gn"),
            Language::Gujarati => Some("gu"),
            Language::Haitian => Some("ht"),
            Language::Hausa => Some("ha"),
            Language::Hebrew => Some("he"),
            Language::Herero => Some("hz"),
            Language::Hindi => Some("hi"),
            Language::HiriMotu => Some("ho"),
            Language::Hungarian => Some("hu"),
            Language::Icelandic => Some("is"),
            Language::Ido => Some("io"),
            Language::Igbo => Some("ig"),
            Language::Indonesian => Some("id"),
            Language::Interlingua => Some("ia"),
            Language::Interlingue => Some("ie"),
            Language::Inuktitut => Some("iu"),
            Language::Inupiaq => Some("ik"),
            Language::Irish => Some("ga"),
            Language::Italian => Some("it"),
            Language::Japanese => Some("ja"),
            Language::Javanese => Some("jv"),
            Language::Kalaallisut => Some("kl"),
            Language::Kannada => Some("kn"),
            Language::Kanuri => Some("kr"),
            Language::Kashmiri => Some("ks"),
            Language::Kazakh => Some("kk"),
            Language::Kikuyu => Some("ki"),
            Language::Kinyarwanda => Some("rw"),
            Language::Kirghiz => Some("ky"),
            Language::Komi => Some("kv"),
            Language::Kongo => Some("kg"),
            Language::Korean => Some("ko"),
            Language::Kuanyama => Some("kj"),
            Language::Kurdish => Some("ku"),
            Language::Lao => Some("lo"),
            Language::Latin => Some("la"),
            Language::Latvian => Some("lv"),
            Language::Limburgan => Some("li"),
            Language::Lingala => Some("ln"),
            Language::Lithuanian => Some("lt"),
            Language::LubaKatanga => Some("lu"),
            Language::Luxembourgish => Some("lb"),
            Language::Macedonian => Some("mk"),
            Language::Malagasy => Some("mg"),
            Language::Malay => Some("ms"),
            Language::Malayalam => Some("ml"),
            Language::Maltese => Some("mt"),
            Language::Manx => Some("gv"),
            Language::Maori => Some("mi"),
            Language::Marathi => Some("mr"),
            Language::Marshallese => Some("mh"),
            Language::Mongolian => Some("mn"),
            Language::Nauru => Some("na"),
            Language::Navajo => Some("nv"),
            Language::NdebeleNorth => Some("nd"),
            Language::NdebeleSouth => Some("nr"),
            Language::Ndonga => Some("ng"),
            Language::Nepali => Some("ne"),
            Language::NorthernSami => Some("se"),
            Language::Norwegian => Some("no"),
            Language::NorwegianNynorsk => Some("nn"),
            Language::Occitan => Some("oc"),
            Language::Ojibwa => Some("oj"),
            Language::Oriya => Some("or"),
            Language::Oromo => Some("om"),
            Language::Ossetian => Some("os"),
            Language::Pali => Some("pi"),
            Language::Panjabi => Some("pa"),
            Language::Persian => Some("fa"),
            Language::Polish => Some("pl"),
            Language::Portuguese => Some("pt"),
            Language::Pushto => Some("ps"),
            Language::Quechua => Some("qu"),
            Language::Romanian => Some("ro"),
            Language::Romansh => Some("rm"),
            Language::Rundi => Some("rn"),
            Language::Russian => Some("ru"),
            Language::Samoan => Some("sm"),
            Language::Sango => Some("sg"),
            Language::Sanskrit => Some("sa"),
            Language::Sardinian => Some("sc"),
            Language::Serbian => Some("sr"),
            Language::Shona => Some("sn"),
            Language::SichuanYi => Some("ii"),
            Language::Sindhi => Some("sd"),
            Language::Sinhala => Some("si"),
            Language::Slovak => Some("sk"),
            Language::Slovenian => Some("sl"),
            Language::Somali => Some("so"),
            Language::SothoSouthern => Some("st"),
            Language::Spanish => Some("es"),
            Language::Sundanese => Some("su"),
            Language::Swahili => Some("sw"),
            Language::Swati => Some("ss"),
            Language::Swedish => Some("sv"),
            Language::Tagalog => Some("tl"),
            Language::Tahitian => Some("ty"),
            Language::Tajik => Some("tg"),
            Language::Tamil => Some("ta"),
            Language::Tatar => Some("tt"),
            Language::Telugu => Some("te"),
            Language::Thai => Some("th"),
            Language::Tibetan => Some("bo"),
            Language::Tigrinya => Some("ti"),
            Language::TongaTongaIslands => Some("to"),
            Language::Tsonga => Some("ts"),
            Language::Tswana => Some("tn"),
            Language::Turkish => Some("tr"),
            Language::Turkmen => Some("tk"),
            Language::Twi => Some("tw"),
            Language::Uighur => Some("ug"),
            Language::Ukrainian => Some("uk"),
            Language::Urdu => Some("ur"),
            Language::Uzbek => Some("uz"),
            Language::Venda => Some("ve"),
            Language::Vietnamese => Some("vi"),
            Language::Volapuk => Some("vo"),
            Language::Walloon => Some("wa"),
            Language::Welsh => Some("cy"),
            Language::WesternFrisian => Some("fy"),
            Language::Wolof => Some("wo"),
            Language::Xhosa => Some("xh"),
            Language::Yiddish => Some("yi"),
            Language::Yoruba => Some("yo"),
            Language::Zhuang => Some("za"),
            Language::Zulu => Some("zu"),
            _ => None,
        }
    }
/// Returns the ISO-639 alpha3-b code
/// ```
/// use iso_639::Language;
///
/// let french = Language::French.alpha3_b().unwrap();
/// ```

    pub fn alpha3_b(&self) -> Option<&str> {
        match self {
            Language::Abkhazian => Some("abk"),
            Language::Achinese => Some("ace"),
            Language::Acoli => Some("ach"),
            Language::Adangme => Some("ada"),
            Language::Adyghe => Some("ady"),
            Language::Afar => Some("aar"),
            Language::Afrihili => Some("afh"),
            Language::Afrikaans => Some("afr"),
            Language::AfroAsiaticLanguages => Some("afa"),
            Language::Ainu => Some("ain"),
            Language::Akan => Some("aka"),
            Language::Akkadian => Some("akk"),
            Language::Albanian => Some("alb"),
            Language::Aleut => Some("ale"),
            Language::AlgonquianLanguages => Some("alg"),
            Language::AltaicLanguages => Some("tut"),
            Language::Amharic => Some("amh"),
            Language::Angika => Some("anp"),
            Language::ApacheLanguages => Some("apa"),
            Language::Arabic => Some("ara"),
            Language::Aragonese => Some("arg"),
            Language::Arapaho => Some("arp"),
            Language::Arawak => Some("arw"),
            Language::Armenian => Some("arm"),
            Language::Aromanian => Some("rup"),
            Language::ArtificialLanguages => Some("art"),
            Language::Assamese => Some("asm"),
            Language::Asturian => Some("ast"),
            Language::AthapascanLanguages => Some("ath"),
            Language::AustralianLanguages => Some("aus"),
            Language::AustronesianLanguages => Some("map"),
            Language::Avaric => Some("ava"),
            Language::Avestan => Some("ave"),
            Language::Awadhi => Some("awa"),
            Language::Aymara => Some("aym"),
            Language::Azerbaijani => Some("aze"),
            Language::Balinese => Some("ban"),
            Language::BalticLanguages => Some("bat"),
            Language::Baluchi => Some("bal"),
            Language::Bambara => Some("bam"),
            Language::BamilekeLanguages => Some("bai"),
            Language::BandaLanguages => Some("bad"),
            Language::BantuLanguages => Some("bnt"),
            Language::Basa => Some("bas"),
            Language::Bashkir => Some("bak"),
            Language::Basque => Some("baq"),
            Language::BatakLanguages => Some("btk"),
            Language::Beja => Some("bej"),
            Language::Belarusian => Some("bel"),
            Language::Bemba => Some("bem"),
            Language::Bengali => Some("ben"),
            Language::BerberLanguages => Some("ber"),
            Language::Bhojpuri => Some("bho"),
            Language::BihariLanguages => Some("bih"),
            Language::Bikol => Some("bik"),
            Language::Bini => Some("bin"),
            Language::Bislama => Some("bis"),
            Language::Blin => Some("byn"),
            Language::Blissymbols => Some("zbl"),
            Language::BokmålNorwegian => Some("nob"),
            Language::Bosnian => Some("bos"),
            Language::Braj => Some("bra"),
            Language::Breton => Some("bre"),
            Language::Buginese => Some("bug"),
            Language::Bulgarian => Some("bul"),
            Language::Buriat => Some("bua"),
            Language::Burmese => Some("bur"),
            Language::Caddo => Some("cad"),
            Language::Catalan => Some("cat"),
            Language::CaucasianLanguages => Some("cau"),
            Language::Cebuano => Some("ceb"),
            Language::CelticLanguages => Some("cel"),
            Language::CentralAmericanIndianLanguages => Some("cai"),
            Language::CentralKhmer => Some("khm"),
            Language::Chagatai => Some("chg"),
            Language::ChamicLanguages => Some("cmc"),
            Language::Chamorro => Some("cha"),
            Language::Chechen => Some("che"),
            Language::Cherokee => Some("chr"),
            Language::Cheyenne => Some("chy"),
            Language::Chibcha => Some("chb"),
            Language::Chichewa => Some("nya"),
            Language::Chinese => Some("chi"),
            Language::Chinook => Some("chn"),
            Language::Chipewyan => Some("chp"),
            Language::Choctaw => Some("cho"),
            Language::ChurchSlavic => Some("chu"),
            Language::Chuukese => Some("chk"),
            Language::Chuvash => Some("chv"),
            Language::ClassicalNewari => Some("nwc"),
            Language::ClassicalSyriac => Some("syc"),
            Language::Coptic => Some("cop"),
            Language::Cornish => Some("cor"),
            Language::Corsican => Some("cos"),
            Language::Cree => Some("cre"),
            Language::Creek => Some("mus"),
            Language::CreolesAndPidgins => Some("crp"),
            Language::CreolesAndPidginsEnglishBased => Some("cpe"),
            Language::CreolesAndPidginsFrenchBased => Some("cpf"),
            Language::CreolesAndPidginsPortugueseBased => Some("cpp"),
            Language::CrimeanTatar => Some("crh"),
            Language::Croatian => Some("hrv"),
            Language::CushiticLanguages => Some("cus"),
            Language::Czech => Some("cze"),
            Language::Dakota => Some("dak"),
            Language::Danish => Some("dan"),
            Language::Dargwa => Some("dar"),
            Language::Delaware => Some("del"),
            Language::Dinka => Some("din"),
            Language::Divehi => Some("div"),
            Language::Dogri => Some("doi"),
            Language::Dogrib => Some("dgr"),
            Language::DravidianLanguages => Some("dra"),
            Language::Duala => Some("dua"),
            Language::Dutch => Some("dut"),
            Language::DutchMiddle => Some("dum"),
            Language::Dyula => Some("dyu"),
            Language::Dzongkha => Some("dzo"),
            Language::EasternFrisian => Some("frs"),
            Language::Efik => Some("efi"),
            Language::EgyptianAncient => Some("egy"),
            Language::Ekajuk => Some("eka"),
            Language::Elamite => Some("elx"),
            Language::English => Some("eng"),
            Language::EnglishMiddle => Some("enm"),
            Language::EnglishOld => Some("ang"),
            Language::Erzya => Some("myv"),
            Language::Esperanto => Some("epo"),
            Language::Estonian => Some("est"),
            Language::Ewe => Some("ewe"),
            Language::Ewondo => Some("ewo"),
            Language::Fang => Some("fan"),
            Language::Fanti => Some("fat"),
            Language::Faroese => Some("fao"),
            Language::Fijian => Some("fij"),
            Language::Filipino => Some("fil"),
            Language::Finnish => Some("fin"),
            Language::FinnoUgrianLanguages => Some("fiu"),
            Language::Fon => Some("fon"),
            Language::French => Some("fre"),
            Language::FrenchMiddle => Some("frm"),
            Language::FrenchOld => Some("fro"),
            Language::Friulian => Some("fur"),
            Language::Fulah => Some("ful"),
            Language::Ga => Some("gaa"),
            Language::Gaelic => Some("gla"),
            Language::GalibiCarib => Some("car"),
            Language::Galician => Some("glg"),
            Language::Ganda => Some("lug"),
            Language::Gayo => Some("gay"),
            Language::Gbaya => Some("gba"),
            Language::Geez => Some("gez"),
            Language::Georgian => Some("geo"),
            Language::German => Some("ger"),
            Language::GermanMiddleHigh => Some("gmh"),
            Language::GermanOldHigh => Some("goh"),
            Language::GermanicLanguages => Some("gem"),
            Language::Gilbertese => Some("gil"),
            Language::Gondi => Some("gon"),
            Language::Gorontalo => Some("gor"),
            Language::Gothic => Some("got"),
            Language::Grebo => Some("grb"),
            Language::GreekAncient => Some("grc"),
            Language::GreekModern => Some("gre"),
            Language::Guarani => Some("grn"),
            Language::Gujarati => Some("guj"),
            Language::GwichIn => Some("gwi"),
            Language::Haida => Some("hai"),
            Language::Haitian => Some("hat"),
            Language::Hausa => Some("hau"),
            Language::Hawaiian => Some("haw"),
            Language::Hebrew => Some("heb"),
            Language::Herero => Some("her"),
            Language::Hiligaynon => Some("hil"),
            Language::HimachaliLanguages => Some("him"),
            Language::Hindi => Some("hin"),
            Language::HiriMotu => Some("hmo"),
            Language::Hittite => Some("hit"),
            Language::Hmong => Some("hmn"),
            Language::Hungarian => Some("hun"),
            Language::Hupa => Some("hup"),
            Language::Iban => Some("iba"),
            Language::Icelandic => Some("ice"),
            Language::Ido => Some("ido"),
            Language::Igbo => Some("ibo"),
            Language::IjoLanguages => Some("ijo"),
            Language::Iloko => Some("ilo"),
            Language::InariSami => Some("smn"),
            Language::IndicLanguages => Some("inc"),
            Language::IndoEuropeanLanguages => Some("ine"),
            Language::Indonesian => Some("ind"),
            Language::Ingush => Some("inh"),
            Language::Interlingua => Some("ina"),
            Language::Interlingue => Some("ile"),
            Language::Inuktitut => Some("iku"),
            Language::Inupiaq => Some("ipk"),
            Language::IranianLanguages => Some("ira"),
            Language::Irish => Some("gle"),
            Language::IrishMiddle => Some("mga"),
            Language::IrishOld => Some("sga"),
            Language::IroquoianLanguages => Some("iro"),
            Language::Italian => Some("ita"),
            Language::Japanese => Some("jpn"),
            Language::Javanese => Some("jav"),
            Language::JudeoArabic => Some("jrb"),
            Language::JudeoPersian => Some("jpr"),
            Language::Kabardian => Some("kbd"),
            Language::Kabyle => Some("kab"),
            Language::Kachin => Some("kac"),
            Language::Kalaallisut => Some("kal"),
            Language::Kalmyk => Some("xal"),
            Language::Kamba => Some("kam"),
            Language::Kannada => Some("kan"),
            Language::Kanuri => Some("kau"),
            Language::KaraKalpak => Some("kaa"),
            Language::KarachayBalkar => Some("krc"),
            Language::Karelian => Some("krl"),
            Language::KarenLanguages => Some("kar"),
            Language::Kashmiri => Some("kas"),
            Language::Kashubian => Some("csb"),
            Language::Kawi => Some("kaw"),
            Language::Kazakh => Some("kaz"),
            Language::Khasi => Some("kha"),
            Language::KhoisanLanguages => Some("khi"),
            Language::Khotanese => Some("kho"),
            Language::Kikuyu => Some("kik"),
            Language::Kimbundu => Some("kmb"),
            Language::Kinyarwanda => Some("kin"),
            Language::Kirghiz => Some("kir"),
            Language::Klingon => Some("tlh"),
            Language::Komi => Some("kom"),
            Language::Kongo => Some("kon"),
            Language::Konkani => Some("kok"),
            Language::Korean => Some("kor"),
            Language::Kosraean => Some("kos"),
            Language::Kpelle => Some("kpe"),
            Language::KruLanguages => Some("kro"),
            Language::Kuanyama => Some("kua"),
            Language::Kumyk => Some("kum"),
            Language::Kurdish => Some("kur"),
            Language::Kurukh => Some("kru"),
            Language::Kutenai => Some("kut"),
            Language::Ladino => Some("lad"),
            Language::Lahnda => Some("lah"),
            Language::Lamba => Some("lam"),
            Language::LandDayakLanguages => Some("day"),
            Language::Lao => Some("lao"),
            Language::Latin => Some("lat"),
            Language::Latvian => Some("lav"),
            Language::Lezghian => Some("lez"),
            Language::Limburgan => Some("lim"),
            Language::Lingala => Some("lin"),
            Language::Lithuanian => Some("lit"),
            Language::Lojban => Some("jbo"),
            Language::LowGerman => Some("nds"),
            Language::LowerSorbian => Some("dsb"),
            Language::Lozi => Some("loz"),
            Language::LubaKatanga => Some("lub"),
            Language::LubaLulua => Some("lua"),
            Language::Luiseno => Some("lui"),
            Language::LuleSami => Some("smj"),
            Language::Lunda => Some("lun"),
            Language::LuoKenyaAndTanzania => Some("luo"),
            Language::Lushai => Some("lus"),
            Language::Luxembourgish => Some("ltz"),
            Language::Macedonian => Some("mac"),
            Language::Madurese => Some("mad"),
            Language::Magahi => Some("mag"),
            Language::Maithili => Some("mai"),
            Language::Makasar => Some("mak"),
            Language::Malagasy => Some("mlg"),
            Language::Malay => Some("may"),
            Language::Malayalam => Some("mal"),
            Language::Maltese => Some("mlt"),
            Language::Manchu => Some("mnc"),
            Language::Mandar => Some("mdr"),
            Language::Mandingo => Some("man"),
            Language::Manipuri => Some("mni"),
            Language::ManoboLanguages => Some("mno"),
            Language::Manx => Some("glv"),
            Language::Maori => Some("mao"),
            Language::Mapudungun => Some("arn"),
            Language::Marathi => Some("mar"),
            Language::Mari => Some("chm"),
            Language::Marshallese => Some("mah"),
            Language::Marwari => Some("mwr"),
            Language::Masai => Some("mas"),
            Language::MayanLanguages => Some("myn"),
            Language::Mende => Some("men"),
            Language::MiKmaq => Some("mic"),
            Language::Minangkabau => Some("min"),
            Language::Mirandese => Some("mwl"),
            Language::Mohawk => Some("moh"),
            Language::Moksha => Some("mdf"),
            Language::MonKhmerLanguages => Some("mkh"),
            Language::Mongo => Some("lol"),
            Language::Mongolian => Some("mon"),
            Language::Montenegrin => Some("cnr"),
            Language::Mossi => Some("mos"),
            Language::MultipleLanguages => Some("mul"),
            Language::MundaLanguages => Some("mun"),
            Language::NKo => Some("nqo"),
            Language::NahuatlLanguages => Some("nah"),
            Language::Nauru => Some("nau"),
            Language::Navajo => Some("nav"),
            Language::NdebeleNorth => Some("nde"),
            Language::NdebeleSouth => Some("nbl"),
            Language::Ndonga => Some("ndo"),
            Language::Neapolitan => Some("nap"),
            Language::NepalBhasa => Some("new"),
            Language::Nepali => Some("nep"),
            Language::Nias => Some("nia"),
            Language::NigerKordofanianLanguages => Some("nic"),
            Language::NiloSaharanLanguages => Some("ssa"),
            Language::Niuean => Some("niu"),
            Language::NoLinguisticContent => Some("zxx"),
            Language::Nogai => Some("nog"),
            Language::NorseOld => Some("non"),
            Language::NorthAmericanIndianLanguages => Some("nai"),
            Language::NorthernFrisian => Some("frr"),
            Language::NorthernSami => Some("sme"),
            Language::Norwegian => Some("nor"),
            Language::NorwegianNynorsk => Some("nno"),
            Language::NubianLanguages => Some("nub"),
            Language::Nyamwezi => Some("nym"),
            Language::Nyankole => Some("nyn"),
            Language::Nyoro => Some("nyo"),
            Language::Nzima => Some("nzi"),
            Language::Occitan => Some("oci"),
            Language::OfficialAramaic => Some("arc"),
            Language::Ojibwa => Some("oji"),
            Language::Oriya => Some("ori"),
            Language::Oromo => Some("orm"),
            Language::Osage => Some("osa"),
            Language::Ossetian => Some("oss"),
            Language::OtomianLanguages => Some("oto"),
            Language::Pahlavi => Some("pal"),
            Language::Palauan => Some("pau"),
            Language::Pali => Some("pli"),
            Language::Pampanga => Some("pam"),
            Language::Pangasinan => Some("pag"),
            Language::Panjabi => Some("pan"),
            Language::Papiamento => Some("pap"),
            Language::PapuanLanguages => Some("paa"),
            Language::Pedi => Some("nso"),
            Language::Persian => Some("per"),
            Language::PersianOld => Some("peo"),
            Language::PhilippineLanguages => Some("phi"),
            Language::Phoenician => Some("phn"),
            Language::Pohnpeian => Some("pon"),
            Language::Polish => Some("pol"),
            Language::Portuguese => Some("por"),
            Language::PrakritLanguages => Some("pra"),
            Language::ProvencalOld => Some("pro"),
            Language::Pushto => Some("pus"),
            Language::Quechua => Some("que"),
            Language::Rajasthani => Some("raj"),
            Language::Rapanui => Some("rap"),
            Language::Rarotongan => Some("rar"),
            Language::RomanceLanguages => Some("roa"),
            Language::Romanian => Some("rum"),
            Language::Romansh => Some("roh"),
            Language::Romany => Some("rom"),
            Language::Rundi => Some("run"),
            Language::Russian => Some("rus"),
            Language::SalishanLanguages => Some("sal"),
            Language::SamaritanAramaic => Some("sam"),
            Language::SamiLanguages => Some("smi"),
            Language::Samoan => Some("smo"),
            Language::Sandawe => Some("sad"),
            Language::Sango => Some("sag"),
            Language::Sanskrit => Some("san"),
            Language::Santali => Some("sat"),
            Language::Sardinian => Some("srd"),
            Language::Sasak => Some("sas"),
            Language::Scots => Some("sco"),
            Language::Selkup => Some("sel"),
            Language::SemiticLanguages => Some("sem"),
            Language::Serbian => Some("srp"),
            Language::Serer => Some("srr"),
            Language::Shan => Some("shn"),
            Language::Shona => Some("sna"),
            Language::SichuanYi => Some("iii"),
            Language::Sicilian => Some("scn"),
            Language::Sidamo => Some("sid"),
            Language::SignLanguages => Some("sgn"),
            Language::Siksika => Some("bla"),
            Language::Sindhi => Some("snd"),
            Language::Sinhala => Some("sin"),
            Language::SinoTibetanLanguages => Some("sit"),
            Language::SiouanLanguages => Some("sio"),
            Language::SkoltSami => Some("sms"),
            Language::SlaveAthapascan => Some("den"),
            Language::SlavicLanguages => Some("sla"),
            Language::Slovak => Some("slo"),
            Language::Slovenian => Some("slv"),
            Language::Sogdian => Some("sog"),
            Language::Somali => Some("som"),
            Language::SonghaiLanguages => Some("son"),
            Language::Soninke => Some("snk"),
            Language::SorbianLanguages => Some("wen"),
            Language::SothoSouthern => Some("sot"),
            Language::SouthAmericanIndianLanguages => Some("sai"),
            Language::SouthernAltai => Some("alt"),
            Language::SouthernSami => Some("sma"),
            Language::Spanish => Some("spa"),
            Language::SrananTongo => Some("srn"),
            Language::StandardMoroccanTamazight => Some("zgh"),
            Language::Sukuma => Some("suk"),
            Language::Sumerian => Some("sux"),
            Language::Sundanese => Some("sun"),
            Language::Susu => Some("sus"),
            Language::Swahili => Some("swa"),
            Language::Swati => Some("ssw"),
            Language::Swedish => Some("swe"),
            Language::SwissGerman => Some("gsw"),
            Language::Syriac => Some("syr"),
            Language::Tagalog => Some("tgl"),
            Language::Tahitian => Some("tah"),
            Language::TaiLanguages => Some("tai"),
            Language::Tajik => Some("tgk"),
            Language::Tamashek => Some("tmh"),
            Language::Tamil => Some("tam"),
            Language::Tatar => Some("tat"),
            Language::Telugu => Some("tel"),
            Language::Tereno => Some("ter"),
            Language::Tetum => Some("tet"),
            Language::Thai => Some("tha"),
            Language::Tibetan => Some("tib"),
            Language::Tigre => Some("tig"),
            Language::Tigrinya => Some("tir"),
            Language::Timne => Some("tem"),
            Language::Tiv => Some("tiv"),
            Language::Tlingit => Some("tli"),
            Language::TokPisin => Some("tpi"),
            Language::Tokelau => Some("tkl"),
            Language::TongaNyasa => Some("tog"),
            Language::TongaTongaIslands => Some("ton"),
            Language::Tsimshian => Some("tsi"),
            Language::Tsonga => Some("tso"),
            Language::Tswana => Some("tsn"),
            Language::Tumbuka => Some("tum"),
            Language::TupiLanguages => Some("tup"),
            Language::Turkish => Some("tur"),
            Language::TurkishOttoman => Some("ota"),
            Language::Turkmen => Some("tuk"),
            Language::Tuvalu => Some("tvl"),
            Language::Tuvinian => Some("tyv"),
            Language::Twi => Some("twi"),
            Language::Udmurt => Some("udm"),
            Language::Ugaritic => Some("uga"),
            Language::Uighur => Some("uig"),
            Language::Ukrainian => Some("ukr"),
            Language::Umbundu => Some("umb"),
            Language::UncodedLanguages => Some("mis"),
            Language::Undetermined => Some("und"),
            Language::UpperSorbian => Some("hsb"),
            Language::Urdu => Some("urd"),
            Language::Uzbek => Some("uzb"),
            Language::Vai => Some("vai"),
            Language::Venda => Some("ven"),
            Language::Vietnamese => Some("vie"),
            Language::Volapuk => Some("vol"),
            Language::Votic => Some("vot"),
            Language::WakashanLanguages => Some("wak"),
            Language::Walloon => Some("wln"),
            Language::Waray => Some("war"),
            Language::Washo => Some("was"),
            Language::Welsh => Some("wel"),
            Language::WesternFrisian => Some("fry"),
            Language::Wolaitta => Some("wal"),
            Language::Wolof => Some("wol"),
            Language::Xhosa => Some("xho"),
            Language::Yakut => Some("sah"),
            Language::Yao => Some("yao"),
            Language::Yapese => Some("yap"),
            Language::Yiddish => Some("yid"),
            Language::Yoruba => Some("yor"),
            Language::YupikLanguages => Some("ypk"),
            Language::ZandeLanguages => Some("znd"),
            Language::Zapotec => Some("zap"),
            Language::Zaza => Some("zza"),
            Language::Zenaga => Some("zen"),
            Language::Zhuang => Some("zha"),
            Language::Zulu => Some("zul"),
            Language::Zuni => Some("zun"),
            _ => None,
        }
    }
/// Returns the ISO-639 alpha3-t code
/// ```
/// use iso_639::Language;
///
/// let french = Language::French.alpha3_t().unwrap();
/// ```

    pub fn alpha3_t(&self) -> Option<&str> {
        match self {
            Language::Albanian => Some("sqi"),
            Language::Armenian => Some("hye"),
            Language::Basque => Some("eus"),
            Language::Burmese => Some("mya"),
            Language::Chinese => Some("zho"),
            Language::Czech => Some("ces"),
            Language::Dutch => Some("nld"),
            Language::French => Some("fra"),
            Language::Georgian => Some("kat"),
            Language::German => Some("deu"),
            Language::GreekModern => Some("ell"),
            Language::Icelandic => Some("isl"),
            Language::Macedonian => Some("mkd"),
            Language::Malay => Some("msa"),
            Language::Maori => Some("mri"),
            Language::Persian => Some("fas"),
            Language::Romanian => Some("ron"),
            Language::Slovak => Some("slk"),
            Language::Tibetan => Some("bod"),
            Language::Welsh => Some("cym"),
            _ => None,
        }
    }
/// Returns the English name
/// ```
/// use iso_639::Language;
///
/// let french = Language::French.english_name();
/// ```

    pub fn english_name(&self) -> &str {
        match self {
            Language::Abkhazian => "Abkhazian",
            Language::Achinese => "Achinese",
            Language::Acoli => "Acoli",
            Language::Adangme => "Adangme",
            Language::Adyghe => "Adyghe; Adygei",
            Language::Afar => "Afar",
            Language::Afrihili => "Afrihili",
            Language::Afrikaans => "Afrikaans",
            Language::AfroAsiaticLanguages => "Afro-Asiatic languages",
            Language::Ainu => "Ainu",
            Language::Akan => "Akan",
            Language::Akkadian => "Akkadian",
            Language::Albanian => "Albanian",
            Language::Aleut => "Aleut",
            Language::AlgonquianLanguages => "Algonquian languages",
            Language::AltaicLanguages => "Altaic languages",
            Language::Amharic => "Amharic",
            Language::Angika => "Angika",
            Language::ApacheLanguages => "Apache languages",
            Language::Arabic => "Arabic",
            Language::Aragonese => "Aragonese",
            Language::Arapaho => "Arapaho",
            Language::Arawak => "Arawak",
            Language::Armenian => "Armenian",
            Language::Aromanian => "Aromanian; Arumanian; Macedo-Romanian",
            Language::ArtificialLanguages => "Artificial languages",
            Language::Assamese => "Assamese",
            Language::Asturian => "Asturian; Bable; Leonese; Asturleonese",
            Language::AthapascanLanguages => "Athapascan languages",
            Language::AustralianLanguages => "Australian languages",
            Language::AustronesianLanguages => "Austronesian languages",
            Language::Avaric => "Avaric",
            Language::Avestan => "Avestan",
            Language::Awadhi => "Awadhi",
            Language::Aymara => "Aymara",
            Language::Azerbaijani => "Azerbaijani",
            Language::Balinese => "Balinese",
            Language::BalticLanguages => "Baltic languages",
            Language::Baluchi => "Baluchi",
            Language::Bambara => "Bambara",
            Language::BamilekeLanguages => "Bamileke languages",
            Language::BandaLanguages => "Banda languages",
            Language::BantuLanguages => "Bantu languages",
            Language::Basa => "Basa",
            Language::Bashkir => "Bashkir",
            Language::Basque => "Basque",
            Language::BatakLanguages => "Batak languages",
            Language::Beja => "Beja; Bedawiyet",
            Language::Belarusian => "Belarusian",
            Language::Bemba => "Bemba",
            Language::Bengali => "Bengali",
            Language::BerberLanguages => "Berber languages",
            Language::Bhojpuri => "Bhojpuri",
            Language::BihariLanguages => "Bihari languages",
            Language::Bikol => "Bikol",
            Language::Bini => "Bini; Edo",
            Language::Bislama => "Bislama",
            Language::Blin => "Blin; Bilin",
            Language::Blissymbols => "Blissymbols; Blissymbolics; Bliss",
            Language::BokmålNorwegian => "Bokmål, Norwegian; Norwegian Bokmål",
            Language::Bosnian => "Bosnian",
            Language::Braj => "Braj",
            Language::Breton => "Breton",
            Language::Buginese => "Buginese",
            Language::Bulgarian => "Bulgarian",
            Language::Buriat => "Buriat",
            Language::Burmese => "Burmese",
            Language::Caddo => "Caddo",
            Language::Catalan => "Catalan; Valencian",
            Language::CaucasianLanguages => "Caucasian languages",
            Language::Cebuano => "Cebuano",
            Language::CelticLanguages => "Celtic languages",
            Language::CentralAmericanIndianLanguages => "Central American Indian languages",
            Language::CentralKhmer => "Central Khmer",
            Language::Chagatai => "Chagatai",
            Language::ChamicLanguages => "Chamic languages",
            Language::Chamorro => "Chamorro",
            Language::Chechen => "Chechen",
            Language::Cherokee => "Cherokee",
            Language::Cheyenne => "Cheyenne",
            Language::Chibcha => "Chibcha",
            Language::Chichewa => "Chichewa; Chewa; Nyanja",
            Language::Chinese => "Chinese",
            Language::Chinook => "Chinook jargon",
            Language::Chipewyan => "Chipewyan; Dene Suline",
            Language::Choctaw => "Choctaw",
            Language::ChurchSlavic => "Church Slavic; Old Slavonic; Church Slavonic; Old Bulgarian; Old Church Slavonic",
            Language::Chuukese => "Chuukese",
            Language::Chuvash => "Chuvash",
            Language::ClassicalNewari => "Classical Newari; Old Newari; Classical Nepal Bhasa",
            Language::ClassicalSyriac => "Classical Syriac",
            Language::Coptic => "Coptic",
            Language::Cornish => "Cornish",
            Language::Corsican => "Corsican",
            Language::Cree => "Cree",
            Language::Creek => "Creek",
            Language::CreolesAndPidgins => "Creoles and pidgins",
            Language::CreolesAndPidginsEnglishBased => "Creoles and pidgins, English based",
            Language::CreolesAndPidginsFrenchBased => "Creoles and pidgins, French-based",
            Language::CreolesAndPidginsPortugueseBased => "Creoles and pidgins, Portuguese-based",
            Language::CrimeanTatar => "Crimean Tatar; Crimean Turkish",
            Language::Croatian => "Croatian",
            Language::CushiticLanguages => "Cushitic languages",
            Language::Czech => "Czech",
            Language::Dakota => "Dakota",
            Language::Danish => "Danish",
            Language::Dargwa => "Dargwa",
            Language::Delaware => "Delaware",
            Language::Dinka => "Dinka",
            Language::Divehi => "Divehi; Dhivehi; Maldivian",
            Language::Dogri => "Dogri",
            Language::Dogrib => "Dogrib",
            Language::DravidianLanguages => "Dravidian languages",
            Language::Duala => "Duala",
            Language::Dutch => "Dutch; Flemish",
            Language::DutchMiddle => "Dutch, Middle (ca.1050-1350)",
            Language::Dyula => "Dyula",
            Language::Dzongkha => "Dzongkha",
            Language::EasternFrisian => "Eastern Frisian",
            Language::Efik => "Efik",
            Language::EgyptianAncient => "Egyptian (Ancient)",
            Language::Ekajuk => "Ekajuk",
            Language::Elamite => "Elamite",
            Language::English => "English",
            Language::EnglishMiddle => "English, Middle (1100-1500)",
            Language::EnglishOld => "English, Old (ca.450-1100)",
            Language::Erzya => "Erzya",
            Language::Esperanto => "Esperanto",
            Language::Estonian => "Estonian",
            Language::Ewe => "Ewe",
            Language::Ewondo => "Ewondo",
            Language::Fang => "Fang",
            Language::Fanti => "Fanti",
            Language::Faroese => "Faroese",
            Language::Fijian => "Fijian",
            Language::Filipino => "Filipino; Pilipino",
            Language::Finnish => "Finnish",
            Language::FinnoUgrianLanguages => "Finno-Ugrian languages",
            Language::Fon => "Fon",
            Language::French => "French",
            Language::FrenchMiddle => "French, Middle (ca.1400-1600)",
            Language::FrenchOld => "French, Old (842-ca.1400)",
            Language::Friulian => "Friulian",
            Language::Fulah => "Fulah",
            Language::Ga => "Ga",
            Language::Gaelic => "Gaelic; Scottish Gaelic",
            Language::GalibiCarib => "Galibi Carib",
            Language::Galician => "Galician",
            Language::Ganda => "Ganda",
            Language::Gayo => "Gayo",
            Language::Gbaya => "Gbaya",
            Language::Geez => "Geez",
            Language::Georgian => "Georgian",
            Language::German => "German",
            Language::GermanMiddleHigh => "German, Middle High (ca.1050-1500)",
            Language::GermanOldHigh => "German, Old High (ca.750-1050)",
            Language::GermanicLanguages => "Germanic languages",
            Language::Gilbertese => "Gilbertese",
            Language::Gondi => "Gondi",
            Language::Gorontalo => "Gorontalo",
            Language::Gothic => "Gothic",
            Language::Grebo => "Grebo",
            Language::GreekAncient => "Greek, Ancient (to 1453)",
            Language::GreekModern => "Greek, Modern (1453-)",
            Language::Guarani => "Guarani",
            Language::Gujarati => "Gujarati",
            Language::GwichIn => "Gwich'in",
            Language::Haida => "Haida",
            Language::Haitian => "Haitian; Haitian Creole",
            Language::Hausa => "Hausa",
            Language::Hawaiian => "Hawaiian",
            Language::Hebrew => "Hebrew",
            Language::Herero => "Herero",
            Language::Hiligaynon => "Hiligaynon",
            Language::HimachaliLanguages => "Himachali languages; Western Pahari languages",
            Language::Hindi => "Hindi",
            Language::HiriMotu => "Hiri Motu",
            Language::Hittite => "Hittite",
            Language::Hmong => "Hmong; Mong",
            Language::Hungarian => "Hungarian",
            Language::Hupa => "Hupa",
            Language::Iban => "Iban",
            Language::Icelandic => "Icelandic",
            Language::Ido => "Ido",
            Language::Igbo => "Igbo",
            Language::IjoLanguages => "Ijo languages",
            Language::Iloko => "Iloko",
            Language::InariSami => "Inari Sami",
            Language::IndicLanguages => "Indic languages",
            Language::IndoEuropeanLanguages => "Indo-European languages",
            Language::Indonesian => "Indonesian",
            Language::Ingush => "Ingush",
            Language::Interlingua => "Interlingua (International Auxiliary Language Association)",
            Language::Interlingue => "Interlingue; Occidental",
            Language::Inuktitut => "Inuktitut",
            Language::Inupiaq => "Inupiaq",
            Language::IranianLanguages => "Iranian languages",
            Language::Irish => "Irish",
            Language::IrishMiddle => "Irish, Middle (900-1200)",
            Language::IrishOld => "Irish, Old (to 900)",
            Language::IroquoianLanguages => "Iroquoian languages",
            Language::Italian => "Italian",
            Language::Japanese => "Japanese",
            Language::Javanese => "Javanese",
            Language::JudeoArabic => "Judeo-Arabic",
            Language::JudeoPersian => "Judeo-Persian",
            Language::Kabardian => "Kabardian",
            Language::Kabyle => "Kabyle",
            Language::Kachin => "Kachin; Jingpho",
            Language::Kalaallisut => "Kalaallisut; Greenlandic",
            Language::Kalmyk => "Kalmyk; Oirat",
            Language::Kamba => "Kamba",
            Language::Kannada => "Kannada",
            Language::Kanuri => "Kanuri",
            Language::KaraKalpak => "Kara-Kalpak",
            Language::KarachayBalkar => "Karachay-Balkar",
            Language::Karelian => "Karelian",
            Language::KarenLanguages => "Karen languages",
            Language::Kashmiri => "Kashmiri",
            Language::Kashubian => "Kashubian",
            Language::Kawi => "Kawi",
            Language::Kazakh => "Kazakh",
            Language::Khasi => "Khasi",
            Language::KhoisanLanguages => "Khoisan languages",
            Language::Khotanese => "Khotanese; Sakan",
            Language::Kikuyu => "Kikuyu; Gikuyu",
            Language::Kimbundu => "Kimbundu",
            Language::Kinyarwanda => "Kinyarwanda",
            Language::Kirghiz => "Kirghiz; Kyrgyz",
            Language::Klingon => "Klingon; tlhIngan-Hol",
            Language::Komi => "Komi",
            Language::Kongo => "Kongo",
            Language::Konkani => "Konkani",
            Language::Korean => "Korean",
            Language::Kosraean => "Kosraean",
            Language::Kpelle => "Kpelle",
            Language::KruLanguages => "Kru languages",
            Language::Kuanyama => "Kuanyama; Kwanyama",
            Language::Kumyk => "Kumyk",
            Language::Kurdish => "Kurdish",
            Language::Kurukh => "Kurukh",
            Language::Kutenai => "Kutenai",
            Language::Ladino => "Ladino",
            Language::Lahnda => "Lahnda",
            Language::Lamba => "Lamba",
            Language::LandDayakLanguages => "Land Dayak languages",
            Language::Lao => "Lao",
            Language::Latin => "Latin",
            Language::Latvian => "Latvian",
            Language::Lezghian => "Lezghian",
            Language::Limburgan => "Limburgan; Limburger; Limburgish",
            Language::Lingala => "Lingala",
            Language::Lithuanian => "Lithuanian",
            Language::Lojban => "Lojban",
            Language::LowGerman => "Low German; Low Saxon; German, Low; Saxon, Low",
            Language::LowerSorbian => "Lower Sorbian",
            Language::Lozi => "Lozi",
            Language::LubaKatanga => "Luba-Katanga",
            Language::LubaLulua => "Luba-Lulua",
            Language::Luiseno => "Luiseno",
            Language::LuleSami => "Lule Sami",
            Language::Lunda => "Lunda",
            Language::LuoKenyaAndTanzania => "Luo (Kenya and Tanzania)",
            Language::Lushai => "Lushai",
            Language::Luxembourgish => "Luxembourgish; Letzeburgesch",
            Language::Macedonian => "Macedonian",
            Language::Madurese => "Madurese",
            Language::Magahi => "Magahi",
            Language::Maithili => "Maithili",
            Language::Makasar => "Makasar",
            Language::Malagasy => "Malagasy",
            Language::Malay => "Malay",
            Language::Malayalam => "Malayalam",
            Language::Maltese => "Maltese",
            Language::Manchu => "Manchu",
            Language::Mandar => "Mandar",
            Language::Mandingo => "Mandingo",
            Language::Manipuri => "Manipuri",
            Language::ManoboLanguages => "Manobo languages",
            Language::Manx => "Manx",
            Language::Maori => "Maori",
            Language::Mapudungun => "Mapudungun; Mapuche",
            Language::Marathi => "Marathi",
            Language::Mari => "Mari",
            Language::Marshallese => "Marshallese",
            Language::Marwari => "Marwari",
            Language::Masai => "Masai",
            Language::MayanLanguages => "Mayan languages",
            Language::Mende => "Mende",
            Language::MiKmaq => "Mi'kmaq; Micmac",
            Language::Minangkabau => "Minangkabau",
            Language::Mirandese => "Mirandese",
            Language::Mohawk => "Mohawk",
            Language::Moksha => "Moksha",
            Language::MonKhmerLanguages => "Mon-Khmer languages",
            Language::Mongo => "Mongo",
            Language::Mongolian => "Mongolian",
            Language::Montenegrin => "Montenegrin",
            Language::Mossi => "Mossi",
            Language::MultipleLanguages => "Multiple languages",
            Language::MundaLanguages => "Munda languages",
            Language::NKo => "N'Ko",
            Language::NahuatlLanguages => "Nahuatl languages",
            Language::Nauru => "Nauru",
            Language::Navajo => "Navajo; Navaho",
            Language::NdebeleNorth => "Ndebele, North; North Ndebele",
            Language::NdebeleSouth => "Ndebele, South; South Ndebele",
            Language::Ndonga => "Ndonga",
            Language::Neapolitan => "Neapolitan",
            Language::NepalBhasa => "Nepal Bhasa; Newari",
            Language::Nepali => "Nepali",
            Language::Nias => "Nias",
            Language::NigerKordofanianLanguages => "Niger-Kordofanian languages",
            Language::NiloSaharanLanguages => "Nilo-Saharan languages",
            Language::Niuean => "Niuean",
            Language::NoLinguisticContent => "No linguistic content; Not applicable",
            Language::Nogai => "Nogai",
            Language::NorseOld => "Norse, Old",
            Language::NorthAmericanIndianLanguages => "North American Indian languages",
            Language::NorthernFrisian => "Northern Frisian",
            Language::NorthernSami => "Northern Sami",
            Language::Norwegian => "Norwegian",
            Language::NorwegianNynorsk => "Norwegian Nynorsk; Nynorsk, Norwegian",
            Language::NubianLanguages => "Nubian languages",
            Language::Nyamwezi => "Nyamwezi",
            Language::Nyankole => "Nyankole",
            Language::Nyoro => "Nyoro",
            Language::Nzima => "Nzima",
            Language::Occitan => "Occitan (post 1500)",
            Language::OfficialAramaic => "Official Aramaic (700-300 BCE); Imperial Aramaic (700-300 BCE)",
            Language::Ojibwa => "Ojibwa",
            Language::Oriya => "Oriya",
            Language::Oromo => "Oromo",
            Language::Osage => "Osage",
            Language::Ossetian => "Ossetian; Ossetic",
            Language::OtomianLanguages => "Otomian languages",
            Language::Pahlavi => "Pahlavi",
            Language::Palauan => "Palauan",
            Language::Pali => "Pali",
            Language::Pampanga => "Pampanga; Kapampangan",
            Language::Pangasinan => "Pangasinan",
            Language::Panjabi => "Panjabi; Punjabi",
            Language::Papiamento => "Papiamento",
            Language::PapuanLanguages => "Papuan languages",
            Language::Pedi => "Pedi; Sepedi; Northern Sotho",
            Language::Persian => "Persian",
            Language::PersianOld => "Persian, Old (ca.600-400 B.C.)",
            Language::PhilippineLanguages => "Philippine languages",
            Language::Phoenician => "Phoenician",
            Language::Pohnpeian => "Pohnpeian",
            Language::Polish => "Polish",
            Language::Portuguese => "Portuguese",
            Language::PrakritLanguages => "Prakrit languages",
            Language::ProvencalOld => "Provençal, Old (to 1500); Occitan, Old (to 1500)",
            Language::Pushto => "Pushto; Pashto",
            Language::Quechua => "Quechua",
            Language::Rajasthani => "Rajasthani",
            Language::Rapanui => "Rapanui",
            Language::Rarotongan => "Rarotongan; Cook Islands Maori",
            Language::RomanceLanguages => "Romance languages",
            Language::Romanian => "Romanian; Moldavian; Moldovan",
            Language::Romansh => "Romansh",
            Language::Romany => "Romany",
            Language::Rundi => "Rundi",
            Language::Russian => "Russian",
            Language::SalishanLanguages => "Salishan languages",
            Language::SamaritanAramaic => "Samaritan Aramaic",
            Language::SamiLanguages => "Sami languages",
            Language::Samoan => "Samoan",
            Language::Sandawe => "Sandawe",
            Language::Sango => "Sango",
            Language::Sanskrit => "Sanskrit",
            Language::Santali => "Santali",
            Language::Sardinian => "Sardinian",
            Language::Sasak => "Sasak",
            Language::Scots => "Scots",
            Language::Selkup => "Selkup",
            Language::SemiticLanguages => "Semitic languages",
            Language::Serbian => "Serbian",
            Language::Serer => "Serer",
            Language::Shan => "Shan",
            Language::Shona => "Shona",
            Language::SichuanYi => "Sichuan Yi; Nuosu",
            Language::Sicilian => "Sicilian",
            Language::Sidamo => "Sidamo",
            Language::SignLanguages => "Sign Languages",
            Language::Siksika => "Siksika",
            Language::Sindhi => "Sindhi",
            Language::Sinhala => "Sinhala; Sinhalese",
            Language::SinoTibetanLanguages => "Sino-Tibetan languages",
            Language::SiouanLanguages => "Siouan languages",
            Language::SkoltSami => "Skolt Sami",
            Language::SlaveAthapascan => "Slave (Athapascan)",
            Language::SlavicLanguages => "Slavic languages",
            Language::Slovak => "Slovak",
            Language::Slovenian => "Slovenian",
            Language::Sogdian => "Sogdian",
            Language::Somali => "Somali",
            Language::SonghaiLanguages => "Songhai languages",
            Language::Soninke => "Soninke",
            Language::SorbianLanguages => "Sorbian languages",
            Language::SothoSouthern => "Sotho, Southern",
            Language::SouthAmericanIndianLanguages => "South American Indian languages",
            Language::SouthernAltai => "Southern Altai",
            Language::SouthernSami => "Southern Sami",
            Language::Spanish => "Spanish; Castilian",
            Language::SrananTongo => "Sranan Tongo",
            Language::StandardMoroccanTamazight => "Standard Moroccan Tamazight",
            Language::Sukuma => "Sukuma",
            Language::Sumerian => "Sumerian",
            Language::Sundanese => "Sundanese",
            Language::Susu => "Susu",
            Language::Swahili => "Swahili",
            Language::Swati => "Swati",
            Language::Swedish => "Swedish",
            Language::SwissGerman => "Swiss German; Alemannic; Alsatian",
            Language::Syriac => "Syriac",
            Language::Tagalog => "Tagalog",
            Language::Tahitian => "Tahitian",
            Language::TaiLanguages => "Tai languages",
            Language::Tajik => "Tajik",
            Language::Tamashek => "Tamashek",
            Language::Tamil => "Tamil",
            Language::Tatar => "Tatar",
            Language::Telugu => "Telugu",
            Language::Tereno => "Tereno",
            Language::Tetum => "Tetum",
            Language::Thai => "Thai",
            Language::Tibetan => "Tibetan",
            Language::Tigre => "Tigre",
            Language::Tigrinya => "Tigrinya",
            Language::Timne => "Timne",
            Language::Tiv => "Tiv",
            Language::Tlingit => "Tlingit",
            Language::TokPisin => "Tok Pisin",
            Language::Tokelau => "Tokelau",
            Language::TongaNyasa => "Tonga (Nyasa)",
            Language::TongaTongaIslands => "Tonga (Tonga Islands)",
            Language::Tsimshian => "Tsimshian",
            Language::Tsonga => "Tsonga",
            Language::Tswana => "Tswana",
            Language::Tumbuka => "Tumbuka",
            Language::TupiLanguages => "Tupi languages",
            Language::Turkish => "Turkish",
            Language::TurkishOttoman => "Turkish, Ottoman (1500-1928)",
            Language::Turkmen => "Turkmen",
            Language::Tuvalu => "Tuvalu",
            Language::Tuvinian => "Tuvinian",
            Language::Twi => "Twi",
            Language::Udmurt => "Udmurt",
            Language::Ugaritic => "Ugaritic",
            Language::Uighur => "Uighur; Uyghur",
            Language::Ukrainian => "Ukrainian",
            Language::Umbundu => "Umbundu",
            Language::UncodedLanguages => "Uncoded languages",
            Language::Undetermined => "Undetermined",
            Language::UpperSorbian => "Upper Sorbian",
            Language::Urdu => "Urdu",
            Language::Uzbek => "Uzbek",
            Language::Vai => "Vai",
            Language::Venda => "Venda",
            Language::Vietnamese => "Vietnamese",
            Language::Volapuk => "Volapük",
            Language::Votic => "Votic",
            Language::WakashanLanguages => "Wakashan languages",
            Language::Walloon => "Walloon",
            Language::Waray => "Waray",
            Language::Washo => "Washo",
            Language::Welsh => "Welsh",
            Language::WesternFrisian => "Western Frisian",
            Language::Wolaitta => "Wolaitta; Wolaytta",
            Language::Wolof => "Wolof",
            Language::Xhosa => "Xhosa",
            Language::Yakut => "Yakut",
            Language::Yao => "Yao",
            Language::Yapese => "Yapese",
            Language::Yiddish => "Yiddish",
            Language::Yoruba => "Yoruba",
            Language::YupikLanguages => "Yupik languages",
            Language::ZandeLanguages => "Zande languages",
            Language::Zapotec => "Zapotec",
            Language::Zaza => "Zaza; Dimili; Dimli; Kirdki; Kirmanjki; Zazaki",
            Language::Zenaga => "Zenaga",
            Language::Zhuang => "Zhuang; Chuang",
            Language::Zulu => "Zulu",
            Language::Zuni => "Zuni",
        }
    }
}
