//! Fast arbitrary length prime number checker using the Miller-Rabin primality test algorithm
//!
//! This module implements the [Miller-Rabin primality
//! test](https://en.wikipedia.org/wiki/Miller%E2%80%93Rabin_primality_test) algorithm. Given an
//! arbitrary length integer specified within a string, apply the probabilistic algorithm to check
//! if the integer may be prime.
//!
//! # Examples
//!
//!```
//!extern crate is_prime;
//!
//!use is_prime::*;
//!
//!fn main() {
//!    // The first RSA Prime
//!    assert!(is_prime("37975227936943673922808872755445627854565536638199") == true);
//!
//!    // The first RSA Prime + 1
//!    assert!(is_prime("37975227936943673922808872755445627854565536638200") == false);
//!}
//!```

extern crate num_bigint;
extern crate num_traits;

use num_bigint::BigUint;
use std::str::FromStr;

/// Tests if the given integer within a string is prime
///
/// # Parameters
///
/// `n_str` is the string containing the integer to test its primality
///
/// # Examples
///
///```
///extern crate is_prime;
///
///use is_prime::*;
///
///fn main() {
///    // The first RSA Prime
///    assert!(is_prime("37975227936943673922808872755445627854565536638199") == true);
///
///    // The first RSA Prime + 1
///    assert!(is_prime("37975227936943673922808872755445627854565536638200") == false);
///}
///```
pub fn is_prime(n_str: &str) -> bool {
    // Translated from
    // https://rosettacode.org/wiki/Miller%E2%80%93Rabin_primality_test#Perl

    let n = BigUint::from_str(n_str).unwrap();

    if n < BigUint::from(2u32) {
        return false;
    }

    if n == BigUint::from(2u32) || n == BigUint::from(3u32) || n == BigUint::from(5u32) {
        return true;
    }

    if (&n % BigUint::from(2u32)) == BigUint::from(0u32) {
        return false;
    }

    let n_sub = n.clone() - BigUint::from(1u32);
    let mut exponent = n_sub.clone();
    let mut trials = 0;

    while (&exponent % BigUint::from(2u32)) == BigUint::from(0u32) {
        exponent /= 2u32;
        trials += 1;
    }

    'LOOP: for i in 1..((n.to_string().len()) + 2) {
        let mut result = bmodpow(&(BigUint::from(2u32) + i), &exponent, &n);

        if result == BigUint::from(1u32) || result == n_sub {
            continue;
        }

        for _ in 1..trials {
            result = result.pow(2) % &n;

            if result == BigUint::from(1u32) {
                return false;
            }

            if result == n_sub {
                continue 'LOOP;
            }
        }

        return false;
    }

    true
}

fn bmodpow(base: &BigUint, exponent: &BigUint, modulus: &BigUint) -> BigUint {
    // Translated from
    // http://search.cpan.org/~pjacklam/Math-BigInt-1.999810/lib/Math/BigInt.pm#Arithmetic_methods

    if *base == BigUint::from(0u32) {
        return match *exponent == BigUint::from(0u32) {
            true => BigUint::from(1u32),
            false => BigUint::from(0u32),
        };
    }

    if *modulus == BigUint::from(1u32) {
        return BigUint::from(0u32);
    }

    let exponent_in_binary = exponent.to_radix_le(2);
    let mut my_base = base.clone();
    let mut result = BigUint::from(1u32);

    for next_bit in exponent_in_binary {
        if next_bit == 1 {
            result = (result * my_base.clone()) % modulus;
        }

        my_base = my_base.pow(2) % modulus;
    }

    result
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn small_primes() {
        let small_primes = vec![
            2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83,
            89, 97, 101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179,
            181, 191, 193, 197, 199, 211, 223, 227, 229, 233, 239, 241, 251, 257, 263, 269, 271,
            277, 281, 283, 293, 307, 311, 313, 317, 331, 337, 347, 349, 353, 359, 367, 373, 379,
            383, 389, 397, 401, 409, 419, 421, 431, 433, 439, 443, 449, 457, 461, 463, 467, 479,
            487, 491, 499, 503, 509, 521, 523, 541, 547, 557, 563, 569, 571, 577, 587, 593, 599,
            601, 607, 613, 617, 619, 631, 641, 643, 647, 653, 659, 661, 673, 677, 683, 691, 701,
            709, 719, 727, 733, 739, 743, 751, 757, 761, 769, 773, 787, 797, 809, 811, 821, 823,
            827, 829, 839, 853, 857, 859, 863, 877, 881, 883, 887, 907, 911, 919, 929, 937, 941,
            947, 953, 967, 971, 977, 983, 991, 997, 1009, 1013, 1019, 1021, 1031, 1033, 1039, 1049,
            1051, 1061, 1063, 1069, 1087, 1091, 1093, 1097, 1103, 1109, 1117, 1123, 1129, 1151,
            1153, 1163, 1171, 1181, 1187, 1193, 1201, 1213, 1217, 1223, 1229, 1231, 1237, 1249,
            1259, 1277, 1279, 1283, 1289, 1291, 1297, 1301, 1303, 1307, 1319, 1321, 1327, 1361,
            1367, 1373, 1381, 1399, 1409, 1423, 1427, 1429, 1433, 1439, 1447, 1451, 1453, 1459,
            1471, 1481, 1483, 1487, 1489, 1493, 1499, 1511, 1523, 1531, 1543, 1549, 1553, 1559,
            1567, 1571, 1579, 1583, 1597, 1601, 1607, 1609, 1613, 1619, 1621, 1627, 1637, 1657,
            1663, 1667, 1669, 1693, 1697, 1699, 1709, 1721, 1723, 1733, 1741, 1747, 1753, 1759,
            1777, 1783, 1787, 1789, 1801, 1811, 1823, 1831, 1847, 1861, 1867, 1871, 1873, 1877,
            1879, 1889, 1901, 1907, 1913, 1931, 1933, 1949, 1951, 1973, 1979, 1987, 1993, 1997,
            1999, 2003, 2011, 2017, 2027, 2029, 2039, 2053, 2063, 2069, 2081, 2083, 2087, 2089,
            2099, 2111, 2113, 2129, 2131, 2137, 2141, 2143, 2153, 2161, 2179, 2203, 2207, 2213,
            2221, 2237, 2239, 2243, 2251, 2267, 2269, 2273, 2281, 2287, 2293, 2297, 2309, 2311,
            2333, 2339, 2341, 2347, 2351, 2357, 2371, 2377, 2381, 2383, 2389, 2393, 2399, 2411,
            2417, 2423, 2437, 2441, 2447, 2459, 2467, 2473, 2477, 2503, 2521, 2531, 2539, 2543,
            2549, 2551, 2557, 2579, 2591, 2593, 2609, 2617, 2621, 2633, 2647, 2657, 2659, 2663,
            2671, 2677, 2683, 2687, 2689, 2693, 2699, 2707, 2711, 2713, 2719, 2729, 2731, 2741,
            2749, 2753, 2767, 2777, 2789, 2791, 2797, 2801, 2803, 2819, 2833, 2837, 2843, 2851,
            2857, 2861, 2879, 2887, 2897, 2903, 2909, 2917, 2927, 2939, 2953, 2957, 2963, 2969,
            2971, 2999, 3001, 3011, 3019, 3023, 3037, 3041, 3049, 3061, 3067, 3079, 3083, 3089,
            3109, 3119, 3121, 3137, 3163, 3167, 3169, 3181, 3187, 3191, 3203, 3209, 3217, 3221,
            3229, 3251, 3253, 3257, 3259, 3271, 3299, 3301, 3307, 3313, 3319, 3323, 3329, 3331,
            3343, 3347, 3359, 3361, 3371, 3373, 3389, 3391, 3407, 3413, 3433, 3449, 3457, 3461,
            3463, 3467, 3469, 3491, 3499, 3511, 3517, 3527, 3529, 3533, 3539, 3541, 3547, 3557,
            3559, 3571, 3581, 3583, 3593, 3607, 3613, 3617, 3623, 3631, 3637, 3643, 3659, 3671,
            3673, 3677, 3691, 3697, 3701, 3709, 3719, 3727, 3733, 3739, 3761, 3767, 3769, 3779,
            3793, 3797, 3803, 3821, 3823, 3833, 3847, 3851, 3853, 3863, 3877, 3881, 3889, 3907,
            3911, 3917, 3919, 3923, 3929, 3931, 3943, 3947, 3967, 3989, 4001, 4003, 4007, 4013,
            4019, 4021, 4027, 4049, 4051, 4057, 4073, 4079, 4091, 4093, 4099, 4111, 4127, 4129,
            4133, 4139, 4153, 4157, 4159, 4177, 4201, 4211, 4217, 4219, 4229, 4231, 4241, 4243,
            4253, 4259, 4261, 4271, 4273, 4283, 4289, 4297, 4327, 4337, 4339, 4349, 4357, 4363,
            4373, 4391, 4397, 4409, 4421, 4423, 4441, 4447, 4451, 4457, 4463, 4481, 4483, 4493,
            4507, 4513, 4517, 4519, 4523, 4547, 4549, 4561, 4567, 4583, 4591, 4597, 4603, 4621,
            4637, 4639, 4643, 4649, 4651, 4657, 4663, 4673, 4679, 4691, 4703, 4721, 4723, 4729,
            4733, 4751, 4759, 4783, 4787, 4789, 4793, 4799, 4801, 4813, 4817, 4831, 4861, 4871,
            4877, 4889, 4903, 4909, 4919, 4931, 4933, 4937, 4943, 4951, 4957, 4967, 4969, 4973,
            4987, 4993, 4999, 5003, 5009, 5011, 5021, 5023, 5039, 5051, 5059, 5077, 5081, 5087,
            5099, 5101, 5107, 5113, 5119, 5147, 5153, 5167, 5171, 5179, 5189, 5197, 5209, 5227,
            5231, 5233, 5237, 5261, 5273, 5279, 5281, 5297, 5303, 5309, 5323, 5333, 5347, 5351,
            5381, 5387, 5393, 5399, 5407, 5413, 5417, 5419, 5431, 5437, 5441, 5443, 5449, 5471,
            5477, 5479, 5483, 5501, 5503, 5507, 5519, 5521, 5527, 5531, 5557, 5563, 5569, 5573,
            5581, 5591, 5623, 5639, 5641, 5647, 5651, 5653, 5657, 5659, 5669, 5683, 5689, 5693,
            5701, 5711, 5717, 5737, 5741, 5743, 5749, 5779, 5783, 5791, 5801, 5807, 5813, 5821,
            5827, 5839, 5843, 5849, 5851, 5857, 5861, 5867, 5869, 5879, 5881, 5897, 5903, 5923,
            5927, 5939, 5953, 5981, 5987, 6007, 6011, 6029, 6037, 6043, 6047, 6053, 6067, 6073,
            6079, 6089, 6091, 6101, 6113, 6121, 6131, 6133, 6143, 6151, 6163, 6173, 6197, 6199,
            6203, 6211, 6217, 6221, 6229, 6247, 6257, 6263, 6269, 6271, 6277, 6287, 6299, 6301,
            6311, 6317, 6323, 6329, 6337, 6343, 6353, 6359, 6361, 6367, 6373, 6379, 6389, 6397,
            6421, 6427, 6449, 6451, 6469, 6473, 6481, 6491, 6521, 6529, 6547, 6551, 6553, 6563,
            6569, 6571, 6577, 6581, 6599, 6607, 6619, 6637, 6653, 6659, 6661, 6673, 6679, 6689,
            6691, 6701, 6703, 6709, 6719, 6733, 6737, 6761, 6763, 6779, 6781, 6791, 6793, 6803,
            6823, 6827, 6829, 6833, 6841, 6857, 6863, 6869, 6871, 6883, 6899, 6907, 6911, 6917,
            6947, 6949, 6959, 6961, 6967, 6971, 6977, 6983, 6991, 6997, 7001, 7013, 7019, 7027,
            7039, 7043, 7057, 7069, 7079, 7103, 7109, 7121, 7127, 7129, 7151, 7159, 7177, 7187,
            7193, 7207, 7211, 7213, 7219, 7229, 7237, 7243, 7247, 7253, 7283, 7297, 7307, 7309,
            7321, 7331, 7333, 7349, 7351, 7369, 7393, 7411, 7417, 7433, 7451, 7457, 7459, 7477,
            7481, 7487, 7489, 7499, 7507, 7517, 7523, 7529, 7537, 7541, 7547, 7549, 7559, 7561,
            7573, 7577, 7583, 7589, 7591, 7603, 7607, 7621, 7639, 7643, 7649, 7669, 7673, 7681,
            7687, 7691, 7699, 7703, 7717, 7723, 7727, 7741, 7753, 7757, 7759, 7789, 7793, 7817,
            7823, 7829, 7841, 7853, 7867, 7873, 7877, 7879, 7883, 7901, 7907, 7919, 7927, 7933,
            7937, 7949, 7951, 7963, 7993, 8009, 8011, 8017, 8039, 8053, 8059, 8069, 8081, 8087,
            8089, 8093, 8101, 8111, 8117, 8123, 8147, 8161, 8167, 8171, 8179, 8191, 8209, 8219,
            8221, 8231, 8233, 8237, 8243, 8263, 8269, 8273, 8287, 8291, 8293, 8297, 8311, 8317,
            8329, 8353, 8363, 8369, 8377, 8387, 8389, 8419, 8423, 8429, 8431, 8443, 8447, 8461,
            8467, 8501, 8513, 8521, 8527, 8537, 8539, 8543, 8563, 8573, 8581, 8597, 8599, 8609,
            8623, 8627, 8629, 8641, 8647, 8663, 8669, 8677, 8681, 8689, 8693, 8699, 8707, 8713,
            8719, 8731, 8737, 8741, 8747, 8753, 8761, 8779, 8783, 8803, 8807, 8819, 8821, 8831,
            8837, 8839, 8849, 8861, 8863, 8867, 8887, 8893, 8923, 8929, 8933, 8941, 8951, 8963,
            8969, 8971, 8999, 9001, 9007, 9011, 9013, 9029, 9041, 9043, 9049, 9059, 9067, 9091,
            9103, 9109, 9127, 9133, 9137, 9151, 9157, 9161, 9173, 9181, 9187, 9199, 9203, 9209,
            9221, 9227, 9239, 9241, 9257, 9277, 9281, 9283, 9293, 9311, 9319, 9323, 9337, 9341,
            9343, 9349, 9371, 9377, 9391, 9397, 9403, 9413, 9419, 9421, 9431, 9433, 9437, 9439,
            9461, 9463, 9467, 9473, 9479, 9491, 9497, 9511, 9521, 9533, 9539, 9547, 9551, 9587,
            9601, 9613, 9619, 9623, 9629, 9631, 9643, 9649, 9661, 9677, 9679, 9689, 9697, 9719,
            9721, 9733, 9739, 9743, 9749, 9767, 9769, 9781, 9787, 9791, 9803, 9811, 9817, 9829,
            9833, 9839, 9851, 9857, 9859, 9871, 9883, 9887, 9901, 9907, 9923, 9929, 9931, 9941,
            9949, 9967, 9973,
        ];

        for p in small_primes {
            assert!(is_prime(p.to_string().as_str()) == true);
        }
    }

    #[test]
    fn small_non_primes() {
        let small_non_primes = vec![
            0, 1, 4, 6, 8, 9, 10, 12, 14, 15, 16, 18, 20, 21, 22, 24, 25, 26, 27, 28, 30, 32, 33,
            34, 35, 36, 38, 39, 40, 42, 44, 45, 46, 48, 49, 50, 51, 52, 54, 55, 56, 57, 58, 60, 62,
            63, 64, 65, 66, 68, 69, 70, 72, 74, 75, 76, 77, 78, 80, 81, 82, 84, 85, 86, 87, 88, 90,
            91, 92, 93, 94, 95, 96, 98, 99, 100, 102, 104, 105, 106, 108, 110, 111, 112, 114, 115,
            116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 128, 129, 130, 132, 133, 134,
            135, 136, 138, 140, 141, 142, 143, 144, 145, 146, 147, 148, 150, 152, 153, 154, 155,
            156, 158, 159, 160, 161, 162, 164, 165, 166, 168, 169, 170, 171, 172, 174, 175, 176,
            177, 178, 180, 182, 183, 184, 185, 186, 187, 188, 189, 190, 192, 194, 195, 196, 198,
            200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 212, 213, 214, 215, 216, 217,
            218, 219, 220, 221, 222, 224, 225, 226, 228, 230, 231, 232, 234, 235, 236, 237, 238,
            240, 242, 243, 244, 245, 246, 247, 248, 249, 250, 252, 253, 254, 255, 256, 258, 259,
            260, 261, 262, 264, 265, 266, 267, 268, 270, 272, 273, 274, 275, 276, 278, 279, 280,
            282, 284, 285, 286, 287, 288, 289, 290, 291, 292, 294, 295, 296, 297, 298, 299, 300,
            301, 302, 303, 304, 305, 306, 308, 309, 310, 312, 314, 315, 316, 318, 319, 320, 321,
            322, 323, 324, 325, 326, 327, 328, 329, 330, 332, 333, 334, 335, 336, 338, 339, 340,
            341, 342, 343, 344, 345, 346, 348, 350, 351, 352, 354, 355, 356, 357, 358, 360, 361,
            362, 363, 364, 365, 366, 368, 369, 370, 371, 372, 374, 375, 376, 377, 378, 380, 381,
            382, 384, 385, 386, 387, 388, 390, 391, 392, 393, 394, 395, 396, 398, 399, 400, 402,
            403, 404, 405, 406, 407, 408, 410, 411, 412, 413, 414, 415, 416, 417, 418, 420, 422,
            423, 424, 425, 426, 427, 428, 429, 430, 432, 434, 435, 436, 437, 438, 440, 441, 442,
            444, 445, 446, 447, 448, 450, 451, 452, 453, 454, 455, 456, 458, 459, 460, 462, 464,
            465, 466, 468, 469, 470, 471, 472, 473, 474, 475, 476, 477, 478, 480, 481, 482, 483,
            484, 485, 486, 488, 489, 490, 492, 493, 494, 495, 496, 497, 498, 500, 501, 502, 504,
            505, 506, 507, 508, 510, 511, 512, 513, 514, 515, 516, 517, 518, 519, 520, 522, 524,
            525, 526, 527, 528, 529, 530, 531, 532, 533, 534, 535, 536, 537, 538, 539, 540, 542,
            543, 544, 545, 546, 548, 549, 550, 551, 552, 553, 554, 555, 556, 558, 559, 560, 561,
            562, 564, 565, 566, 567, 568, 570, 572, 573, 574, 575, 576, 578, 579, 580, 581, 582,
            583, 584, 585, 586, 588, 589, 590, 591, 592, 594, 595, 596, 597, 598, 600, 602, 603,
            604, 605, 606, 608, 609, 610, 611, 612, 614, 615, 616, 618, 620, 621, 622, 623, 624,
            625, 626, 627, 628, 629, 630, 632, 633, 634, 635, 636, 637, 638, 639, 640, 642, 644,
            645, 646, 648, 649, 650, 651, 652, 654, 655, 656, 657, 658, 660, 662, 663, 664, 665,
            666, 667, 668, 669, 670, 671, 672, 674, 675, 676, 678, 679, 680, 681, 682, 684, 685,
            686, 687, 688, 689, 690, 692, 693, 694, 695, 696, 697, 698, 699, 700, 702, 703, 704,
            705, 706, 707, 708, 710, 711, 712, 713, 714, 715, 716, 717, 718, 720, 721, 722, 723,
            724, 725, 726, 728, 729, 730, 731, 732, 734, 735, 736, 737, 738, 740, 741, 742, 744,
            745, 746, 747, 748, 749, 750, 752, 753, 754, 755, 756, 758, 759, 760, 762, 763, 764,
            765, 766, 767, 768, 770, 771, 772, 774, 775, 776, 777, 778, 779, 780, 781, 782, 783,
            784, 785, 786, 788, 789, 790, 791, 792, 793, 794, 795, 796, 798, 799, 800, 801, 802,
            803, 804, 805, 806, 807, 808, 810, 812, 813, 814, 815, 816, 817, 818, 819, 820, 822,
            824, 825, 826, 828, 830, 831, 832, 833, 834, 835, 836, 837, 838, 840, 841, 842, 843,
            844, 845, 846, 847, 848, 849, 850, 851, 852, 854, 855, 856, 858, 860, 861, 862, 864,
            865, 866, 867, 868, 869, 870, 871, 872, 873, 874, 875, 876, 878, 879, 880, 882, 884,
            885, 886, 888, 889, 890, 891, 892, 893, 894, 895, 896, 897, 898, 899, 900, 901, 902,
            903, 904, 905, 906, 908, 909, 910, 912, 913, 914, 915, 916, 917, 918, 920, 921, 922,
            923, 924, 925, 926, 927, 928, 930, 931, 932, 933, 934, 935, 936, 938, 939, 940, 942,
            943, 944, 945, 946, 948, 949, 950, 951, 952, 954, 955, 956, 957, 958, 959, 960, 961,
            962, 963, 964, 965, 966, 968, 969, 970, 972, 973, 974, 975, 976, 978, 979, 980, 981,
            982, 984, 985, 986, 987, 988, 989, 990, 992, 993, 994, 995, 996, 998, 999, 1000, 1001,
            1002, 1003, 1004, 1005, 1006, 1007, 1008, 1010, 1011, 1012, 1014, 1015, 1016, 1017,
            1018, 1020, 1022, 1023, 1024, 1025, 1026, 1027, 1028, 1029, 1030, 1032, 1034, 1035,
            1036, 1037, 1038, 1040, 1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1050, 1052,
            1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1062, 1064, 1065, 1066, 1067, 1068,
            1070, 1071, 1072, 1073, 1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083,
            1084, 1085, 1086, 1088, 1089, 1090, 1092, 1094, 1095, 1096, 1098, 1099, 1100, 1101,
            1102, 1104, 1105, 1106, 1107, 1108, 1110, 1111, 1112, 1113, 1114, 1115, 1116, 1118,
            1119, 1120, 1121, 1122, 1124, 1125, 1126, 1127, 1128, 1130, 1131, 1132, 1133, 1134,
            1135, 1136, 1137, 1138, 1139, 1140, 1141, 1142, 1143, 1144, 1145, 1146, 1147, 1148,
            1149, 1150, 1152, 1154, 1155, 1156, 1157, 1158, 1159, 1160, 1161, 1162, 1164, 1165,
            1166, 1167, 1168, 1169, 1170, 1172, 1173, 1174, 1175, 1176, 1177, 1178, 1179, 1180,
            1182, 1183, 1184, 1185, 1186, 1188, 1189, 1190, 1191, 1192, 1194, 1195, 1196, 1197,
            1198, 1199, 1200, 1202, 1203, 1204, 1205, 1206, 1207, 1208, 1209, 1210, 1211, 1212,
            1214, 1215, 1216, 1218, 1219, 1220, 1221, 1222, 1224, 1225, 1226, 1227, 1228, 1230,
            1232, 1233, 1234, 1235, 1236, 1238, 1239, 1240, 1241, 1242, 1243, 1244, 1245, 1246,
            1247, 1248, 1250, 1251, 1252, 1253, 1254, 1255, 1256, 1257, 1258, 1260, 1261, 1262,
            1263, 1264, 1265, 1266, 1267, 1268, 1269, 1270, 1271, 1272, 1273, 1274, 1275, 1276,
            1278, 1280, 1281, 1282, 1284, 1285, 1286, 1287, 1288, 1290, 1292, 1293, 1294, 1295,
            1296, 1298, 1299, 1300, 1302, 1304, 1305, 1306, 1308, 1309, 1310, 1311, 1312, 1313,
            1314, 1315, 1316, 1317, 1318, 1320, 1322, 1323, 1324, 1325, 1326, 1328, 1329, 1330,
            1331, 1332, 1333, 1334, 1335, 1336, 1337, 1338, 1339, 1340, 1341, 1342, 1343, 1344,
            1345, 1346, 1347, 1348, 1349, 1350, 1351, 1352, 1353, 1354, 1355, 1356, 1357, 1358,
            1359, 1360, 1362, 1363, 1364, 1365, 1366, 1368, 1369, 1370, 1371, 1372, 1374, 1375,
            1376, 1377, 1378, 1379, 1380, 1382, 1383, 1384, 1385, 1386, 1387, 1388, 1389, 1390,
            1391, 1392, 1393, 1394, 1395, 1396, 1397, 1398, 1400, 1401, 1402, 1403, 1404, 1405,
            1406, 1407, 1408, 1410, 1411, 1412, 1413, 1414, 1415, 1416, 1417, 1418, 1419, 1420,
            1421, 1422, 1424, 1425, 1426, 1428, 1430, 1431, 1432, 1434, 1435, 1436, 1437, 1438,
            1440, 1441, 1442, 1443, 1444, 1445, 1446, 1448, 1449, 1450, 1452, 1454, 1455, 1456,
            1457, 1458, 1460, 1461, 1462, 1463, 1464, 1465, 1466, 1467, 1468, 1469, 1470, 1472,
            1473, 1474, 1475, 1476, 1477, 1478, 1479, 1480, 1482, 1484, 1485, 1486, 1488, 1490,
            1491, 1492, 1494, 1495, 1496, 1497, 1498, 1500, 1501, 1502, 1503, 1504, 1505, 1506,
            1507, 1508, 1509, 1510, 1512, 1513, 1514, 1515, 1516, 1517, 1518, 1519, 1520, 1521,
            1522, 1524, 1525, 1526, 1527, 1528, 1529, 1530, 1532, 1533, 1534, 1535, 1536, 1537,
            1538, 1539, 1540, 1541, 1542, 1544, 1545, 1546, 1547, 1548, 1550, 1551, 1552, 1554,
            1555, 1556, 1557, 1558, 1560, 1561, 1562, 1563, 1564, 1565, 1566, 1568, 1569, 1570,
            1572, 1573, 1574, 1575, 1576, 1577, 1578, 1580, 1581, 1582, 1584, 1585, 1586, 1587,
            1588, 1589, 1590, 1591, 1592, 1593, 1594, 1595, 1596, 1598, 1599, 1600, 1602, 1603,
            1604, 1605, 1606, 1608, 1610, 1611, 1612, 1614, 1615, 1616, 1617, 1618, 1620, 1622,
            1623, 1624, 1625, 1626, 1628, 1629, 1630, 1631, 1632, 1633, 1634, 1635, 1636, 1638,
            1639, 1640, 1641, 1642, 1643, 1644, 1645, 1646, 1647, 1648, 1649, 1650, 1651, 1652,
            1653, 1654, 1655, 1656, 1658, 1659, 1660, 1661, 1662, 1664, 1665, 1666, 1668, 1670,
            1671, 1672, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 1680, 1681, 1682, 1683, 1684,
            1685, 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1694, 1695, 1696, 1698, 1700, 1701,
            1702, 1703, 1704, 1705, 1706, 1707, 1708, 1710, 1711, 1712, 1713, 1714, 1715, 1716,
            1717, 1718, 1719, 1720, 1722, 1724, 1725, 1726, 1727, 1728, 1729, 1730, 1731, 1732,
            1734, 1735, 1736, 1737, 1738, 1739, 1740, 1742, 1743, 1744, 1745, 1746, 1748, 1749,
            1750, 1751, 1752, 1754, 1755, 1756, 1757, 1758, 1760, 1761, 1762, 1763, 1764, 1765,
            1766, 1767, 1768, 1769, 1770, 1771, 1772, 1773, 1774, 1775, 1776, 1778, 1779, 1780,
            1781, 1782, 1784, 1785, 1786, 1788, 1790, 1791, 1792, 1793, 1794, 1795, 1796, 1797,
            1798, 1799, 1800, 1802, 1803, 1804, 1805, 1806, 1807, 1808, 1809, 1810, 1812, 1813,
            1814, 1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1824, 1825, 1826, 1827, 1828,
            1829, 1830, 1832, 1833, 1834, 1835, 1836, 1837, 1838, 1839, 1840, 1841, 1842, 1843,
            1844, 1845, 1846, 1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858,
            1859, 1860, 1862, 1863, 1864, 1865, 1866, 1868, 1869, 1870, 1872, 1874, 1875, 1876,
            1878, 1880, 1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1890, 1891, 1892, 1893,
            1894, 1895, 1896, 1897, 1898, 1899, 1900, 1902, 1903, 1904, 1905, 1906, 1908, 1909,
            1910, 1911, 1912, 1914, 1915, 1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924,
            1925, 1926, 1927, 1928, 1929, 1930, 1932, 1934, 1935, 1936, 1937, 1938, 1939, 1940,
            1941, 1942, 1943, 1944, 1945, 1946, 1947, 1948, 1950, 1952, 1953, 1954, 1955, 1956,
            1957, 1958, 1959, 1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968, 1969, 1970,
            1971, 1972, 1974, 1975, 1976, 1977, 1978, 1980, 1981, 1982, 1983, 1984, 1985, 1986,
            1988, 1989, 1990, 1991, 1992, 1994, 1995, 1996, 1998, 2000, 2001, 2002, 2004, 2005,
            2006, 2007, 2008, 2009, 2010, 2012, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2021,
            2022, 2023, 2024, 2025, 2026, 2028, 2030, 2031, 2032, 2033, 2034, 2035, 2036, 2037,
            2038, 2040, 2041, 2042, 2043, 2044, 2045, 2046, 2047, 2048, 2049, 2050, 2051, 2052,
            2054, 2055, 2056, 2057, 2058, 2059, 2060, 2061, 2062, 2064, 2065, 2066, 2067, 2068,
            2070, 2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078, 2079, 2080, 2082, 2084, 2085,
            2086, 2088, 2090, 2091, 2092, 2093, 2094, 2095, 2096, 2097, 2098, 2100, 2101, 2102,
            2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2112, 2114, 2115, 2116, 2117, 2118,
            2119, 2120, 2121, 2122, 2123, 2124, 2125, 2126, 2127, 2128, 2130, 2132, 2133, 2134,
            2135, 2136, 2138, 2139, 2140, 2142, 2144, 2145, 2146, 2147, 2148, 2149, 2150, 2151,
            2152, 2154, 2155, 2156, 2157, 2158, 2159, 2160, 2162, 2163, 2164, 2165, 2166, 2167,
            2168, 2169, 2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177, 2178, 2180, 2181, 2182,
            2183, 2184, 2185, 2186, 2187, 2188, 2189, 2190, 2191, 2192, 2193, 2194, 2195, 2196,
            2197, 2198, 2199, 2200, 2201, 2202, 2204, 2205, 2206, 2208, 2209, 2210, 2211, 2212,
            2214, 2215, 2216, 2217, 2218, 2219, 2220, 2222, 2223, 2224, 2225, 2226, 2227, 2228,
            2229, 2230, 2231, 2232, 2233, 2234, 2235, 2236, 2238, 2240, 2241, 2242, 2244, 2245,
            2246, 2247, 2248, 2249, 2250, 2252, 2253, 2254, 2255, 2256, 2257, 2258, 2259, 2260,
            2261, 2262, 2263, 2264, 2265, 2266, 2268, 2270, 2271, 2272, 2274, 2275, 2276, 2277,
            2278, 2279, 2280, 2282, 2283, 2284, 2285, 2286, 2288, 2289, 2290, 2291, 2292, 2294,
            2295, 2296, 2298, 2299, 2300, 2301, 2302, 2303, 2304, 2305, 2306, 2307, 2308, 2310,
            2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320, 2321, 2322, 2323, 2324, 2325,
            2326, 2327, 2328, 2329, 2330, 2331, 2332, 2334, 2335, 2336, 2337, 2338, 2340, 2342,
            2343, 2344, 2345, 2346, 2348, 2349, 2350, 2352, 2353, 2354, 2355, 2356, 2358, 2359,
            2360, 2361, 2362, 2363, 2364, 2365, 2366, 2367, 2368, 2369, 2370, 2372, 2373, 2374,
            2375, 2376, 2378, 2379, 2380, 2382, 2384, 2385, 2386, 2387, 2388, 2390, 2391, 2392,
            2394, 2395, 2396, 2397, 2398, 2400, 2401, 2402, 2403, 2404, 2405, 2406, 2407, 2408,
            2409, 2410, 2412, 2413, 2414, 2415, 2416, 2418, 2419, 2420, 2421, 2422, 2424, 2425,
            2426, 2427, 2428, 2429, 2430, 2431, 2432, 2433, 2434, 2435, 2436, 2438, 2439, 2440,
            2442, 2443, 2444, 2445, 2446, 2448, 2449, 2450, 2451, 2452, 2453, 2454, 2455, 2456,
            2457, 2458, 2460, 2461, 2462, 2463, 2464, 2465, 2466, 2468, 2469, 2470, 2471, 2472,
            2474, 2475, 2476, 2478, 2479, 2480, 2481, 2482, 2483, 2484, 2485, 2486, 2487, 2488,
            2489, 2490, 2491, 2492, 2493, 2494, 2495, 2496, 2497, 2498, 2499, 2500, 2501, 2502,
            2504, 2505, 2506, 2507, 2508, 2509, 2510, 2511, 2512, 2513, 2514, 2515, 2516, 2517,
            2518, 2519, 2520, 2522, 2523, 2524, 2525, 2526, 2527, 2528, 2529, 2530, 2532, 2533,
            2534, 2535, 2536, 2537, 2538, 2540, 2541, 2542, 2544, 2545, 2546, 2547, 2548, 2550,
            2552, 2553, 2554, 2555, 2556, 2558, 2559, 2560, 2561, 2562, 2563, 2564, 2565, 2566,
            2567, 2568, 2569, 2570, 2571, 2572, 2573, 2574, 2575, 2576, 2577, 2578, 2580, 2581,
            2582, 2583, 2584, 2585, 2586, 2587, 2588, 2589, 2590, 2592, 2594, 2595, 2596, 2597,
            2598, 2599, 2600, 2601, 2602, 2603, 2604, 2605, 2606, 2607, 2608, 2610, 2611, 2612,
            2613, 2614, 2615, 2616, 2618, 2619, 2620, 2622, 2623, 2624, 2625, 2626, 2627, 2628,
            2629, 2630, 2631, 2632, 2634, 2635, 2636, 2637, 2638, 2639, 2640, 2641, 2642, 2643,
            2644, 2645, 2646, 2648, 2649, 2650, 2651, 2652, 2653, 2654, 2655, 2656, 2658, 2660,
            2661, 2662, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2672, 2673, 2674, 2675, 2676,
            2678, 2679, 2680, 2681, 2682, 2684, 2685, 2686, 2688, 2690, 2691, 2692, 2694, 2695,
            2696, 2697, 2698, 2700, 2701, 2702, 2703, 2704, 2705, 2706, 2708, 2709, 2710, 2712,
            2714, 2715, 2716, 2717, 2718, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727, 2728,
            2730, 2732, 2733, 2734, 2735, 2736, 2737, 2738, 2739, 2740, 2742, 2743, 2744, 2745,
            2746, 2747, 2748, 2750, 2751, 2752, 2754, 2755, 2756, 2757, 2758, 2759, 2760, 2761,
            2762, 2763, 2764, 2765, 2766, 2768, 2769, 2770, 2771, 2772, 2773, 2774, 2775, 2776,
            2778, 2779, 2780, 2781, 2782, 2783, 2784, 2785, 2786, 2787, 2788, 2790, 2792, 2793,
            2794, 2795, 2796, 2798, 2799, 2800, 2802, 2804, 2805, 2806, 2807, 2808, 2809, 2810,
            2811, 2812, 2813, 2814, 2815, 2816, 2817, 2818, 2820, 2821, 2822, 2823, 2824, 2825,
            2826, 2827, 2828, 2829, 2830, 2831, 2832, 2834, 2835, 2836, 2838, 2839, 2840, 2841,
            2842, 2844, 2845, 2846, 2847, 2848, 2849, 2850, 2852, 2853, 2854, 2855, 2856, 2858,
            2859, 2860, 2862, 2863, 2864, 2865, 2866, 2867, 2868, 2869, 2870, 2871, 2872, 2873,
            2874, 2875, 2876, 2877, 2878, 2880, 2881, 2882, 2883, 2884, 2885, 2886, 2888, 2889,
            2890, 2891, 2892, 2893, 2894, 2895, 2896, 2898, 2899, 2900, 2901, 2902, 2904, 2905,
            2906, 2907, 2908, 2910, 2911, 2912, 2913, 2914, 2915, 2916, 2918, 2919, 2920, 2921,
            2922, 2923, 2924, 2925, 2926, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935, 2936,
            2937, 2938, 2940, 2941, 2942, 2943, 2944, 2945, 2946, 2947, 2948, 2949, 2950, 2951,
            2952, 2954, 2955, 2956, 2958, 2959, 2960, 2961, 2962, 2964, 2965, 2966, 2967, 2968,
            2970, 2972, 2973, 2974, 2975, 2976, 2977, 2978, 2979, 2980, 2981, 2982, 2983, 2984,
            2985, 2986, 2987, 2988, 2989, 2990, 2991, 2992, 2993, 2994, 2995, 2996, 2997, 2998,
            3000, 3002, 3003, 3004, 3005, 3006, 3007, 3008, 3009, 3010, 3012, 3013, 3014, 3015,
            3016, 3017, 3018, 3020, 3021, 3022, 3024, 3025, 3026, 3027, 3028, 3029, 3030, 3031,
            3032, 3033, 3034, 3035, 3036, 3038, 3039, 3040, 3042, 3043, 3044, 3045, 3046, 3047,
            3048, 3050, 3051, 3052, 3053, 3054, 3055, 3056, 3057, 3058, 3059, 3060, 3062, 3063,
            3064, 3065, 3066, 3068, 3069, 3070, 3071, 3072, 3073, 3074, 3075, 3076, 3077, 3078,
            3080, 3081, 3082, 3084, 3085, 3086, 3087, 3088, 3090, 3091, 3092, 3093, 3094, 3095,
            3096, 3097, 3098, 3099, 3100, 3101, 3102, 3103, 3104, 3105, 3106, 3107, 3108, 3110,
            3111, 3112, 3113, 3114, 3115, 3116, 3117, 3118, 3120, 3122, 3123, 3124, 3125, 3126,
            3127, 3128, 3129, 3130, 3131, 3132, 3133, 3134, 3135, 3136, 3138, 3139, 3140, 3141,
            3142, 3143, 3144, 3145, 3146, 3147, 3148, 3149, 3150, 3151, 3152, 3153, 3154, 3155,
            3156, 3157, 3158, 3159, 3160, 3161, 3162, 3164, 3165, 3166, 3168, 3170, 3171, 3172,
            3173, 3174, 3175, 3176, 3177, 3178, 3179, 3180, 3182, 3183, 3184, 3185, 3186, 3188,
            3189, 3190, 3192, 3193, 3194, 3195, 3196, 3197, 3198, 3199, 3200, 3201, 3202, 3204,
            3205, 3206, 3207, 3208, 3210, 3211, 3212, 3213, 3214, 3215, 3216, 3218, 3219, 3220,
            3222, 3223, 3224, 3225, 3226, 3227, 3228, 3230, 3231, 3232, 3233, 3234, 3235, 3236,
            3237, 3238, 3239, 3240, 3241, 3242, 3243, 3244, 3245, 3246, 3247, 3248, 3249, 3250,
            3252, 3254, 3255, 3256, 3258, 3260, 3261, 3262, 3263, 3264, 3265, 3266, 3267, 3268,
            3269, 3270, 3272, 3273, 3274, 3275, 3276, 3277, 3278, 3279, 3280, 3281, 3282, 3283,
            3284, 3285, 3286, 3287, 3288, 3289, 3290, 3291, 3292, 3293, 3294, 3295, 3296, 3297,
            3298, 3300, 3302, 3303, 3304, 3305, 3306, 3308, 3309, 3310, 3311, 3312, 3314, 3315,
            3316, 3317, 3318, 3320, 3321, 3322, 3324, 3325, 3326, 3327, 3328, 3330, 3332, 3333,
            3334, 3335, 3336, 3337, 3338, 3339, 3340, 3341, 3342, 3344, 3345, 3346, 3348, 3349,
            3350, 3351, 3352, 3353, 3354, 3355, 3356, 3357, 3358, 3360, 3362, 3363, 3364, 3365,
            3366, 3367, 3368, 3369, 3370, 3372, 3374, 3375, 3376, 3377, 3378, 3379, 3380, 3381,
            3382, 3383, 3384, 3385, 3386, 3387, 3388, 3390, 3392, 3393, 3394, 3395, 3396, 3397,
            3398, 3399, 3400, 3401, 3402, 3403, 3404, 3405, 3406, 3408, 3409, 3410, 3411, 3412,
            3414, 3415, 3416, 3417, 3418, 3419, 3420, 3421, 3422, 3423, 3424, 3425, 3426, 3427,
            3428, 3429, 3430, 3431, 3432, 3434, 3435, 3436, 3437, 3438, 3439, 3440, 3441, 3442,
            3443, 3444, 3445, 3446, 3447, 3448, 3450, 3451, 3452, 3453, 3454, 3455, 3456, 3458,
            3459, 3460, 3462, 3464, 3465, 3466, 3468, 3470, 3471, 3472, 3473, 3474, 3475, 3476,
            3477, 3478, 3479, 3480, 3481, 3482, 3483, 3484, 3485, 3486, 3487, 3488, 3489, 3490,
            3492, 3493, 3494, 3495, 3496, 3497, 3498, 3500, 3501, 3502, 3503, 3504, 3505, 3506,
            3507, 3508, 3509, 3510, 3512, 3513, 3514, 3515, 3516, 3518, 3519, 3520, 3521, 3522,
            3523, 3524, 3525, 3526, 3528, 3530, 3531, 3532, 3534, 3535, 3536, 3537, 3538, 3540,
            3542, 3543, 3544, 3545, 3546, 3548, 3549, 3550, 3551, 3552, 3553, 3554, 3555, 3556,
            3558, 3560, 3561, 3562, 3563, 3564, 3565, 3566, 3567, 3568, 3569, 3570, 3572, 3573,
            3574, 3575, 3576, 3577, 3578, 3579, 3580, 3582, 3584, 3585, 3586, 3587, 3588, 3589,
            3590, 3591, 3592, 3594, 3595, 3596, 3597, 3598, 3599, 3600, 3601, 3602, 3603, 3604,
            3605, 3606, 3608, 3609, 3610, 3611, 3612, 3614, 3615, 3616, 3618, 3619, 3620, 3621,
            3622, 3624, 3625, 3626, 3627, 3628, 3629, 3630, 3632, 3633, 3634, 3635, 3636, 3638,
            3639, 3640, 3641, 3642, 3644, 3645, 3646, 3647, 3648, 3649, 3650, 3651, 3652, 3653,
            3654, 3655, 3656, 3657, 3658, 3660, 3661, 3662, 3663, 3664, 3665, 3666, 3667, 3668,
            3669, 3670, 3672, 3674, 3675, 3676, 3678, 3679, 3680, 3681, 3682, 3683, 3684, 3685,
            3686, 3687, 3688, 3689, 3690, 3692, 3693, 3694, 3695, 3696, 3698, 3699, 3700, 3702,
            3703, 3704, 3705, 3706, 3707, 3708, 3710, 3711, 3712, 3713, 3714, 3715, 3716, 3717,
            3718, 3720, 3721, 3722, 3723, 3724, 3725, 3726, 3728, 3729, 3730, 3731, 3732, 3734,
            3735, 3736, 3737, 3738, 3740, 3741, 3742, 3743, 3744, 3745, 3746, 3747, 3748, 3749,
            3750, 3751, 3752, 3753, 3754, 3755, 3756, 3757, 3758, 3759, 3760, 3762, 3763, 3764,
            3765, 3766, 3768, 3770, 3771, 3772, 3773, 3774, 3775, 3776, 3777, 3778, 3780, 3781,
            3782, 3783, 3784, 3785, 3786, 3787, 3788, 3789, 3790, 3791, 3792, 3794, 3795, 3796,
            3798, 3799, 3800, 3801, 3802, 3804, 3805, 3806, 3807, 3808, 3809, 3810, 3811, 3812,
            3813, 3814, 3815, 3816, 3817, 3818, 3819, 3820, 3822, 3824, 3825, 3826, 3827, 3828,
            3829, 3830, 3831, 3832, 3834, 3835, 3836, 3837, 3838, 3839, 3840, 3841, 3842, 3843,
            3844, 3845, 3846, 3848, 3849, 3850, 3852, 3854, 3855, 3856, 3857, 3858, 3859, 3860,
            3861, 3862, 3864, 3865, 3866, 3867, 3868, 3869, 3870, 3871, 3872, 3873, 3874, 3875,
            3876, 3878, 3879, 3880, 3882, 3883, 3884, 3885, 3886, 3887, 3888, 3890, 3891, 3892,
            3893, 3894, 3895, 3896, 3897, 3898, 3899, 3900, 3901, 3902, 3903, 3904, 3905, 3906,
            3908, 3909, 3910, 3912, 3913, 3914, 3915, 3916, 3918, 3920, 3921, 3922, 3924, 3925,
            3926, 3927, 3928, 3930, 3932, 3933, 3934, 3935, 3936, 3937, 3938, 3939, 3940, 3941,
            3942, 3944, 3945, 3946, 3948, 3949, 3950, 3951, 3952, 3953, 3954, 3955, 3956, 3957,
            3958, 3959, 3960, 3961, 3962, 3963, 3964, 3965, 3966, 3968, 3969, 3970, 3971, 3972,
            3973, 3974, 3975, 3976, 3977, 3978, 3979, 3980, 3981, 3982, 3983, 3984, 3985, 3986,
            3987, 3988, 3990, 3991, 3992, 3993, 3994, 3995, 3996, 3997, 3998, 3999, 4000, 4002,
            4004, 4005, 4006, 4008, 4009, 4010, 4011, 4012, 4014, 4015, 4016, 4017, 4018, 4020,
            4022, 4023, 4024, 4025, 4026, 4028, 4029, 4030, 4031, 4032, 4033, 4034, 4035, 4036,
            4037, 4038, 4039, 4040, 4041, 4042, 4043, 4044, 4045, 4046, 4047, 4048, 4050, 4052,
            4053, 4054, 4055, 4056, 4058, 4059, 4060, 4061, 4062, 4063, 4064, 4065, 4066, 4067,
            4068, 4069, 4070, 4071, 4072, 4074, 4075, 4076, 4077, 4078, 4080, 4081, 4082, 4083,
            4084, 4085, 4086, 4087, 4088, 4089, 4090, 4092, 4094, 4095, 4096, 4097, 4098, 4100,
            4101, 4102, 4103, 4104, 4105, 4106, 4107, 4108, 4109, 4110, 4112, 4113, 4114, 4115,
            4116, 4117, 4118, 4119, 4120, 4121, 4122, 4123, 4124, 4125, 4126, 4128, 4130, 4131,
            4132, 4134, 4135, 4136, 4137, 4138, 4140, 4141, 4142, 4143, 4144, 4145, 4146, 4147,
            4148, 4149, 4150, 4151, 4152, 4154, 4155, 4156, 4158, 4160, 4161, 4162, 4163, 4164,
            4165, 4166, 4167, 4168, 4169, 4170, 4171, 4172, 4173, 4174, 4175, 4176, 4178, 4179,
            4180, 4181, 4182, 4183, 4184, 4185, 4186, 4187, 4188, 4189, 4190, 4191, 4192, 4193,
            4194, 4195, 4196, 4197, 4198, 4199, 4200, 4202, 4203, 4204, 4205, 4206, 4207, 4208,
            4209, 4210, 4212, 4213, 4214, 4215, 4216, 4218, 4220, 4221, 4222, 4223, 4224, 4225,
            4226, 4227, 4228, 4230, 4232, 4233, 4234, 4235, 4236, 4237, 4238, 4239, 4240, 4242,
            4244, 4245, 4246, 4247, 4248, 4249, 4250, 4251, 4252, 4254, 4255, 4256, 4257, 4258,
            4260, 4262, 4263, 4264, 4265, 4266, 4267, 4268, 4269, 4270, 4272, 4274, 4275, 4276,
            4277, 4278, 4279, 4280, 4281, 4282, 4284, 4285, 4286, 4287, 4288, 4290, 4291, 4292,
            4293, 4294, 4295, 4296, 4298, 4299, 4300, 4301, 4302, 4303, 4304, 4305, 4306, 4307,
            4308, 4309, 4310, 4311, 4312, 4313, 4314, 4315, 4316, 4317, 4318, 4319, 4320, 4321,
            4322, 4323, 4324, 4325, 4326, 4328, 4329, 4330, 4331, 4332, 4333, 4334, 4335, 4336,
            4338, 4340, 4341, 4342, 4343, 4344, 4345, 4346, 4347, 4348, 4350, 4351, 4352, 4353,
            4354, 4355, 4356, 4358, 4359, 4360, 4361, 4362, 4364, 4365, 4366, 4367, 4368, 4369,
            4370, 4371, 4372, 4374, 4375, 4376, 4377, 4378, 4379, 4380, 4381, 4382, 4383, 4384,
            4385, 4386, 4387, 4388, 4389, 4390, 4392, 4393, 4394, 4395, 4396, 4398, 4399, 4400,
            4401, 4402, 4403, 4404, 4405, 4406, 4407, 4408, 4410, 4411, 4412, 4413, 4414, 4415,
            4416, 4417, 4418, 4419, 4420, 4422, 4424, 4425, 4426, 4427, 4428, 4429, 4430, 4431,
            4432, 4433, 4434, 4435, 4436, 4437, 4438, 4439, 4440, 4442, 4443, 4444, 4445, 4446,
            4448, 4449, 4450, 4452, 4453, 4454, 4455, 4456, 4458, 4459, 4460, 4461, 4462, 4464,
            4465, 4466, 4467, 4468, 4469, 4470, 4471, 4472, 4473, 4474, 4475, 4476, 4477, 4478,
            4479, 4480, 4482, 4484, 4485, 4486, 4487, 4488, 4489, 4490, 4491, 4492, 4494, 4495,
            4496, 4497, 4498, 4499, 4500, 4501, 4502, 4503, 4504, 4505, 4506, 4508, 4509, 4510,
            4511, 4512, 4514, 4515, 4516, 4518, 4520, 4521, 4522, 4524, 4525, 4526, 4527, 4528,
            4529, 4530, 4531, 4532, 4533, 4534, 4535, 4536, 4537, 4538, 4539, 4540, 4541, 4542,
            4543, 4544, 4545, 4546, 4548, 4550, 4551, 4552, 4553, 4554, 4555, 4556, 4557, 4558,
            4559, 4560, 4562, 4563, 4564, 4565, 4566, 4568, 4569, 4570, 4571, 4572, 4573, 4574,
            4575, 4576, 4577, 4578, 4579, 4580, 4581, 4582, 4584, 4585, 4586, 4587, 4588, 4589,
            4590, 4592, 4593, 4594, 4595, 4596, 4598, 4599, 4600, 4601, 4602, 4604, 4605, 4606,
            4607, 4608, 4609, 4610, 4611, 4612, 4613, 4614, 4615, 4616, 4617, 4618, 4619, 4620,
            4622, 4623, 4624, 4625, 4626, 4627, 4628, 4629, 4630, 4631, 4632, 4633, 4634, 4635,
            4636, 4638, 4640, 4641, 4642, 4644, 4645, 4646, 4647, 4648, 4650, 4652, 4653, 4654,
            4655, 4656, 4658, 4659, 4660, 4661, 4662, 4664, 4665, 4666, 4667, 4668, 4669, 4670,
            4671, 4672, 4674, 4675, 4676, 4677, 4678, 4680, 4681, 4682, 4683, 4684, 4685, 4686,
            4687, 4688, 4689, 4690, 4692, 4693, 4694, 4695, 4696, 4697, 4698, 4699, 4700, 4701,
            4702, 4704, 4705, 4706, 4707, 4708, 4709, 4710, 4711, 4712, 4713, 4714, 4715, 4716,
            4717, 4718, 4719, 4720, 4722, 4724, 4725, 4726, 4727, 4728, 4730, 4731, 4732, 4734,
            4735, 4736, 4737, 4738, 4739, 4740, 4741, 4742, 4743, 4744, 4745, 4746, 4747, 4748,
            4749, 4750, 4752, 4753, 4754, 4755, 4756, 4757, 4758, 4760, 4761, 4762, 4763, 4764,
            4765, 4766, 4767, 4768, 4769, 4770, 4771, 4772, 4773, 4774, 4775, 4776, 4777, 4778,
            4779, 4780, 4781, 4782, 4784, 4785, 4786, 4788, 4790, 4791, 4792, 4794, 4795, 4796,
            4797, 4798, 4800, 4802, 4803, 4804, 4805, 4806, 4807, 4808, 4809, 4810, 4811, 4812,
            4814, 4815, 4816, 4818, 4819, 4820, 4821, 4822, 4823, 4824, 4825, 4826, 4827, 4828,
            4829, 4830, 4832, 4833, 4834, 4835, 4836, 4837, 4838, 4839, 4840, 4841, 4842, 4843,
            4844, 4845, 4846, 4847, 4848, 4849, 4850, 4851, 4852, 4853, 4854, 4855, 4856, 4857,
            4858, 4859, 4860, 4862, 4863, 4864, 4865, 4866, 4867, 4868, 4869, 4870, 4872, 4873,
            4874, 4875, 4876, 4878, 4879, 4880, 4881, 4882, 4883, 4884, 4885, 4886, 4887, 4888,
            4890, 4891, 4892, 4893, 4894, 4895, 4896, 4897, 4898, 4899, 4900, 4901, 4902, 4904,
            4905, 4906, 4907, 4908, 4910, 4911, 4912, 4913, 4914, 4915, 4916, 4917, 4918, 4920,
            4921, 4922, 4923, 4924, 4925, 4926, 4927, 4928, 4929, 4930, 4932, 4934, 4935, 4936,
            4938, 4939, 4940, 4941, 4942, 4944, 4945, 4946, 4947, 4948, 4949, 4950, 4952, 4953,
            4954, 4955, 4956, 4958, 4959, 4960, 4961, 4962, 4963, 4964, 4965, 4966, 4968, 4970,
            4971, 4972, 4974, 4975, 4976, 4977, 4978, 4979, 4980, 4981, 4982, 4983, 4984, 4985,
            4986, 4988, 4989, 4990, 4991, 4992, 4994, 4995, 4996, 4997, 4998, 5000, 5001, 5002,
            5004, 5005, 5006, 5007, 5008, 5010, 5012, 5013, 5014, 5015, 5016, 5017, 5018, 5019,
            5020, 5022, 5024, 5025, 5026, 5027, 5028, 5029, 5030, 5031, 5032, 5033, 5034, 5035,
            5036, 5037, 5038, 5040, 5041, 5042, 5043, 5044, 5045, 5046, 5047, 5048, 5049, 5050,
            5052, 5053, 5054, 5055, 5056, 5057, 5058, 5060, 5061, 5062, 5063, 5064, 5065, 5066,
            5067, 5068, 5069, 5070, 5071, 5072, 5073, 5074, 5075, 5076, 5078, 5079, 5080, 5082,
            5083, 5084, 5085, 5086, 5088, 5089, 5090, 5091, 5092, 5093, 5094, 5095, 5096, 5097,
            5098, 5100, 5102, 5103, 5104, 5105, 5106, 5108, 5109, 5110, 5111, 5112, 5114, 5115,
            5116, 5117, 5118, 5120, 5121, 5122, 5123, 5124, 5125, 5126, 5127, 5128, 5129, 5130,
            5131, 5132, 5133, 5134, 5135, 5136, 5137, 5138, 5139, 5140, 5141, 5142, 5143, 5144,
            5145, 5146, 5148, 5149, 5150, 5151, 5152, 5154, 5155, 5156, 5157, 5158, 5159, 5160,
            5161, 5162, 5163, 5164, 5165, 5166, 5168, 5169, 5170, 5172, 5173, 5174, 5175, 5176,
            5177, 5178, 5180, 5181, 5182, 5183, 5184, 5185, 5186, 5187, 5188, 5190, 5191, 5192,
            5193, 5194, 5195, 5196, 5198, 5199, 5200, 5201, 5202, 5203, 5204, 5205, 5206, 5207,
            5208, 5210, 5211, 5212, 5213, 5214, 5215, 5216, 5217, 5218, 5219, 5220, 5221, 5222,
            5223, 5224, 5225, 5226, 5228, 5229, 5230, 5232, 5234, 5235, 5236, 5238, 5239, 5240,
            5241, 5242, 5243, 5244, 5245, 5246, 5247, 5248, 5249, 5250, 5251, 5252, 5253, 5254,
            5255, 5256, 5257, 5258, 5259, 5260, 5262, 5263, 5264, 5265, 5266, 5267, 5268, 5269,
            5270, 5271, 5272, 5274, 5275, 5276, 5277, 5278, 5280, 5282, 5283, 5284, 5285, 5286,
            5287, 5288, 5289, 5290, 5291, 5292, 5293, 5294, 5295, 5296, 5298, 5299, 5300, 5301,
            5302, 5304, 5305, 5306, 5307, 5308, 5310, 5311, 5312, 5313, 5314, 5315, 5316, 5317,
            5318, 5319, 5320, 5321, 5322, 5324, 5325, 5326, 5327, 5328, 5329, 5330, 5331, 5332,
            5334, 5335, 5336, 5337, 5338, 5339, 5340, 5341, 5342, 5343, 5344, 5345, 5346, 5348,
            5349, 5350, 5352, 5353, 5354, 5355, 5356, 5357, 5358, 5359, 5360, 5361, 5362, 5363,
            5364, 5365, 5366, 5367, 5368, 5369, 5370, 5371, 5372, 5373, 5374, 5375, 5376, 5377,
            5378, 5379, 5380, 5382, 5383, 5384, 5385, 5386, 5388, 5389, 5390, 5391, 5392, 5394,
            5395, 5396, 5397, 5398, 5400, 5401, 5402, 5403, 5404, 5405, 5406, 5408, 5409, 5410,
            5411, 5412, 5414, 5415, 5416, 5418, 5420, 5421, 5422, 5423, 5424, 5425, 5426, 5427,
            5428, 5429, 5430, 5432, 5433, 5434, 5435, 5436, 5438, 5439, 5440, 5442, 5444, 5445,
            5446, 5447, 5448, 5450, 5451, 5452, 5453, 5454, 5455, 5456, 5457, 5458, 5459, 5460,
            5461, 5462, 5463, 5464, 5465, 5466, 5467, 5468, 5469, 5470, 5472, 5473, 5474, 5475,
            5476, 5478, 5480, 5481, 5482, 5484, 5485, 5486, 5487, 5488, 5489, 5490, 5491, 5492,
            5493, 5494, 5495, 5496, 5497, 5498, 5499, 5500, 5502, 5504, 5505, 5506, 5508, 5509,
            5510, 5511, 5512, 5513, 5514, 5515, 5516, 5517, 5518, 5520, 5522, 5523, 5524, 5525,
            5526, 5528, 5529, 5530, 5532, 5533, 5534, 5535, 5536, 5537, 5538, 5539, 5540, 5541,
            5542, 5543, 5544, 5545, 5546, 5547, 5548, 5549, 5550, 5551, 5552, 5553, 5554, 5555,
            5556, 5558, 5559, 5560, 5561, 5562, 5564, 5565, 5566, 5567, 5568, 5570, 5571, 5572,
            5574, 5575, 5576, 5577, 5578, 5579, 5580, 5582, 5583, 5584, 5585, 5586, 5587, 5588,
            5589, 5590, 5592, 5593, 5594, 5595, 5596, 5597, 5598, 5599, 5600, 5601, 5602, 5603,
            5604, 5605, 5606, 5607, 5608, 5609, 5610, 5611, 5612, 5613, 5614, 5615, 5616, 5617,
            5618, 5619, 5620, 5621, 5622, 5624, 5625, 5626, 5627, 5628, 5629, 5630, 5631, 5632,
            5633, 5634, 5635, 5636, 5637, 5638, 5640, 5642, 5643, 5644, 5645, 5646, 5648, 5649,
            5650, 5652, 5654, 5655, 5656, 5658, 5660, 5661, 5662, 5663, 5664, 5665, 5666, 5667,
            5668, 5670, 5671, 5672, 5673, 5674, 5675, 5676, 5677, 5678, 5679, 5680, 5681, 5682,
            5684, 5685, 5686, 5687, 5688, 5690, 5691, 5692, 5694, 5695, 5696, 5697, 5698, 5699,
            5700, 5702, 5703, 5704, 5705, 5706, 5707, 5708, 5709, 5710, 5712, 5713, 5714, 5715,
            5716, 5718, 5719, 5720, 5721, 5722, 5723, 5724, 5725, 5726, 5727, 5728, 5729, 5730,
            5731, 5732, 5733, 5734, 5735, 5736, 5738, 5739, 5740, 5742, 5744, 5745, 5746, 5747,
            5748, 5750, 5751, 5752, 5753, 5754, 5755, 5756, 5757, 5758, 5759, 5760, 5761, 5762,
            5763, 5764, 5765, 5766, 5767, 5768, 5769, 5770, 5771, 5772, 5773, 5774, 5775, 5776,
            5777, 5778, 5780, 5781, 5782, 5784, 5785, 5786, 5787, 5788, 5789, 5790, 5792, 5793,
            5794, 5795, 5796, 5797, 5798, 5799, 5800, 5802, 5803, 5804, 5805, 5806, 5808, 5809,
            5810, 5811, 5812, 5814, 5815, 5816, 5817, 5818, 5819, 5820, 5822, 5823, 5824, 5825,
            5826, 5828, 5829, 5830, 5831, 5832, 5833, 5834, 5835, 5836, 5837, 5838, 5840, 5841,
            5842, 5844, 5845, 5846, 5847, 5848, 5850, 5852, 5853, 5854, 5855, 5856, 5858, 5859,
            5860, 5862, 5863, 5864, 5865, 5866, 5868, 5870, 5871, 5872, 5873, 5874, 5875, 5876,
            5877, 5878, 5880, 5882, 5883, 5884, 5885, 5886, 5887, 5888, 5889, 5890, 5891, 5892,
            5893, 5894, 5895, 5896, 5898, 5899, 5900, 5901, 5902, 5904, 5905, 5906, 5907, 5908,
            5909, 5910, 5911, 5912, 5913, 5914, 5915, 5916, 5917, 5918, 5919, 5920, 5921, 5922,
            5924, 5925, 5926, 5928, 5929, 5930, 5931, 5932, 5933, 5934, 5935, 5936, 5937, 5938,
            5940, 5941, 5942, 5943, 5944, 5945, 5946, 5947, 5948, 5949, 5950, 5951, 5952, 5954,
            5955, 5956, 5957, 5958, 5959, 5960, 5961, 5962, 5963, 5964, 5965, 5966, 5967, 5968,
            5969, 5970, 5971, 5972, 5973, 5974, 5975, 5976, 5977, 5978, 5979, 5980, 5982, 5983,
            5984, 5985, 5986, 5988, 5989, 5990, 5991, 5992, 5993, 5994, 5995, 5996, 5997, 5998,
            5999, 6000, 6001, 6002, 6003, 6004, 6005, 6006, 6008, 6009, 6010, 6012, 6013, 6014,
            6015, 6016, 6017, 6018, 6019, 6020, 6021, 6022, 6023, 6024, 6025, 6026, 6027, 6028,
            6030, 6031, 6032, 6033, 6034, 6035, 6036, 6038, 6039, 6040, 6041, 6042, 6044, 6045,
            6046, 6048, 6049, 6050, 6051, 6052, 6054, 6055, 6056, 6057, 6058, 6059, 6060, 6061,
            6062, 6063, 6064, 6065, 6066, 6068, 6069, 6070, 6071, 6072, 6074, 6075, 6076, 6077,
            6078, 6080, 6081, 6082, 6083, 6084, 6085, 6086, 6087, 6088, 6090, 6092, 6093, 6094,
            6095, 6096, 6097, 6098, 6099, 6100, 6102, 6103, 6104, 6105, 6106, 6107, 6108, 6109,
            6110, 6111, 6112, 6114, 6115, 6116, 6117, 6118, 6119, 6120, 6122, 6123, 6124, 6125,
            6126, 6127, 6128, 6129, 6130, 6132, 6134, 6135, 6136, 6137, 6138, 6139, 6140, 6141,
            6142, 6144, 6145, 6146, 6147, 6148, 6149, 6150, 6152, 6153, 6154, 6155, 6156, 6157,
            6158, 6159, 6160, 6161, 6162, 6164, 6165, 6166, 6167, 6168, 6169, 6170, 6171, 6172,
            6174, 6175, 6176, 6177, 6178, 6179, 6180, 6181, 6182, 6183, 6184, 6185, 6186, 6187,
            6188, 6189, 6190, 6191, 6192, 6193, 6194, 6195, 6196, 6198, 6200, 6201, 6202, 6204,
            6205, 6206, 6207, 6208, 6209, 6210, 6212, 6213, 6214, 6215, 6216, 6218, 6219, 6220,
            6222, 6223, 6224, 6225, 6226, 6227, 6228, 6230, 6231, 6232, 6233, 6234, 6235, 6236,
            6237, 6238, 6239, 6240, 6241, 6242, 6243, 6244, 6245, 6246, 6248, 6249, 6250, 6251,
            6252, 6253, 6254, 6255, 6256, 6258, 6259, 6260, 6261, 6262, 6264, 6265, 6266, 6267,
            6268, 6270, 6272, 6273, 6274, 6275, 6276, 6278, 6279, 6280, 6281, 6282, 6283, 6284,
            6285, 6286, 6288, 6289, 6290, 6291, 6292, 6293, 6294, 6295, 6296, 6297, 6298, 6300,
            6302, 6303, 6304, 6305, 6306, 6307, 6308, 6309, 6310, 6312, 6313, 6314, 6315, 6316,
            6318, 6319, 6320, 6321, 6322, 6324, 6325, 6326, 6327, 6328, 6330, 6331, 6332, 6333,
            6334, 6335, 6336, 6338, 6339, 6340, 6341, 6342, 6344, 6345, 6346, 6347, 6348, 6349,
            6350, 6351, 6352, 6354, 6355, 6356, 6357, 6358, 6360, 6362, 6363, 6364, 6365, 6366,
            6368, 6369, 6370, 6371, 6372, 6374, 6375, 6376, 6377, 6378, 6380, 6381, 6382, 6383,
            6384, 6385, 6386, 6387, 6388, 6390, 6391, 6392, 6393, 6394, 6395, 6396, 6398, 6399,
            6400, 6401, 6402, 6403, 6404, 6405, 6406, 6407, 6408, 6409, 6410, 6411, 6412, 6413,
            6414, 6415, 6416, 6417, 6418, 6419, 6420, 6422, 6423, 6424, 6425, 6426, 6428, 6429,
            6430, 6431, 6432, 6433, 6434, 6435, 6436, 6437, 6438, 6439, 6440, 6441, 6442, 6443,
            6444, 6445, 6446, 6447, 6448, 6450, 6452, 6453, 6454, 6455, 6456, 6457, 6458, 6459,
            6460, 6461, 6462, 6463, 6464, 6465, 6466, 6467, 6468, 6470, 6471, 6472, 6474, 6475,
            6476, 6477, 6478, 6479, 6480, 6482, 6483, 6484, 6485, 6486, 6487, 6488, 6489, 6490,
            6492, 6493, 6494, 6495, 6496, 6497, 6498, 6499, 6500, 6501, 6502, 6503, 6504, 6505,
            6506, 6507, 6508, 6509, 6510, 6511, 6512, 6513, 6514, 6515, 6516, 6517, 6518, 6519,
            6520, 6522, 6523, 6524, 6525, 6526, 6527, 6528, 6530, 6531, 6532, 6533, 6534, 6535,
            6536, 6537, 6538, 6539, 6540, 6541, 6542, 6543, 6544, 6545, 6546, 6548, 6549, 6550,
            6552, 6554, 6555, 6556, 6557, 6558, 6559, 6560, 6561, 6562, 6564, 6565, 6566, 6567,
            6568, 6570, 6572, 6573, 6574, 6575, 6576, 6578, 6579, 6580, 6582, 6583, 6584, 6585,
            6586, 6587, 6588, 6589, 6590, 6591, 6592, 6593, 6594, 6595, 6596, 6597, 6598, 6600,
            6601, 6602, 6603, 6604, 6605, 6606, 6608, 6609, 6610, 6611, 6612, 6613, 6614, 6615,
            6616, 6617, 6618, 6620, 6621, 6622, 6623, 6624, 6625, 6626, 6627, 6628, 6629, 6630,
            6631, 6632, 6633, 6634, 6635, 6636, 6638, 6639, 6640, 6641, 6642, 6643, 6644, 6645,
            6646, 6647, 6648, 6649, 6650, 6651, 6652, 6654, 6655, 6656, 6657, 6658, 6660, 6662,
            6663, 6664, 6665, 6666, 6667, 6668, 6669, 6670, 6671, 6672, 6674, 6675, 6676, 6677,
            6678, 6680, 6681, 6682, 6683, 6684, 6685, 6686, 6687, 6688, 6690, 6692, 6693, 6694,
            6695, 6696, 6697, 6698, 6699, 6700, 6702, 6704, 6705, 6706, 6707, 6708, 6710, 6711,
            6712, 6713, 6714, 6715, 6716, 6717, 6718, 6720, 6721, 6722, 6723, 6724, 6725, 6726,
            6727, 6728, 6729, 6730, 6731, 6732, 6734, 6735, 6736, 6738, 6739, 6740, 6741, 6742,
            6743, 6744, 6745, 6746, 6747, 6748, 6749, 6750, 6751, 6752, 6753, 6754, 6755, 6756,
            6757, 6758, 6759, 6760, 6762, 6764, 6765, 6766, 6767, 6768, 6769, 6770, 6771, 6772,
            6773, 6774, 6775, 6776, 6777, 6778, 6780, 6782, 6783, 6784, 6785, 6786, 6787, 6788,
            6789, 6790, 6792, 6794, 6795, 6796, 6797, 6798, 6799, 6800, 6801, 6802, 6804, 6805,
            6806, 6807, 6808, 6809, 6810, 6811, 6812, 6813, 6814, 6815, 6816, 6817, 6818, 6819,
            6820, 6821, 6822, 6824, 6825, 6826, 6828, 6830, 6831, 6832, 6834, 6835, 6836, 6837,
            6838, 6839, 6840, 6842, 6843, 6844, 6845, 6846, 6847, 6848, 6849, 6850, 6851, 6852,
            6853, 6854, 6855, 6856, 6858, 6859, 6860, 6861, 6862, 6864, 6865, 6866, 6867, 6868,
            6870, 6872, 6873, 6874, 6875, 6876, 6877, 6878, 6879, 6880, 6881, 6882, 6884, 6885,
            6886, 6887, 6888, 6889, 6890, 6891, 6892, 6893, 6894, 6895, 6896, 6897, 6898, 6900,
            6901, 6902, 6903, 6904, 6905, 6906, 6908, 6909, 6910, 6912, 6913, 6914, 6915, 6916,
            6918, 6919, 6920, 6921, 6922, 6923, 6924, 6925, 6926, 6927, 6928, 6929, 6930, 6931,
            6932, 6933, 6934, 6935, 6936, 6937, 6938, 6939, 6940, 6941, 6942, 6943, 6944, 6945,
            6946, 6948, 6950, 6951, 6952, 6953, 6954, 6955, 6956, 6957, 6958, 6960, 6962, 6963,
            6964, 6965, 6966, 6968, 6969, 6970, 6972, 6973, 6974, 6975, 6976, 6978, 6979, 6980,
            6981, 6982, 6984, 6985, 6986, 6987, 6988, 6989, 6990, 6992, 6993, 6994, 6995, 6996,
            6998, 6999, 7000, 7002, 7003, 7004, 7005, 7006, 7007, 7008, 7009, 7010, 7011, 7012,
            7014, 7015, 7016, 7017, 7018, 7020, 7021, 7022, 7023, 7024, 7025, 7026, 7028, 7029,
            7030, 7031, 7032, 7033, 7034, 7035, 7036, 7037, 7038, 7040, 7041, 7042, 7044, 7045,
            7046, 7047, 7048, 7049, 7050, 7051, 7052, 7053, 7054, 7055, 7056, 7058, 7059, 7060,
            7061, 7062, 7063, 7064, 7065, 7066, 7067, 7068, 7070, 7071, 7072, 7073, 7074, 7075,
            7076, 7077, 7078, 7080, 7081, 7082, 7083, 7084, 7085, 7086, 7087, 7088, 7089, 7090,
            7091, 7092, 7093, 7094, 7095, 7096, 7097, 7098, 7099, 7100, 7101, 7102, 7104, 7105,
            7106, 7107, 7108, 7110, 7111, 7112, 7113, 7114, 7115, 7116, 7117, 7118, 7119, 7120,
            7122, 7123, 7124, 7125, 7126, 7128, 7130, 7131, 7132, 7133, 7134, 7135, 7136, 7137,
            7138, 7139, 7140, 7141, 7142, 7143, 7144, 7145, 7146, 7147, 7148, 7149, 7150, 7152,
            7153, 7154, 7155, 7156, 7157, 7158, 7160, 7161, 7162, 7163, 7164, 7165, 7166, 7167,
            7168, 7169, 7170, 7171, 7172, 7173, 7174, 7175, 7176, 7178, 7179, 7180, 7181, 7182,
            7183, 7184, 7185, 7186, 7188, 7189, 7190, 7191, 7192, 7194, 7195, 7196, 7197, 7198,
            7199, 7200, 7201, 7202, 7203, 7204, 7205, 7206, 7208, 7209, 7210, 7212, 7214, 7215,
            7216, 7217, 7218, 7220, 7221, 7222, 7223, 7224, 7225, 7226, 7227, 7228, 7230, 7231,
            7232, 7233, 7234, 7235, 7236, 7238, 7239, 7240, 7241, 7242, 7244, 7245, 7246, 7248,
            7249, 7250, 7251, 7252, 7254, 7255, 7256, 7257, 7258, 7259, 7260, 7261, 7262, 7263,
            7264, 7265, 7266, 7267, 7268, 7269, 7270, 7271, 7272, 7273, 7274, 7275, 7276, 7277,
            7278, 7279, 7280, 7281, 7282, 7284, 7285, 7286, 7287, 7288, 7289, 7290, 7291, 7292,
            7293, 7294, 7295, 7296, 7298, 7299, 7300, 7301, 7302, 7303, 7304, 7305, 7306, 7308,
            7310, 7311, 7312, 7313, 7314, 7315, 7316, 7317, 7318, 7319, 7320, 7322, 7323, 7324,
            7325, 7326, 7327, 7328, 7329, 7330, 7332, 7334, 7335, 7336, 7337, 7338, 7339, 7340,
            7341, 7342, 7343, 7344, 7345, 7346, 7347, 7348, 7350, 7352, 7353, 7354, 7355, 7356,
            7357, 7358, 7359, 7360, 7361, 7362, 7363, 7364, 7365, 7366, 7367, 7368, 7370, 7371,
            7372, 7373, 7374, 7375, 7376, 7377, 7378, 7379, 7380, 7381, 7382, 7383, 7384, 7385,
            7386, 7387, 7388, 7389, 7390, 7391, 7392, 7394, 7395, 7396, 7397, 7398, 7399, 7400,
            7401, 7402, 7403, 7404, 7405, 7406, 7407, 7408, 7409, 7410, 7412, 7413, 7414, 7415,
            7416, 7418, 7419, 7420, 7421, 7422, 7423, 7424, 7425, 7426, 7427, 7428, 7429, 7430,
            7431, 7432, 7434, 7435, 7436, 7437, 7438, 7439, 7440, 7441, 7442, 7443, 7444, 7445,
            7446, 7447, 7448, 7449, 7450, 7452, 7453, 7454, 7455, 7456, 7458, 7460, 7461, 7462,
            7463, 7464, 7465, 7466, 7467, 7468, 7469, 7470, 7471, 7472, 7473, 7474, 7475, 7476,
            7478, 7479, 7480, 7482, 7483, 7484, 7485, 7486, 7488, 7490, 7491, 7492, 7493, 7494,
            7495, 7496, 7497, 7498, 7500, 7501, 7502, 7503, 7504, 7505, 7506, 7508, 7509, 7510,
            7511, 7512, 7513, 7514, 7515, 7516, 7518, 7519, 7520, 7521, 7522, 7524, 7525, 7526,
            7527, 7528, 7530, 7531, 7532, 7533, 7534, 7535, 7536, 7538, 7539, 7540, 7542, 7543,
            7544, 7545, 7546, 7548, 7550, 7551, 7552, 7553, 7554, 7555, 7556, 7557, 7558, 7560,
            7562, 7563, 7564, 7565, 7566, 7567, 7568, 7569, 7570, 7571, 7572, 7574, 7575, 7576,
            7578, 7579, 7580, 7581, 7582, 7584, 7585, 7586, 7587, 7588, 7590, 7592, 7593, 7594,
            7595, 7596, 7597, 7598, 7599, 7600, 7601, 7602, 7604, 7605, 7606, 7608, 7609, 7610,
            7611, 7612, 7613, 7614, 7615, 7616, 7617, 7618, 7619, 7620, 7622, 7623, 7624, 7625,
            7626, 7627, 7628, 7629, 7630, 7631, 7632, 7633, 7634, 7635, 7636, 7637, 7638, 7640,
            7641, 7642, 7644, 7645, 7646, 7647, 7648, 7650, 7651, 7652, 7653, 7654, 7655, 7656,
            7657, 7658, 7659, 7660, 7661, 7662, 7663, 7664, 7665, 7666, 7667, 7668, 7670, 7671,
            7672, 7674, 7675, 7676, 7677, 7678, 7679, 7680, 7682, 7683, 7684, 7685, 7686, 7688,
            7689, 7690, 7692, 7693, 7694, 7695, 7696, 7697, 7698, 7700, 7701, 7702, 7704, 7705,
            7706, 7707, 7708, 7709, 7710, 7711, 7712, 7713, 7714, 7715, 7716, 7718, 7719, 7720,
            7721, 7722, 7724, 7725, 7726, 7728, 7729, 7730, 7731, 7732, 7733, 7734, 7735, 7736,
            7737, 7738, 7739, 7740, 7742, 7743, 7744, 7745, 7746, 7747, 7748, 7749, 7750, 7751,
            7752, 7754, 7755, 7756, 7758, 7760, 7761, 7762, 7763, 7764, 7765, 7766, 7767, 7768,
            7769, 7770, 7771, 7772, 7773, 7774, 7775, 7776, 7777, 7778, 7779, 7780, 7781, 7782,
            7783, 7784, 7785, 7786, 7787, 7788, 7790, 7791, 7792, 7794, 7795, 7796, 7797, 7798,
            7799, 7800, 7801, 7802, 7803, 7804, 7805, 7806, 7807, 7808, 7809, 7810, 7811, 7812,
            7813, 7814, 7815, 7816, 7818, 7819, 7820, 7821, 7822, 7824, 7825, 7826, 7827, 7828,
            7830, 7831, 7832, 7833, 7834, 7835, 7836, 7837, 7838, 7839, 7840, 7842, 7843, 7844,
            7845, 7846, 7847, 7848, 7849, 7850, 7851, 7852, 7854, 7855, 7856, 7857, 7858, 7859,
            7860, 7861, 7862, 7863, 7864, 7865, 7866, 7868, 7869, 7870, 7871, 7872, 7874, 7875,
            7876, 7878, 7880, 7881, 7882, 7884, 7885, 7886, 7887, 7888, 7889, 7890, 7891, 7892,
            7893, 7894, 7895, 7896, 7897, 7898, 7899, 7900, 7902, 7903, 7904, 7905, 7906, 7908,
            7909, 7910, 7911, 7912, 7913, 7914, 7915, 7916, 7917, 7918, 7920, 7921, 7922, 7923,
            7924, 7925, 7926, 7928, 7929, 7930, 7931, 7932, 7934, 7935, 7936, 7938, 7939, 7940,
            7941, 7942, 7943, 7944, 7945, 7946, 7947, 7948, 7950, 7952, 7953, 7954, 7955, 7956,
            7957, 7958, 7959, 7960, 7961, 7962, 7964, 7965, 7966, 7967, 7968, 7969, 7970, 7971,
            7972, 7973, 7974, 7975, 7976, 7977, 7978, 7979, 7980, 7981, 7982, 7983, 7984, 7985,
            7986, 7987, 7988, 7989, 7990, 7991, 7992, 7994, 7995, 7996, 7997, 7998, 7999, 8000,
            8001, 8002, 8003, 8004, 8005, 8006, 8007, 8008, 8010, 8012, 8013, 8014, 8015, 8016,
            8018, 8019, 8020, 8021, 8022, 8023, 8024, 8025, 8026, 8027, 8028, 8029, 8030, 8031,
            8032, 8033, 8034, 8035, 8036, 8037, 8038, 8040, 8041, 8042, 8043, 8044, 8045, 8046,
            8047, 8048, 8049, 8050, 8051, 8052, 8054, 8055, 8056, 8057, 8058, 8060, 8061, 8062,
            8063, 8064, 8065, 8066, 8067, 8068, 8070, 8071, 8072, 8073, 8074, 8075, 8076, 8077,
            8078, 8079, 8080, 8082, 8083, 8084, 8085, 8086, 8088, 8090, 8091, 8092, 8094, 8095,
            8096, 8097, 8098, 8099, 8100, 8102, 8103, 8104, 8105, 8106, 8107, 8108, 8109, 8110,
            8112, 8113, 8114, 8115, 8116, 8118, 8119, 8120, 8121, 8122, 8124, 8125, 8126, 8127,
            8128, 8129, 8130, 8131, 8132, 8133, 8134, 8135, 8136, 8137, 8138, 8139, 8140, 8141,
            8142, 8143, 8144, 8145, 8146, 8148, 8149, 8150, 8151, 8152, 8153, 8154, 8155, 8156,
            8157, 8158, 8159, 8160, 8162, 8163, 8164, 8165, 8166, 8168, 8169, 8170, 8172, 8173,
            8174, 8175, 8176, 8177, 8178, 8180, 8181, 8182, 8183, 8184, 8185, 8186, 8187, 8188,
            8189, 8190, 8192, 8193, 8194, 8195, 8196, 8197, 8198, 8199, 8200, 8201, 8202, 8203,
            8204, 8205, 8206, 8207, 8208, 8210, 8211, 8212, 8213, 8214, 8215, 8216, 8217, 8218,
            8220, 8222, 8223, 8224, 8225, 8226, 8227, 8228, 8229, 8230, 8232, 8234, 8235, 8236,
            8238, 8239, 8240, 8241, 8242, 8244, 8245, 8246, 8247, 8248, 8249, 8250, 8251, 8252,
            8253, 8254, 8255, 8256, 8257, 8258, 8259, 8260, 8261, 8262, 8264, 8265, 8266, 8267,
            8268, 8270, 8271, 8272, 8274, 8275, 8276, 8277, 8278, 8279, 8280, 8281, 8282, 8283,
            8284, 8285, 8286, 8288, 8289, 8290, 8292, 8294, 8295, 8296, 8298, 8299, 8300, 8301,
            8302, 8303, 8304, 8305, 8306, 8307, 8308, 8309, 8310, 8312, 8313, 8314, 8315, 8316,
            8318, 8319, 8320, 8321, 8322, 8323, 8324, 8325, 8326, 8327, 8328, 8330, 8331, 8332,
            8333, 8334, 8335, 8336, 8337, 8338, 8339, 8340, 8341, 8342, 8343, 8344, 8345, 8346,
            8347, 8348, 8349, 8350, 8351, 8352, 8354, 8355, 8356, 8357, 8358, 8359, 8360, 8361,
            8362, 8364, 8365, 8366, 8367, 8368, 8370, 8371, 8372, 8373, 8374, 8375, 8376, 8378,
            8379, 8380, 8381, 8382, 8383, 8384, 8385, 8386, 8388, 8390, 8391, 8392, 8393, 8394,
            8395, 8396, 8397, 8398, 8399, 8400, 8401, 8402, 8403, 8404, 8405, 8406, 8407, 8408,
            8409, 8410, 8411, 8412, 8413, 8414, 8415, 8416, 8417, 8418, 8420, 8421, 8422, 8424,
            8425, 8426, 8427, 8428, 8430, 8432, 8433, 8434, 8435, 8436, 8437, 8438, 8439, 8440,
            8441, 8442, 8444, 8445, 8446, 8448, 8449, 8450, 8451, 8452, 8453, 8454, 8455, 8456,
            8457, 8458, 8459, 8460, 8462, 8463, 8464, 8465, 8466, 8468, 8469, 8470, 8471, 8472,
            8473, 8474, 8475, 8476, 8477, 8478, 8479, 8480, 8481, 8482, 8483, 8484, 8485, 8486,
            8487, 8488, 8489, 8490, 8491, 8492, 8493, 8494, 8495, 8496, 8497, 8498, 8499, 8500,
            8502, 8503, 8504, 8505, 8506, 8507, 8508, 8509, 8510, 8511, 8512, 8514, 8515, 8516,
            8517, 8518, 8519, 8520, 8522, 8523, 8524, 8525, 8526, 8528, 8529, 8530, 8531, 8532,
            8533, 8534, 8535, 8536, 8538, 8540, 8541, 8542, 8544, 8545, 8546, 8547, 8548, 8549,
            8550, 8551, 8552, 8553, 8554, 8555, 8556, 8557, 8558, 8559, 8560, 8561, 8562, 8564,
            8565, 8566, 8567, 8568, 8569, 8570, 8571, 8572, 8574, 8575, 8576, 8577, 8578, 8579,
            8580, 8582, 8583, 8584, 8585, 8586, 8587, 8588, 8589, 8590, 8591, 8592, 8593, 8594,
            8595, 8596, 8598, 8600, 8601, 8602, 8603, 8604, 8605, 8606, 8607, 8608, 8610, 8611,
            8612, 8613, 8614, 8615, 8616, 8617, 8618, 8619, 8620, 8621, 8622, 8624, 8625, 8626,
            8628, 8630, 8631, 8632, 8633, 8634, 8635, 8636, 8637, 8638, 8639, 8640, 8642, 8643,
            8644, 8645, 8646, 8648, 8649, 8650, 8651, 8652, 8653, 8654, 8655, 8656, 8657, 8658,
            8659, 8660, 8661, 8662, 8664, 8665, 8666, 8667, 8668, 8670, 8671, 8672, 8673, 8674,
            8675, 8676, 8678, 8679, 8680, 8682, 8683, 8684, 8685, 8686, 8687, 8688, 8690, 8691,
            8692, 8694, 8695, 8696, 8697, 8698, 8700, 8701, 8702, 8703, 8704, 8705, 8706, 8708,
            8709, 8710, 8711, 8712, 8714, 8715, 8716, 8717, 8718, 8720, 8721, 8722, 8723, 8724,
            8725, 8726, 8727, 8728, 8729, 8730, 8732, 8733, 8734, 8735, 8736, 8738, 8739, 8740,
            8742, 8743, 8744, 8745, 8746, 8748, 8749, 8750, 8751, 8752, 8754, 8755, 8756, 8757,
            8758, 8759, 8760, 8762, 8763, 8764, 8765, 8766, 8767, 8768, 8769, 8770, 8771, 8772,
            8773, 8774, 8775, 8776, 8777, 8778, 8780, 8781, 8782, 8784, 8785, 8786, 8787, 8788,
            8789, 8790, 8791, 8792, 8793, 8794, 8795, 8796, 8797, 8798, 8799, 8800, 8801, 8802,
            8804, 8805, 8806, 8808, 8809, 8810, 8811, 8812, 8813, 8814, 8815, 8816, 8817, 8818,
            8820, 8822, 8823, 8824, 8825, 8826, 8827, 8828, 8829, 8830, 8832, 8833, 8834, 8835,
            8836, 8838, 8840, 8841, 8842, 8843, 8844, 8845, 8846, 8847, 8848, 8850, 8851, 8852,
            8853, 8854, 8855, 8856, 8857, 8858, 8859, 8860, 8862, 8864, 8865, 8866, 8868, 8869,
            8870, 8871, 8872, 8873, 8874, 8875, 8876, 8877, 8878, 8879, 8880, 8881, 8882, 8883,
            8884, 8885, 8886, 8888, 8889, 8890, 8891, 8892, 8894, 8895, 8896, 8897, 8898, 8899,
            8900, 8901, 8902, 8903, 8904, 8905, 8906, 8907, 8908, 8909, 8910, 8911, 8912, 8913,
            8914, 8915, 8916, 8917, 8918, 8919, 8920, 8921, 8922, 8924, 8925, 8926, 8927, 8928,
            8930, 8931, 8932, 8934, 8935, 8936, 8937, 8938, 8939, 8940, 8942, 8943, 8944, 8945,
            8946, 8947, 8948, 8949, 8950, 8952, 8953, 8954, 8955, 8956, 8957, 8958, 8959, 8960,
            8961, 8962, 8964, 8965, 8966, 8967, 8968, 8970, 8972, 8973, 8974, 8975, 8976, 8977,
            8978, 8979, 8980, 8981, 8982, 8983, 8984, 8985, 8986, 8987, 8988, 8989, 8990, 8991,
            8992, 8993, 8994, 8995, 8996, 8997, 8998, 9000, 9002, 9003, 9004, 9005, 9006, 9008,
            9009, 9010, 9012, 9014, 9015, 9016, 9017, 9018, 9019, 9020, 9021, 9022, 9023, 9024,
            9025, 9026, 9027, 9028, 9030, 9031, 9032, 9033, 9034, 9035, 9036, 9037, 9038, 9039,
            9040, 9042, 9044, 9045, 9046, 9047, 9048, 9050, 9051, 9052, 9053, 9054, 9055, 9056,
            9057, 9058, 9060, 9061, 9062, 9063, 9064, 9065, 9066, 9068, 9069, 9070, 9071, 9072,
            9073, 9074, 9075, 9076, 9077, 9078, 9079, 9080, 9081, 9082, 9083, 9084, 9085, 9086,
            9087, 9088, 9089, 9090, 9092, 9093, 9094, 9095, 9096, 9097, 9098, 9099, 9100, 9101,
            9102, 9104, 9105, 9106, 9107, 9108, 9110, 9111, 9112, 9113, 9114, 9115, 9116, 9117,
            9118, 9119, 9120, 9121, 9122, 9123, 9124, 9125, 9126, 9128, 9129, 9130, 9131, 9132,
            9134, 9135, 9136, 9138, 9139, 9140, 9141, 9142, 9143, 9144, 9145, 9146, 9147, 9148,
            9149, 9150, 9152, 9153, 9154, 9155, 9156, 9158, 9159, 9160, 9162, 9163, 9164, 9165,
            9166, 9167, 9168, 9169, 9170, 9171, 9172, 9174, 9175, 9176, 9177, 9178, 9179, 9180,
            9182, 9183, 9184, 9185, 9186, 9188, 9189, 9190, 9191, 9192, 9193, 9194, 9195, 9196,
            9197, 9198, 9200, 9201, 9202, 9204, 9205, 9206, 9207, 9208, 9210, 9211, 9212, 9213,
            9214, 9215, 9216, 9217, 9218, 9219, 9220, 9222, 9223, 9224, 9225, 9226, 9228, 9229,
            9230, 9231, 9232, 9233, 9234, 9235, 9236, 9237, 9238, 9240, 9242, 9243, 9244, 9245,
            9246, 9247, 9248, 9249, 9250, 9251, 9252, 9253, 9254, 9255, 9256, 9258, 9259, 9260,
            9261, 9262, 9263, 9264, 9265, 9266, 9267, 9268, 9269, 9270, 9271, 9272, 9273, 9274,
            9275, 9276, 9278, 9279, 9280, 9282, 9284, 9285, 9286, 9287, 9288, 9289, 9290, 9291,
            9292, 9294, 9295, 9296, 9297, 9298, 9299, 9300, 9301, 9302, 9303, 9304, 9305, 9306,
            9307, 9308, 9309, 9310, 9312, 9313, 9314, 9315, 9316, 9317, 9318, 9320, 9321, 9322,
            9324, 9325, 9326, 9327, 9328, 9329, 9330, 9331, 9332, 9333, 9334, 9335, 9336, 9338,
            9339, 9340, 9342, 9344, 9345, 9346, 9347, 9348, 9350, 9351, 9352, 9353, 9354, 9355,
            9356, 9357, 9358, 9359, 9360, 9361, 9362, 9363, 9364, 9365, 9366, 9367, 9368, 9369,
            9370, 9372, 9373, 9374, 9375, 9376, 9378, 9379, 9380, 9381, 9382, 9383, 9384, 9385,
            9386, 9387, 9388, 9389, 9390, 9392, 9393, 9394, 9395, 9396, 9398, 9399, 9400, 9401,
            9402, 9404, 9405, 9406, 9407, 9408, 9409, 9410, 9411, 9412, 9414, 9415, 9416, 9417,
            9418, 9420, 9422, 9423, 9424, 9425, 9426, 9427, 9428, 9429, 9430, 9432, 9434, 9435,
            9436, 9438, 9440, 9441, 9442, 9443, 9444, 9445, 9446, 9447, 9448, 9449, 9450, 9451,
            9452, 9453, 9454, 9455, 9456, 9457, 9458, 9459, 9460, 9462, 9464, 9465, 9466, 9468,
            9469, 9470, 9471, 9472, 9474, 9475, 9476, 9477, 9478, 9480, 9481, 9482, 9483, 9484,
            9485, 9486, 9487, 9488, 9489, 9490, 9492, 9493, 9494, 9495, 9496, 9498, 9499, 9500,
            9501, 9502, 9503, 9504, 9505, 9506, 9507, 9508, 9509, 9510, 9512, 9513, 9514, 9515,
            9516, 9517, 9518, 9519, 9520, 9522, 9523, 9524, 9525, 9526, 9527, 9528, 9529, 9530,
            9531, 9532, 9534, 9535, 9536, 9537, 9538, 9540, 9541, 9542, 9543, 9544, 9545, 9546,
            9548, 9549, 9550, 9552, 9553, 9554, 9555, 9556, 9557, 9558, 9559, 9560, 9561, 9562,
            9563, 9564, 9565, 9566, 9567, 9568, 9569, 9570, 9571, 9572, 9573, 9574, 9575, 9576,
            9577, 9578, 9579, 9580, 9581, 9582, 9583, 9584, 9585, 9586, 9588, 9589, 9590, 9591,
            9592, 9593, 9594, 9595, 9596, 9597, 9598, 9599, 9600, 9602, 9603, 9604, 9605, 9606,
            9607, 9608, 9609, 9610, 9611, 9612, 9614, 9615, 9616, 9617, 9618, 9620, 9621, 9622,
            9624, 9625, 9626, 9627, 9628, 9630, 9632, 9633, 9634, 9635, 9636, 9637, 9638, 9639,
            9640, 9641, 9642, 9644, 9645, 9646, 9647, 9648, 9650, 9651, 9652, 9653, 9654, 9655,
            9656, 9657, 9658, 9659, 9660, 9662, 9663, 9664, 9665, 9666, 9667, 9668, 9669, 9670,
            9671, 9672, 9673, 9674, 9675, 9676, 9678, 9680, 9681, 9682, 9683, 9684, 9685, 9686,
            9687, 9688, 9690, 9691, 9692, 9693, 9694, 9695, 9696, 9698, 9699, 9700, 9701, 9702,
            9703, 9704, 9705, 9706, 9707, 9708, 9709, 9710, 9711, 9712, 9713, 9714, 9715, 9716,
            9717, 9718, 9720, 9722, 9723, 9724, 9725, 9726, 9727, 9728, 9729, 9730, 9731, 9732,
            9734, 9735, 9736, 9737, 9738, 9740, 9741, 9742, 9744, 9745, 9746, 9747, 9748, 9750,
            9751, 9752, 9753, 9754, 9755, 9756, 9757, 9758, 9759, 9760, 9761, 9762, 9763, 9764,
            9765, 9766, 9768, 9770, 9771, 9772, 9773, 9774, 9775, 9776, 9777, 9778, 9779, 9780,
            9782, 9783, 9784, 9785, 9786, 9788, 9789, 9790, 9792, 9793, 9794, 9795, 9796, 9797,
            9798, 9799, 9800, 9801, 9802, 9804, 9805, 9806, 9807, 9808, 9809, 9810, 9812, 9813,
            9814, 9815, 9816, 9818, 9819, 9820, 9821, 9822, 9823, 9824, 9825, 9826, 9827, 9828,
            9830, 9831, 9832, 9834, 9835, 9836, 9837, 9838, 9840, 9841, 9842, 9843, 9844, 9845,
            9846, 9847, 9848, 9849, 9850, 9852, 9853, 9854, 9855, 9856, 9858, 9860, 9861, 9862,
            9863, 9864, 9865, 9866, 9867, 9868, 9869, 9870, 9872, 9873, 9874, 9875, 9876, 9877,
            9878, 9879, 9880, 9881, 9882, 9884, 9885, 9886, 9888, 9889, 9890, 9891, 9892, 9893,
            9894, 9895, 9896, 9897, 9898, 9899, 9900, 9902, 9903, 9904, 9905, 9906, 9908, 9909,
            9910, 9911, 9912, 9913, 9914, 9915, 9916, 9917, 9918, 9919, 9920, 9921, 9922, 9924,
            9925, 9926, 9927, 9928, 9930, 9932, 9933, 9934, 9935, 9936, 9937, 9938, 9939, 9940,
            9942, 9943, 9944, 9945, 9946, 9947, 9948, 9950, 9951, 9952, 9953, 9954, 9955, 9956,
            9957, 9958, 9959, 9960, 9961, 9962, 9963, 9964, 9965, 9966, 9968, 9969, 9970, 9971,
            9972, 9974, 9975, 9976, 9977, 9978, 9979, 9980, 9981, 9982, 9983, 9984, 9985, 9986,
            9987, 9988, 9989, 9990, 9991, 9992, 9993, 9994, 9995, 9996, 9997, 9998, 9999, 10000,
        ];

        for n in small_non_primes {
            assert!(is_prime(n.to_string().as_str()) == false);
        }
    }

    #[test]
    fn rsa_primes() {
        let rsa_primes = vec![
            "37975227936943673922808872755445627854565536638199",
            "40094690950920881030683735292761468389214899724061",
            "6122421090493547576937037317561418841225758554253106999",
            "5846418214406154678836553182979162384198610505601062333",
            "327414555693498015751146303749141488063642403240171463406883",
            "693342667110830181197325401899700641361965863127336680673013",
            "3490529510847650949147849619903898133417764638493387843990820577",
            "32769132993266709549961988190834461413177642967992942539798288533",
            "39685999459597454290161126162883786067576449112810064832555157243",
            "45534498646735972188403686897274408864356301263205069600999044599",
            "3398717423028438554530123627613875835633986495969597423490929302771479",
            "6264200187401285096151654948264442219302037178623509019111660653946049",
            "348009867102283695483970451047593424831012817350385456889559637548278410717",
            "445647744903640741533241125787086176005442536297766153493419724532460296199",
            "102639592829741105772054196573991675900716567808038066803341933521790711307779",
            "106603488380168454820927220360012878679207958575989291522270608237193062808643",
            "45427892858481394071686190649738831656137145778469793250959984709250004157335359",
            "47388090603832016196633832303788951973268922921040957944741354648812028493909367",
            "3586420730428501486799804587268520423291459681059978161140231860633948450858040593963",
            "7267029064107019078863797763923946264136137803856996670313708936002281582249587494493",
            "398075086424064937397125500550386491199064362342526708406385189575946388957261768583317",
            "472772146107435302536223071973048224632914695302097116459852171130520711256363590397527",
            "400780082329750877952581339104100572526829317815807176564882178998497572771950624613470377",
            "476939688738611836995535477357070857939902076027788232031989775824606225595773435668861833",
            "31711952576901527094851712897404759298051473160294503277847619278327936427981256542415724309619",
            "60152600204445616415876416855266761832435433594718110725997638280836157040460481625355619404899",
            "1634733645809253848443133883865090859841783670033092312181110852389333100104508151212118167511579",
            "1900871281664822113126851573935413975471896789968515493666638539088027103802104498957191261465571",
            "3532461934402770121272604978198464368671197400197625023649303468776121253679423200058547956528088349",
            "7925869954478333033347085841480059687737975857364219960734330341455767872818152135381409304740185467",
            "435958568325940791799951965387214406385470910265220196318705482144524085345275999740244625255428455944579",
            "562545761726884103756277007304447481743876944007510545104946851094548396577479473472146228550799322939273",
            "9091213529597818878440658302600437485892608310328358720428512168960411528640933367824950788367956756806141",
            "8143859259110045265727809126284429335877899002167627883200914172429324360133004116702003240828777970252499",
            "68636564122675662743823714992884378001308422399791648446212449933215410614414642667938213644208420192054999687",
            "32929074394863498120493015492129352919164551965362339524626860511692903493094652463337824866390738191765712603",
            "33478071698956898786044169848212690817704794983713768568912431388982883793878002287614711652531743087737814467999489",
            "36746043666799590428244633799627952632279158164343087642676032283815739666511279233373417143396810270092798736308917",
        ];

        for p in rsa_primes {
            assert!(is_prime(&p) == true);
        }
    }

    #[test]
    fn rsa_semiprimes() {
        let rsa_semiprimes = vec![
            "1522605027922533360535618378132637429718068114961380688657908494580122963258952897654000350692006139",
            "35794234179725868774991807832568455403003778024228226193532908190484670252364677411513516111204504060317568667",
            "227010481295437363334259960947493668895875336466084780038173258247009162675779735389791151574049166747880487470296548479",
            "114381625757888867669235779976146612010218296721242362562561842935706935245733897830597123563958705058989075147599290026879543541",
            "1807082088687404805951656164405905566278102516769401349170127021450056662540244048387341127590812303371781887966563182013214880557",
            "21290246318258757547497882016271517497806703963277216278233383215381949984056495911366573853021918316783107387995317230889569230873441936471",
            "155089812478348440509606754370011861770654545830995430655466945774312632703463465954363335027577729025391453996787414027003501631772186840890795964683",
            "10941738641570527421809707322040357612003732945449205990913842131476349984288934784717997257891267332497625752899781833797076537244027146743531593354333897",
            "2152741102718889701896015201312825429257773588845675980170497676778133145218859135673011059773491059602497907111585214302079314665202840140619946994927570407753",
            "26062623684139844921529879266674432197085925380486406416164785191859999628542069361450283931914514618683512198164805919882053057222974116478065095809832377336510711545759",
            "188198812920607963838697239461650439807163563379417382700763356422988859715234665485319060606504743045317388011303396716199692321205734031879550656996221305168759307650257059",
            "191147927718986609689229466631454649812986246276667354864188503638807260703436799058776201365135161278134258296128109200046702912984568752800330221777752773957404540495707851421041",
            "1907556405060696491061450432646028861081179759533184460647975622318915025587184175754054976155121593293492260464152630093238509246603207417124726121580858185985938946945490481721756401423481",
            "3107418240490043721350750035888567930037346022842727545720161948823206440518081504556346829671723286782437916272838033415471073108501919548529007337724822783525742386454014691736602477652346609",
            "27997833911221327870829467638722601621070446786955428537560009929326128400107609345671052955360856061822351910951365788637105954482006576775098580557613579098734950144178863178946295187237869221823983",
            "245246644900278211976517663573088018467026787678332759743414451715061600830038587216952208399332071549103626827191679864079776723243005600592035631246561218465817904100131859299619933817012149335034875870551067",
            "74037563479561712828046796097429573142593188889231289084936232638972765034028266276891996419625117843995894330502127585370118968098286733173273108930900552505116877063299072396380786710086096962537934650563796359",
            "2260138526203405784941654048610197513508038915719776718321197768109445641817966676608593121306582577250631562886676970448070001811149711863002112487928199487482066070131066586646083327982803560379205391980139946496955261",
            "17969491597941066732916128449573246156367561808012600070888918835531726460341490933493372247868650755230855864199929221814436684722874052065257937495694348389263171152522525654410980819170611742509702440718010364831638288518852689",
            "1230186684530117755130494958384962720772853569595334792197322452151726400507263657518745202199786469389956474942774063845925192557326303453731548268507917026122142913461670429214311602221240479274737794080665351419597459856902143413",
            "124620366781718784065835044608106590434820374651678805754818788883289666801188210855036039570272508747509864768438458621054865537970253930571891217684318286362846948405301614416430468066875699415246993185704183030512549594371372159029236099",
            "2140324650240744961264423072839333563008614715144755017797754920881418023447140136643345519095804679610992851872470914587687396261921557363047454770520805119056493106687691590019759405693457452230589325976697471681738069364894699871578494975937497937",
            "22112825529529666435281085255026230927612089502470015394413748319128822941402001986512729726569746599085900330031400051170742204560859276357953757185954298838958709229238491006703034124620545784566413664540684214361293017694020846391065875914794251435144458199",
            "233108530344407544527637656910680524145619812480305449042948611968495918245135782867888369318577116418213919268572658314913060672626911354027609793166341626693946596196427744273886601876896313468704059066746903123910748277606548649151920812699309766587514735456594993207",
            "412023436986659543855531365332575948179811699844327982845455626433876445565248426198098870423161841879261420247188869492560931776375033421130982397485150944909106910269861031862704114880866970564902903653658867433731720813104105190864254793282601391257624033946373269391",
            "1790707753365795418841729699379193276395981524363782327873718589639655966058578374254964039644910359346857311359948708984278578450069871685344678652553655035251602806563637363071753327728754995053415389279785107516999221971781597724733184279534477239566789173532366357270583106789",
            "30502351862940031577691995198949664002982179597487683486715266186733160876943419156362946151249328917515864630224371171221716993844781534383325603218163254920110064990807393285889718524383600251199650576597076902947432221039432760575157628357292075495937664206199565578681309135044121854119",
            "276931556780344213902868906164723309223760836398395325400503672280937582471494739461900602187562551243171865731050750745462388288171212746300721613469564396741836389979086904304472476001839015983033451909174663464663867829125664459895575157178816900228792711267471958357574416714366499722090015674047",
            "133294399882575758380143779458803658621711224322668460285458826191727627667054255404674269333491950155273493343140718228407463573528003686665212740575911870128339157499072351179666739658503429931021985160714113146720277365006623692721807916355914275519065334791400296725853788916042959771420436564784273910949",
            "135066410865995223349603216278805969938881475605667027524485143851526510604859533833940287150571909441798207282164471551373680419703964191743046496589274256239341020864383202110372958725762358509643110564073501508187510676594629205563685529475213500852879416377328533906109750544334999811150056977236890927563",
            "1848210397825850670380148517702559371400899745254512521925707445580334710601412527675708297932857843901388104766898429433126419139462696524583464983724651631481888473364151368736236317783587518465017087145416734026424615690611620116380982484120857688483676576094865930188367141388795454378671343386258291687641",
            "21368106964100717960120874145003772958637679383727933523150686203631965523578837094085435000951700943373838321997220564166302488321590128061531285010636857163897899811712284013921068534616772684717323224436400485097837112174432182703436548357540610175031371364893034379963672249152120447044722997996160892591129924218437",
            "121870863310605869313817398014332524915771068622605522040866660001748138323813524568024259035558807228052611110790898823037176326388561409009333778630890634828167900405006112727432172179976427017137792606951424995281839383708354636468483926114931976844939654102090966520978986231260960498370992377930421701862444655244698696759267",
            "2690987062294695111996484658008361875931308730357496490239672429933215694995275858877122326330883664971511275673199794677960841323240693443353204889858591766765807522315638843948076220761775866259739752361275228111366001104150630004691128152106812042872285697735145105026966830649540003659922618399694276990464815739966698956947129133275233",
            "26507199951735394734498120973736811015297864642115831624674545482293445855043495841191504413349124560193160478146528433707807716865391982823061751419151606849655575049676468644737917071142487312863146816801954812702917123189212728868259282632393834443989482096498000219878377420094983472636679089765013603382322972552204068806061829535529820731640151",
            "218682020234317263146640637228579265464915856482838406521712186637422774544877649638896808173342116436377521579949695169845394824866781413047516721975240052350576247238785129338002757406892629970748212734663781952170745916609168935837235996278783280225742175701130252626518426356562342682345652253987471761591019113926725623095606566457918240614767013806590649",
            "1888287707234383972842703127997127272470910519387718062380985523004987076701721281993726195254903980001896112258671262466144228850274568145436317048469073794495250347974943216943521462713202965796237266310948224934556725414915442700993152879235272779266578292207161032746297546080025793864030543617862620878802244305286292772467355603044265985905970622730682658082529621",
            "30135004431202116003565860241012769924921679977958392035283632366105785657918270750937407901898070219843622821090980641477056850056514799336625349678549218794180711634478735831265177285887805862071748980072533360656419736316535822377792634235019526468475796787118257207337327341698664061454252865816657556977260763553328252421574633011335112031733393397168350585519524478541747311",
            "268040194118238845450103707934665606536694174908285267872982242439770917825046230024728489676042825623316763136454136724676849961188128997344512282129891630084759485063423604911639099585186833094019957687550377834977803400653628695534490436743728187025341405841406315236881249848600505622302828534189804007954474358650330462487514752974123986970880843210371763922883127855444022091083492089",
            "2014096878945207511726700485783442547915321782072704356103039129009966793396141985086509455102260403208695558793091390340438867513766123418942845301603261911930567685648626153212566300102683464717478365971313989431406854640516317519403149294308737302321684840956395183222117468443578509847947119995373645360710979599471328761075043464682551112058642299370598078702810603300890715874500584758146849481",
            "19653601479938761414239452741787457079262692944398807468279711209925174217701079138139324539033381077755540830342989643633394137538983355218902490897764441296847433275460853182355059915490590169155909870689251647778520385568812706350693720915645943335281565012939241331867051414851378568457417661501594376063244163040088180887087028771717321932252992567756075264441680858665410918431223215368025334985424358839",
            "209136630247651073165255642316333073700965362660524505479852295994129273025818983735700761887526097496489535254849254663948005091692193449062731454136342427186266197097846022969248579454916155633686388106962365337549155747268356466658384680996435419155013602317010591744105651749369012554532024258150373034059528878269258139126839427564311148202923131937053527161657901326732705143817744164107601735413785886836578207979",
            "3534635645620271361541209209607897224734887106182307093292005188843884213420695035531516325888970426873310130582000012467805106432116010499008974138677724241907444538851271730464985654882214412422106879451855659755824580313513382070785777831859308900851761495284515874808406228585310317964648830289141496328996622685469256041007506727884038380871660866837794704723632316890465023570092246473915442026549955865931709542468648109541",
            "26014282119556025900707884873713205505398108045952352894235085896633912708374310252674800592426746319007978890065337573160541942868114065643853327229484502994233222617112392660635752325773689366745234119224790516838789368452481803077294973049597108473379738051456732631199164835297036074054327529666307812234597766390750441445314408171802070904072739275930410299359006059619305590701939627725296116299946059898442103959412221518213407370491",
            "198463423714283662349723072186113142778946286925886208987853800987159869256900787915916842423672625297046526736867114939854460034942655873583931553781158032447061155145160770580926824366573211993981662614635734812647448360573856313224749171552699727811551490561895325344395743588150359341484236709604618276434347948498243152515106628556992696242074513657383842554978233909962839183287667419172988072221996532403300258906083211160744508191024837057033",
            "1786856020404004433262103789212844585886400086993882955081051578507634807524146407881981216968139444577147633460848868774625431829282860339614956262303635645546753552581286559710032014178315212224644686666427660441466419337888368932452217321354860484353296131403821175862890998598653858373835628654351880480636223164308238684873105235011577671552114945370886842810830301698313339004163655154668570049008475016448080768256389182668489641536264864604484300734909",
            "1847699703211741474306835620200164403018549338663410171471785774910651696711161249859337684305435744585616061544571794052229717732524660960646946071249623720442022269756756687378427562389508764678440933285157496578843415088475528298186726451339863364931908084671990431874381283363502795470282653297802934916155811881049844908319545009848393775227257052578591944993870073695755688436933812779613089230392569695253261620823676490316036551371447913932347169566988069",
            "17051473784681185209081599238887028025183255852149159683588918369809675398036897711442383602526314519192366612270595815510311970886116763177669964411814095748660238871306469830461919135901638237924444074122866545522954536883748558744552128950445218096208188788876324395049362376806579941053305386217595984047709603954312447692725276887594590658792939924609261264788572032212334726855302571883565912645432522077138010357669555555071044090857089539320564963576770285413369",
            "302657075295090869739730250315591803589112283576939858395529632634305976144571441696598170401251852159138533455982172343712313383247732107268535247763784105186549246199888070331088462855743520880671299302895546822695492968577380706795842802200829411198422297326020823369315258921162990168697393348736236081296604185145690639952829781767901497605213955485328141965346769742597479306858645849268328985687423881853632604706175564461719396117318298679820785491875674946700413680932103",
            "1860239127076846517198369354026076875269515930592839150201028353837031025971373852216474332794920643399906822553185507255460678213880084116286603739332465781718042017172224499540303152935478714013629615010650024865526886634157459758925793594165651020789220067311416926076949777767604906107061937873540601594274731617619377537419071307115490065850326946551649682856865437718319058695376406980449326388934924579147508558589808491904883853150769224537555274811376719096144119390052199027715691",
            "18971941337486266563305347433172025272371835919534283031845811230624504588707687605943212347625766427494554764419515427586743205659317254669946604982419730160103812521528540068803151640161162396312837062979326593940508107758169447860417214110246410380402787011098086642148000255604546876251377453934182215494821277335671735153472656328448001134940926442438440198910908603252678814785060113207728717281994244511323201949222955423789860663107489107472242561739680319169243814676235712934292299974411361",
            "22701801293785014193580405120204586741061235962766583907094021879215171483119139894870133091111044901683400949483846818299518041763507948922590774925466088171879259465921026597046700449819899096862039460017743094473811056991294128542891880855362707407670722593737772666973440977361243336397308051763091506836310795312607239520365290032105848839507981452307299417185715796297454995023505316040919859193718023307414880446217922800831766040938656344571034778553457121080530736394535923932651866030515041060966437313323672831539323500067937107541955437362433248361242525945868802353916766181532375855504886901432221349733",
            "25195908475657893494027183240048398571429282126204032027777137836043662020707595556264018525880784406918290641249515082189298559149176184502808489120072844992687392807287776735971418347270261896375014971824691165077613379859095700097330459748808428401797429100642458691817195118746121515172654632282216869987549182422433637259085141865462043576798423387184774447920739934236584823824281198163815010674810451660377306056201619676256133844143603833904414952634432190114657544454178424020924616515723350778707749817125772467962926386356373289912154831438167899885040445364023527381951378636564391212010397122822120720357",
        ];

        for n in rsa_semiprimes {
            assert!(is_prime(&n) == false);
        }
    }
}
