use crate::Readonly;

use std::{
  error::Error,
  fmt::{self, Display, Formatter},
};

#[derive(Copy, Clone, Debug)]
pub struct OutOfRangeError(Readonly<usize>);

impl Display for OutOfRangeError {
  fn fmt(&self, formatter: &mut Formatter<'_>) -> Result<(), fmt::Error> {
    write!(
      formatter,
      "The id {} has not been generated by IDManager::next() yet!",
      *self.0
    )
  }
}

impl Error for OutOfRangeError {}

/// It manages and recycles id numbers to be used for any purposes so that id numbers will not become depleted if users
/// keep retrieving and discarding id numbers.
#[derive(Clone, Debug, Default)]
pub struct IDManager {
  id_stack: Vec<usize>,
  next_id: usize,
}

impl IDManager {
  pub const fn new() -> Self {
    Self {
      id_stack: vec![],
      next_id: 0,
    }
  }

  /// It makes the allocated id number to become available again and ready to be used in the future.
  ///
  /// `id`: The allocated id number to return.
  pub fn free(&mut self, id: usize) -> Result<(), OutOfRangeError> {
    debug_assert!(
      !self.id_stack.contains(&id),
      "id {id} has already been freed and cannot be freed again!"
    );
    if id < self.next_id {
      self.id_stack.push(id);
      Ok(())
    } else {
      Err(OutOfRangeError(id.into()))
    }
  }
}

impl Iterator for IDManager {
  type Item = usize;

  fn next(&mut self) -> Option<Self::Item> {
    if let Some(id) = self.id_stack.pop() {
      Some(id)
    } else {
      let result = self.next_id;
      self.next_id += 1;
      Some(result)
    }
  }
}
