use std::{fmt, io};

#[derive(PartialEq)]
pub enum Error {
    GenericError(String),
    IoError(String),
    RecordNotFound,
}

impl From<io::Error> for Error {
    fn from(err: io::Error) -> Error {
        Error::IoError(err.to_string())
    }
}

impl From<&str> for Error {
    fn from(err: &str) -> Error {
        Error::GenericError(err.to_string())
    }
}

impl From<std::string::FromUtf8Error> for Error {
    fn from(err: std::string::FromUtf8Error) -> Error {
        Error::GenericError(err.to_string())
    }
}

impl From<std::array::TryFromSliceError> for Error {
    fn from(err: std::array::TryFromSliceError) -> Error {
        Error::GenericError(err.to_string())
    }
}

impl From<std::net::AddrParseError> for Error {
    fn from(err: std::net::AddrParseError) -> Error {
        Error::GenericError(err.to_string())
    }
}

impl fmt::Debug for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Error::GenericError(msg) => write!(f, "GenericError: {}", msg)?,
            Error::IoError(msg) => write!(f, "IoError: {}", msg)?,
            Error::RecordNotFound => write!(f, "RecordNotFound: no record found")?,
        }
        Ok(())
    }
}
