// Copyright (c) 2021 Soni L.
//
// Licensed under the MIT license.
// Documentation and comments licensed under CC BY-SA 4.0.

use ::std::marker::PhantomData;

use ::iosonism::strcursor::ReadError;
use ::iosonism::strcursor::StringReader;

/// An error callback.
pub trait ErrorFunc<'a, C: StringReader<'a>> {
    fn call<'b>(context: &C, ty: ErrorType<'b>);
}

/// An implementation of various Iosonism errors that calls T.
pub struct ErrorCall<T>(PhantomData<T>);

#[non_exhaustive]
#[derive(PartialEq, Eq, Debug)]
pub enum ErrorType<'a> {
    InvalidInteger(&'a str),
    ExpectedInteger,
    InvalidFloat(&'a str),
    ExpectedFloat,
    InvalidBool(&'a str),
    ExpectedBool,
    ExpectedStartOfQuote,
    ExpectedEndOfQuote,
    InvalidEscape(&'a str),
    ExpectedSymbol(&'a str),
}

impl<T> ::std::fmt::Display for ErrorCall<T> {
    fn fmt(&self, f: &mut ::std::fmt::Formatter) -> ::std::fmt::Result {
        write!(f, "error!")
    }
}

impl<T> ::std::fmt::Debug for ErrorCall<T> {
    fn fmt(&self, f: &mut ::std::fmt::Formatter) -> ::std::fmt::Result {
        write!(f, "ErrorCall")
    }
}

impl<T> ::std::error::Error for ErrorCall<T> {
}

impl<'a, C: StringReader<'a>, T> ReadError<'a, C> for ErrorCall<T>
where T: ErrorFunc<'a, C> {
    fn invalid_integer(context: &C, from: &str) -> Self {
        T::call(context, ErrorType::InvalidInteger(from));
        Self(PhantomData)
    }
    fn expected_integer(context: &C) -> Self {
        T::call(context, ErrorType::ExpectedInteger);
        Self(PhantomData)
    }
    fn invalid_float(context: &C, from: &str) -> Self {
        T::call(context, ErrorType::InvalidFloat(from));
        Self(PhantomData)
    }
    fn expected_float(context: &C) -> Self {
        T::call(context, ErrorType::ExpectedFloat);
        Self(PhantomData)
    }
    fn invalid_bool(context: &C, from: &str) -> Self {
        T::call(context, ErrorType::InvalidBool(from));
        Self(PhantomData)
    }
    fn expected_bool(context: &C) -> Self {
        T::call(context, ErrorType::ExpectedBool);
        Self(PhantomData)
    }
    fn expected_start_of_quote(context: &C) -> Self {
        T::call(context, ErrorType::ExpectedStartOfQuote);
        Self(PhantomData)
    }
    fn expected_end_of_quote(context: &C) -> Self {
        T::call(context, ErrorType::ExpectedEndOfQuote);
        Self(PhantomData)
    }
    fn invalid_escape(context: &C, from: &str) -> Self {
        T::call(context, ErrorType::InvalidEscape(from));
        Self(PhantomData)
    }
    fn expected_symbol(context: &C, from: &str) -> Self {
        T::call(context, ErrorType::ExpectedSymbol(from));
        Self(PhantomData)
    }
}

