#![crate_name = "intset"]
// Copyright 2020 Rob King
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//! This crate provides a collection of data structures for storing sets of integers.
//! The different data structures are designed to make different operations efficient.
//!
//! All of the data structures in this crate support the following operations with the
//! associated complexity:
//!
//! - **contains** - check if an integer is in the set in O(1) time
//! - **iterate** - iterate over the members of the set in O(*n*) time, where *n* is the number of
//!                 elements in the set
//! - **len** - return the number of elements in the set in O(1) time
//!
//! Individual set data structures support additional operations, as documented below.
//!
//! All of the set data structures in this crate have a maximum capacity, specified as the
//! largest integer that can be stored in the set plus one. Once a set is created, it does
//! no further allocations.

/// A `GrowSet` is a set of integers that supports efficient addition and clearing.
///
/// It supports the following additional operations with the associated complexity:
///
/// - **add** - add an integer to the set in O(1) time
/// - **clear** - remove all members from the set in O(1) time
/// - **pop** - remove and return a random member of the set in O(1) time
///
/// `GrowSet`s are useful for sets that need to be cleared frequently and rebuilt.
///
/// The implementation of `GrowSet` is based on "An Efficient Represtation of Sparse Sets"
/// by Briggs and Torczon (1993).
#[derive(Debug)]
pub struct GrowSet {
    n: usize,
    sparse: Vec<usize>,
    dense: Vec<usize>,
}

impl GrowSet {
    /// Returns a GrowSet with the given capacity.
    ///
    /// # Arguments
    ///
    /// * `size` - the set can store all integers up to, but not including, this value.
    ///
    /// # Examples
    ///
    /// ```
    /// // This set can store the integers 0, 1, 2, 3, and 4.
    /// use intset::GrowSet;
    /// let mut set = GrowSet::with_capacity(5);
    /// set.add(3);
    /// assert!(set.contains(3));
    /// ```
    pub fn with_capacity(size: usize) -> Self {
        Self {
            n: 0,
            sparse: vec![0; size],
            dense: vec![0; size],
        }
    }

    /// Returns one greater than the largest integer that can be stored in this set.
    /// That is, a set that can store integers from 0 to 5 will have a capacity of 6.
    pub fn capacity(&self) -> usize {
        self.sparse.capacity()
    }

    /// Return the number of integers in the set.
    ///
    /// # Examples
    ///
    /// ```
    /// use intset::GrowSet;
    /// let mut set = GrowSet::with_capacity(5);
    /// set.add(1);
    /// set.add(4);
    /// assert_eq!(set.len(), 2);
    /// ```
    pub fn len(&self) -> usize {
        self.n
    }

    /// Returns true if the specified value is in the set.
    ///
    /// # Arguments
    /// * `value` - the element to test
    ///
    /// # Examples
    ///
    /// ```
    /// use intset::GrowSet;
    /// let mut set = GrowSet::with_capacity(5);
    /// set.add(1);
    /// assert!(set.contains(1));
    /// assert!(!set.contains(3));
    /// ```
    pub fn contains(&self, value: usize) -> bool {
        value < self.sparse.len()
            && self.sparse[value] < self.n
            && self.dense[self.sparse[value]] == value
    }

    /// Remove all items from the set.
    pub fn clear(&mut self) {
        self.n = 0;
    }

    /// Add an item to the set. The item must be less than the capacity of the set.
    ///
    /// # Arguments
    ///
    /// * `value` - the value to store
    ///
    /// # Examples
    ///
    /// ```
    /// use intset::GrowSet;
    /// let mut set = GrowSet::with_capacity(100);
    /// set.add(42);
    /// assert!(set.contains(42));
    /// ```
    pub fn add(&mut self, value: usize) {
        self.dense[self.n] = value;
        self.sparse[value] = self.n;
        self.n += 1;
    }

    /// Returns an iterator over the values in the set.
    /// Uniqueness is guaranteed; ordering is not.
    pub fn iter(&self) -> impl Iterator<Item = &usize> + '_ {
        self.dense.iter().take(self.n)
    }

    /// Returns a random element of the set in constant
    /// time, or None if the set is empty.
    pub fn pop(&mut self) -> Option<usize> {
        if self.n == 0 {
            None
        } else {
            self.n -= 1;
            Some(self.dense[self.n])
        }
    }
}

#[cfg(test)]
mod grow_set_tests {
    use super::*;

    #[test]
    fn test_add() {
        let mut set = GrowSet::with_capacity(6);
        set.add(3);
        set.add(4);
        assert!(!set.contains(0));
        assert!(!set.contains(1));
        assert!(!set.contains(2));
        assert!(set.contains(3));
        assert!(set.contains(4));
        assert!(!set.contains(6));
    }

    #[test]
    fn test_clear() {
        let mut set = GrowSet::with_capacity(6);
        set.add(3);
        set.add(4);
        assert!(!set.contains(0));
        assert!(!set.contains(1));
        assert!(!set.contains(2));
        assert!(set.contains(3));
        assert!(set.contains(4));
        assert!(!set.contains(6));

        set.clear();
        assert!(!set.contains(0));
        assert!(!set.contains(1));
        assert!(!set.contains(2));
        assert!(!set.contains(3));
        assert!(!set.contains(4));
        assert!(!set.contains(6));
    }

    #[test]
    fn test_iter() {
        let mut set = GrowSet::with_capacity(6);
        set.add(0);
        set.add(2);
        set.add(4);

        for i in set.iter() {
            match i {
                0 | 2 | 4 => assert!(true),
                _ => unreachable!(),
            }
        }

        set.clear();

        for _ in set.iter() {
            unreachable!();
        }
    }

    #[test]
    fn test_pop() {
        let mut set = GrowSet::with_capacity(6);
        set.add(0);
        set.add(2);
        set.add(4);

        assert_eq!(set.pop(), Some(4));
        assert_eq!(set.pop(), Some(2));
        assert_eq!(set.pop(), Some(0));
        assert_eq!(set.pop(), None);

        set.add(4);
        assert_eq!(set.pop(), Some(4));
        assert_eq!(set.pop(), None);
    }
}

/// A `ShrinkSet` is a set of integers that supports efficient removal and refilling.
///
/// `ShrinkSet`s are automatically initialized such that they contain all integers
/// up to, but not including, their capacity. For example, a `ShrinkSet` with a capacity
/// of 5 will contain the integers 0, 1, 2, 3, and 4 upon initialization.
///
/// A `ShrinkSet` supports the following additional operations with the associated
/// time complexity:
///
/// - **remove** - remove an integer from the set in O(1) time
/// - **refill** - adds all removed elements back into the set in O(1) time
/// - **pop** - remove and return a random member of the set in O(1) time
///
/// `ShrinkSet`s are useful for situations where we want to prune search spaces or
/// work queues (for example) and then reset them to their initial state efficiently.
///
/// The algorithm used by `ShrinkSet` is believed to be novel, but if there is existing
/// literature, please let me know so I can reference it here.
#[derive(Debug)]
pub struct ShrinkSet {
    p: usize,
    map: Vec<usize>,
    values: Vec<usize>,
}

impl ShrinkSet {
    /// Returns a `ShrinkSet` with the given capacity.
    ///
    /// # Arguments
    ///
    /// * `size` - at creation time, the set will contain all integers up to, but not including, this value
    ///
    /// # Examples
    ///
    /// ```
    /// // This set stores all integers 0..=4.
    /// use intset::ShrinkSet;
    /// let mut set = ShrinkSet::new(5);
    /// assert!(set.contains(3));
    /// assert!(set.contains(4));
    /// ```
    pub fn new(size: usize) -> Self {
        Self {
            p: size,
            map: (0..size).collect(),
            values: (0..size).collect(),
        }
    }

    /// Returns one greater than the largest integer that can be stored in this set.
    /// That is, a set that can store integers from 0 to 5 will have a capacity of 6.
    pub fn capacity(&self) -> usize {
        self.map.capacity()
    }

    /// Return the number of integers in the set.
    ///
    /// # Examples
    ///
    /// ```
    /// use intset::ShrinkSet;
    /// let mut set = ShrinkSet::new(5);
    /// set.remove(1);
    /// set.remove(4);
    /// assert_eq!(set.len(), 3);
    /// ```
    pub fn len(&self) -> usize {
        self.p
    }

    /// Returns true if the specified value is in the set.
    ///
    /// # Arguments
    /// * `value` - the element to test
    ///
    /// # Examples
    ///
    /// ```
    /// use intset::ShrinkSet;
    /// let mut set = ShrinkSet::new(5);
    /// set.remove(1);
    /// assert!(!set.contains(1));
    /// assert!(set.contains(3));
    /// ```
    pub fn contains(&self, value: usize) -> bool {
        value < self.map.len() && self.map[value] < self.p
    }

    /// Refills the set by adding all removed elements back.
    ///
    /// # Examples
    ///
    /// ```
    /// use intset::ShrinkSet;
    /// let mut set = ShrinkSet::new(5);
    /// set.remove(0);
    /// set.remove(2);
    /// assert!(set.contains(1));
    /// assert!(set.contains(3));
    /// assert!(set.contains(4));
    /// assert!(!set.contains(0));
    /// assert!(!set.contains(2));
    ///
    /// set.refill();
    /// assert!(set.contains(1));
    /// assert!(set.contains(3));
    /// assert!(set.contains(4));
    /// assert!(set.contains(0));
    /// assert!(set.contains(2));
    /// ```
    pub fn refill(&mut self) {
        self.p = self.map.len();
    }

    /// Returns an iterator over the values in the set.
    /// Uniqueness is guaranteed; ordering is not.
    pub fn iter(&self) -> impl Iterator<Item = &usize> + '_ {
        self.values.iter().take(self.p)
    }

    /// Remove an element from the set.
    ///
    /// # Arguments
    /// - `item` - the item to remove
    ///
    /// # Examples
    ///
    /// ```
    /// use intset::ShrinkSet;
    /// let mut set = ShrinkSet::new(5);
    /// set.remove(0);
    /// set.remove(2);
    /// assert!(set.contains(1));
    /// assert!(set.contains(3));
    /// assert!(!set.contains(0));
    /// assert!(!set.contains(2));
    /// ```
    pub fn remove(&mut self, item: usize) -> usize {
        if self.contains(item) {
            let item_index = self.map[item];
            let last_item = self.values[self.p - 1];
            let last_item_index = self.map[last_item];

            self.values[last_item_index] = item;
            self.values[item_index] = last_item;
            self.map[last_item] = item_index;
            self.map[item] = last_item_index;
            self.p -= 1;
        }

        item
    }

    /// Remove and return a random element from the set
    /// in constant time, or None if the set is empty.
    pub fn pop(&mut self) -> Option<usize> {
        if self.p == 0 {
            None
        } else {
            Some(self.remove(self.values[0]))
        }
    }
}

#[cfg(test)]
mod shrink_set_tests {
    use super::*;

    #[test]
    fn test_new() {
        let set = ShrinkSet::new(5);
        assert!(set.contains(0));
        assert!(set.contains(1));
        assert!(set.contains(2));
        assert!(set.contains(3));
        assert!(set.contains(4));
    }

    #[test]
    fn test_remove() {
        let mut set = ShrinkSet::new(6);
        assert_eq!(set.remove(1), 1);
        assert_eq!(set.remove(3), 3);
        assert_eq!(set.remove(5), 5);

        assert!(set.contains(0));
        assert!(!set.contains(1));
        assert!(set.contains(2));
        assert!(!set.contains(3));
        assert!(set.contains(4));
        assert!(!set.contains(5));
    }

    #[test]
    fn test_refill() {
        let mut set = ShrinkSet::new(6);
        set.remove(1);
        set.remove(3);
        set.remove(5);

        assert!(set.contains(0));
        assert!(!set.contains(1));
        assert!(set.contains(2));
        assert!(!set.contains(3));
        assert!(set.contains(4));
        assert!(!set.contains(5));

        set.refill();

        assert!(set.contains(0));
        assert!(set.contains(1));
        assert!(set.contains(2));
        assert!(set.contains(3));
        assert!(set.contains(4));
        assert!(set.contains(5));
    }

    #[test]
    fn test_iter() {
        let mut set = ShrinkSet::new(6);
        set.remove(1);
        set.remove(3);
        set.remove(5);

        for i in set.iter() {
            match i {
                1 | 3 | 5 => unreachable!(),
                _ => assert!(true),
            }
        }

        set.refill();

        assert!(set.contains(0));
        assert!(set.contains(1));
        assert!(set.contains(2));
        assert!(set.contains(3));
        assert!(set.contains(4));
        assert!(set.contains(5));
    }

    #[test]
    fn test_pop() {
        let mut set = ShrinkSet::new(4);
        set.remove(3);

        let i = set.pop();
        assert!(i == Some(0) || i == Some(1) || i == Some(2));

        let j = set.pop();
        assert!(j == Some(0) || j == Some(1) || j == Some(2));
        assert!(j != i);

        let k = set.pop();
        assert!(k == Some(0) || k == Some(1) || k == Some(2));
        assert!(k != i && k != j);

        assert_eq!(set.pop(), None);

        set.refill();

        set.remove(3);

        let i = set.pop();
        assert!(i == Some(0) || i == Some(1) || i == Some(2));

        let j = set.pop();
        assert!(j == Some(0) || j == Some(1) || j == Some(2));
        assert!(j != i);

        let k = set.pop();
        assert!(k == Some(0) || k == Some(1) || k == Some(2));
        assert!(k != i && k != j);

        assert_eq!(set.pop(), None);
    }
}
