use proc_macro2::{Span, TokenStream};
use syn::spanned::Spanned;
use syn::{Fields, ItemEnum};

use crate::input::{InputEnum, Variant, VariantFields};
use crate::parse_enum::InputParseError::{
    HasGenerics, VariantHasMultipleFields, VariantHasNamedFields, VariantHasNoFields,
};

#[derive(Clone, Debug)]
pub enum InputParseError {
    ParseError(syn::Error),
    HasGenerics(Span),
    VariantHasNamedFields(Span),
    VariantHasNoFields(Span),
    VariantHasMultipleFields(Span),
}

pub fn parse_enum(input: TokenStream) -> Result<InputEnum, Vec<InputParseError>> {
    let parsed =
        syn::parse2::<ItemEnum>(input).map_err(|e| vec![InputParseError::ParseError(e)])?;

    if !parsed.generics.params.is_empty() {
        return Err(vec![HasGenerics(parsed.generics.span())]);
    }

    let variants: Vec<_> = parsed.variants.iter().map(parse_variant).collect();

    let errors: Vec<_> = variants
        .iter()
        .filter_map(|result| result.as_ref().err().cloned())
        .collect();
    if !errors.is_empty() {
        return Err(errors);
    }

    let unwraped = variants
        .into_iter()
        .map(|result| result.expect("unreachable: checked for errors"))
        .collect();

    Ok(InputEnum::new(parsed.ident, unwraped))
}

fn parse_variant(variant: &syn::Variant) -> Result<Variant, InputParseError> {
    match &variant.fields {
        Fields::Unnamed(fields) => {
            let field_vec: Vec<_> = fields.unnamed.iter().collect();
            if field_vec.is_empty() {
                return Err(VariantHasNoFields(fields.span()));
            }
            if field_vec.len() > 1 {
                return Err(VariantHasMultipleFields(fields.span()));
            }

            return Ok(Variant::new(
                variant.ident.clone(),
                VariantFields::Single(field_vec[0].ty.clone()),
            ));
        }
        Fields::Named(_) => Err(VariantHasNamedFields(variant.span())),
        Fields::Unit => Err(VariantHasNoFields(variant.span())),
    }
}
