//! Generates C bindings for [Interoptopus](https://github.com/ralfbiedert/interoptopus).
//!
//! ## Usage
//!
//! In your library or a support project add this:
//!
//! ```
//! # mod my_crate { use interoptopus::{Library}; pub fn ffi_inventory() -> Library { todo!() } }
//! use my_crate::ffi_inventory;
//!
//! #[test]
//! fn generate_c_bindings() {
//!     use interoptopus::Interop;
//!     use interoptopus_backend_c::{Generator, InteropC, Config};
//!
//!     // Converts an `ffi_inventory()` into Python interop definitions.
//!     Generator::new(Config::default(), ffi_inventory()).write_to("module.h")
//! }
//! ```
//!
//! And we might produce something like this:
//!
//! ```c
//!
//! #ifndef module
//! #define module
//!
//! #ifdef __cplusplus
//! extern "C" {
//! #endif
//!
//! #include <stdint.h>
//! #include <stdbool.h>
//!
//! typedef struct Vec3f32
//! {
//!     float x;
//!     float y;
//!     float z;
//! } Vec3f32;
//!
//! Vec3f32 my_game_function(Vec3f32* input);
//!
//! #ifdef __cplusplus
//! }
//! #endif
//!
//! #endif /* module */
//!
//! ```

use interoptopus::lang::c::{
    CType, CompositeType, Constant, ConstantValue, Documentation, EnumType, Field, FnPointerType, Function, OpaqueType, Parameter, PrimitiveType, PrimitiveValue, Variant,
};
use interoptopus::patterns::TypePattern;
use interoptopus::util::{safe_name, sort_types_by_dependencies};
use interoptopus::writer::IndentWriter;
use interoptopus::Interop;
use interoptopus::{Error, Library};

/// Configures C code generation.
#[derive(Clone, Debug)]
pub struct Config {
    /// Whether to write conditional directives like `#ifndef _X`.
    pub directives: bool,
    /// Whether to write `#include <>` directives.
    pub imports: bool,
    /// The `_X` in `#ifndef _X` to be used.
    pub ifndef: String,
    /// Multiline string with custom `#define` values.
    pub custom_defines: String,
    /// Prefix to be applied to any function, e.g., `__DLLATTR`.
    pub function_attribute: String,
    /// Comment at the very beginning of the file, e.g., `// (c) My Company.`
    pub file_header_comment: String,
}

impl Default for Config {
    fn default() -> Self {
        Self {
            directives: true,
            imports: true,
            file_header_comment: "// Automatically generated by Interoptopus.".to_string(),
            ifndef: "interoptopus_generated".to_string(),
            custom_defines: "".to_string(),
            function_attribute: "".to_string(),
        }
    }
}

/// Helper type implementing [`InteropC`] and [`Interop`].
pub struct Generator {
    config: Config,
    library: Library,
}

impl Generator {
    pub fn new(config: Config, library: Library) -> Self {
        Self { config, library }
    }
}

/// Contains all C generators, create sub-trait to customize.
pub trait InteropC {
    /// Returns the user config.
    fn config(&self) -> &Config;

    /// Returns the library to produce bindings for.
    fn library(&self) -> &Library;

    /// Converts a primitive (Rust) type to a native C# type name, e.g., `f32` to `float`.
    fn type_primitive_to_typename(&self, x: &PrimitiveType) -> String {
        match x {
            PrimitiveType::Void => "void".to_string(),
            PrimitiveType::Bool => "bool".to_string(),
            PrimitiveType::U8 => "uint8_t".to_string(),
            PrimitiveType::U16 => "uint16_t".to_string(),
            PrimitiveType::U32 => "uint32_t".to_string(),
            PrimitiveType::U64 => "uint64_t".to_string(),
            PrimitiveType::I8 => "int8_t".to_string(),
            PrimitiveType::I16 => "int16_t".to_string(),
            PrimitiveType::I32 => "int32_t".to_string(),
            PrimitiveType::I64 => "int64_t".to_string(),
            PrimitiveType::F32 => "float".to_string(),
            PrimitiveType::F64 => "double".to_string(),
        }
    }

    /// Converts a Rust enum name such as `Error` to a C# enum name `Error`.
    fn type_enum_to_typename(&self, x: &EnumType) -> String {
        x.rust_name().to_string()
    }

    /// TODO Converts an opaque Rust struct `Context` to a C# struct ``.
    fn type_opaque_to_typename(&self, opaque: &OpaqueType) -> String {
        // x.name().to_string()
        opaque.rust_name().to_string()
    }

    /// Converts an Rust struct name `Vec2` to a C# struct name `Vec2`.
    fn type_composite_to_typename(&self, x: &CompositeType) -> String {
        x.rust_name().to_string()
    }

    /// Converts an Rust `fn()` to a C# delegate name such as `InteropDelegate`.
    fn type_fnpointer_to_typename(&self, x: &FnPointerType) -> String {
        vec!["fptr".to_string(), safe_name(&x.internal_name())].join("_")
    }

    /// Converts the `u32` part in a Rust paramter `x: u32` to a C# equivalent. Might convert pointers to `out X` or `ref X`.
    fn type_to_type_specifier(&self, x: &CType) -> String {
        match x {
            CType::Primitive(x) => self.type_primitive_to_typename(x),
            CType::Enum(x) => self.type_enum_to_typename(x),
            CType::Opaque(x) => self.type_opaque_to_typename(x),
            CType::Composite(x) => self.type_composite_to_typename(x),
            CType::ReadPointer(x) => format!("{}*", self.type_to_type_specifier(x)),
            CType::ReadWritePointer(x) => format!("{}*", self.type_to_type_specifier(x)),
            CType::FnPointer(x) => self.type_fnpointer_to_typename(x),
            CType::Pattern(x) => self.type_to_type_specifier(&x.fallback_type()),
        }
    }

    fn constant_value_to_value(&self, value: &ConstantValue) -> String {
        match value {
            ConstantValue::Primitive(x) => match x {
                PrimitiveValue::Bool(x) => format!("{}", x),
                PrimitiveValue::U8(x) => format!("{}", x),
                PrimitiveValue::U16(x) => format!("{}", x),
                PrimitiveValue::U32(x) => format!("{}", x),
                PrimitiveValue::U64(x) => format!("{}", x),
                PrimitiveValue::I8(x) => format!("{}", x),
                PrimitiveValue::I16(x) => format!("{}", x),
                PrimitiveValue::I32(x) => format!("{}", x),
                PrimitiveValue::I64(x) => format!("{}", x),
                PrimitiveValue::F32(x) => format!("{}", x),
                PrimitiveValue::F64(x) => format!("{}", x),
            },
        }
    }

    fn function_parameter_to_csharp_typename(&self, x: &Parameter, _function: &Function) -> String {
        self.type_to_type_specifier(x.the_type())
    }

    fn function_name_to_csharp_name(&self, function: &Function) -> String {
        function.name().to_string()
    }

    fn write_custom_defines(&self, w: &mut IndentWriter) -> Result<(), Error> {
        writeln!(w.writer(), "{}", &self.config().custom_defines)?;
        Ok(())
    }

    fn write_file_header_comments(&self, w: &mut IndentWriter) -> Result<(), Error> {
        writeln!(w.writer(), "{}", &self.config().file_header_comment)?;
        Ok(())
    }

    fn write_imports(&self, w: &mut IndentWriter) -> Result<(), Error> {
        w.indented(|w| writeln!(w, r#"#include <stdint.h>"#))?;
        w.indented(|w| writeln!(w, r#"#include <stdbool.h>"#))?;

        Ok(())
    }

    fn write_constants(&self, w: &mut IndentWriter) -> Result<(), Error> {
        for constant in self.library().constants() {
            self.write_constant(w, constant)?;
            w.newline()?;
        }

        Ok(())
    }

    fn write_constant(&self, w: &mut IndentWriter, constant: &Constant) -> Result<(), Error> {
        w.indented(|w| write!(w, r#"#define "#))?;

        write!(w.writer(), "{} ", constant.name())?;
        write!(w.writer(), "{}", self.constant_value_to_value(constant.value()))?;

        Ok(())
    }

    fn write_functions(&self, w: &mut IndentWriter) -> Result<(), Error> {
        for function in self.library().functions() {
            self.write_function(w, function)?;
        }

        Ok(())
    }

    fn write_function(&self, w: &mut IndentWriter, function: &Function) -> Result<(), Error> {
        self.write_function_declaration(w, function)?;
        Ok(())
    }

    fn write_documentation(&self, w: &mut IndentWriter, documentation: &Documentation) -> Result<(), Error> {
        for line in documentation.lines() {
            w.indented(|w| writeln!(w, r#"/// {}"#, line))?;
        }

        Ok(())
    }

    fn write_function_declaration(&self, w: &mut IndentWriter, function: &Function) -> Result<(), Error> {
        w.indented(|w| {
            write!(
                w,
                r#"{}{} "#,
                self.config().function_attribute,
                self.type_to_type_specifier(function.signature().rval())
            )
        })?;

        write!(w.writer(), "{}(", self.function_name_to_csharp_name(function))?;

        let params = function.signature().params();
        for (i, p) in params.iter().enumerate() {
            write!(w.writer(), "{}", self.function_parameter_to_csharp_typename(p, function))?;
            write!(w.writer(), " {}", p.name())?;
            if i < params.len() - 1 {
                write!(w.writer(), ", ")?;
            }
        }

        writeln!(w.writer(), ");")?;
        Ok(())
    }

    fn write_type_definitions(&self, w: &mut IndentWriter) -> Result<(), Error> {
        for the_type in &sort_types_by_dependencies(self.library().ctypes().to_vec()) {
            self.write_type_definition(w, the_type)?;
        }

        Ok(())
    }

    fn write_type_definition(&self, w: &mut IndentWriter, the_type: &CType) -> Result<(), Error> {
        match the_type {
            CType::Primitive(_) => {}
            CType::Enum(e) => {
                self.write_type_definition_enum(w, e)?;
                w.newline()?;
            }
            CType::Opaque(o) => {
                self.write_type_definition_opaque(w, o)?;
                w.newline()?;
            }
            CType::Composite(c) => {
                self.write_type_definition_composite(w, c)?;
                w.newline()?;
            }
            CType::FnPointer(f) => {
                self.write_type_definition_fn_pointer(w, f)?;
                w.newline()?;
            }
            CType::ReadPointer(_) => {}
            CType::ReadWritePointer(_) => {}
            CType::Pattern(p) => match p {
                TypePattern::AsciiPointer => {}
                TypePattern::SuccessEnum(e) => {
                    self.write_type_definition_enum(w, e.the_enum())?;
                    w.newline()?;
                }
                TypePattern::Slice(_) => {}
            },
        }
        Ok(())
    }

    fn write_type_definition_fn_pointer(&self, w: &mut IndentWriter, the_type: &FnPointerType) -> Result<(), Error> {
        self.write_type_definition_fn_pointer_body(w, the_type)?;
        Ok(())
    }

    fn write_type_definition_fn_pointer_body(&self, w: &mut IndentWriter, the_type: &FnPointerType) -> Result<(), Error> {
        w.indented(|w| write!(w, "typedef {} ", self.type_to_type_specifier(the_type.signature().rval())))?;
        write!(w.writer(), "(*{})(", self.type_fnpointer_to_typename(the_type))?;

        let params = the_type.signature().params();
        for (i, param) in params.iter().enumerate() {
            write!(w.writer(), "{} x{}", self.type_to_type_specifier(param.the_type()), i)?;

            if i < params.len() - 1 {
                write!(w.writer(), ", ")?;
            }
        }

        writeln!(w.writer(), ");")?;
        Ok(())
    }

    fn write_type_definition_enum(&self, w: &mut IndentWriter, the_type: &EnumType) -> Result<(), Error> {
        w.indented(|w| writeln!(w, r#"typedef enum {}"#, the_type.rust_name()))?;
        w.indent();
        w.indented(|w| writeln!(w, r#"{{"#))?;

        for variant in the_type.variants() {
            self.write_type_definition_enum_variant(w, variant, the_type)?;
        }

        w.indented(|w| writeln!(w, r#"}} {};"#, the_type.rust_name()))?;
        w.unindent();
        Ok(())
    }

    fn write_type_definition_enum_variant(&self, w: &mut IndentWriter, variant: &Variant, _the_type: &EnumType) -> Result<(), Error> {
        let variant_name = variant.name();
        let variant_value = variant.value();
        w.indented(|w| writeln!(w, r#"{} = {},"#, variant_name, variant_value))?;
        Ok(())
    }

    fn write_type_definition_opaque(&self, w: &mut IndentWriter, the_type: &OpaqueType) -> Result<(), Error> {
        self.write_type_definition_opaque_body(w, the_type)?;
        Ok(())
    }

    fn write_type_definition_opaque_body(&self, w: &mut IndentWriter, the_type: &OpaqueType) -> Result<(), Error> {
        w.indented(|w| writeln!(w, r#"typedef struct {} {};"#, the_type.rust_name(), the_type.rust_name()))?;
        Ok(())
    }

    fn write_type_definition_composite(&self, w: &mut IndentWriter, the_type: &CompositeType) -> Result<(), Error> {
        if the_type.is_empty() {
            // C doesn't allow us writing empty structs.
            w.indented(|w| writeln!(w, r#"typedef struct {} {};"#, the_type.rust_name(), the_type.rust_name()))?;
            Ok(())
        } else {
            self.write_type_definition_composite_body(w, the_type)
        }
    }

    fn write_type_definition_composite_body(&self, w: &mut IndentWriter, the_type: &CompositeType) -> Result<(), Error> {
        w.indented(|w| writeln!(w, r#"typedef struct {}"#, the_type.rust_name()))?;
        w.indent();
        w.indented(|w| writeln!(w, r#"{{"#))?;

        for field in the_type.fields() {
            self.write_type_definition_composite_body_field(w, field, the_type)?;
        }

        w.indented(|w| writeln!(w, r#"}} {};"#, the_type.rust_name()))?;
        w.unindent();
        Ok(())
    }

    fn write_type_definition_composite_body_field(&self, w: &mut IndentWriter, field: &Field, _the_type: &CompositeType) -> Result<(), Error> {
        let field_name = field.name();
        let type_name = self.type_to_type_specifier(field.the_type());
        w.indented(|w| writeln!(w, r#"{} {};"#, type_name, field_name))?;
        Ok(())
    }

    fn write_ifndef(&self, w: &mut IndentWriter, f: impl FnOnce(&mut IndentWriter) -> Result<(), Error>) -> Result<(), Error> {
        if self.config().directives {
            w.indented(|w| writeln!(w, r#"#ifndef {}"#, self.config().ifndef))?;
            w.indented(|w| writeln!(w, r#"#define {}"#, self.config().ifndef))?;
            w.newline()?;
        }

        f(w)?;

        if self.config().directives {
            w.newline()?;
            w.indented(|w| writeln!(w, r#"#endif /* {} */"#, self.config().ifndef))?;
        }

        Ok(())
    }

    fn write_ifdefcpp(&self, w: &mut IndentWriter, f: impl FnOnce(&mut IndentWriter) -> Result<(), Error>) -> Result<(), Error> {
        if self.config().directives {
            w.indented(|w| writeln!(w, r#"#ifdef __cplusplus"#))?;
            w.indented(|w| writeln!(w, r#"extern "C" {{"#))?;
            w.indented(|w| writeln!(w, r#"#endif"#))?;
            w.newline()?;
        }

        f(w)?;

        if self.config().directives {
            w.newline()?;
            w.indented(|w| writeln!(w, r#"#ifdef __cplusplus"#))?;
            w.indented(|w| writeln!(w, r#"}}"#))?;
            w.indented(|w| writeln!(w, r#"#endif"#))?;
        }
        Ok(())
    }
}

impl Interop for Generator {
    fn write_to(&self, w: &mut IndentWriter) -> Result<(), Error> {
        self.write_file_header_comments(w)?;
        w.newline()?;

        self.write_ifndef(w, |w| {
            self.write_ifdefcpp(w, |w| {
                if self.config().imports {
                    self.write_imports(w)?;
                    w.newline()?;
                }

                self.write_custom_defines(w)?;
                w.newline()?;

                self.write_constants(w)?;
                w.newline()?;

                self.write_type_definitions(w)?;
                w.newline()?;

                self.write_functions(w)?;

                Ok(())
            })?;

            Ok(())
        })?;

        Ok(())
    }
}

impl InteropC for Generator {
    fn config(&self) -> &Config {
        &self.config
    }

    fn library(&self) -> &Library {
        &self.library
    }
}
