//! Fit2D spline file parser and calculator of the spline curves.
//!
//! Only text files with spline coefficients are supported.
//!
use cryiorust::frame::Array;
use std::fs::File;
use std::io;
use std::path::Path;

/// Main exported structure.
#[derive(PartialEq, Debug)]
pub struct Spline {
    x_max: f64,
    x_min: f64,
    y_max: f64,
    y_min: f64,
    grid: f64,
    pixel1: f64,
    pixel2: f64,
    x_knots: Vec<f64>,
    y_knots: Vec<f64>,
    x_order: [usize; 2],
    y_order: [usize; 2],
    x: Vec<f64>,
    y: Vec<f64>,
}

enum ParseState {
    Begin,
    ValidRegion,
    Pixels,
    XDistortion,
    YDistortion,
}

const FLOAT_ITEM_SIZE: usize = 14;

impl Spline {
    /// Creates a new empty [Spline].
    fn new() -> Spline {
        Spline {
            x_max: 0.0,
            x_min: 0.0,
            y_max: 0.0,
            y_min: 0.0,
            grid: 0.0,
            pixel1: 0.0,
            pixel2: 0.0,
            x_knots: vec![],
            y_knots: vec![],
            x_order: [0, 0],
            y_order: [0, 0],
            x: vec![],
            y: vec![],
        }
    }

    /// Creates a new [Spline] reading a buffer and using an [Array] to calculate coefficients.
    pub fn init<R: io::Read + io::BufRead>(reader: R, array: &Array) -> io::Result<Spline> {
        let mut s = Spline::parse(reader)?;
        s.calculate(array);
        Ok(s)
    }

    /// Opens a spline from a text [File]. The spline curves must be calculated after.
    pub fn open<P: AsRef<Path>>(filename: P) -> io::Result<Spline> {
        Spline::parse(io::BufReader::new(File::open(filename)?))
    }

    /// Parses a buffer to get the [Spline]. The spline curves must be calculated after.
    pub fn parse<R: io::Read + io::BufRead>(reader: R) -> io::Result<Spline> {
        let mut s = Spline::new();
        s.parse_reader(reader)?;
        Ok(s)
    }

    fn parse_reader<R: io::Read + io::BufRead>(&mut self, reader: R) -> io::Result<()> {
        let mut ss = ParseState::Begin;
        for line in reader.lines() {
            let line = line?;
            match line.trim() {
                "" => continue,
                "VALID REGION" => ss = ParseState::ValidRegion,
                "GRID SPACING, X-PIXEL SIZE, Y-PIXEL SIZE" => ss = ParseState::Pixels,
                "X-DISTORTION" => ss = ParseState::XDistortion,
                "Y-DISTORTION" => ss = ParseState::YDistortion,
                _ => match ss {
                    ParseState::Begin => (),
                    ParseState::ValidRegion => self.parse_valid_region(&line)?,
                    ParseState::Pixels => self.parse_pixels(&line)?,
                    ParseState::XDistortion => self.parse_x_distortion(&line)?,
                    ParseState::YDistortion => self.parse_y_distortion(&line)?,
                },
            }
        }
        if self.x_knots.len() < self.x_order.iter().sum()
            || self.y_knots.len() < self.y_order.iter().sum()
        {
            return Err(io::Error::new(
                io::ErrorKind::InvalidInput,
                "not enough coefficient in spline file",
            ));
        }
        if self.x_max == 0. || self.y_max == 0. {
            return Err(io::Error::new(
                io::ErrorKind::InvalidInput,
                "not a spline file",
            ));
        }
        Ok(())
    }

    fn parse_valid_region(&mut self, line: &str) -> io::Result<()> {
        let mut refs = [
            &mut self.x_min,
            &mut self.y_min,
            &mut self.x_max,
            &mut self.y_max,
        ];
        line.parse_floats(&mut refs)
    }

    fn parse_pixels(&mut self, line: &str) -> io::Result<()> {
        let mut refs = [&mut self.grid, &mut self.pixel1, &mut self.pixel2];
        line.parse_floats(&mut refs)
    }

    fn parse_x_distortion(&mut self, line: &str) -> io::Result<()> {
        if self.x_order == [0, 0] {
            return line.parse_ints(&mut self.x_order);
        }
        line.parse_knots(&mut self.x_knots)
    }

    fn parse_y_distortion(&mut self, line: &str) -> io::Result<()> {
        if self.y_order == [0, 0] {
            return line.parse_ints(&mut self.y_order);
        }
        line.parse_knots(&mut self.y_knots)
    }

    /// Calculates [Spline] curves using a certain [Array].
    pub fn calculate(&mut self, array: &Array) {
        let x_end = self.x_order.iter().sum();
        let y_end = self.y_order.iter().sum();
        let mut b = Bisplev {
            dim1: array.dim2() + 1,
            dim2: array.dim1() + 1,
            tx: &self.x_knots[0..self.x_order[0]],
            ty: &self.x_knots[self.x_order[0]..x_end],
            c: &self.x_knots[x_end..],
        };
        self.x = b.calculate();
        b.tx = &self.y_knots[0..self.y_order[0]];
        b.ty = &self.y_knots[self.y_order[0]..y_end];
        b.c = &self.y_knots[y_end..];
        self.y = b.calculate();
        for i in 0..b.dim2 {
            let k = i * b.dim1;
            for j in 0..b.dim1 {
                let n = k + j;
                self.x[n] += j as f64;
                self.y[n] += i as f64;
            }
        }
    }

    /// Returns [Spline] ```x``` curve.
    pub fn x(&self) -> &[f64] {
        &self.x
    }

    /// Returns [Spline] ```y``` curve.
    pub fn y(&self) -> &[f64] {
        &self.y
    }
}

const SPLINE_ORDER: usize = 3;

struct Bisplev<'a> {
    dim1: usize,
    dim2: usize,
    tx: &'a [f64],
    ty: &'a [f64],
    c: &'a [f64],
}

impl<'a> Bisplev<'a> {
    fn calculate(&self) -> Vec<f64> {
        let nky1 = self.ty.len() - SPLINE_ORDER - 1;
        let mut z = Vec::with_capacity(self.dim1 * self.dim2);
        let (lx, wx) = self.tx.init(self.dim1);
        let (ly, wy) = self.ty.init(self.dim2);
        for j in 0..self.dim2 {
            let k = j * (SPLINE_ORDER + 1);
            for i in 0..self.dim1 {
                let n = i * (SPLINE_ORDER + 1);
                let l3 = lx[i] * nky1 + ly[j];
                let mut sp = 0.;
                let mut err = 0.;
                for i1 in 0..=SPLINE_ORDER {
                    let l = n + i1;
                    let l4 = l3 + i1 * nky1;
                    for j1 in 0..=SPLINE_ORDER {
                        let l2 = l4 + j1;
                        let a = self.c[l2] * wx[l] * wy[k + j1] - err;
                        let tmp = sp + a;
                        err = tmp - sp - a;
                        sp = tmp;
                    }
                }
                z.push(sp);
            }
        }
        z
    }
}

trait Auxiliary {
    fn init(&self, dim: usize) -> (Vec<usize>, Vec<f64>);

    fn fpbspl(&self, x: f64, l: usize, h: &mut [f64], hh: &mut [f64]);
}

impl Auxiliary for [f64] {
    fn init(&self, dim: usize) -> (Vec<usize>, Vec<f64>) {
        let mut h = [0.; 6];
        let mut hh = [0.; 5];
        let mut ll = Vec::with_capacity(dim);
        let mut ww = Vec::with_capacity(dim * (SPLINE_ORDER + 1));
        let tb = self[SPLINE_ORDER];
        let k = self.len() - SPLINE_ORDER - 1;
        let te = self[k];
        let mut l = SPLINE_ORDER + 1;
        let mut l1 = l + 1;
        for i in 0..dim {
            let mut arg = i as f64;
            if arg < tb {
                arg = tb;
            }
            if arg > te {
                arg = te;
            }
            while !(arg < self[l] || l == k) {
                l = l1;
                l1 = l + 1;
            }
            self.fpbspl(arg, l, &mut h, &mut hh);
            ll.push(l - SPLINE_ORDER - 1);
            for j in 0..=SPLINE_ORDER {
                ww.push(h[j]);
            }
        }
        (ll, ww)
    }

    fn fpbspl(&self, x: f64, l: usize, h: &mut [f64], hh: &mut [f64]) {
        h[0] = 1.;
        for j in 1..=SPLINE_ORDER {
            for i in 0..j {
                hh[i] = h[i];
            }
            h[0] = 0.;
            for i in 0..j {
                let k = l + i;
                let n = k - j;
                let f = hh[i] / (self[k] - self[n]);
                h[i] = h[i] + f * (self[k] - x);
                h[i + 1] = f * (x - self[n]);
            }
        }
    }
}

trait LineParser {
    fn parse_knots(&self, vec: &mut Vec<f64>) -> io::Result<()>;

    fn parse_floats(&self, refs: &mut [&mut f64]) -> io::Result<()>;

    fn parse_ints(&self, slice: &mut [usize]) -> io::Result<()>;
}

impl LineParser for str {
    fn parse_knots(&self, vec: &mut Vec<f64>) -> io::Result<()> {
        let line = self.trim_end();
        if line.len() % FLOAT_ITEM_SIZE != 0 {
            return Err(io::Error::new(
                io::ErrorKind::InvalidData,
                format!("corrupted spline: bad line of float numbers: {}", line),
            ));
        }
        for i in (0..line.len()).step_by(FLOAT_ITEM_SIZE) {
            let item = line[i..i + FLOAT_ITEM_SIZE].trim();
            match item.parse::<f64>() {
                Ok(v) => vec.push(v),
                Err(e) => {
                    return Err(io::Error::new(
                        io::ErrorKind::InvalidData,
                        format!(
                            "corrupted spline: could not convert string to float: {}: {}; line: {}",
                            e, item, line
                        ),
                    ));
                }
            }
        }
        Ok(())
    }

    fn parse_floats(&self, refs: &mut [&mut f64]) -> io::Result<()> {
        let line = self.trim_end();
        if line.len() != refs.len() * FLOAT_ITEM_SIZE {
            return Err(io::Error::new(
                io::ErrorKind::InvalidInput,
                format!("corrupted spline: line is too short: {}", line),
            ));
        }
        for i in 0..refs.len() {
            let literal = line[i * FLOAT_ITEM_SIZE..(i + 1) * FLOAT_ITEM_SIZE].trim();
            match literal.parse::<f64>() {
                Ok(v) => *refs[i] = v,
                Err(e) => {
                    return Err(io::Error::new(
                        io::ErrorKind::InvalidData,
                        format!(
                            "corrupted spline: could not convert string to float: {}: {}; line: {}",
                            e, literal, line
                        ),
                    ));
                }
            }
        }
        Ok(())
    }

    fn parse_ints(&self, slice: &mut [usize]) -> io::Result<()> {
        let vec: Vec<&str> = self.split(' ').filter(|s| !s.trim().is_empty()).collect();
        if vec.len() != slice.len() {
            return Err(io::Error::new(
                io::ErrorKind::InvalidData,
                format!(
                    "corrupted spline: should be {} items in line '{}'",
                    slice.len(),
                    self
                ),
            ));
        }
        for i in 0..slice.len() {
            match vec[i].parse::<usize>() {
                Ok(v) => slice[i] = v,
                Err(e) => {
                    return Err(io::Error::new(
                        io::ErrorKind::InvalidData,
                        format!(
                            "corrupted spline: cannot convert chars to numbers: {}: in line '{}'",
                            e, self
                        ),
                    ));
                }
            }
        }
        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use crate::spline::Spline;
    use cryiorust::cbf::Cbf;
    use cryiorust::frame::{Array, Frame};
    use std::fs::File;
    use std::io::BufReader;

    fn get_test_spline() -> Spline {
        Spline {
            x_max: 2048.0,
            x_min: 0.0,
            y_max: 2048.0,
            y_min: 0.0,
            grid: 500.0,
            pixel1: 47.2,
            pixel2: 47.2,
            x_knots: vec![
                0., 0., 0., 0., 360.6083, 762.521, 1551.555, 2048., 2048., 2048., 2048., 0., 0.,
                0., 0., 505.3899, 1060.756, 1526.2, 2048., 2048., 2048., 2048., 24.68002, 19.52121,
                13.59002, 5.055466, -1.68399, -4.280879, -3.653514, 20.82285, 17.07395, 12.083,
                2.499447, -2.838655, -4.644655, -4.496968, 15.18211, 12.35614, 6.724404, -2.78299,
                -6.281503, -5.694461, -5.12374, 4.927905, 3.169188, -0.4696169, -4.024407,
                -6.167011, -4.815168, -3.586679, -1.87131, -0.8669506, 2.795098, 7.825379,
                9.104835, 5.50268, 3.52069, -1.427343, -0.2707842, 4.009722, 14.09914, 18.57977,
                15.92535, 13.79433, -3.723788, 0.09381421, 4.841388, 15.12116, 22.11635, 20.53245,
                17.63961,
            ],
            y_knots: vec![
                0., 0., 0., 0., 512.2123, 795.7775, 1064.048, 1559.302, 2048., 2048., 2048., 2048.,
                0., 0., 0., 0., 269.1021, 677.3147, 1070.146, 1468.136, 1773.886, 2048., 2048.,
                2048., 2048., 29.95735, 26.03971, 15.67342, 5.778813, 3.002887, 8.506763, 16.46872,
                20.81678, 21.94076, 32.25115, 28.48376, 17.81215, 5.932275, 2.193315, 6.591226,
                13.27262, 18.76158, 20.90431, 33.88808, 29.2512, 16.85436, 3.755977, 0.9809836,
                5.792869, 11.49634, 15.51587, 17.76502, 32.63147, 27.12963, 14.00474, 1.129311,
                -0.2720726, 5.885967, 12.26527, 15.26805, 17.03015, 30.10333, 24.4333, 11.08872,
                -1.139364, -1.281094, 6.123082, 12.88706, 15.98727, 17.37502, 26.4933, 21.8373,
                9.715258, -2.074174, -1.406571, 6.242774, 13.4685, 16.86897, 17.7566, 21.88243,
                17.89145, 8.327894, -1.851751, -1.294551, 6.26944, 13.67262, 18.3722, 19.83007,
                19.85607, 15.6295, 6.204237, -2.203498, 0.2285182, 7.514933, 15.79158, 20.17148,
                21.24315,
            ],
            x_order: [11, 11],
            y_order: [12, 13],
            x: vec![],
            y: vec![],
        }
    }

    fn get_test_dx() -> Vec<f64> {
        vec![
            24.68001938,
            25.64795794,
            26.61595125,
            27.58399918,
            28.55210159,
            29.52025835,
            24.64942401,
            25.61743197,
            26.58549441,
            27.55361121,
            28.52178222,
            29.49000732,
            24.61888334,
            25.58696051,
            26.55509190,
            27.52327737,
            28.49151681,
            29.45981006,
            24.58839725,
            25.55654344,
            26.52474359,
            27.49299756,
            28.46130523,
            29.42966645,
            24.55796564,
            25.52618067,
            26.49444939,
            27.46277167,
            28.43114738,
            29.39957638,
        ]
    }

    fn get_test_dy() -> Vec<f64> {
        vec![
            29.95734978,
            29.97077025,
            29.98416237,
            29.99752615,
            30.01086159,
            30.02416869,
            30.91366691,
            30.92709704,
            30.94049864,
            30.95387171,
            30.96721626,
            30.98053230,
            31.86996766,
            31.88340732,
            31.89681826,
            31.91020049,
            31.92355403,
            31.93687888,
            32.82625228,
            32.83970133,
            32.85312148,
            32.86651275,
            32.87987515,
            32.89320868,
            33.78252103,
            33.79597934,
            33.80940858,
            33.82280876,
            33.83617988,
            33.84952196,
        ]
    }

    #[test]
    fn test_spline_parser() {
        let expected = get_test_spline();
        let actual = Spline::open("testdata/F21newEO.spline").unwrap();
        assert_eq!(actual, expected);
    }

    fn get_frame(dim1: usize, dim2: usize) -> Cbf {
        let mut frame = Cbf::new();
        let a = Array::with_dims(dim1, dim2);
        frame.set_array(a);
        frame
    }

    #[test]
    fn test_spline_bispev() {
        let dx_expected = get_test_dx();
        let dy_expected = get_test_dy();
        let frame = get_frame(4, 5);
        let spline = Spline::init(
            BufReader::new(File::open("testdata/F21newEO.spline").unwrap()),
            frame.array(),
        )
        .unwrap();
        for (i, (c, e)) in spline.y().iter().zip(dy_expected).enumerate() {
            let c = *c;
            let v = (c - e).abs();
            assert!(v < 1e-6, "dy[{}]: calc = {}; exp = {}", i, c, e);
        }
        for (i, (c, e)) in spline.x().iter().zip(dx_expected).enumerate() {
            let c = *c;
            let v = (c - e).abs();
            assert!(v < 1e-6, "dx[{}]: calc = {}; exp = {}", i, c, e);
        }
    }
}
