//! Parser of the PONI files (pyFAI-calib output).
//!
//!
//! PONI is an acronym for Point Of Normal Incidence.
//! It is the pixel coordinates of the orthogonal projection
//! of the sample position on the detector (for plane detectors,
//! or the plane ```z=0``` for other). The PONI coincide with the beam
//! center in the case of orthogonal setup but most of the time differs.
//!
//! Usage example:
//! ```rust
//! use integrustio::poni::Poni;
//! use std::path::Path;
//! use std::fmt;
//!
//! fn read_poni1<P: AsRef<Path> + fmt::Debug>(poni: P) {
//!     let poni1 = Poni {
//!         pixel1: 0.00017199999999999998,
//!         pixel2: 0.00017199999999999998,
//!         distance: 0.142266095244,
//!         poni1: 0.284775045579,
//!         poni2: 0.126207280557,
//!         rot1: -0.000575699365876,
//!         rot2: -0.0123630633278,
//!         rot3: 7.40180673316e-06,
//!         wavelength: 7.22191445315e-11,
//!         version: 1,
//!     };
//!     let poni2 = match Poni::read_file(&poni) {
//!         Ok(poni) => poni,
//!         Err(e) => panic!("Could not read file {:?}: {}", poni, e),
//!     };
//!     assert_eq!(poni1, poni2);
//! }
//! ```
extern crate serde_json;
use serde::Deserialize;
use serde_json::Result;
use std::fs::File;
use std::io::{self, BufReader};
use std::path::Path;

/// PONI structure.
#[derive(PartialEq, Debug)]
pub struct Poni {
    /// Distance in meters.
    pub distance: f64,
    /// PONI 1 in meters.
    pub poni1: f64,
    /// PONI 2 in meters.
    pub poni2: f64,
    /// Pixel size 1 in meters.
    pub pixel1: f64,
    /// Pixel size 2 in meters.
    pub pixel2: f64,
    /// Rotation 1 in radians.
    pub rot1: f64,
    /// Rotation 2 in radians.
    pub rot2: f64,
    /// Rotation 3 in radians.
    pub rot3: f64,
    /// Wavelength of the incoming x-ray beam in meters.
    pub wavelength: f64,
    /// PONI file version.
    pub version: i32,
}

impl Poni {
    /// Creates a new empty [Poni].
    pub fn new() -> Poni {
        Poni {
            distance: 0.,
            poni1: 0.,
            poni2: 0.,
            pixel1: 0.,
            pixel2: 0.,
            rot1: 0.,
            rot2: 0.,
            rot3: 0.,
            wavelength: 0.,
            version: 1,
        }
    }

    /// Reads a buffer as [Poni].
    pub fn read_buffer<R: io::BufRead>(reader: R) -> io::Result<Poni> {
        let mut poni = Poni::new();
        let mut detector = String::new();
        for line in reader.lines() {
            let line = line?;
            let item = line.to_poni_item();
            match item.value {
                PoniValue::Empty => continue,
                PoniValue::Number(v) => match item.key {
                    "PixelSize1" => poni.pixel1 = v,
                    "PixelSize2" => poni.pixel2 = v,
                    "Wavelength" => poni.wavelength = v,
                    "Distance" => poni.distance = v,
                    "Poni1" => poni.poni1 = v,
                    "Poni2" => poni.poni2 = v,
                    "Rot1" => poni.rot1 = v,
                    "Rot2" => poni.rot2 = v,
                    "Rot3" => poni.rot3 = v,
                    "poni_version" => poni.version = v as i32,
                    &_ => continue,
                },
                PoniValue::Str(v) => match item.key {
                    "Detector" => detector = v.to_lowercase().replace(' ', "_"),
                    &_ => continue,
                },
                PoniValue::Pixels(v) => {
                    poni.pixel1 = v.pixel1;
                    poni.pixel2 = v.pixel2;
                }
            }
        }
        poni.parse_detector(&detector);
        Ok(poni)
    }

    /// Reads a file as [Poni].
    pub fn read_file<P: AsRef<Path>>(filename: P) -> io::Result<Poni> {
        Poni::read_buffer(BufReader::new(File::open(filename)?))
    }

    fn parse_detector(&mut self, detector: &str) {
        if self.pixel1 != 0. && self.pixel2 != 0. {
            return;
        }
        let pixel = detector.to_pixel();
        self.pixel1 = pixel.0;
        self.pixel2 = pixel.1;
    }
}

enum PoniValue<'a> {
    Empty,
    Number(f64),
    Pixels(DetectorConfig),
    Str(&'a str),
}

struct PoniItem<'a> {
    key: &'a str,
    value: PoniValue<'a>,
}

#[derive(Deserialize)]
struct DetectorConfig {
    pixel1: f64,
    pixel2: f64,
}

trait PoniItemParser {
    fn to_poni_item(&self) -> PoniItem;
}

impl PoniItemParser for str {
    fn to_poni_item(&self) -> PoniItem {
        let mut item = PoniItem {
            key: self,
            value: PoniValue::Empty,
        };
        if self.starts_with('#') {
            return item;
        }
        let vec: Vec<&str> = self.splitn(2, ':').map(|s| s.trim()).collect();
        if vec.len() != 2 {
            return item;
        }
        item.key = vec[0];
        if let Ok(value) = vec[1].parse::<f64>() {
            item.value = PoniValue::Number(value);
            return item;
        }
        let result: Result<DetectorConfig> = serde_json::from_str(vec[1]);
        if let Ok(dc) = result {
            item.value = PoniValue::Pixels(dc);
            return item;
        }
        item.value = PoniValue::Str(vec[1]);
        return item;
    }
}

/// Tries to guess a pixel size from the detector name
pub trait ToPixel {
    fn to_pixel(&self) -> (f64, f64);
}

impl ToPixel for str {
    fn to_pixel(&self) -> (f64, f64) {
        match self {
            "pilatus100k" | "pilatus_100k" | "pilatus200k" | "pilatus_200k" | "pilatus300k"
            | "pilatus_300k" | "pilatus300kw" | "pilatus_300kw" | "pilatus1m" | "pilatus_1m"
            | "pilatus2m" | "pilatus_2m" | "pilatus6m" | "pilatus_6m" | "pilatuscdte300k"
            | "pilatus_cdte_300k" | "pilatus_300k_cdte" | "pilatus300kcdte"
            | "pilatus300k_cdte" | "pilatuscdte300kw" | "pilatus_cdte_300kw"
            | "pilatus_300kw_cdte" | "pilatus300kwcdte" | "pilatus300kw_cdte"
            | "pilatuscdte900kw" | "pilatus_cdte_900kw" | "pilatus_900kw_cdte"
            | "pilatus900kwcdte" | "pilatus900kw_cdte" | "pilatuscdte1m" | "pilatus_cdte_1m"
            | "pilatus_1m_cdte" | "pilatus1mcdte" | "pilatus1m_cdte" | "pilatuscdte2m"
            | "pilatus_cdte_2m" | "pilatus_2m_cdte" | "pilatus2mcdte" | "pilatus2m_cdte" => {
                (172e-6, 172e-6)
            }
            "eiger500k" | "eiger_500k" | "eiger1m" | "eiger_1m" | "eiger4m" | "eiger_4m"
            | "eiger9m" | "eiger_9m" | "eiger16m" | "eiger_16m" | "eiger2_500k" | "eiger2500k"
            | "eiger2_1m" | "eiger21m" | "eiger2_1mw" | "eiger2_1m-w" | "eiger21m-w"
            | "eiger2_2mw" | "eiger2_2m-w" | "eiger22m-w" | "eiger2_4m" | "eiger24m"
            | "eiger2_9m" | "eiger29m" | "eiger2_16m" | "eiger216m" | "eiger2cdte_500k"
            | "eiger2_cdte_500k" | "eiger2cdte500k" | "eiger2cdte_1m" | "eiger2_cdte_1m"
            | "eiger2cdte1m" | "eiger2cdte_1mw" | "eiger2_cdte_1m-w" | "eiger2cdte1m-w"
            | "eiger2cdte_2mw" | "eiger2_cdte_2m-w" | "eiger2cdte2m-w" | "eiger2cdte_4m"
            | "eiger2_cdte_4m" | "eiger2cdte4m" | "eiger2cdte_9m" | "eiger2_cdte_9m"
            | "eiger2cdte9m" | "eiger2cdte_16m" | "eiger2_cdte_16m" | "eiger2cdte16m"
            | "jungfrau" | "jungfrau_500k" | "jungfrau500k" | "jungfrau_16m_cor"
            | "jungfrau16mcor" | "dexela2923" | "dexela_2923" => (75e-6, 75e-6),
            "frelon" => (50e-6, 50e-6),
            "maxipix" | "maxipix_1x1" | "maxipix1x1" | "maxipix2x2" | "maxipix_2x2"
            | "maxipix5x1" | "maxipix_5x1" => (55e-6, 55e-6),
            "mythen" | "mythen_1280" | "mythen1280" => (8e-3, 50e-6),
            "adsc_q315" | "quantum_315" | "quantum315" | "adsc_q210" | "quantum_210"
            | "quantum210" => (51e-6, 51e-6),
            "adsc_q270" | "quantum_270" | "quantum270" => (64.8e-6, 64.8e-6),
            "adsc_q4" | "quantum_4" | "quantum4" => (64.8e-6, 64.8e-6),
            "hf_130k" | "hf-130k" | "hf_262k" | "hf-262k" | "hf_1m" | "hf-1m" | "hf_2m"
            | "hf-2.4m" | "hf_4m" | "hf-4m" | "hf_9m" | "hf-9.4m" => (150e-6, 150e-6),
            "imxpads10" | "imxpad_s10" | "imxpads70" | "imxpad_s70" | "imxpads70v"
            | "imxpad_s70_v" | "imxpads140" | "imxpad_s140" | "xpad_flat" | "xpad_s540_flat"
            | "xpads540flat" | "d5" | "cirpad" | "xcirpad" => (130e-6, 130e-6),
            "fairchild" | "condor" | "fairchild_condor_486:90" | "fairchildcondor486:90" => {
                (15e-6, 15e-6)
            }
            "titan" | "titan_2k_x_2k" | "titan2kx2k" | "oxd_titan" | "oxdtitan"
            | "agilent_titan" | "agilenttitan" => (60e-6, 60e-6),
            "basler" | "aca1300" => (3.75e-6, 3.75e-6),
            "perkin" | "perkin_detector" | "perkindetector" | "perkin_elmer" | "perkinelmer" => {
                (200e-6, 200e-6)
            }
            "pixium"
            | "pixium_4700"
            | "pixium4700"
            | "pixium_4700_detector"
            | "pixium4700detector"
            | "thales_electronics"
            | "thaleselectronics" => (308e-6, 308e-6),
            _ => {
                eprintln!("Unknown detector name '{}'!", self);
                (0., 0.)
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use crate::poni::Poni;

    #[test]
    fn read_poni1() {
        let poni1 = Poni {
            pixel1: 0.00017199999999999998,
            pixel2: 0.00017199999999999998,
            distance: 0.142266095244,
            poni1: 0.284775045579,
            poni2: 0.126207280557,
            rot1: -0.000575699365876,
            rot2: -0.0123630633278,
            rot3: 7.40180673316e-06,
            wavelength: 7.22191445315e-11,
            version: 1,
        };
        let test = "testdata/testv1.poni";
        let poni2 = match Poni::read_file(test) {
            Ok(poni) => poni,
            Err(e) => panic!("Could not read file {}: {}", test, e),
        };
        assert_eq!(poni1, poni2);
    }

    #[test]
    fn read_poni2() {
        let poni1 = Poni {
            pixel1: 0.00017199999999999998,
            pixel2: 0.00017199999999999998,
            distance: 0.13807196533727795,
            poni1: 0.28380515495915426,
            poni2: 0.1264715897469189,
            rot1: -5.112234990518272e-05,
            rot2: -0.012195862516792824,
            rot3: -5.194437754410024e-10,
            wavelength: 6.867253033883946e-11,
            version: 2,
        };
        let test = "testdata/testv2.poni";
        let poni2 = match Poni::read_file(test) {
            Ok(poni) => poni,
            Err(e) => panic!("Could not read file {}: {}", test, e),
        };
        assert_eq!(poni1, poni2);
    }

    #[test]
    fn read_poni3() {
        let poni1 = Poni {
            pixel1: 0.000172,
            pixel2: 0.000172,
            distance: 0.13898527790976398,
            poni1: 0.20430869943415775,
            poni2: 0.12623379778647795,
            rot1: -0.0007599337147390097,
            rot2: -0.012005434170665762,
            rot3: -5.42184694367652e-05,
            wavelength: 6.410033999301687e-11,
            version: 2,
        };
        let test = "testdata/testv2.1.poni";
        let poni2 = match Poni::read_file(test) {
            Ok(poni) => poni,
            Err(e) => panic!("Could not read file {}: {}", test, e),
        };
        assert_eq!(poni1, poni2);
    }

    #[test]
    fn read_poni4() {
        let poni1 = Poni {
            pixel1: 0.000172,
            pixel2: 0.000172,
            distance: 0.13845051835684719,
            poni1: 0.28531406827737266,
            poni2: 0.12408547301822423,
            rot1: 0.0016209524088685516,
            rot2: 0.0020720728627343903,
            rot3: -9.219510971787646e-07,
            wavelength: 9.533639999999999e-11,
            version: 2,
        };
        let test = "testdata/testv2.2.poni";
        let poni2 = Poni::read_file(test).unwrap();
        assert_eq!(poni1, poni2);
    }
}
