//! Azimuthal integrator of diffraction data acquired with 2D detectors.
//!
use crate::poni::Poni;
use crate::utils::GoFloatMath;
use cryiorust::frame::{Array, FrameError, FrameResult};
use itertools::izip;
use std::cmp;
use std::f64::consts::{PI, TAU};
use std::sync::Arc;

const TTH2Q: f64 = 4e-9 * PI;
const DSA: i32 = 3;

/// Enum with possible integration units.
#[derive(PartialEq)]
pub enum Units {
    /// 2θ in degrees
    TwoTheta,
    /// Q in nanometers
    Qnm,
    /// Q in Ångströms
    QA,
}

/// Structure filled with user values to be integrated.
pub struct Integrable<'a> {
    pub array: &'a Array,
    pub radial_range: &'a [f64],
    pub azimuthal_range: &'a [f64],
    pub integration_type: IntegrationType,
}

/// Enum for integration results.
pub enum PatternType {
    /// Nothing.
    None,
    /// Radial [Pattern]
    Radial(Pattern),
    /// Azimuthal [Pattern]
    Azimuthal(Pattern),
    /// 2D Radial-Azimuthal [Pattern] (cake)
    Cake(Cake),
}

/// Enum for possible integration type.
#[derive(PartialEq)]
pub enum IntegrationType {
    Radial,
    Azimuthal,
    Cake,
}

/// Result for [Cake] integration.
pub struct Cake {
    /// Radial positions in [Units].
    pub radial_positions: Arc<Vec<f64>>,
    /// Azimuthal positions in degrees.
    pub azimuthal_positions: Arc<Vec<f64>>,
    /// 2D [Cake] [Array].
    pub cake: Array,
}

/// Result for Azimuthal or Radial integrations.
pub struct Pattern {
    /// Position bins in [Units].
    pub positions: Arc<Vec<f64>>,
    /// Intensity.
    pub intensity: Vec<f64>,
    /// Intensity error.
    pub sigma: Vec<f64>,
}

/// Returned result as [Diffractogram].
pub struct Diffractogram {
    /// Data in [PatternType].
    pub data: PatternType,
    /// Normalized and corrected [Array].
    pub image: Array,
    /// Radial min boundary.
    pub radial_min: f64,
    /// Radial max boundary.
    pub radial_max: f64,
    /// Azimuthal min boundary.
    pub azimuth_min: f64,
    /// Azimuthal max boundary.
    pub azimuth_max: f64,
    azimuth: bool,
    radial: bool,
    i_type: IntegrationType,
}

/// Main [Integrator] object.
pub struct Integrator {
    params: Params,
    rbins: usize,
    abins: usize,
    use_pol: bool,
    radial: Positions,
    azimuth: Positions,
    solid_angle: Vec<f64>,
    use_solid_angle: bool,
    polar: Vec<f64>,
}

impl Integrator {
    /// Creates a new empty [Integrator].
    pub fn new() -> Integrator {
        Integrator {
            params: Params::new(),
            rbins: 0,
            abins: 0,
            use_pol: false,
            radial: Positions::new(0, 0),
            azimuth: Positions::new(0, 0),
            solid_angle: Vec::new(),
            use_solid_angle: true,
            polar: Vec::new(),
        }
    }

    /// Checks whether [Integrator] is initialized with the curren [Array].
    pub fn is_initialized(&self, array: &Array) -> bool {
        self.params.dim1 == array.dim1() && self.params.dim2 == array.dim2()
    }

    /// Sets correction on the solid angle.
    pub fn set_solid_angle(&mut self, enable: bool) {
        self.use_solid_angle = enable;
    }

    /// Sets units for integration.
    pub fn set_units(&mut self, units: Units) {
        self.params.set_units(units);
    }

    /// Sets [Poni].
    pub fn set_poni(&mut self, poni: Poni) {
        self.params.set_poni(poni);
    }

    /// Sets the number of radial bins.
    pub fn set_radial_bins(&mut self, bins: usize) {
        if self.rbins != bins {
            self.rbins = bins;
            self.params.dim1 = 0;
        }
    }

    /// Sets the number of azimuthal bins.
    pub fn set_azimuthal_bins(&mut self, bins: usize) {
        if self.abins != bins {
            self.abins = bins;
            self.params.dim1 = 0;
        }
    }

    /// Sets polarization correction factor.
    pub fn set_polarization(&mut self, pol_fac: f64) {
        self.use_pol = pol_fac < 1. && pol_fac > -1.;
        self.params.set_pol(pol_fac);
    }

    fn init_pos(&mut self) {
        let corners = Corners::new(&self.params);
        if self.rbins == 0 {
            self.rbins = cmp::max(self.params.dim1, self.params.dim2) * 2;
        }
        if self.abins == 0 {
            self.abins = 360;
        }
        self.radial = Positions::new(self.params.size, self.rbins);
        self.azimuth = Positions::new(self.params.size, self.abins);
        self.radial.start = *corners.tth.first().unwrap();
        self.radial.stop = self.radial.start;
        self.azimuth.start = *corners.chi.first().unwrap();
        self.azimuth.stop = self.azimuth.start;
        for i in 0..self.params.size {
            let (max_tth, max_chi) = corners.maxd(i);
            self.radial.uplow(max_tth, corners.tth[i]);
            self.azimuth.uplow(max_chi, corners.chi[i]);
        }
        if self.radial.start < 0. {
            self.radial.start = 0.
        }
        if self.azimuth.stop > TAU {
            self.azimuth.stop = TAU
        }
        if self.azimuth.start < 0. {
            self.azimuth.start = 0.
        }
        self.polar = corners.polar;
        self.solid_angle = corners.sa;
        self.radial.calc_bins(&self.params.units);
        self.azimuth.calc_bins(&Units::TwoTheta);
    }

    /// Initializes [Integrator] for a certain [Array].
    pub fn init(&mut self, array: &Array) {
        if self.params.set_dims(array.dim1(), array.dim2()) {
            self.init_pos()
        }
    }

    /// Integrates the user supplied data.
    pub fn integrate(&self, data: &Integrable) -> FrameResult<Diffractogram> {
        if self.params.dim1 != data.array.dim1() || self.params.dim2 != data.array.dim2() {
            return Err(FrameError::FormatError(
                format!(
                    "integration is set for {}x{} arrays but supplied array is {}x{}",
                    self.params.dim1,
                    self.params.dim2,
                    data.array.dim1(),
                    data.array.dim2()
                )
                .into(),
            ));
        }
        let mut p = match data.integration_type {
            IntegrationType::Radial => Diffractogram::new_radial(&self, data),
            IntegrationType::Azimuthal => Diffractogram::new_azimuthal(&self, data),
            IntegrationType::Cake => Diffractogram::new_cake(&self, data),
        };
        p.azimuth_min = p.azimuth_min.to_degrees();
        p.azimuth_max = p.azimuth_max.to_degrees();
        if self.params.units == Units::TwoTheta {
            p.radial_max = p.radial_max.to_degrees();
            p.radial_min = p.radial_min.to_degrees();
        }
        Ok(p)
    }

    fn correct(&self, mut intensity: f64, j: usize) -> f64 {
        if self.use_solid_angle {
            intensity /= self.solid_angle[j];
        }
        if self.use_pol {
            intensity /= self.polar[j];
        }
        return intensity;
    }
}

struct Positions {
    bins: usize,
    start: f64,
    stop: f64,
    step: f64,
    lower: Vec<f64>,
    upper: Vec<f64>,
    pos: Arc<Vec<f64>>,
}

impl Positions {
    fn new(size: usize, bins: usize) -> Positions {
        Positions {
            bins,
            start: 0.,
            stop: 0.,
            step: 0.,
            lower: Vec::with_capacity(size),
            upper: Vec::with_capacity(size),
            pos: Arc::new(vec![]),
        }
    }

    fn uplow(&mut self, max: f64, cur: f64) {
        let sum = cur + max;
        let diff = cur - max;
        self.upper.push(sum);
        self.lower.push(diff);
        if diff < self.start {
            self.start = diff;
        }
        if sum > self.stop {
            self.stop = sum;
        }
    }

    fn calc_bins(&mut self, units: &Units) {
        self.step = (self.stop - self.start) / self.bins as f64;
        let start = self.start + 0.5 * self.step;
        let stop = self.stop - 0.5 * self.step;
        let step = (stop - start) / (self.bins - 1) as f64;
        self.pos = Arc::new(match units {
            Units::TwoTheta => self.linspace(start.to_degrees(), step.to_degrees()),
            Units::Qnm | Units::QA => self.linspace(start, step),
        });
    }

    fn linspace(&self, mut start: f64, step: f64) -> Vec<f64> {
        let mut vec: Vec<f64> = Vec::with_capacity(self.bins);
        unsafe { vec.set_len(self.bins) };
        for val in &mut vec {
            *val = start;
            start += step;
        }
        vec
    }
}

struct PositionConstants {
    s1: f64,
    s2: f64,
    s3: f64,
    c1: f64,
    c2: f64,
    c3: f64,
    c2c3: f64,
    c3s1s2: f64,
    c1s3: f64,
    c1c3s2: f64,
    s1s3: f64,
    c2s3: f64,
    c1c3: f64,
    s1s2s3: f64,
    c3s1: f64,
    c1s2s3: f64,
    c2s1: f64,
    c1c2: f64,
    dt1: f64,
    dt2: f64,
    dt3: f64,
    c3s1s2c1s3: f64,
    c1c3s1s2s3: f64,
    pp1: f64,
    pp2: f64,
    dist2: f64,
    tth2q: f64,
}

impl PositionConstants {
    fn new(p: &Poni) -> PositionConstants {
        let mut c = PositionConstants {
            s1: p.rot1.sin(),
            s2: p.rot2.sin(),
            s3: p.rot3.sin(),
            c1: p.rot1.cos(),
            c2: p.rot2.cos(),
            c3: p.rot3.cos(),
            c2c3: 0.,
            c3s1s2: 0.,
            c1s3: 0.,
            c1c3s2: 0.,
            s1s3: 0.,
            c2s3: 0.,
            c1c3: 0.,
            s1s2s3: 0.,
            c3s1: 0.,
            c1s2s3: 0.,
            c2s1: 0.,
            c1c2: 0.,
            dt1: 0.,
            dt2: 0.,
            dt3: 0.,
            c3s1s2c1s3: 0.,
            c1c3s1s2s3: 0.,
            pp1: p.pixel1 * 0.5 - p.poni1,
            pp2: p.pixel2 * 0.5 - p.poni2,
            dist2: p.distance * p.distance,
            tth2q: TTH2Q / p.wavelength,
        };
        c.c2c3 = c.c2 * c.c3;
        c.c3s1s2 = c.c3 * c.s1 * c.s2;
        c.c1s3 = c.c1 * c.s3;
        c.c1c3s2 = c.c1 * c.c3 * c.s2;
        c.s1s3 = c.s1 * c.s3;
        c.c2s3 = c.c2 * c.s3;
        c.c1c3 = c.c1 * c.c3;
        c.s1s2s3 = c.s1 * c.s2 * c.s3;
        c.c3s1 = c.c3 * c.s1;
        c.c1s2s3 = c.c1 * c.s2 * c.s3;
        c.c2s1 = c.c2 * c.s1;
        c.c1c2 = c.c1 * c.c2;
        c.dt1 = p.distance * (c.c1c3s2 + c.s1s3);
        c.dt2 = p.distance * (c.c3s1 - c.c1s2s3);
        c.dt3 = p.distance * c.c1c2;
        c.c3s1s2c1s3 = c.c3s1s2 - c.c1s3;
        c.c1c3s1s2s3 = c.c1c3 + c.s1s2s3;
        return c;
    }
}

struct Corners<'a> {
    params: &'a Params,
    tth: Vec<f64>,
    chi: Vec<f64>,
    dtth: Vec<f64>,
    dchi: Vec<f64>,
    counters: [usize; 4],
    ctth: [Vec<f64>; 4],
    cchi: [Vec<f64>; 4],
    pc: PositionConstants,
    sa: Vec<f64>,
    polar: Vec<f64>,
}

impl<'a> Corners<'a> {
    fn new(params: &Params) -> Corners {
        let size = (params.dim1 + 1) * (params.dim2 + 1);
        let mut corners = Corners {
            params,
            pc: PositionConstants::new(&params.poni),
            tth: Vec::with_capacity(params.size),
            chi: Vec::with_capacity(params.size),
            dtth: Vec::with_capacity(size),
            dchi: Vec::with_capacity(size),
            counters: [0; 4],
            ctth: [
                Vec::with_capacity(params.size),
                Vec::with_capacity(params.size),
                Vec::with_capacity(params.size),
                Vec::with_capacity(params.size),
            ],
            cchi: [
                Vec::with_capacity(params.size),
                Vec::with_capacity(params.size),
                Vec::with_capacity(params.size),
                Vec::with_capacity(params.size),
            ],
            sa: Vec::with_capacity(params.size),
            polar: Vec::with_capacity(params.size),
        };
        corners.calc_pos();
        return corners;
    }

    fn calc_pos(&mut self) {
        for i in 0..=self.params.dim1 {
            let p1i = self.params.poni.pixel1 * i as f64;
            let p1 = p1i + self.pc.pp1;
            let p11 = self.pc.dist2 + p1 * p1;
            let dp1 = p1i - self.params.poni.poni1;
            let t11 = p1 * self.pc.c2c3 - self.pc.dt1;
            let dt11 = dp1 * self.pc.c2c3 - self.pc.dt1;
            let t21 = p1 * self.pc.c2s3 + self.pc.dt2;
            let dt21 = dp1 * self.pc.c2s3 + self.pc.dt2;
            let t31 = p1 * self.pc.s2 + self.pc.dt3;
            let dt31 = dp1 * self.pc.s2 + self.pc.dt3;
            for j in 0..=self.params.dim2 {
                let p2j = self.params.poni.pixel2 * j as f64;
                let p2 = p2j + self.pc.pp2;
                let dp2 = p2j - self.params.poni.poni2;
                let t1 = t11 + p2 * self.pc.c3s1s2c1s3;
                let dt1 = dt11 + dp2 * self.pc.c3s1s2c1s3;
                let t2 = t21 + p2 * self.pc.c1c3s1s2s3;
                let dt2 = dt21 + dp2 * self.pc.c1c3s1s2s3;
                let t3 = t31 - p2 * self.pc.c2s1;
                let dt3 = dt31 - dp2 * self.pc.c2s1;
                self.dtth.push(self.tth2q(dt1.hypot(dt2).atan2(dt3)));
                let mut chi = dt1.atan2(dt2);
                if chi < 0. {
                    chi += TAU;
                }
                self.dchi.push(chi);
                if i != self.params.dim1 && j != self.params.dim2 {
                    self.sa
                        .push((self.params.poni.distance / (p11 + p2 * p2).sqrt()).powi(DSA));
                    let tth = t1.hypot(t2).atan2(t3);
                    let mut chi = t1.atan2(t2);
                    let ctth = tth.cos().powi(2);
                    self.polar.push(
                        0.5 * (1. + ctth - self.params.pol_fac * (2. * chi).cos() * (1. - ctth)),
                    );
                    self.tth.push(self.tth2q(tth));
                    if chi < 0. {
                        chi += TAU;
                    }
                    self.chi.push(chi);
                }
                self.calc_corners(i, j);
            }
        }
    }

    fn calc_corners(&mut self, i: usize, j: usize) {
        if i != self.params.dim1 && j != self.params.dim2 {
            self.calc_part(0);
        }
        if i != 0 && j != self.params.dim2 {
            self.calc_part(1);
        }
        if i != 0 && j != 0 {
            self.calc_part(2);
        }
        if i != self.params.dim1 && j != 0 {
            self.calc_part(3);
        }
    }

    fn calc_part(&mut self, n: usize) {
        let i = self.counters[n];
        self.ctth[n].push((*self.dtth.last().unwrap() - self.tth[i]).abs());
        self.cchi[n].push(
            (*self.dchi.last().unwrap() - self.chi[i])
                .abs()
                .remainder(TAU), // see comments in utils.rs
        );
        self.counters[n] = i + 1;
    }

    fn nm(&self, value: f64) -> f64 {
        self.pc.tth2q * (value * 0.5).sin()
    }

    fn tth2q(&self, value: f64) -> f64 {
        match self.params.units {
            Units::TwoTheta => value,
            Units::Qnm => self.nm(value),
            Units::QA => self.nm(value) * 0.1,
        }
    }

    fn maxd(&self, i: usize) -> (f64, f64) {
        let mut tth = f64::MIN;
        let mut chi = f64::MIN;
        for (ctth, cchi) in izip!(self.ctth.iter(), self.cchi.iter()) {
            let ctth = ctth[i];
            let cchi = cchi[i];
            if ctth > tth {
                tth = ctth;
            }
            if cchi > chi {
                chi = cchi;
            }
        }
        return (tth, chi);
    }
}

struct Params {
    dim1: usize,
    dim2: usize,
    size: usize,
    poni: Poni,
    units: Units,
    pol_fac: f64,
}

impl Params {
    fn new() -> Params {
        Params {
            dim1: 0,
            dim2: 0,
            size: 0,
            units: Units::TwoTheta,
            poni: Poni::new(),
            pol_fac: 0.99,
        }
    }

    fn set_poni(&mut self, poni: Poni) {
        if self.poni != poni {
            self.poni = poni;
            self.dim1 = 0;
        }
    }

    fn set_dims(&mut self, dim1: usize, dim2: usize) -> bool {
        if dim1 == self.dim1 && dim2 == self.dim2 {
            false
        } else {
            self.dim1 = dim1;
            self.dim2 = dim2;
            self.size = dim1 * dim2;
            true
        }
    }

    fn set_units(&mut self, units: Units) {
        if self.units != units {
            self.units = units;
            self.dim1 = 0;
        }
    }

    fn set_pol(&mut self, pol: f64) {
        if self.pol_fac != pol {
            self.pol_fac = pol;
            self.dim1 = 0;
        }
    }
}

struct Buffers {
    counts: Vec<f64>,
    values: Vec<f64>,
}

impl Buffers {
    fn new(bins: usize) -> Buffers {
        Buffers {
            counts: vec![0.; bins],
            values: vec![0.; bins],
        }
    }
}

impl Diffractogram {
    fn new(i: &Integrator, it: IntegrationType) -> Diffractogram {
        Diffractogram {
            data: PatternType::None,
            image: Array::with_dims(i.params.dim1, i.params.dim2),
            radial_min: 0.0,
            radial_max: 0.0,
            azimuth_min: 0.0,
            azimuth_max: 0.0,
            azimuth: false,
            radial: false,
            i_type: it,
        }
    }

    fn new_radial(i: &Integrator, data: &Integrable) -> Diffractogram {
        let mut buffers = Buffers::new(i.rbins);
        let mut d = Diffractogram::new(i, IntegrationType::Radial);
        d.integrate(i, data, &mut buffers);
        d.data = PatternType::Radial(Pattern {
            positions: i.radial.pos.clone(),
            intensity: buffers.counts,
            sigma: buffers.values,
        });
        d
    }

    fn new_azimuthal(i: &Integrator, data: &Integrable) -> Diffractogram {
        let mut buffers = Buffers::new(i.abins);
        let mut d = Diffractogram::new(i, IntegrationType::Azimuthal);
        d.integrate(i, data, &mut buffers);
        d.data = PatternType::Azimuthal(Pattern {
            positions: i.azimuth.pos.clone(),
            intensity: buffers.counts,
            sigma: buffers.values,
        });
        d
    }

    fn new_cake(i: &Integrator, data: &Integrable) -> Diffractogram {
        let mut buffers = Buffers::new(i.rbins * i.abins);
        let mut d = Diffractogram::new(i, IntegrationType::Cake);
        d.integrate_cake(i, data, &mut buffers);
        d.data = PatternType::Cake(Cake {
            radial_positions: i.radial.pos.clone(),
            azimuthal_positions: i.azimuth.pos.clone(),
            cake: Array::with_data(i.rbins, i.abins, buffers.counts),
        });
        d
    }

    pub fn consume_image(self) -> Array {
        self.image
    }

    fn integrate_cake(&mut self, i: &Integrator, data: &Integrable, mut buffers: &mut Buffers) {
        self.set_limits(i, data);
        let mut a = Bin {
            count: i.abins,
            min: 0,
            max: 0,
            count_f: i.abins as f64,
            min_f: 0.0,
            max_f: 0.0,
            pos: &i.azimuth,
        };
        let mut r = Bin {
            count: i.rbins,
            min: 0,
            max: 0,
            count_f: i.rbins as f64,
            min_f: 0.0,
            max_f: 0.0,
            pos: &i.radial,
        };
        for (j, intensity) in data.array.data().iter().enumerate() {
            if self.in_limits(*intensity, i, j) && a.valid(j) && r.valid(j) {
                let intensity = i.correct(*intensity, j);
                self.split_pixel_2d(intensity, &r, &a, &mut buffers);
                self.image.push(intensity);
            } else {
                self.image.push(0.0);
            }
        }
        self.sum_bins(&mut buffers);
    }

    fn integrate(&mut self, i: &Integrator, data: &Integrable, mut buffers: &mut Buffers) {
        self.set_limits(i, data);
        let (bins, pos) = match &self.i_type {
            IntegrationType::Azimuthal => (i.azimuth.pos.len(), &i.azimuth),
            IntegrationType::Radial => (i.radial.pos.len(), &i.radial),
            IntegrationType::Cake => unimplemented!(),
        };
        let mut bin = Bin {
            count: bins,
            min: 0,
            max: 0,
            count_f: bins as f64,
            min_f: 0.0,
            max_f: 0.0,
            pos,
        };
        for (j, intensity) in data.array.data().iter().enumerate() {
            if self.in_limits(*intensity, i, j) && bin.valid(j) {
                let intensity = i.correct(*intensity, j);
                self.split_pixel(intensity, &bin, &mut buffers);
                self.image.push(intensity);
            } else {
                self.image.push(0.0);
            }
        }
        self.sum_bins(&mut buffers);
    }

    fn split_pixel(&self, intensity: f64, bin: &Bin, buffers: &mut Buffers) {
        if bin.min == bin.max {
            buffers.counts[bin.min] += 1.;
            buffers.values[bin.min] += intensity;
        } else {
            let d_a = (bin.max_f - bin.min_f).recip();
            let d_l = (bin.min + 1) as f64 - bin.min_f;
            let d_r = bin.max_f - bin.max as f64;
            let d_al = d_a * d_l;
            let d_ar = d_a * d_r;
            buffers.counts[bin.min] += d_al;
            buffers.values[bin.min] += intensity * d_al;
            buffers.counts[bin.max] += d_ar;
            buffers.values[bin.max] += intensity * d_ar;
            if bin.min + 1 < bin.max {
                let d_ai = intensity * d_a;
                for j in (bin.min + 1)..bin.max {
                    buffers.counts[j] += d_a;
                    buffers.values[j] += d_ai;
                }
            }
        }
    }

    fn split_pixel_2d(&self, intensity: f64, p1: &Bin, p2: &Bin, buffers: &mut Buffers) {
        let i = p1.min * p2.count;
        let j = i + p2.min;
        let k = i + p2.max;
        let n = p1.max * p2.count;
        let o = n + p2.min;
        let pp = n + p2.max;
        if p1.min == p1.max {
            if p2.min == p2.max {
                buffers.counts[j] += 1.;
                buffers.values[j] += intensity;
            } else {
                let d_d = (p2.min + 1) as f64 - p2.min_f;
                let d_u = p2.max_f - p2.max as f64;
                let d_a = (p2.max_f - p2.min_f).recip();
                buffers.counts[j] += d_a * d_d;
                buffers.values[j] += intensity * d_a * d_d;
                buffers.counts[k] += d_a * d_u;
                buffers.values[k] += intensity * d_a * d_u;
                for l in p2.min + 1..p2.max {
                    let m = i + l;
                    buffers.counts[m] += d_a;
                    buffers.values[m] += intensity * d_a;
                }
            }
        } else {
            if p2.min == p2.max {
                let d_a = (p1.max_f - p1.min_f).recip();
                let d_l = (p1.min + 1) as f64 - p1.min_f;
                let d_r = p1.max_f - p1.max as f64;
                buffers.counts[j] += d_a * d_l;
                buffers.values[j] += intensity * d_a * d_l;
                buffers.counts[o] += d_a * d_r;
                buffers.values[o] += intensity * d_a * d_r;
                for l in p1.min + 1..p1.max {
                    let m = l * p2.count + p2.min;
                    buffers.counts[m] += d_a;
                    buffers.values[m] += intensity * d_a;
                }
            } else {
                let d_l = (p1.min + 1) as f64 - p1.min_f;
                let d_r = p1.max_f - p1.max as f64;
                let d_d = (p2.min + 1) as f64 - p2.min_f;
                let d_u = p2.max_f - p2.max as f64;
                let d_a = ((p1.max_f - p1.min_f) * (p2.max_f - p2.min_f)).recip();
                buffers.counts[j] += d_a * d_l * d_d;
                buffers.values[j] += intensity * d_a * d_l * d_d;
                buffers.counts[k] += d_a * d_l * d_u;
                buffers.values[k] += intensity * d_a * d_l * d_u;
                buffers.counts[o] += d_a * d_r * d_d;
                buffers.values[o] += intensity * d_a * d_r * d_d;
                buffers.counts[pp] += d_a * d_r * d_u;
                buffers.values[pp] += intensity * d_a * d_r * d_u;
                for l in p1.min + 1..p1.max {
                    let m = l * p2.count + p2.min;
                    buffers.counts[m] += d_a * d_d;
                    buffers.values[m] += intensity * d_a * d_d;
                    for q in p2.min + 1..p2.max {
                        let s = l * p2.count + q;
                        buffers.counts[s] += d_a;
                        buffers.values[s] += intensity * d_a;
                    }
                    let s = l * p2.count + p2.max;
                    buffers.counts[s] += d_a * d_u;
                    buffers.values[s] += intensity * d_a * d_u;
                }
                for l in p2.min + 1..p2.max {
                    let q = i + l;
                    let s = n + l;
                    buffers.counts[q] += d_a * d_l;
                    buffers.values[q] += intensity * d_a * d_l;
                    buffers.counts[s] += d_a * d_r;
                    buffers.values[s] += intensity * d_a * d_r;
                }
            }
        }
    }

    fn sum_bins(&self, buffers: &mut Buffers) {
        for (counts, value) in buffers.counts.iter_mut().zip(buffers.values.iter_mut()) {
            if *counts == 0. {
                *value = 0.;
            } else {
                let i = *value / *counts;
                let e = value.sqrt() / *counts;
                *counts = i;
                *value = e;
            }
        }
    }

    fn in_limits(&self, intensity: f64, i: &Integrator, j: usize) -> bool {
        if intensity < 0. {
            return false;
        }
        if self.azimuth
            && (i.azimuth.lower[j] < self.azimuth_min || i.azimuth.upper[j] > self.azimuth_max)
        {
            return false;
        }
        if self.radial
            && (i.radial.lower[j] < self.radial_min || i.radial.upper[j] > self.radial_max)
        {
            return false;
        }
        return true;
    }

    fn set_limits(&mut self, i: &Integrator, data: &Integrable) {
        self.azimuth_min = data.azimuthal_range[0].to_radians();
        self.azimuth_max = data.azimuthal_range[1].to_radians();
        self.azimuth = self.azimuth_max != self.azimuth_min;
        match i.params.units {
            Units::TwoTheta => {
                self.radial_min = data.radial_range[0].to_radians();
                self.radial_max = data.radial_range[1].to_radians();
            }
            Units::QA | Units::Qnm => {
                self.radial_min = data.radial_range[0];
                self.radial_max = data.radial_range[1];
            }
        }
        self.radial = self.radial_min != self.radial_max;
    }
}

struct Bin<'a> {
    count: usize,
    min: usize,
    max: usize,
    count_f: f64,
    min_f: f64,
    max_f: f64,
    pos: &'a Positions,
}

impl<'a> Bin<'a> {
    fn valid(&mut self, j: usize) -> bool {
        self.min_f = self.bin(self.pos.lower[j]);
        self.max_f = self.bin(self.pos.upper[j]);
        if self.max_f < 0. || self.min_f >= self.count_f {
            return false;
        }
        self.max = if self.max_f >= self.count_f {
            self.count - 1
        } else {
            self.max_f as usize
        };
        self.min = if self.min_f < 0. {
            0
        } else {
            self.min_f as usize
        };
        return true;
    }

    fn bin(&self, stop: f64) -> f64 {
        (stop - self.pos.start) / self.pos.step
    }
}

impl Cake {
    /// Fast access to the first radial value if any
    pub fn radial_first(&self) -> f64 {
        match self.radial_positions.first() {
            None => 0.,
            Some(v) => *v,
        }
    }

    /// Fast access to the last radial value if any
    pub fn radial_last(&self) -> f64 {
        match self.radial_positions.last() {
            None => 0.,
            Some(v) => *v,
        }
    }

    /// Fast access to the first azimuthal value if any
    pub fn azimuthal_first(&self) -> f64 {
        match self.azimuthal_positions.first() {
            None => 0.,
            Some(v) => *v,
        }
    }

    /// Fast access to the last azimuthal value if any
    pub fn azimuthal_last(&self) -> f64 {
        match self.azimuthal_positions.last() {
            None => 0.,
            Some(v) => *v,
        }
    }
}

#[cfg(test)]
mod tests {
    use crate::integrator::{Integrable, IntegrationType, Integrator, PatternType, Units};
    use crate::poni::Poni;
    use cryiorust::cbf::Cbf;
    use cryiorust::frame::{Array, Frame};
    use std::fs::File;
    use std::io::{self, BufRead, BufReader};

    fn get_data(filename: &str) -> io::Result<(Vec<f64>, Vec<f64>, Vec<f64>)> {
        let mut pos: Vec<f64> = Vec::new();
        let mut int: Vec<f64> = Vec::new();
        let mut sigma: Vec<f64> = Vec::new();
        let file = File::open(filename)?;
        for line in BufReader::new(file).lines() {
            for (i, sline) in line?.split_whitespace().enumerate() {
                let value = sline.trim().parse::<f64>().unwrap();
                match i % 3 {
                    0 => pos.push(value),
                    1 => int.push(value),
                    2 => sigma.push(value),
                    _ => {}
                }
            }
        }
        return Ok((pos, int, sigma));
    }

    #[test]
    fn test_position_2th() {
        let test = "testdata/test.poni";
        let poni = match Poni::read_file(test) {
            Ok(poni) => poni,
            Err(e) => panic!("Could not read file {}: {}", test, e),
        };
        let mut i = Integrator::new();
        i.set_poni(poni);
        i.set_radial_bins(3500);
        i.set_polarization(0.99);
        i.init(get_frame_data("testdata/sample_frame.dat").array());
        let (pos, _, _) = get_data("testdata/sample.dat").unwrap();
        compare(&pos, &i.radial.pos, "2theta");
    }

    #[test]
    fn test_position_q() {
        let test = "testdata/test.poni";
        let poni = match Poni::read_file(test) {
            Ok(poni) => poni,
            Err(e) => panic!("Could not read file {}: {}", test, e),
        };
        let mut i = Integrator::new();
        i.set_poni(poni);
        i.set_radial_bins(3500);
        i.set_polarization(0.99);
        i.set_units(Units::Qnm);
        i.init(get_frame_data("testdata/sample_frame.dat").array());
        let (pos, _, _) = get_data("testdata/sample_q.dat").unwrap();
        compare(&pos, &i.radial.pos, "q");
    }

    fn get_frame_data(filename: &str) -> Cbf {
        let file = File::open(filename).unwrap();
        let mut image = Vec::with_capacity(1679 * 1475);
        for line in BufReader::new(file).lines() {
            image.push(line.unwrap().parse::<f64>().unwrap());
        }
        let a = Array::with_data(1679, 1475, image);
        let mut cbf = Cbf::new();
        cbf.set_array(a);
        cbf
    }

    fn compare(exp: &[f64], calc: &[f64], what: &str) {
        assert_eq!(
            exp.len(),
            calc.len(),
            "{} length expected = {}, length calculated = {}",
            what,
            exp.len(),
            calc.len()
        );
        for (i, (e, c)) in exp.iter().zip(calc.iter()).enumerate() {
            if (*c - *e).abs() > 1e-9 {
                panic!("{}: {} expected = {}, calculated = {}", i, what, *e, *c);
            }
        }
    }

    #[test]
    fn test_integrate_radial() {
        let frame = get_frame_data("testdata/sample_frame.dat");
        let ranges = [0., 0.];
        let data = Integrable {
            array: frame.array(),
            radial_range: &ranges,
            azimuthal_range: &ranges,
            integration_type: IntegrationType::Radial,
        };
        let mut i = Integrator::new();
        i.set_poni(Poni::read_file("testdata/test.poni").unwrap());
        i.set_radial_bins(3500);
        i.set_polarization(0.99);
        i.init(frame.array());
        let p = i.integrate(&data).unwrap();
        let (pos, int, sigma) = get_data("testdata/sample.dat").unwrap();
        let p = if let PatternType::Radial(p) = p.data {
            p
        } else {
            panic!("Wrong integration result");
        };
        compare(&pos, &i.radial.pos, "2th");
        compare(&int, &p.intensity, "intensity");
        compare(&sigma, &p.sigma, "intensity");
    }

    #[test]
    fn test_integrate_azimuthal() {
        let frame = get_frame_data("testdata/sample_frame.dat");
        let ranges = [0., 0.];
        let data = Integrable {
            array: frame.array(),
            radial_range: &ranges,
            azimuthal_range: &ranges,
            integration_type: IntegrationType::Azimuthal,
        };
        let mut i = Integrator::new();
        i.set_poni(Poni::read_file("testdata/test.poni").unwrap());
        i.set_polarization(0.99);
        i.init(frame.array());
        let p = i.integrate(&data).unwrap();
        let p = if let PatternType::Azimuthal(p) = p.data {
            p
        } else {
            panic!("Wrong integration result");
        };
        let (pos, int, sigma) = get_data("testdata/sample_azimuthal.dat").unwrap();
        compare(&pos, &i.azimuth.pos, "azimuthal angle");
        compare(&int, &p.intensity, "intensity");
        compare(&sigma, &p.sigma, "sigma");
    }

    #[test]
    fn test_integrate_cake() {
        let frame = get_frame_data("testdata/sample_frame.dat");
        let ranges = [0., 0.];
        let data = Integrable {
            array: frame.array(),
            radial_range: &ranges,
            azimuthal_range: &ranges,
            integration_type: IntegrationType::Cake,
        };
        let mut i = Integrator::new();
        i.set_poni(Poni::read_file("testdata/test.poni").unwrap());
        i.set_polarization(0.99);
        i.set_radial_bins(360);
        i.set_azimuthal_bins(360);
        i.init(frame.array());
        let p = i.integrate(&data).unwrap();
        let p = if let PatternType::Cake(p) = p.data {
            p
        } else {
            panic!("Wrong integration result");
        };
        let sample = get_frame_data("testdata/sample_cake.dat");
        compare(&sample.array().data(), &p.cake.data(), "cake");
    }
}
