#[macro_use]
extern crate criterion;
use criterion::{black_box, Benchmark, Criterion};
use cryiorust::cbf::Cbf;
use cryiorust::frame::{Array, Frame};
use integrustio::distortion::Distortion;
use integrustio::integrator::{Integrable, IntegrationType, Integrator, Units};
use integrustio::poni::Poni;
use integrustio::spline::Spline;
use std::fs::File;
use std::io::{BufRead, BufReader};

fn get_frame_data(filename: &str) -> Cbf {
    let file = File::open(filename).unwrap();
    let mut image = Vec::with_capacity(1679 * 1475);
    for line in BufReader::new(file).lines() {
        image.push(line.unwrap().parse::<f64>().unwrap());
    }
    let a = Array::with_data(1679, 1475, image);
    let mut cbf = Cbf::new();
    cbf.set_array(a);
    cbf
}

fn get_empty_frame(dim1: usize, dim2: usize) -> Cbf {
    let image = vec![0.; dim1 * dim2];
    let a = Array::with_data(dim1, dim2, image);
    let mut cbf = Cbf::new();
    cbf.set_array(a);
    cbf
}

fn init_integrator_2th() -> Integrator {
    let test = "testdata/test.poni";
    let poni = match Poni::read_file(test) {
        Ok(poni) => poni,
        Err(e) => panic!("Could not read file {}: {}", test, e),
    };
    let mut i = Integrator::new();
    i.set_poni(poni);
    i.set_radial_bins(3500);
    i.set_polarization(0.99);
    i
}

fn init_integrator_q() -> Integrator {
    let test = "testdata/test.poni";
    let poni = match Poni::read_file(test) {
        Ok(poni) => poni,
        Err(e) => panic!("Could not read file {}: {}", test, e),
    };
    let mut i = Integrator::new();
    i.set_poni(poni);
    i.set_radial_bins(3500);
    i.set_polarization(0.99);
    i.set_units(Units::Qnm);
    i
}

fn init_integrator_radial() -> (Integrator, Cbf) {
    let frame = get_frame_data("testdata/sample_frame.dat");
    let mut i = Integrator::new();
    i.set_poni(Poni::read_file("testdata/test.poni").unwrap());
    i.set_radial_bins(3500);
    i.set_polarization(0.99);
    i.init(frame.array());
    (i, frame)
}

fn criterion_benchmark(c: &mut Criterion) {
    c.bench_function("Spline parser", |b| {
        b.iter(|| Spline::open(black_box("testdata/F21newEO.spline")))
    });
    c.bench(
        "Spline calculations",
        Benchmark::new("dx dy", move |b| {
            let mut s = Spline::open("testdata/F21newEO.spline").unwrap();
            let frame = get_empty_frame(2048, 2048);
            b.iter(|| s.calculate(black_box(frame.array())))
        })
        .sample_size(10),
    );
    c.bench(
        "Distortion",
        Benchmark::new("initialization", move |b| {
            let mut d1 = Distortion::new();
            let frame = get_empty_frame(2048, 2048);
            let mut s = Spline::open("testdata/F21newEO.spline").unwrap();
            s.calculate(frame.array());
            b.iter(|| d1.init(black_box(frame.array()), black_box(&s)))
        })
        .with_function("correction", move |b| {
            let mut d2 = Distortion::new();
            let frame = get_empty_frame(2048, 2048);
            let mut s = Spline::open("testdata/F21newEO.spline").unwrap();
            s.calculate(frame.array());
            d2.init(frame.array(), &s);
            b.iter(|| d2.correct(black_box(frame.array())))
        })
        .sample_size(10),
    );
    c.bench_function("Poni 1", |b| {
        b.iter(|| Poni::read_file(black_box("testdata/testv1.poni")))
    });
    c.bench_function("Poni 2", |b| {
        b.iter(|| Poni::read_file(black_box("testdata/testv2.poni")))
    });
    c.bench_function("Poni 3", |b| {
        b.iter(|| Poni::read_file(black_box("testdata/testv2.1.poni")))
    });
    c.bench(
        "Positions",
        Benchmark::new("2theta", move |b| {
            let mut i1 = init_integrator_2th();
            let frame = get_frame_data("testdata/sample_frame.dat");
            b.iter(|| i1.init(black_box(frame.array())))
        })
        .with_function("Q", move |b| {
            let mut i2 = init_integrator_q();
            let frame = get_frame_data("testdata/sample_frame.dat");
            b.iter(|| i2.init(black_box(frame.array())))
        })
        .sample_size(10),
    );
    c.bench(
        "Integration",
        Benchmark::new("Radial", move |b| {
            let (i1, data) = init_integrator_radial();
            let ranges = [0., 0.];
            let data = Integrable {
                array: &data.array(),
                radial_range: &ranges,
                azimuthal_range: &ranges,
                integration_type: IntegrationType::Radial,
            };
            b.iter(|| i1.integrate(black_box(&data)))
        })
        .sample_size(10),
    );
}

criterion_group!(benches, criterion_benchmark);
criterion_main!(benches);
