# integrustio

This crate provides an azimuthal integrator and distortion corrector for diffraction data
acquired with 2D detectors. This is mainly redo of the great Python library
[pyFAI](https://pyfai.readthedocs.io) in the Rust programming language.

This is library is intended to be used mainly in the [SNBL](https://snbl.eu) integration server
[Bubble](https://soft.snbl.eu/bubble.html). Integrustio does not try to redo all the possible
integration algorithms implemented in [pyFAI](https://pyfai.readthedocs.io), its goal
is to choose one robust enough to be used for the data measured at [SNBL](https://snbl.eu).

Currently the BoundingBox with pixel splitting is chosen.

## Usage examples

* Poni parser:
```rust
use integrustio::poni::Poni;
use std::path::Path;
use std::fmt;

fn read_poni<P: AsRef<Path> + fmt::Debug>(path: P) {
    let test = path;
    let poni2 = match Poni::read_file(&test) {
        Ok(poni) => poni,
        Err(e) => panic!("Could not read file {:?}: {}", test, e),
    };
}
```

* Distortion correction together with the Spine file:
```rust
use std::io;
use integrustio::spline::Spline;
use integrustio::distortion::Distortion;
use std::io::BufReader;
use std::fs::File;
use cryiorust::frame::{Array, Frame};
use cryiorust::cbf::Cbf;
use std::path::Path;

fn correct_file<P: AsRef<Path>>(frame: P, spline: P) -> io::Result<Array> {
    let frame = Cbf::read_file(frame)?;
    let spline = Spline::init(BufReader::new(File::open(spline)?), frame.array())?;
    let mut d = Distortion::new();
    d.init(frame.array(), &spline);
    let corrected_array: Array = d.correct(frame.array())?;
    Ok(corrected_array)
}
```

* Integration:
```rust
use integrustio::integrator::{Integrable, IntegrationType, Integrator, Diffractogram};
use integrustio::poni::Poni;
use cryiorust::cbf::Cbf;
use cryiorust::frame::{Frame, FrameResult};
use std::path::Path;

fn integrate<P: AsRef<Path>>(frame: P, poni: P) -> FrameResult<Diffractogram> {
    let frame = Cbf::read_file(frame)?;
    let ranges = [0., 0.];
    let data = Integrable {
        array: frame.array(),
        radial_range: &ranges,
        azimuthal_range: &ranges,
        integration_type: IntegrationType::Radial,
    };
    let mut i = Integrator::new();
    i.set_poni(Poni::read_file(poni)?);
    i.set_radial_bins(3500);
    i.set_polarization(0.99);
    i.init(frame.array());
    Ok(i.integrate(&data).unwrap())
}
```


License: GPL-3.0+
