#[allow(unused)]
use log::{debug, error, info, warn};

use bytes::Bytes;
use serde::{de::DeserializeOwned, Serialize};

use {
    super::InfoCon,
    ipcon_sys::ipcon::{
        Ipcon, IPCON_KERNEL_GROUP_NAME, IPCON_KERNEL_NAME, IPF_DISABLE_KEVENT_FILTER, IPF_RCV_IF,
        IPF_SND_IF,
    },
    ipcon_sys::ipcon_msg::{IpconMsg, IpconMsgType},
    std::io::{Error, ErrorKind, Result},
    std::sync::mpsc::{Receiver, Sender},
    std::thread::JoinHandle,
};

#[derive(Default)]
pub struct IpconInfomgr {
    handler: Option<JoinHandle<Result<()>>>,
}


impl InfoCon for IpconInfomgr {
    /* source : group@peer */
    fn sender<T>(&mut self, source: String, receiver: Receiver<T>) -> Result<()>
    where
        T: 'static + Serialize + Send,
    {
        let handler = std::thread::spawn(move || -> Result<()> {
            let (group, peer) = source.split_once('@').map_or(
                Err(Error::new(
                    ErrorKind::InvalidInput,
                    format!("Invalid ipcon source :{}", source),
                )),
                |(g, p)| Ok((g, p)),
            )?;

            let ih = Ipcon::new(Some(peer), Some(IPF_SND_IF)).ok_or(Error::new(
                ErrorKind::InvalidData,
                "Failed to create handler",
            ))?;
            ih.register_group(group)?;

            loop {
                let report = receiver
                    .recv()
                    .map_err(|e| Error::new(ErrorKind::Other, format!("{}", e)))?;

                let buf = serde_json::to_string(&report)
                    .map_err(|e| Error::new(ErrorKind::InvalidData, format!("{}", e)))?;

                ih.send_multicast(group, Bytes::from(buf.into_bytes()), false)?;
            }
        });

        self.handler = Some(handler);
        Ok(())
    }

    /* source : group@peer */
    fn receiver<T>(&mut self, source: String, sender: Sender<Option<T>>) -> Result<()>
    where
        T: 'static + DeserializeOwned + Send,
    {
        let handler = std::thread::spawn(move || -> Result<()> {
            let (group, peer) = source.split_once('@').map_or(
                Err(Error::new(
                    ErrorKind::InvalidInput,
                    format!("Invalid ipcon source :{}", source),
                )),
                |(g, p)| Ok((g, p)),
            )?;
            let matched = |p: &str, g: &str| -> bool { p == peer && g == group };

            let ih = Ipcon::new(None, Some(IPF_RCV_IF | IPF_DISABLE_KEVENT_FILTER)).ok_or(
                Error::new(
                    ErrorKind::Other,
                    "Failed to create Ipcon handler.".to_string(),
                ),
            )?;

            ih.join_group(IPCON_KERNEL_NAME, IPCON_KERNEL_GROUP_NAME)?;
            if ih.join_group(peer, group).is_ok() {
                debug!("Infomgr: {}@{} connected", group, peer);
            }

            loop {
                let msg = ih.receive_msg()?;

                match msg {
                    IpconMsg::IpconMsgKevent(kevent) => {
                        if let Some((p, g)) = kevent.group_added() {
                            if matched(&p, &g) {
                                if ih.join_group(&p, &g).is_ok() {
                                    debug!("Infomgr: {}@{} connected", g, p);
                                }
                            }

                            continue;
                        }

                        if let Some((p, g)) = kevent.group_removed() {
                            if matched(&p, &g) {
                                debug!("Infomgr {}@{} lost.", g, p);
                                sender
                                    .send(None)
                                    .map_err(|e| Error::new(ErrorKind::Other, format!("{}", e)))?;
                                continue;
                            }
                        }
                    }

                    IpconMsg::IpconMsgUser(m)
                        if matches!(m.msg_type, IpconMsgType::IpconMsgTypeGroup) =>
                    {
                        let group = m.group.unwrap();
                        if matched(&m.peer, &group) {
                            if let Ok(s) = std::str::from_utf8(&m.buf) {
                                if let Ok(ir) = serde_json::from_str(s) {
                                    debug!("Infomgr report received from {}@{}.", group, m.peer);
                                    sender.send(Some(ir)).map_err(|e| {
                                        Error::new(ErrorKind::Other, format!("{}", e))
                                    })?;
                                }
                            }
                        }
                    }
                    _ => {}
                }
            }
        });
        self.handler = Some(handler);
        Ok(())
    }
}
