#[allow(unused)]
use {
    clap::Parser,
    infomgr_con::util::{
        cpuload::{CpuReport, CpuTime, CpuTimeDiff},
        meminfo::{MemInfo, MemoryReport},
        report::{InfomgrRawReport, InfomgrReport},
        InfoCon,
    },
    jlogger::JloggerBuilder,
    std::io::{Error, ErrorKind, Result},
    std::sync::mpsc,
};

#[cfg(feature = "enable_ipcon")]
use infomgr_con::util::ipcon::IpconInfomgr;

#[derive(Parser, Debug)]
#[clap(author, version, about, long_about=None)]
struct Cli {
    /// Name of the Sender.
    /// It should be "<group name>@<peer name)" in case of using IPCON.
    #[clap(short, long)]
    name: String,

    /// Sending interval.
    #[clap(short, long, default_value_t = 1)]
    interval: u64,

    /// Sending raw infomgr report.
    #[clap(short, long)]
    raw: bool,
}

#[allow(unused)]
fn report(name: String, interval: u64) {
    let (sender, receiver) = mpsc::channel();

    cfg_if::cfg_if! {
        if #[cfg(feature = "enable_ipcon")] {
            let mut ipcon: IpconInfomgr = Default::default();
            let _ = ipcon.sender(name, receiver);
        } else {
            let _ = std::thread::spawn(move || -> Result<()> {
                loop {
                    let ir = receiver.recv()
                        .map_err(|e| Error::new(ErrorKind::ConnectionReset, format!("{}",e)))?;

                    println!("{}", ir);

                }
            });
        }
    }

    let mut cpu_start = CpuTime::new().unwrap();
    loop {
        std::thread::sleep(std::time::Duration::from_secs(interval));
        let cpu_end = CpuTime::new().unwrap();
        let mut report: InfomgrReport = Default::default();
        report.cpu = (cpu_end - cpu_start).cpu_report();
        report.memory = MemInfo::load();
        if sender.send(report).is_err() {
            break;
        }
        cpu_start = cpu_end;
    }
}

#[allow(unused)]
fn raw_report(name: String, interval: u64) {
    let (sender, receiver) = mpsc::channel();

    cfg_if::cfg_if! {
        if #[cfg(feature = "enable_ipcon")] {
            let mut ipcon: IpconInfomgr = Default::default();
            let _ = ipcon.sender(name, receiver);
        } else {
            let _ = std::thread::spawn(move || -> Result<()> {
                loop {
                    let ir = receiver.recv()
                        .map_err(|e| Error::new(ErrorKind::ConnectionReset, format!("{}",e)))?;

                    println!("{}", ir);

                }
            });
        }
    }

    loop {
        let mut report: InfomgrRawReport = Default::default();
        let cpu = CpuTime::new().unwrap();
        report.cpu = cpu;
        report.memory = MemInfo::load();
        if sender.send(report).is_err() {
            break;
        }
        std::thread::sleep(std::time::Duration::from_secs(interval));
    }
}

fn main() {
    let cli = Cli::parse();

    JloggerBuilder::new()
        .max_level(log::LevelFilter::Debug)
        .build();

    if cli.raw {
        raw_report(cli.name, cli.interval);
    } else {
        report(cli.name, cli.interval);
    }
}
