#[allow(unused)]
use {
    clap::Parser,
    infomgr_con::util::{
        cpuload::{CpuReport, CpuTime, CpuTimeDiff},
        meminfo::{MemInfo, MemoryReport},
        report::{InfomgrRawReport, InfomgrReport},
        InfoCon,
    },
    jlogger::{jdebug, jerror, jinfo, jwarn, JloggerBuilder},
    log::{debug, error, info, warn},
    rand::prelude::*,
    std::io::{Error, ErrorKind, Result},
    std::sync::mpsc,
};

#[cfg(feature = "enable_ipcon")]
use infomgr_con::util::ipcon::IpconInfomgr;

#[derive(Parser, Debug)]
#[clap(author, version, about, long_about=None)]
struct Cli {
    /// Name of the Receiver. Format should be "<group name>@<peer name)" in case of IPCON and
    /// <FakeSource>@<Interval in secondes>)" in case of using FakeSource.
    #[clap(short, long)]
    name: String,

    /// Sending raw infomgr report.
    #[clap(short, long)]
    raw: bool,
}

fn report(name: String) {
    let (sender, receiver) = mpsc::channel::<Option<InfomgrReport>>();
    cfg_if::cfg_if! {
        if #[cfg(feature = "enable_ipcon")] {
            let mut ipcon: IpconInfomgr = Default::default();
            let _ = ipcon.receiver(name.clone(), sender);
        } else {
            let _ = std::thread::spawn(move || -> Result<()> {
                let mut rng = rand::thread_rng();
                let mut fake_cpu = || -> f32 {
                    rng.gen()
                };
                let total :u64 = 1024_u64 * 1024_u64 * 1024_u64;

                let mut rng_mem = rand::thread_rng();
                let mut fake_memory= || -> Option<u64> {
                    Some(rng_mem.gen::<u64>() % total)
                };

                loop {
                    let ir = InfomgrReport {
                        cpu:  CpuTimeDiff {
                            usr: fake_cpu(),
                            nice: fake_cpu(),
                            system: fake_cpu(),
                            idle: fake_cpu(),
                            iowait: fake_cpu(),
                            irq: fake_cpu(),
                            softirq: fake_cpu(),
                        }.cpu_report(),
                        memory: MemoryReport{
                            total: Some(total),
                            anon: fake_memory(),
                            avail: fake_memory(),
                            slab: fake_memory(),
                            ..Default::default()
                        }
                    };

                    sender.send(Some(ir))
                        .map_err(|e| Error::new(ErrorKind::ConnectionReset, format!("{}",e)))?;

                    std::thread::sleep(std::time::Duration::from_secs(1));

                }
            });
        }
    }

    loop {
        match receiver.recv() {
            Ok(Some(report)) => println!("{}", report),
            Ok(None) => error!("Source {} is lost", name),
            Err(e) => {
                jerror!("Error: {}", e);
                break;
            }
        }
    }
}

fn raw_report(name: String) {
    let (sender, receiver) = mpsc::channel::<Option<InfomgrRawReport>>();
    cfg_if::cfg_if! {
        if #[cfg(feature = "enable_ipcon")] {
            let mut ipcon: IpconInfomgr = Default::default();
            let _ = ipcon.receiver(name.clone(), sender);
        } else {
            let _ = std::thread::spawn(move || -> Result<()> {
                let mut rng = rand::thread_rng();
                let mut fake_cpu = || -> i64 {
                    rng.gen::<i64>().abs() % 10000
                };
                let total :u64 = 1024_u64 * 1024_u64 * 1024_u64;

                let mut rng_mem = rand::thread_rng();
                let mut fake_memory= || -> Option<u64> {
                    Some(rng_mem.gen::<u64>() % total)
                };

                loop {
                    let ir = InfomgrRawReport {
                        cpu:  CpuTime{
                            usr: fake_cpu(),
                            nice: fake_cpu(),
                            system: fake_cpu(),
                            idle: fake_cpu(),
                            iowait: fake_cpu(),
                            irq: fake_cpu(),
                            softirq: fake_cpu(),
                        },
                        memory: MemoryReport{
                            total: Some(total),
                            anon: fake_memory(),
                            avail: fake_memory(),
                            slab: fake_memory(),
                            ..Default::default()
                        }
                    };

                    sender.send(Some(ir))
                        .map_err(|e| Error::new(ErrorKind::ConnectionReset, format!("{}",e)))?;

                    std::thread::sleep(std::time::Duration::from_secs(1));

                }
            });
        }
    }

    loop {
        match receiver.recv() {
            Ok(Some(report)) => println!("{}", report),
            Ok(None) => error!("Source {} is lost", name),
            Err(e) => {
                jerror!("Error: {}", e);
                break;
            }
        }
    }
}

fn main() {
    let cli = Cli::parse();

    JloggerBuilder::new()
        .max_level(log::LevelFilter::Debug)
        .build();

    if cli.raw {
        raw_report(cli.name);
    } else {
        report(cli.name);
    }
}
