use std::env;
use std::fs::{self, File};
use std::io::Write;
use std::os::unix::fs::symlink;
use std::path::Path;

fn env_string(env: &str) -> String {
    format!(
        "{:30} : {}",
        env,
        env::var(env).unwrap_or_else(|_| String::from("null"))
    )
}

fn main() {
    let out_dir = env::var("OUT_DIR").unwrap();
    let build_script_log=format!("{}/build_script.log", out_dir);
    let _ = fs::remove_file(&build_script_log);
    let mut f = File::options()
        .create_new(true)
        .write(true)
        .open(&build_script_log)
        .unwrap();

    writeln!(f, "{}", env_string("CARGO")).unwrap();
    writeln!(f, "{}", env_string("CARGO_MANIFEST_LINKS")).unwrap();

    writeln!(
        f,
        "{}",
        env_string("CARGO_FEATURE_ENABLE_IPCON")
            .replace('1', "Enabled")
            .replace("null", "Disabled")
    )
    .unwrap();
    writeln!(f, "{}", env_string("TARGET")).unwrap();
    writeln!(f, "{}", env_string("OUT_DIR")).unwrap();

    let target = env::var("TARGET").unwrap();

    if target == "aarch64-unknown-linux-gnu" {
        let env_dir_name = format!("{}/build_env", out_dir);
        let env_dir = Path::new(env_dir_name.as_str());
        writeln!(f, "{:30} : {}", "env_dir", env_dir.display()).unwrap();

        if env_dir.exists() {
            if !env_dir.is_dir() {
                panic!("{} is not a directory.", env_dir.display())
            }
        } else {
            fs::create_dir(env_dir).expect("Failed to create env dir")
        }

        let ipcon_lib_orig = env::var("LIBIPCON").expect("No LIBIPCON specified.");
        let ipcon_lib_name = format!("{}/libipcon.so", env_dir_name);
        let ipcon_lib = Path::new(ipcon_lib_name.as_str());
        if ipcon_lib.exists() {
            fs::remove_file(ipcon_lib).expect("Failed to remove old ipcon lib.");
        }
        symlink(ipcon_lib_orig.as_str(), ipcon_lib_name.as_str())
            .expect("Failed to created libipcon link");

        let nl_lib_orig = env::var("LIBNL").expect("No LIBNL specified.");
        let nl_lib_name = format!("{}/libnl.so", env_dir_name);
        let nl_lib = Path::new(nl_lib_name.as_str());
        if nl_lib.exists() {
            fs::remove_file(nl_lib).expect("Failed to remove old ipcon lib.");
        }
        symlink(nl_lib_orig.as_str(), nl_lib_name.as_str())
            .expect("Failed to created libipcon link");

        let cargo_search_path = format!("cargo:rustc-link-search={}", env_dir_name);
        writeln!(f, "{:30} : {}", "cargo_search_path", cargo_search_path).unwrap();
        println!("{}", cargo_search_path.as_str());
        println!("cargo:rustc-link-lib=nl");
    }
}
