#![deny(
    warnings,
    missing_docs,
    rustdoc::all,
    clippy::pedantic,
    clippy::dbg_macro,
    clippy::semicolon_if_nothing_returned
)]
#![forbid(unsafe_code)]
#![doc(test(attr(deny(warnings), forbid(unsafe_code))))]
#![cfg_attr(any(not(feature = "no-core"), doc, test), no_std)]
#![cfg_attr(
    all(feature = "no-core", not(any(doc, test))),
    feature(no_core),
    no_core
)]
//! "Infer" the lengths of arrays in array [`static`](
//! https://doc.rust-lang.org/nightly/std/keyword.static.html)/[`const`](
//! https://doc.rust-lang.org/nightly/std/keyword.const.html) items.
//!
//! ## Crate feature
//! - `no-core`: Make this crate [`#![no_core]`](https://github.com/rust-lang/rust/issues/29639).
//!   <details>
//!   <summary><code>#![no_core]</code> example</summary>
//!
#![cfg_attr(feature = "no-core", doc = "  ```")]
#![cfg_attr(not(feature = "no-core"), doc = "  ```compile_fail")]
//!   #![feature(no_core, lang_items, start)]
//!   #![no_core]
//!
//!   #[cfg_attr(not(windows), link(name = "c"))]
//!   #[cfg_attr(windows, link(name = "msvcrt"))]
//!   extern "C" {}
//!
//!   use infer_len::infer_len;
//!
//!   #[lang = "sized"]
//!   trait Sized {}
//!
//!   #[lang = "eh_personality"]
//!   fn eh_personality() -> ! {
//!       loop {}
//!   }
//!
//!   #[lang = "receiver"]
//!   trait Receiver {}
//!
//!   impl<T: ?Sized> Receiver for &T {}
//!
//!   #[lang = "copy"]
//!   trait Copy {}
//!
//!   impl Copy for i32 {}
//!
//!   #[lang = "array"]
//!   impl [i32; 5] {
//!       const fn len(self) -> usize {
//!           5
//!       }
//!   }
//!
//!   infer_len! {
//!       const FOO: [i32; _] = [1, 2, 3, 4, 5];
//!       pub(crate) const BAR: &[i32; _] = &FOO;
//!       pub(self) const BAZ: [i32; _] = [6, 7, 8, 9, 10];
//!   }
//!
//!   #[start]
//!   fn main(_: isize, _: *const *const u8) -> isize {
//!       let _: [i32; 5] = FOO;
//!       let _: &[i32; 5] = BAR;
//!       let _: [i32; 5] = BAZ;
//!       0
//!   }
//!   ```
//!   </details>
//!
//! ## Example
//! ```
//! mod foo {
//!     # use infer_len::infer_len;
//!     use std::convert;
//!
//!     infer_len! {
//!         pub(self) const FOO: &[i128; _] = &[-1, -2, -3, -4];
//!         const BAR: [i128; _] = *FOO;
//!         pub static BAZ: [i128; _] = BAR;
//!         pub(super) static HELLO_WORLD: &[u8; _] = b"Hello World";
//!         #[allow(non_upper_case_globals)] // attribute works
//!         static qux: [i128; _] = convert::identity(BAR); // arbitrary expression works
//!         #[allow(non_upper_case_globals)]
//!         #[used] // multiple attributes works
//!         static used: [i128; _] = convert::identity(*FOO);
//!         pub(crate) const QUZ: &[i128; _] = FOO;
//!     }
//!
//!     fn _foo() {
//!         let _: &[i128; 4] = FOO;
//!         let _: [i128; 4] = BAR;
//!         let _: [i128; 4] = qux;
//!     }
//! }
//!
//! let _: &[u8; 11] = foo::HELLO_WORLD;
//! let _: [i128; 4] = foo::BAZ;
//! let _: &[i128; 4] = foo::QUZ;
//! ```

/// The main macro of this crate.
///
/// This macro accepts one or more `const` or `static` definitions of arrays or array references as
/// input. Attributes and visibility are allowed. Trailing semicolon is required, so these does not
/// compile:
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len!(const NO_TRAILING_SEMICOLON: [i32; _] = [1, 2, 3]);
/// ```
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len!(const NO_TRAILING_SEMICOLON: &[i32; _] = &[1, 2, 3]);
/// ```
///
/// Empty input is allowed:
/// ```
/// # use infer_len::infer_len;
/// infer_len! {}
/// ```
///
/// Unnamed `const` is also not allowed:
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len! {
///     const _: [i32; _] = [1, 2, 3];
/// }
/// infer_len! {
///     static FOO: [i32; _] = [1, 2, 3];
///     const BAZ: [i32; _] = [13, 14, 15];
///     pub(self) const _: &[i32; _] = [4, 5, 6];
///     static BAR: [i32; _] = [7, 8, 9];
/// }
/// ```
///
/// And visibility works as usual: ✨
/// ```compile_fail
/// mod foo {
///     # use infer_len::infer_len;
///     infer_len! {
///         pub(self) static FOO: [i32; _] = [1, 2, 3, 4, 5, 6];
///     }
/// }
/// let _: [i32; 6] = foo::FOO;
/// ```
/// ```compile_fail
/// mod foo {
///     # use infer_len::infer_len;
///     infer_len! {
///         const FOO: &[i32; _] = &[1, 2, 3, 4, 5, 6, 7];
///     }
/// }
/// let _: &[i32; 7] = foo::FOO;
/// ```
///
/// ## Implementation note
/// This macro obtains lengths of the arrays by calling their `.len()` methods.
///
/// ## Limitation
/// `[]` doesn't work:
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len! {
///     const FOO: [i32; _] = [];
/// }
/// ```
///
/// ## See also
/// The [example in the crate-level documentation](crate#example).
#[macro_export]
macro_rules! infer_len {
    { $(#[$attr:meta])* $v:vis $sc:ident $NAME:ident: [$T:ty; _] = $init:expr; $($rem:tt)* } => {
        $(#[$attr])*
        $v $sc $NAME: [$T; $init.len()] = $init;
        $crate::infer_len! { $($rem)* }
    };
    { $(#[$attr:meta])* $v:vis $sc:ident $NAME:ident: &[$T:ty; _] = $init:expr; $($rem:tt)* } => {
        $(#[$attr])*
        $v $sc $NAME: &[$T; $init.len()] = $init;
        $crate::infer_len! { $($rem)* }
    };
    () => {};
}

#[cfg(test)]
mod tests {
    use super::infer_len;

    struct Element;

    #[test]
    fn single_input() {
        infer_len! {
            const FOO: [Element; _] = [Element, Element, Element];
        }
        infer_len! {
            const BAR: &[u8; _] = b"Hello World";
        }
        let _: [Element; 3] = FOO;
        let _: &[u8; 11] = BAR;
    }

    #[test]
    fn zero_len() {
        infer_len! {
            const FOO: [Element; _] = [Element; 0];
            const BAR: &[Element; _] = &[Element; 0];
        }
        let _: [Element; 0] = FOO;
        let _: &[Element; 0] = BAR;
    }
}
