#![deny(
    warnings,
    missing_docs,
    clippy::pedantic,
    clippy::dbg_macro,
    clippy::semicolon_if_nothing_returned
)]
#![forbid(unsafe_code)]
#![doc(test(attr(deny(warnings), forbid(unsafe_code))))]
#![cfg_attr(any(not(feature = "no-core"), doc, test), no_std)]
#![cfg_attr(
    all(feature = "no-core", not(any(doc, test))),
    feature(no_core),
    no_core
)]
//! "Infer" the lengths of arrays in [`static`](
//! https://doc.rust-lang.org/nightly/std/keyword.static.html)
//! /[`const`](https://doc.rust-lang.org/nightly/std/keyword.const.html) items.
//!
//! ## Crate feature
//! - `no-core`: Make this crate [`#![no_core]`](https://github.com/rust-lang/rust/issues/29639).
//!   <details>
//!   <summary><code>#![no_core]</code> example</summary>
//!
#![cfg_attr(feature = "no-core", doc = "  ```")]
#![cfg_attr(not(feature = "no-core"), doc = "  ```compile_fail")]
//!   #![feature(no_core, lang_items, start)]
//!   #![no_core]
//!
//!   #[cfg_attr(not(windows), link(name = "c"))]
//!   #[cfg_attr(windows, link(name = "msvcrt"))]
//!   extern "C" {}
//!
//!   use infer_len::infer_len;
//!
//!   #[lang = "sized"]
//!   trait Sized {}
//!
//!   #[lang = "eh_personality"]
//!   fn eh_personality() -> ! {
//!       loop {}
//!   }
//!
//!   #[lang = "receiver"]
//!   trait Receiver {}
//!
//!   impl<T: ?Sized> Receiver for &T {}
//!
//!   #[lang = "array"]
//!   impl [i32; 5] {
//!       const fn len(&self) -> usize {
//!           5
//!       }
//!   }
//!
//!   infer_len! {
//!       const FOO: [i32] = [1, 2, 3, 4, 5];
//!       const BAR: [i32] = [6, 7, 8, 9, 10];
//!   }
//!
//!   #[start]
//!   fn main(_: isize, _: *const *const u8) -> isize {
//!       let _: [i32; 5] = FOO;
//!       let _: [i32; 5] = BAR;
//!       0
//!   }
//!   ```
//!   </details>
//!
//! ## Example
//! ```
//! # use infer_len::infer_len;
//! use std::convert;
//!
//! infer_len! {
//!     pub(self) const FOO: [i128] = [-1, -2, -3, -4];
//!     const BAR: [i128] = FOO;
//!     pub static BAZ: [i128] = BAR;
//!     #[allow(non_upper_case_globals)] // attribute works
//!     static qux: [i128] = convert::identity(BAR); // arbitrary expression works
//!     #[allow(non_upper_case_globals)]
//!     #[used] // multiple attributes works
//!     static used: [i128] = FOO;
//! }
//!
//! let _: [i128; 4] = FOO;
//! let _: [i128; 4] = BAR;
//! let _: [i128; 4] = BAZ;
//! let _: [i128; 4] = qux;
//! ```

/// The main macro of this crate.
///
/// This macro accepts one or more `const` or `static` definitions as input. Attributes and
/// visibility are allowed. Trailing semicolon is required:
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len!(const NO_TRAILING_SEMICOLON: [i32] = [1, 2, 3]);
/// ```
///
/// Empty input is not allowed:
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len! {}
/// ```
///
/// Unnamed `const` is also not allowed:
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len! {
///     const _: [i32] = [1, 2, 3];
/// }
/// infer_len! {
///     static FOO: [i32] = [1, 2, 3];
///     const BAZ: [i32] = [13, 14, 15];
///     pub(self) const _: [i32] = [4, 5, 6];
///     static BAR: [i32] = [7, 8, 9];
/// }
/// ```
///
/// The types of definitions need to be [slices](slice), although the generated `const` or
/// `static` items have array types.
///
/// ## Implementation note
/// This macro obtains lengths of the arrays by calling their `.len()` methods.
///
/// ## Limitation
/// `[]` doesn't work:
/// ```compile_fail
/// # use infer_len::infer_len;
/// infer_len! {
///     const FOO: [i32] = [];
/// }
/// ```
///
/// ## See also
/// The [example in the crate-level documentation](crate#example).
#[macro_export]
macro_rules! infer_len {
    { $($(#[$attr:meta])* $v:vis $sc:ident $NAME:ident: [$T:ty] = $init:expr;)+ } => {
        $(
            $(#[$attr])*
            $v $sc $NAME: [$T; $init.len()] = $init;
        )+
    };
}

#[cfg(test)]
mod tests {
    use super::infer_len;

    struct Element;

    #[test]
    fn single_input() {
        infer_len! {
            const FOO: [Element] = [Element, Element, Element];
        }
        let _: [Element; 3] = FOO;
    }

    #[test]
    fn zero_len() {
        infer_len! {
            const FOO: [Element] = [Element; 0];
        }
        let _: [Element; 0] = FOO;
    }
}
