use crate::message::{TryFromByteError, TryFromIndexError};
use std::{
    collections::HashMap,
    ffi::{CStr, CString},
};

/// The milter stage.
#[derive(Clone, Copy, Debug, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub enum Stage {
    /// The `connect` stage.
    Connect,
    /// The `helo` stage.
    Helo,
    /// The `mail` stage.
    Mail,
    /// The `rcpt` stage.
    Rcpt,
    /// The `data` stage.
    Data,
    /// The `eoh` stage.
    Eoh,
    /// The `eom` stage.
    Eom,
}

impl Stage {
    /// Returns an iterator of all stages.
    pub fn all_stages() -> impl DoubleEndedIterator<Item = Self> {
        use Stage::*;
        [Connect, Helo, Mail, Rcpt, Data, Eoh, Eom].into_iter()
    }

    /// Returns an iterator of all stages, sorted by their index value.
    pub fn all_stages_sorted_by_index() -> impl DoubleEndedIterator<Item = Self> {
        // In libmilter, Eoh and Eom are out of order. This function is provided
        // to be able to iterate in the same order as libmilter.
        use Stage::*;
        [Connect, Helo, Mail, Rcpt, Data, Eom, Eoh].into_iter()
    }
}

impl From<Stage> for i32 {
    fn from(stage: Stage) -> Self {
        match stage {
            Stage::Connect => 0,
            Stage::Helo => 1,
            Stage::Mail => 2,
            Stage::Rcpt => 3,
            Stage::Data => 4,
            Stage::Eoh => 6,
            Stage::Eom => 5,
        }
    }
}

impl TryFrom<i32> for Stage {
    type Error = TryFromIndexError;

    fn try_from(value: i32) -> Result<Self, Self::Error> {
        match value {
            0 => Ok(Self::Connect),
            1 => Ok(Self::Helo),
            2 => Ok(Self::Mail),
            3 => Ok(Self::Rcpt),
            4 => Ok(Self::Data),
            6 => Ok(Self::Eoh),
            5 => Ok(Self::Eom),
            value => Err(TryFromIndexError::new(value)),
        }
    }
}

impl From<Stage> for u8 {
    fn from(stage: Stage) -> Self {
        match stage {
            Stage::Connect => b'C',
            Stage::Helo => b'H',
            Stage::Mail => b'M',
            Stage::Rcpt => b'R',
            Stage::Data => b'T',
            Stage::Eoh => b'N',
            Stage::Eom => b'E',
        }
    }
}

impl TryFrom<u8> for Stage {
    type Error = TryFromByteError;

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            b'C' => Ok(Self::Connect),
            b'H' => Ok(Self::Helo),
            b'M' => Ok(Self::Mail),
            b'R' => Ok(Self::Rcpt),
            b'T' => Ok(Self::Data),
            b'N' => Ok(Self::Eoh),
            b'E' => Ok(Self::Eom),
            value => Err(TryFromByteError::new(value)),
        }
    }
}

// Note: This struct does *not* implement `Default` or `Clone`. Callbacks
// receive a mutable reference to this struct but must not be able to replace
// it.
/// Currently defined macros.
#[derive(Debug, Eq, PartialEq)]
pub struct Macros(HashMap<Stage, HashMap<CString, CString>>);

impl Macros {
    pub(crate) fn new() -> Self {
        Self(
            Stage::all_stages()
                .map(|stage| (stage, Default::default()))
                .collect(),
        )
    }

    pub(crate) fn clone_internal(&self) -> Self {
        Self(self.0.clone())
    }

    /// Returns the value associated with the given macro name, if any.
    ///
    /// Single-character macros like `i` or `{i}` can be looked up in either
    /// form (both names `i` and `{i}` return the same result).
    pub fn get(&self, name: &CStr) -> Option<&CStr> {
        let alt_name;
        let alt_name = match *name.to_bytes() {
            [x] => {
                alt_name = [b'{', x, b'}', 0];
                Some(CStr::from_bytes_with_nul(&alt_name[..4]).unwrap())
            }
            [b'{', x, b'}'] => {
                alt_name = [x, 0, 0, 0];
                Some(CStr::from_bytes_with_nul(&alt_name[..2]).unwrap())
            }
            _ => None,
        };

        Stage::all_stages().rev().find_map(|stage| {
            let m = &self.0[&stage];
            m.get(name)
                .or_else(|| alt_name.and_then(|name| m.get(name)))
                .map(|v| v.as_ref())
        })
    }

    /// Returns a new hash map with all defined macros.
    pub fn to_hash_map(&self) -> HashMap<CString, CString> {
        Stage::all_stages()
            .flat_map(|stage| self.0[&stage].clone())
            .collect()
    }

    pub(crate) fn insert(&mut self, stage: Stage, entries: HashMap<CString, CString>) {
        self.0.insert(stage, entries);
    }

    pub(crate) fn clear(&mut self) {
        for s in Stage::all_stages() {
            self.0.get_mut(&s).unwrap().clear();
        }
    }

    pub(crate) fn clear_after(&mut self, stage: Stage) {
        for s in Stage::all_stages().rev().take_while(|&s| s != stage) {
            self.0.get_mut(&s).unwrap().clear();
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use byte_strings::c_str;

    #[test]
    fn macro_lookup() {
        let mut macros = Macros::new();

        macros.insert(
            Stage::Connect,
            HashMap::from([
                (c_str!("a").into(), c_str!("b").into()),
                (c_str!("{x}").into(), c_str!("y").into()),
                (c_str!("{name}").into(), c_str!("value").into()),
            ]),
        );

        assert_eq!(macros.get(c_str!("n")), None);
        assert_eq!(macros.get(c_str!("{none}")), None);
        assert_eq!(macros.get(c_str!("{name}")), Some(c_str!("value")));
        assert_eq!(macros.get(c_str!("a")), Some(c_str!("b")));
        assert_eq!(macros.get(c_str!("{a}")), Some(c_str!("b")));
        assert_eq!(macros.get(c_str!("x")), Some(c_str!("y")));
        assert_eq!(macros.get(c_str!("{x}")), Some(c_str!("y")));
    }

    #[test]
    fn to_hash_map_ok() {
        let mut macros = Macros::new();

        macros.insert(
            Stage::Connect,
            HashMap::from([
                (c_str!("{name1}").into(), c_str!("connect1").into()),
                (c_str!("{name2}").into(), c_str!("connect2").into()),
            ]),
        );
        macros.insert(
            Stage::Mail,
            HashMap::from([
                (c_str!("{name2}").into(), c_str!("mail2").into()),
                (c_str!("{name3}").into(), c_str!("mail3").into()),
            ]),
        );

        assert_eq!(
            macros.to_hash_map(),
            HashMap::from([
                (c_str!("{name1}").into(), c_str!("connect1").into()),
                (c_str!("{name2}").into(), c_str!("mail2").into()),
                (c_str!("{name3}").into(), c_str!("mail3").into()),
            ]),
        );

        macros.clear_after(Stage::Helo);

        assert_eq!(
            macros.to_hash_map(),
            HashMap::from([
                (c_str!("{name1}").into(), c_str!("connect1").into()),
                (c_str!("{name2}").into(), c_str!("connect2").into()),
            ]),
        );
    }
}
