use crate::proto_util::Actions;
use std::time::Duration;

/// Milter configuration.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct Config {
    /// The maximum number of connections open at any time.
    ///
    /// The default is 100.
    pub max_connections: usize,

    /// Actions to enable during the `eom` stage.
    ///
    /// This setting is overridden by the actions requested through
    /// [`NegotiateContext::requested_actions`][crate::context::NegotiateContext::requested_actions]
    /// in the `negotiate` stage.
    ///
    /// The default is empty.
    pub actions: Actions,

    /// The connection timeout.
    ///
    /// This timeout concerns reading/writing of entire milter protocol messages
    /// to the connection. This limit would be reached eg when a milter client
    /// idles and does not send the next command. When the timeout expires the
    /// client’s connection is closed.
    ///
    /// Compare Postfix’s `milter_*_timeout` parameters.
    ///
    /// The default is 305 seconds.
    pub connection_timeout: Duration,
}

impl Default for Config {
    fn default() -> Self {
        Self {
            max_connections: 100,
            actions: Default::default(),
            // The default value of 305 seconds (5 minutes plus 5 seconds) is
            // slightly above the longest default timeout value used in Postfix,
            // namely `milter_content_timeout`, which defaults to 300 seconds.
            // (Compare with libmilter, which uses a timeout of 2 hours!)
            connection_timeout: Duration::from_secs(305),
        }
    }
}
