use crate::{
    context::{Context, EomContext, NegotiateContext},
    proto_util::{Actions, ProtoOpts, SocketInfo},
};
use std::{ffi::CString, future::Future, pin::Pin};

/// The status returned from callbacks.
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum Status {
    /// Accept this connection or message.
    Accept,
    /// Continue with the next stage.
    Continue,
    /// Reject this connection, message, or recipient.
    Reject,
    /// Reject this connection, message, or recipient with a temporary failure.
    Tempfail,
    /// Silently discard this message.
    Discard,
    /// Do not send a response to the MTA.
    Noreply,
    /// Skip further invocations of this callback.
    Skip,
    /// Enable all stages and all actions during negotiation.
    AllOpts,
}

impl Default for Status {
    fn default() -> Self {
        Self::Continue
    }
}

/// The type returned by milter callbacks.
pub type CallbackFuture<'a> = Pin<Box<dyn Future<Output = Status> + Send + 'a>>;

// Deriving `Default` enables struct update syntax (`..Default::default()`) when
// type parameter `T` is also `Default`.
/// Callback closures for each milter stage.
#[derive(Default)]
pub struct Callbacks<T: Send> {
    /// The callback for the `negotiate` stage.
    pub negotiate:
        Option<Box<dyn for<'cx> Fn(&'cx mut NegotiateContext<T>, Actions, ProtoOpts) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `connect` stage.
    pub connect:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>, CString, SocketInfo) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `helo` stage.
    pub helo:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>, CString) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `mail` stage.
    pub mail:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>, Vec<CString>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `rcpt` stage.
    pub rcpt:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>, Vec<CString>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `data` stage.
    pub data:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `header` stage.
    pub header:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>, CString, CString) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `eoh` stage.
    pub eoh:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `body` stage.
    pub body:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>, Vec<u8>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `eom` stage.
    pub eom:
        Option<Box<dyn for<'cx> Fn(&'cx mut EomContext<T>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `abort` stage.
    pub abort:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `close` stage.
    pub close:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>) -> CallbackFuture<'cx> + Send + Sync>>,
    /// The callback for the `unknown` stage.
    pub unknown:
        Option<Box<dyn for<'cx> Fn(&'cx mut Context<T>, CString) -> CallbackFuture<'cx> + Send + Sync>>,
}

impl<T: Send> Callbacks<T> {
    /// Returns a new, empty collection of callbacks.
    pub fn new() -> Self {
        Self {
            negotiate: None,
            connect: None,
            helo: None,
            mail: None,
            rcpt: None,
            data: None,
            header: None,
            eoh: None,
            body: None,
            eom: None,
            abort: None,
            close: None,
            unknown: None,
        }
    }

    /// Configures the callback for the `negotiate` stage.
    pub fn on_negotiate(
        mut self,
        callback: impl Fn(&mut NegotiateContext<T>, Actions, ProtoOpts) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.negotiate = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `connect` stage.
    pub fn on_connect(
        mut self,
        callback: impl Fn(&mut Context<T>, CString, SocketInfo) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.connect = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `helo` stage.
    pub fn on_helo(
        mut self,
        callback: impl Fn(&mut Context<T>, CString) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.helo = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `mail` stage.
    pub fn on_mail(
        mut self,
        callback: impl Fn(&mut Context<T>, Vec<CString>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.mail = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `rcpt` stage.
    pub fn on_rcpt(
        mut self,
        callback: impl Fn(&mut Context<T>, Vec<CString>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.rcpt = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `data` stage.
    pub fn on_data(
        mut self,
        callback: impl Fn(&mut Context<T>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.data = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `header` stage.
    pub fn on_header(
        mut self,
        callback: impl Fn(&mut Context<T>, CString, CString) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.header = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `eoh` stage.
    pub fn on_eoh(
        mut self,
        callback: impl Fn(&mut Context<T>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.eoh = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `body` stage.
    pub fn on_body(
        mut self,
        callback: impl Fn(&mut Context<T>, Vec<u8>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.body = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `eom` stage.
    pub fn on_eom(
        mut self,
        callback: impl Fn(&mut EomContext<T>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.eom = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `abort` stage.
    pub fn on_abort(
        mut self,
        callback: impl Fn(&mut Context<T>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.abort = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `close` stage.
    pub fn on_close(
        mut self,
        callback: impl Fn(&mut Context<T>) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.close = Some(Box::new(callback));
        self
    }

    /// Configures the callback for the `unknown` stage.
    pub fn on_unknown(
        mut self,
        callback: impl Fn(&mut Context<T>, CString) -> CallbackFuture<'_> + Send + Sync + 'static,
    ) -> Self {
        self.unknown = Some(Box::new(callback));
        self
    }
}
