//! Milter replies.

use crate::{
    macros::Stage,
    message::{Message, TryFromByteError, Version},
    proto_util::{Actions, ProtoOpts},
};
use bytes::{BufMut, Bytes, BytesMut};
use std::{collections::HashMap, ffi::CString};

/// The kind of a reply.
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum ReplyKind {
    /// The `+` reply.
    AddRcpt,
    /// The `-` reply.
    DeleteRcpt,
    /// The `2` reply.
    AddRcptExt,
    /// The `O` reply.
    OptNeg,
    /// The `a` reply.
    Accept,
    /// The `b` reply.
    ReplaceBody,
    /// The `c` reply.
    Continue,
    /// The `d` reply.
    Discard,
    /// The `e` reply.
    ChangeSender,
    /// The `h` reply.
    AddHeader,
    /// The `i` reply.
    InsertHeader,
    /// The `m` reply.
    ChangeHeader,
    /// The `p` reply.
    Progress,
    /// The `q` reply.
    Quarantine,
    /// The `r` reply.
    Reject,
    /// The `s` reply.
    Skip,
    /// The `t` reply.
    Tempfail,
    /// The `y` reply.
    ReplyCode,
}

impl From<ReplyKind> for u8 {
    fn from(kind: ReplyKind) -> Self {
        match kind {
            ReplyKind::AddRcpt => b'+',
            ReplyKind::DeleteRcpt => b'-',
            ReplyKind::AddRcptExt => b'2',
            ReplyKind::OptNeg => b'O',
            ReplyKind::Accept => b'a',
            ReplyKind::ReplaceBody => b'b',
            ReplyKind::Continue => b'c',
            ReplyKind::Discard => b'd',
            ReplyKind::ChangeSender => b'e',
            ReplyKind::AddHeader => b'h',
            ReplyKind::InsertHeader => b'i',
            ReplyKind::ChangeHeader => b'm',
            ReplyKind::Progress => b'p',
            ReplyKind::Quarantine => b'q',
            ReplyKind::Reject => b'r',
            ReplyKind::Skip => b's',
            ReplyKind::Tempfail => b't',
            ReplyKind::ReplyCode => b'y',
        }
    }
}

impl TryFrom<u8> for ReplyKind {
    type Error = TryFromByteError;

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            b'+' => Ok(Self::AddRcpt),
            b'-' => Ok(Self::DeleteRcpt),
            b'2' => Ok(Self::AddRcptExt),
            b'O' => Ok(Self::OptNeg),
            b'a' => Ok(Self::Accept),
            b'b' => Ok(Self::ReplaceBody),
            b'c' => Ok(Self::Continue),
            b'd' => Ok(Self::Discard),
            b'e' => Ok(Self::ChangeSender),
            b'h' => Ok(Self::AddHeader),
            b'i' => Ok(Self::InsertHeader),
            b'm' => Ok(Self::ChangeHeader),
            b'p' => Ok(Self::Progress),
            b'q' => Ok(Self::Quarantine),
            b'r' => Ok(Self::Reject),
            b's' => Ok(Self::Skip),
            b't' => Ok(Self::Tempfail),
            b'y' => Ok(Self::ReplyCode),
            value => Err(TryFromByteError(value)),
        }
    }
}

/// A reply.
#[derive(Clone, Debug, Eq, PartialEq)]
pub enum Reply {
    /// The `+` reply.
    AddRcpt {
        rcpt: CString,  // non-empty
    },
    /// The `-` reply.
    DeleteRcpt {
        rcpt: CString,  // non-empty
    },
    /// The `2` reply.
    AddRcptExt {
        rcpt: CString,  // non-empty
        args: Option<CString>,
    },
    /// The `O` reply.
    OptNeg {
        version: Version,
        actions: Actions,
        opts: ProtoOpts,
        macros: HashMap<Stage, CString>,
    },
    /// The `a` reply.
    Accept,
    /// The `b` reply.
    ReplaceBody { chunk: Bytes },
    /// The `c` reply.
    Continue,
    /// The `d` reply.
    Discard,
    /// The `e` reply.
    ChangeSender {
        mail: CString,  // non-empty
        args: Option<CString>,
    },
    /// The `h` reply.
    AddHeader {
        name: CString,  // non-empty
        value: CString,
    },
    /// The `i` reply.
    InsertHeader {
        index: i32,  // non-negative
        name: CString,  // non-empty
        value: CString,
    },
    /// The `m` reply.
    ChangeHeader {
        name: CString,  // non-empty
        index: i32,  // non-negative
        value: CString,
    },
    /// The `p` reply.
    Progress,
    /// The `q` reply.
    Quarantine {
        reason: CString,  // non-empty
    },
    /// The `r` reply.
    Reject,
    /// The `s` reply.
    Skip,
    /// The `t` reply.
    Tempfail,
    /// The `y` reply.
    ReplyCode {
        reply: CString,  // conforms to reply as produced in Context, eg "550 ..."
    },
}

impl Reply {
    /// Converts this reply into a milter protocol message.
    pub fn into_message(self) -> Message {
        match self {
            Self::AddRcpt { rcpt } => {
                let rcpt = rcpt.to_bytes_with_nul();

                Message::new(ReplyKind::AddRcpt, Bytes::copy_from_slice(rcpt))
            }
            Self::DeleteRcpt { rcpt } => {
                let rcpt = rcpt.to_bytes_with_nul();

                Message::new(ReplyKind::DeleteRcpt, Bytes::copy_from_slice(rcpt))
            }
            Self::AddRcptExt { rcpt, args } => {
                let rcpt = rcpt.to_bytes_with_nul();

                let mut buf = BytesMut::with_capacity(rcpt.len());

                buf.put(rcpt);
                if let Some(args) = args {
                    buf.put(args.to_bytes_with_nul());
                }

                Message::new(ReplyKind::AddRcptExt, buf)
            }
            Self::OptNeg { version, actions, opts, macros } => {
                let mut buf = BytesMut::with_capacity(12);

                buf.put_u32(version);
                buf.put_u32(actions.bits());
                buf.put_u32(opts.bits());

                for stage in Stage::all_stages_sorted_by_index() {
                    if let Some(macros) = macros.get(&stage) {
                        buf.put_i32(stage.into());
                        buf.put(macros.to_bytes_with_nul());
                    }
                }

                Message::new(ReplyKind::OptNeg, buf)
            }
            Self::Accept => Message::new(ReplyKind::Accept, Bytes::new()),
            Self::ReplaceBody { chunk } => Message::new(ReplyKind::ReplaceBody, chunk),
            Self::Continue => Message::new(ReplyKind::Continue, Bytes::new()),
            Self::Discard => Message::new(ReplyKind::Discard, Bytes::new()),
            Self::ChangeSender { mail, args } => {
                let mail = mail.to_bytes_with_nul();

                let mut buf = BytesMut::with_capacity(mail.len());

                buf.put(mail);
                if let Some(args) = args {
                    buf.put(args.to_bytes_with_nul());
                }

                Message::new(ReplyKind::ChangeSender, buf)
            }
            Self::AddHeader { name, value } => {
                let name = name.to_bytes_with_nul();
                let value = value.to_bytes_with_nul();

                let mut buf = BytesMut::with_capacity(name.len() + value.len());

                buf.put(name);
                buf.put(value);

                Message::new(ReplyKind::AddHeader, buf)
            }
            Self::InsertHeader { index, name, value } => {
                let name = name.to_bytes_with_nul();
                let value = value.to_bytes_with_nul();

                let mut buf = BytesMut::with_capacity(name.len() + value.len() + 4);

                buf.put_i32(index);
                buf.put(name);
                buf.put(value);

                Message::new(ReplyKind::InsertHeader, buf)
            }
            Self::ChangeHeader { name, index, value } => {
                let name = name.to_bytes_with_nul();
                let value = value.to_bytes_with_nul();

                let mut buf = BytesMut::with_capacity(name.len() + value.len() + 4);

                buf.put_i32(index);
                buf.put(name);
                buf.put(value);

                Message::new(ReplyKind::ChangeHeader, buf)
            }
            Self::Progress => Message::new(ReplyKind::Progress, Bytes::new()),
            Self::Quarantine { reason } => {
                let reason = reason.to_bytes_with_nul();

                Message::new(ReplyKind::Quarantine, Bytes::copy_from_slice(reason))
            }
            Self::Reject => Message::new(ReplyKind::Reject, Bytes::new()),
            Self::Skip => Message::new(ReplyKind::Skip, Bytes::new()),
            Self::Tempfail => Message::new(ReplyKind::Tempfail, Bytes::new()),
            Self::ReplyCode { reply } => {
                let reply = reply.to_bytes_with_nul();

                Message::new(ReplyKind::ReplyCode, Bytes::copy_from_slice(reply))
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use byte_strings::c_str;

    #[test]
    fn add_rcpt_into_message() {
        let rcpt = c_str!("<rcpt@example.com>");

        let add_rcpt = Reply::AddRcpt { rcpt: rcpt.into() };
        let add_rcpt_ext = Reply::AddRcptExt {
            rcpt: rcpt.into(),
            args: Some(c_str!("ARGS").into()),
        };

        assert_eq!(
            add_rcpt.into_message(),
            Message::new(ReplyKind::AddRcpt, "<rcpt@example.com>\0")
        );
        assert_eq!(
            add_rcpt_ext.into_message(),
            Message::new(ReplyKind::AddRcptExt, "<rcpt@example.com>\0ARGS\0")
        );
    }
}
