//! Milter protocol messages.

use bytes::Bytes;
use std::{
    ascii,
    error::Error,
    fmt::{self, Debug, Display, Formatter, Write},
};

pub mod commands;
pub mod replies;

/// The type of the milter protocol version.
pub type Version = u32;

/// Milter protocol version.
pub const PROTOCOL_VERSION: Version = 6;

/// A milter protocol message.
#[derive(Clone, Eq, Hash, PartialEq)]
pub struct Message {
    /// The message kind.
    pub kind: u8,
    /// The message payload buffer.
    pub buffer: Bytes,
}

impl Message {
    // Limit length of message buffer somewhat arbitrarily to 1 MB. Else if
    // someone erroneously sends a jumbo message we would spend a lot of time
    // and memory reading/writing it.
    //
    // Support for the experimental, negotiable custom `Maxdatasize` feature of
    // libmilter is not implemented.
    pub(crate) const MAX_BUFFER_LEN: usize = 1024 * 1024 - 1;

    /// Creates a new message with the given kind and payload buffer.
    pub fn new(kind: impl Into<u8>, buffer: impl Into<Bytes>) -> Self {
        Self {
            kind: kind.into(),
            buffer: buffer.into(),
        }
    }
}

impl Debug for Message {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.debug_struct("Message")
            .field("kind", &Byte(self.kind))
            .field("buffer", &self.buffer)
            .finish()
    }
}

struct Byte(u8);

impl Debug for Byte {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.write_str("b'")?;

        match self.0 {
            b'\0' => f.write_str("\\0")?,
            b'"' => f.write_str("\"")?,
            byte => {
                for c in ascii::escape_default(byte) {
                    f.write_char(c.into())?;
                }
            }
        }

        f.write_str("'")?;

        Ok(())
    }
}

/// An error that occurs when conversion from a wire format byte fails.
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub struct TryFromByteError(u8);

impl TryFromByteError {
    pub(crate) fn new(byte: u8) -> Self {
        Self(byte)
    }

    /// Returns the byte that caused the conversion error.
    pub fn byte(&self) -> u8 {
        self.0
    }
}

impl Display for TryFromByteError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "failed to convert byte {:?}", Byte(self.0))
    }
}

impl Error for TryFromByteError {}

/// An error that occurs when conversion from a wire format index fails.
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub struct TryFromIndexError(i32);

impl TryFromIndexError {
    pub(crate) fn new(index: i32) -> Self {
        Self(index)
    }

    /// Returns the index that caused the conversion error.
    pub fn index(&self) -> i32 {
        self.0
    }
}

impl Display for TryFromIndexError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "failed to convert index {}", self.0)
    }
}

impl Error for TryFromIndexError {}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn byte_debug_ok() {
        assert_eq!(format!("{:?}", Byte(b'\0')), r#"b'\0'"#);
        assert_eq!(format!("{:?}", Byte(b'\n')), r#"b'\n'"#);
        assert_eq!(format!("{:?}", Byte(b'"')), r#"b'"'"#);
        assert_eq!(format!("{:?}", Byte(b'a')), r#"b'a'"#);
        assert_eq!(format!("{:?}", Byte(b' ')), r#"b' '"#);
        assert_eq!(format!("{:?}", Byte(b'\x07')), r#"b'\x07'"#);
        assert_eq!(format!("{:?}", Byte(b'\xef')), r#"b'\xef'"#);
    }

    #[test]
    fn message_debug_ok() {
        assert_eq!(
            format!("{:?}", Message::new(b'A', "abc\0")),
            r#"Message { kind: b'A', buffer: b"abc\0" }"#
        );
        assert_eq!(
            format!("{:?}", Message::new(b'\0', "")),
            r#"Message { kind: b'\0', buffer: b"" }"#
        );
    }

    #[test]
    fn try_from_byte_error_debug() {
        assert_eq!(
            TryFromByteError(b'x').to_string(),
            "failed to convert byte b'x'"
        );
    }
}
