use crate::proto_util::Actions;
use std::time::Duration;

/// Milter configuration.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct Config {
    /// The maximum number of connections open at any time.
    ///
    /// The default is 100.
    pub max_connections: usize,

    /// Actions to enable during `eom` stage.
    ///
    /// The default is empty.
    pub actions: Actions,

    /// The connection timeout.
    ///
    /// This timeout concerns reading/writing of entire milter protocol messages
    /// to the connection. This limit would be reached eg when a milter client
    /// idles and does not send the next command.
    ///
    /// Between 30 and 300 seconds may be a reasonable value. See also Postfix’s
    /// `milter_*_timeout` parameters.
    ///
    /// The default is 60 seconds.
    pub connection_timeout: Duration,
}

impl Default for Config {
    fn default() -> Self {
        Self {
            max_connections: 100,
            actions: Default::default(),
            // This is a somewhat ‘aggressive’ timeout: Postfix’s
            // `milter_content_timeout` defaults to 300 seconds. However, a
            // milter processes the body content in chunks, so we simply assume
            // that it takes no more than 60 seconds to send each command/chunk.
            // (Compare with libmilter, which uses a timeout of 2 hours!)
            connection_timeout: Duration::from_secs(60),
        }
    }
}
