mod common;

pub use crate::common::*;
use byte_strings::c_str;
use indymilter::{
    message::{
        commands::{Command, ConnInfoPayload, MacroPayload, OptNegPayload},
        replies::Reply,
        MILTER_VERSION,
    },
    Actions, Callbacks, ProtoOpts, Stage, Status,
};
use std::{
    collections::HashMap,
    sync::{Arc, Mutex},
};

#[tokio::test]
async fn request_macros() {
    let _ = tracing_subscriber::fmt::try_init();

    let callbacks = Callbacks::<()>::new()
        .on_negotiate(|cx, _actions, _opts| {
            Box::pin(async move {
                cx.requested_actions = Actions::empty();
                cx.requested_opts = ProtoOpts::empty();

                cx.requested_macros
                    .insert(Stage::Helo, c_str!("what").into());

                Status::Continue
            })
        });

    let milter = Milter::spawn(LOCALHOST, callbacks, Default::default())
        .await
        .unwrap();

    let mut client = Client::connect(milter.addr()).await.unwrap();

    client
        .write_command(Command::OptNeg(OptNegPayload {
            version: MILTER_VERSION,
            actions: Actions::all(),
            opts: ProtoOpts::all(),
        }))
        .await
        .unwrap();

    let reply = client.read_reply().await.unwrap();
    assert_eq!(
        reply,
        Reply::OptNeg {
            version: MILTER_VERSION,
            actions: Actions::empty(),
            opts: ProtoOpts::empty(),
            macros: HashMap::from([(Stage::Helo, c_str!("what").into())]),
        }
    );

    client.write_command(Command::Quit).await.unwrap();

    client.disconnect().await.unwrap();

    milter.shutdown().await.unwrap();
}

#[tokio::test]
async fn out_of_order_macros() {
    let _ = tracing_subscriber::fmt::try_init();

    let result = Arc::new(Mutex::new(HashMap::new()));

    let callbacks = {
        let result = result.clone();

        Callbacks::<()>::new()
            .on_connect(move |cx, _, _| {
                let result = result.clone();

                Box::pin(async move {
                    *result.lock().unwrap() = cx.macros.to_hash_map();

                    Status::Continue
                })
            })
    };

    let milter = Milter::spawn(LOCALHOST, callbacks, Default::default())
        .await
        .unwrap();

    let mut client = Client::connect(milter.addr()).await.unwrap();

    // Register macros for the Connect stage. However, the OptNeg command clears
    // out the out-of-order macros and no macros are available for connect.

    client
        .write_command(Command::DefMacros(MacroPayload {
            stage: Stage::Connect,
            macros: vec![c_str!("{name}").into(), c_str!("value").into()],
        }))
        .await
        .unwrap();

    client
        .write_command(Command::OptNeg(OptNegPayload {
            version: MILTER_VERSION,
            actions: Actions::all(),
            opts: ProtoOpts::all(),
        }))
        .await
        .unwrap();

    let reply = client.read_reply().await.unwrap();
    assert!(matches!(reply, Reply::OptNeg { .. }));

    client
        .write_command(Command::ConnInfo(ConnInfoPayload {
            hostname: c_str!("mail.example.com").into(),
            socket_info: None,
        }))
        .await
        .unwrap();

    let reply = client.read_reply().await.unwrap();
    assert_eq!(reply, Reply::Continue);

    client.write_command(Command::Quit).await.unwrap();

    client.disconnect().await.unwrap();

    milter.shutdown().await.unwrap();

    let result = Arc::try_unwrap(result).unwrap().into_inner().unwrap();

    assert_eq!(result, HashMap::new());
}
