#![cfg(feature = "miltertest")]

mod common;

pub use crate::common::*;
use indymilter::{Callbacks, Context, ContextActions, SocketInfo, Status};
use std::ffi::CString;
use tracing::info;
use byte_strings::c_str;

#[tokio::test]
async fn basic_miltertest() {
    let _ = tracing_subscriber::fmt::try_init();

    let callbacks = Callbacks::new()
        .on_negotiate(|_cx, actions, opts| {
            Box::pin(async move {
                info!(?actions, ?opts, "inside negotiate callback");
                Status::Continue
            })
        })
        .on_connect(|cx, hostname, addr| Box::pin(connect_callback(cx, hostname, addr)))
        .on_helo(|_cx, hostname| {
            Box::pin(async move {
                info!(?hostname, "inside helo callback");
                Status::Continue
            })
        })
        .on_eom(|cx| {
            Box::pin(async move {
                info!("inside eom callback");
                cx.actions.add_header("name", "value").await.unwrap();
                cx.actions.add_header("name2", "value2").await.unwrap();

                Status::Continue
            })
        })
        .on_close(|_cx| {
            Box::pin(async move {
                info!("inside close callback");
                Status::Continue
            })
        });

    let milter = Milter::spawn(LOCALHOST, callbacks, Default::default())
        .await
        .unwrap();

    let addr = milter.addr();

    let exit_code = run_miltertest("mt_basic.lua", addr).await.unwrap();

    milter.shutdown().await.unwrap();

    assert!(exit_code.success());
}

async fn connect_callback(
    _: &mut Context<()>,
    hostname: CString,
    addr: Option<SocketInfo>,
) -> Status {
    info!(?hostname, ?addr, "inside connect callback");

    Status::Continue
}

#[tokio::test]
async fn basic_milter_macros() {
    let _ = tracing_subscriber::fmt::try_init();

    let callbacks = Callbacks::<()>::new()
        .on_eoh(|_| Box::pin(async { Status::Continue }))
        .on_eom(|cx| {
            Box::pin(async move {
                if let Some(x) = cx.macros.get(c_str!("name")) {
                    if x.to_string_lossy() == "5678" {
                        return Status::Continue;
                    }
                }
                Status::Tempfail
            })
        });

    let milter = Milter::spawn(LOCALHOST, callbacks, Default::default())
        .await
        .unwrap();

    let addr = milter.addr();

    let exit_code = run_miltertest("mt_macros.lua", addr).await.unwrap();

    milter.shutdown().await.unwrap();

    assert!(exit_code.success());
}
