mod common;

pub use crate::common::*;
use byte_strings::c_str;
use indymilter::{
    message::{
        commands::{BodyPayload, Command, EnvAddrPayload, OptNegPayload},
        replies::Reply,
        MILTER_VERSION,
    },
    Actions, Callbacks, Config, ContextActions, ProtoOpts, SetErrorReply, Status,
};

#[tokio::test]
async fn eom_basic() {
    let _ = tracing_subscriber::fmt::try_init();

    let callbacks = Callbacks::<()>::new()
        .on_eom(|cx| {
            Box::pin(async move {
                cx.actions.add_header("name", "value").await.unwrap();

                Status::Continue
            })
        });

    let config = Config {
        actions: Actions::ADDHDRS,
        ..Default::default()
    };

    let milter = Milter::spawn(LOCALHOST, callbacks, config).await.unwrap();

    let mut client = Client::connect(milter.addr()).await.unwrap();

    client
        .write_command(Command::OptNeg(OptNegPayload {
            version: MILTER_VERSION,
            actions: Actions::all(),
            opts: ProtoOpts::all(),
        }))
        .await
        .unwrap();

    let reply = client.read_reply().await.unwrap();
    assert!(matches!(reply, Reply::OptNeg { actions: Actions::ADDHDRS, .. }));

    client
        .write_command(Command::BodyEnd(BodyPayload { chunk: vec![] }))
        .await
        .unwrap();

    let reply = client.read_reply().await.unwrap();
    assert_eq!(
        reply,
        Reply::AddHeader {
            name: c_str!("name").into(),
            value: c_str!("value").into(),
        }
    );

    let reply = client.read_reply().await.unwrap();
    assert_eq!(reply, Reply::Continue);

    client.write_command(Command::Quit).await.unwrap();

    client.disconnect().await.unwrap();

    milter.shutdown().await.unwrap();
}

#[tokio::test]
async fn reply_basic() {
    let _ = tracing_subscriber::fmt::try_init();

    let callbacks = Callbacks::<()>::new()
        .on_mail(|cx, _| {
            Box::pin(async move {
                cx.reply.set_error_reply("550", Some("5.5.0"), ["No!", "Go away."]).unwrap();

                Status::Reject
            })
        });

    let milter = Milter::spawn(LOCALHOST, callbacks, Default::default())
        .await
        .unwrap();

    let mut client = Client::connect(milter.addr()).await.unwrap();

    client
        .write_command(Command::OptNeg(OptNegPayload {
            version: MILTER_VERSION,
            actions: Actions::all(),
            opts: ProtoOpts::all(),
        }))
        .await
        .unwrap();

    let reply = client.read_reply().await.unwrap();
    assert!(matches!(reply, Reply::OptNeg { .. }));

    client
        .write_command(Command::Mail(EnvAddrPayload {
            args: vec![c_str!("me@example.com").into()],
        }))
        .await
        .unwrap();

    let reply = client.read_reply().await.unwrap();
    assert_eq!(
        reply,
        Reply::ReplyCode {
            reply: c_str!("550-5.5.0 No!\r\n550 5.5.0 Go away.").into()
        }
    );

    client.write_command(Command::Quit).await.unwrap();

    client.disconnect().await.unwrap();

    milter.shutdown().await.unwrap();
}
